/*     @(#)init_f.c	1.5 9/23/92  */

/**********************************************************************
FILE   : init_f.c
PURPOSE: SNNS-Kernel Network Initialisation Functions
NOTES  : 
AUTHOR : Niels Mache
DATE   : 18.03.91
VERSION : 1.5  9/23/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <math.h>

#include "kr_typ.h"	 /*  Kernel Types and Constants  */
#include "kr_const.h"	 /*  Constant Declarators for SNNS-Kernel  */
#include "kr_def.h"	 /*  Default Values  */
#include "kernel.h"	 /*  SNNS-Kernel Function Prototypes  */
#include "random.h"	 /*  Randomize Library Function Prototypes  */
#include "kr_mac.h"	 /*  Kernel Macros   */


#define  INIT_PARAM1( param )   param[ 0 ]  /*    contains the 1st initialisation parameter  */
#define  INIT_PARAM2( param )   param[ 1 ]  /*    contains the 2nd initialisation parameter  */
#define  INIT_PARAM3( param )   param[ 2 ]  /*    contains the 3rd initialisation parameter  */
#define  INIT_PARAM4( param )   param[ 3 ]  /*    contains the 4th initialisation parameter  */
#define  INIT_PARAM5( param )   param[ 4 ]  /*    contains the 5th initialisation parameter  */


/*#################################################

GROUP: Global Var's

#################################################*/


extern bool    NetModified,	 /*  TRUE, if the network topology was modified  */
               NetInitialize;    /*  TRUE, if the network has been initialized  */

extern int  NoOfUnits,		 /*  no. of units in the network  */
	    MinUnitNo,		 /*  the first (lowest) used unit number in the network  */
	    MaxUnitNo,		 /*  the last (highest) used unit number in the network  */
	    NoOfPatternPairs,	 /*  no. of defined pattern pairs */
	    NoOfShuffledPatterns,  /*  no. of entries in the pattern_number array  */
	    NoOfInputUnits,	 /*  no. of input units  */
	    NoOfOutputUnits,	 /*  no. of output units  */
	    TopoSortID; 	 /*  topologic mode identifier	*/

extern UnitArray       unit_array;  /*	the unit array	*/

extern TopoPtrArray    topo_ptr_array;	/*  stores pointers to topological sorted units
					    used by kr_topoSort()  */

extern Patterns        in_patterns,	/*  the input pattern array  */
		       out_patterns;	/*  the output pattern array  */
extern PatternNumbers  pattern_numbers;   /*  contains shuffled pattern numbers
					      used by kr_shufflePatterns()  */

extern int     no_of_topo_units;	/*  no. of unit pointers in the topo_ptr_array	*/


/*  --------------  Initialisation  Functions  ----------------------    */



/*  Initializes connection weights with uniform distributed random values.
    Function calls drand48(). <min> must be less then <max>.
*/
krui_err  INIT_randomizeWeights( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register unsigned short    flags;
  register struct Link   *link_ptr;
  register struct Site   *site_ptr;
  register struct Unit   *unit_ptr;
  register FlintType     range, min_weight;
  FlintType   max_weight;

  if ( (unit_array == NULL) || (NoOfUnits == 0) )
    return( KRERR_NO_UNITS );  /*  there is nothing to do  */
  /*
  if (NoOfParams != 2)
    return( KRERR_PARAMETERS );  */ /*  Not the same no. of input parameters  */

  min_weight = INIT_PARAM1( parameterArray );
  max_weight = INIT_PARAM2( parameterArray ); 
  range = max_weight - min_weight;

  if (range == 0.0)  {
    FOR_ALL_UNITS( unit_ptr )  {
      flags = unit_ptr->flags;

      if ( (flags & UFLAG_IN_USE) == UFLAG_IN_USE)
        {  /*  unit is in use  */
        unit_ptr->bias = min_weight;

        if ( (flags & UFLAG_INPUT_PAT) == UFLAG_SITES )
          {  /*  unit has sites  */
	  FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
              link_ptr->weight = min_weight;
        }
        else
          {       /*  unit has no sites   */
          if ( (flags & UFLAG_INPUT_PAT) == UFLAG_DLINKS )
            {     /*  unit has direct links   */
	    FOR_ALL_LINKS( unit_ptr, link_ptr )
              link_ptr->weight = min_weight;
          }
        }
      }
    }
  }
  else  {
    FOR_ALL_UNITS( unit_ptr )  {
      flags = unit_ptr->flags;

      if ( (flags & UFLAG_IN_USE) == UFLAG_IN_USE)
        {     /*  unit is in use  */
        unit_ptr->bias = (FlintType) drand48() * range + min_weight;

        if ( (flags & UFLAG_INPUT_PAT) == UFLAG_SITES )
          {       /*  unit has sites  */
	  FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
            link_ptr->weight = (FlintType) drand48() * range + min_weight;
        }
        else  {
          /*  unit has no sites   */
          if ( (flags & UFLAG_INPUT_PAT) == UFLAG_DLINKS )
            {     /*  unit has direct links   */
	    FOR_ALL_LINKS( unit_ptr, link_ptr )
              link_ptr->weight = (FlintType) drand48() * range + min_weight;
          }
        }
      }
    }
  }

  return( KRERR_NO_ERROR );
}



krui_err INIT_Weights_CPN(parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit *unit_ptr;
  register struct Site *site_ptr;
  register struct Link *link_ptr;
  register TopoPtrArray  topo_ptr;

  register FlintType  sum, amount, range;
  FlintType min, max;
  int  ret_code;


  if ( (unit_array == NULL) || (NoOfUnits == 0) )
    return( KRERR_NO_UNITS );  /*  there is nothing to do  */
  /*
  if (NoOfParams != 2)
    return( KRERR_PARAMETERS );  */ /*  Not the same no. of input parameters  */

  min = INIT_PARAM1( parameterArray );
  max = INIT_PARAM2( parameterArray );
  range = max - min;

  if (NetModified || (TopoSortID != TOPOLOGIC_TYPE))
    {  /*  networt was modified or topologic array isn't initialized  */
    ret_code = kr_topoSort( TOPOLOGIC_TYPE );
    if (ret_code != KRERR_NO_ERROR)
      return( ret_code );

    NetModified = FALSE;
  }

  topo_ptr = topo_ptr_array + (NoOfInputUnits + 1);

  /*  initialize weights of the hidden units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    { /*  this is a hidden unit  */
      /***********************************************************/
      /*     initialize the weights to the Kohonen Layer         */
      /***********************************************************/

    sum = 0.0;
    if UNIT_HAS_SITES( unit_ptr )
      { /* the unit has sites */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )  {
        link_ptr->weight = (FlintType) drand48() * range + min;
        sum += link_ptr->weight * link_ptr->weight;
      }
    }
    else
      { /* the unit has direct links */
      FOR_ALL_LINKS( unit_ptr, link_ptr )  {
        link_ptr->weight = (FlintType) drand48() * range + min;
	sum += link_ptr->weight * link_ptr->weight;
      }
    }
    /* normalize the weightvector to the Kohonen Layer */

    amount = 1.0 / sqrt( sum );

    if UNIT_HAS_SITES( unit_ptr )
      /* the unit has sites */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	  link_ptr->weight = link_ptr->weight * amount;
    else
      /* the unit has direct links */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	link_ptr->weight = link_ptr->weight * amount;
  }


  while ((unit_ptr = *++topo_ptr) != NULL)
    { /* this is a output unit */
      /***********************************************************/
      /*     initialize the weights to the Grossberg Layer       */
      /***********************************************************/

    if UNIT_HAS_SITES( unit_ptr )
      { /* the unit has sites */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
          link_ptr->weight = (FlintType) drand48() * range + min;
    }
    else
      {  /* the unit has direct links */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
        link_ptr->weight = (FlintType) drand48() * range + min;
    }
  }

  return( KRERR_NO_ERROR );
}


/*
 * The pattern is loaded into the input layer and propagated to the output
 * of the INPUT layer. After that, the weights of the links leading to the
 * <hidden_unit> are set to the output values of the corresponding input
 * units. (A center of a RBF is set).
 * The value of <deviation> is used for symmetry breaking: it gives the
 * percentage of the maximum random change of the input pattern. 
 * deviation 1.0 means 100% which means that an input value of x will lead 
 * to a stored weight between 0.0 and 2*x. A value of 0.0 will not change
 * the weights!
 * The bias of the <hidden_unit> is set to 1.0 (parameter of the RBF)
 */

#include "matrix.h"
#define	 RBF_INIT_FULL		0
#define	 RBF_INIT_REINIT	1
#define  RBF_INIT_KOHONEN	2

void RbfInitSetCenter(pattern_no, hidden_unit, deviation, bias)
int		pattern_no;
struct Unit	*hidden_unit;
float		deviation;
float		bias;
{
	register struct Unit	*unit_ptr;
	register struct Link	*link_ptr;
	register Patterns	current_in_pattern;
	register TopoPtrArray	topo_ptr;

	/* calculate index of the input pattern in Pattern array:	*/

	current_in_pattern = in_patterns + pattern_no * NoOfInputUnits;

	/* activate input units with the pattern and calculate		*/
	/* their output value:						*/

	topo_ptr = topo_ptr_array;
	while ((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    /* go through all input units, set activation and calculate */
	    /* output:							*/

	    unit_ptr -> act = *current_in_pattern++;
	    unit_ptr -> Out.output = unit_ptr -> out_func == OUT_IDENTITY 
		? unit_ptr -> act
		: (*unit_ptr -> out_func) (unit_ptr -> act);
	}

	/* set the weights of the links:				*/

	if (deviation == 0.0)
	{
	    FOR_ALL_LINKS(hidden_unit, link_ptr)
	    {
		link_ptr -> weight = link_ptr -> to -> Out.output;
	    }
	}
	else
	{
	    deviation /= 6.3137515;
	    FOR_ALL_LINKS(hidden_unit, link_ptr)
	    {
		link_ptr -> weight = link_ptr -> to -> Out.output *
		    (1.0 + deviation * 
		     tan(((float) drand48() * 2.8274334 - 1.4137167)));
	    }
	}

	hidden_unit -> bias = bias;
}

/*
 * The weights of the links leading to the <hidden_unit> are propagated
 * back to the Output value of the corresponding input units. Only the
 * Out.output of the input units is set. The activation remains unchanged.
 */

void RbfInitBPCenter(hidden_unit)
struct Unit	*hidden_unit;
{
	register struct Link	*curr_link;

	FOR_ALL_LINKS(hidden_unit, curr_link)
	{
		curr_link -> to -> Out.output = curr_link -> weight;
	}
}

/*
 * Initialization Functions for Use of Radial Basis Functions:
 */

/*
 * First initialization of centers and direct calculation of all weights
 * as first step during learning.
 * The initialization of a center consists of setting the coordinates of
 * the center (the weights of the links, leading to it) and setting the
 * RBF function parameter (bias of the unit) to i_bias.
 */

krui_err  RbfInitNetwork(start_pat, end_pat, 
	i_bias, i_devi, i_f_0, i_f_1, i_smooth, init_type)
int	start_pat, end_pat;
float	i_bias, i_devi, i_f_0, i_f_1, i_smooth;
int	init_type;
{
	register struct Unit	*unit_ptr;
	register struct Unit	*h_unit_ptr;
	register struct Link	*link_ptr;
	register Patterns	current_in_pattern;
	register TopoPtrArray	topo_ptr;
	register TopoPtrArray	topo_hidden_ptr;
	register TopoPtrArray	topo_work;
	register int		hidden_units;
	register int		output_units;
	register int		unit_nr;
	register int		h_unit_nr;
	register int		pattern_nr;
	register int		pattern_anz;
	register int		pattern_no;
	register float		deviation;
	register int		abort;
	register int		tmp_err;

	RbfFloatMatrix		hidden_act;
	RbfFloatMatrix		t_hidden_act;
	RbfFloatMatrix		hidden_produkt;
	RbfFloatMatrix		inter_act;
	RbfFloatMatrix		hidden_sum;
	RbfFloatMatrix		m_p_inverse;
	RbfFloatMatrix		y_vektor;
	RbfFloatMatrix		weights_vektor;
#ifdef RBF_MATRIX_TEST
	RbfFloatMatrix		alt_hidden_sum;
	RbfFloatMatrix		soll_einheit_sein;
	int			s,z;
#endif
	int			malloc_fault;

	abort = FALSE;

	if (init_type == RBF_INIT_FULL)
	{
	    fprintf(stderr,"RBF_Weights called, start initialization:\n");
	}
	else
	{
	    fprintf(stderr,"RBF_Weights_Redo called, start initialization:\n");
	}
	
	fprintf(stderr, "... preparing initialization\n");

	/* count the units of the hidden layer (only one hidden layer)	*/
	/* and the output layer						*/
	
	hidden_units = 0;
	output_units = 0;
	FOR_ALL_UNITS(unit_ptr)
	{
	    if ((unit_ptr -> flags & UFLAG_IN_USE) == UFLAG_IN_USE)
	    {
		if (unit_ptr -> flags & UFLAG_TTYP_HIDD)
		    hidden_units++;
		if (unit_ptr -> flags & UFLAG_TTYP_OUT)
		    output_units++; 
	    }
	}

	/* set <unit_ptr> to the NULL between first hidden unit and 	*/
	/* last input unit:						*/
	topo_ptr = topo_ptr_array;
	while ((unit_ptr = *(++topo_ptr)) != NULL);	/* empty loop!	*/

	/* reference to first hidden unit				*/
	topo_hidden_ptr = topo_ptr;
	topo_hidden_ptr++;

	pattern_anz = end_pat - start_pat + 1;

	/* Output bias is treated as additional hidden unit		*/
	hidden_units += 1;

	/*
	fprintf(stderr,"... allocate matrixes\n");
	*/

	/* Allocate memory for all matrixes:				*/
	malloc_fault = 0;
	if (!RbfAllocMatrix(pattern_anz, hidden_units, &hidden_act))
	    malloc_fault = 0;
	else if (!RbfAllocMatrix(hidden_units, pattern_anz, &t_hidden_act))
	    malloc_fault = 1;
	else if (!RbfAllocMatrix(hidden_units, hidden_units, &hidden_produkt))
	    malloc_fault = 2;
	else if (!RbfAllocMatrix(hidden_units, hidden_units, &inter_act))
	    malloc_fault = 3;
	else if (!RbfAllocMatrix(hidden_units, hidden_units, &hidden_sum))
	    malloc_fault = 4;
	else if (!RbfAllocMatrix(pattern_anz, 1, &y_vektor))
	    malloc_fault = 5;
	else if (!RbfAllocMatrix(hidden_units, 1, &weights_vektor))
	    malloc_fault = 6;
	else if (!RbfAllocMatrix(hidden_units, pattern_anz, &m_p_inverse))
	    malloc_fault = 7;
#ifdef RBF_MATRIX_TEST
	else if (!RbfAllocMatrix(hidden_units, hidden_units, &alt_hidden_sum))
	    malloc_fault = 8;
	else if (!RbfAllocMatrix(hidden_units,hidden_units,&soll_einheit_sein))
	    malloc_fault = 9;
#endif


	if (malloc_fault != 0)
	{
	    if (malloc_fault >= 1)
		RbfFreeMatrix(&hidden_act);
	    if (malloc_fault >= 2)
		RbfFreeMatrix(&t_hidden_act);
	    if (malloc_fault >= 3)
		RbfFreeMatrix(&hidden_produkt);
	    if (malloc_fault >= 4)
		RbfFreeMatrix(&inter_act);
	    if (malloc_fault >= 5)
		RbfFreeMatrix(&hidden_sum);
	    if (malloc_fault >= 6)
		RbfFreeMatrix(&y_vektor);
	    if (malloc_fault >= 7)
		RbfFreeMatrix(&weights_vektor);
#ifdef RBF_MATRIX_TEST
	    if (malloc_fault >= 8)
		RbfFreeMatrix(&alt_hidden_sum);
	    if (malloc_fault >= 9)
		RbfFreeMatrix(&soll_einheit_sein);
#endif

	    return KRERR_INSUFFICIENT_MEM;
	}

/* change the following line into '#if 1' to allow deviation only in    */
/* case that there are more hidden units than learn patterns		*/
#if 0
	/* test if more hidden units than learn patterns:		*/
	if (hidden_units - 1 >= pattern_anz)
	{
	    /* more hidden units than learn patterns			*/
	    /* symmetry breaking necessary!				*/
	    deviation = i_devi;		/* maximum change = 		*/
					/* i_devi * activation		*/
	}
	else
	{
	    /* less hidden units than learn patterns			*/
	    deviation = 0.0;		/* no symmetry breaking		*/
	}
#else
	/* set deviation to parameter					*/
	deviation = i_devi;
	
#endif

	fprintf(stderr,"... compute activation of hidden layer on centers\n");

	/* Now set the centers and fill the inter activation matrix:	*/
	unit_nr = 0;
	while ((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    /* set weights of links leading to <unit_ptr>:		*/
	    if (init_type == RBF_INIT_FULL)
	    {
		pattern_no = (((pattern_anz-1)*unit_nr)/(hidden_units-2)) + 
			     start_pat;
		RbfInitSetCenter(pattern_no, unit_ptr, deviation, i_bias);
	    }
	    else
	    {
		RbfInitBPCenter(unit_ptr);
	    }

	    /* calculate activation of previously defined centers on	*/
	    /* the current pattern and store it into the inter		*/
	    /* activation matrix:					*/
	    topo_work = topo_hidden_ptr;
	    for (h_unit_nr = 0; h_unit_nr <= unit_nr; h_unit_nr++)
	    {
		h_unit_ptr = *(topo_work++);

		/* calculate activation:				*/
		h_unit_ptr -> act = h_unit_ptr -> Out.output =
		(*h_unit_ptr -> act_func) (h_unit_ptr);

		/* store it into the symmetric matrix:			*/
		RbfMatrixSetValue(&inter_act, h_unit_nr, unit_nr,
		    h_unit_ptr -> act * i_smooth);
		RbfMatrixSetValue(&inter_act, unit_nr, h_unit_nr,
		    h_unit_ptr -> act * i_smooth);
	    }
	    unit_nr++;
	}
	
	/* Add entrys for the additional bias:				*/
	for (h_unit_nr = 0; h_unit_nr < hidden_units; h_unit_nr++)
	{
	    RbfMatrixSetValue(&inter_act, h_unit_nr, hidden_units - 1, 
		i_smooth);
	    RbfMatrixSetValue(&inter_act, hidden_units - 1, h_unit_nr,
		i_smooth);
	}

	fprintf(stderr,"... compute activation of hidden layer on patterns\n");

	/* Fill the hidden units activation matrix			*/
	for (pattern_no = start_pat; pattern_no <= end_pat; pattern_no++)
	{
	    RbfLearnForward(pattern_no);
	    topo_ptr = topo_hidden_ptr;
	    for (unit_nr = 0; unit_nr < hidden_units - 1; unit_nr++)
	    {
		RbfMatrixSetValue(&hidden_act, pattern_no - start_pat, unit_nr,
		    (*(topo_ptr++)) -> Out.output);
	    }
	}
	
	/* Add entrys for the additional bias:				*/
	for (h_unit_nr = 0; h_unit_nr < pattern_anz; h_unit_nr++)
	{
	    RbfMatrixSetValue(&hidden_act, h_unit_nr, hidden_units - 1, 1.0);
	}
	
#ifdef RBF_MATRIX_TEST
/*
	printf("hidden_act:\n");
	RbfPrintMatrix(&hidden_act, stdout);
	getchar();
	printf("inter_act:\n");
	RbfPrintMatrix(&inter_act, stdout);
	getchar();
*/
#endif

	fprintf(stderr,"... calculate the moore-penrose inverse matrix\n");

	/* Now calculate the Moore-Penrose Pseudoinverse:		*/
	fprintf(stderr,"...... transposing\n");
	RbfTranspMatrix(&t_hidden_act, &hidden_act);
	fprintf(stderr,"...... multiplying\n");
	RbfMulTranspMatrix(&hidden_produkt, &t_hidden_act);
	fprintf(stderr,"...... adding\n");
	RbfAddMatrix(&hidden_sum, &hidden_produkt, &inter_act);
#ifdef RBF_MATRIX_TEST
	RbfSetMatrix(&alt_hidden_sum, &hidden_sum);
#endif
	fprintf(stderr,"...... inverting\n");
	if ((tmp_err = RbfInvMatrix(&hidden_sum)) != 1)
	{
		fprintf(stderr,"... impossible to invert matrix!\n");
		abort = TRUE;
	}
#ifdef RBF_MATRIX_TEST
/*
	printf("alt_hidden_sum:\n");
	RbfPrintMatrix(&alt_hidden_sum, stdout);
	getchar();
	printf("hidden_sum:\n");
	RbfPrintMatrix(&hidden_sum, stdout);
	getchar();
*/
	RbfMulMatrix(&soll_einheit_sein, &alt_hidden_sum, &hidden_sum);
	printf("Einheitsmatrix:\n");
	RbfPrintMatrix(&soll_einheit_sein, stdout);
#endif
	if (!abort)
	{
	fprintf(stderr,"...... multiplying\n");
	RbfMulMatrix(&m_p_inverse, &hidden_sum, &t_hidden_act);

	fprintf(stderr,"... calculate weights between hidden and output layer\n");
	
	/* set topo_ptr to the NULL between hidden and output layer:	*/
	topo_ptr = topo_hidden_ptr;
	while(*(++topo_ptr) != NULL);

	/* direct calculation of all weights of links leading to the	*/
	/* output layer:						*/
	unit_nr = 0;			/* counts the output units	*/
	while((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    /* fill the y_vektor with the desired outputs for all	*/
	    /* patterns:						*/
	    for (pattern_no = start_pat; pattern_no <= end_pat; pattern_no++)
	    {
		RbfMatrixSetValue(&y_vektor, pattern_no - start_pat, 0,
		    i_f_0 + (i_f_1 - i_f_0)*
		    *(out_patterns + pattern_no * NoOfOutputUnits + unit_nr));
	    }

	    /* calculate the weights, leading to the current output unit*/
	    RbfMulMatrix(&weights_vektor, &m_p_inverse, &y_vektor);

	    /* temporarely store the weights in the value_c field of	*/
	    /* the corresponding hidden units:				*/
	    topo_work = topo_hidden_ptr;
	    h_unit_nr = 0;
	    do
	    {
		(*(topo_work++)) -> value_c = 
		    RbfMatrixGetValue(&weights_vektor, h_unit_nr++, 0); 
	    } while (*topo_work != NULL);

	    /* set the bias of the output unit:				*/
	    unit_ptr -> bias = RbfMatrixGetValue(&weights_vektor, 
		hidden_units - 1, 0);

	    /* set the weights of the links:				*/
	    FOR_ALL_LINKS(unit_ptr, link_ptr)
	    {
		link_ptr -> weight = link_ptr -> to -> value_c;
	    }
	    unit_nr++;
	}

	fprintf(stderr,"Initialization done !\n");
	}
	else
	{
	if (tmp_err == 0)
	    fprintf(stderr,"singular matrix !\n");
	fprintf(stderr,"Initialization aborted !\n");
	}
	RbfFreeMatrix(&hidden_act);
	RbfFreeMatrix(&t_hidden_act);
	RbfFreeMatrix(&hidden_produkt);
	RbfFreeMatrix(&inter_act);
	RbfFreeMatrix(&hidden_sum);
	RbfFreeMatrix(&y_vektor);
	RbfFreeMatrix(&weights_vektor);
	RbfFreeMatrix(&m_p_inverse);

	if (abort)
	{
	    return tmp_err == 0 ? KRERR_NO_ERROR : tmp_err;
	}
	else
	{
	    return KRERR_NO_ERROR;
	}
}

#ifdef RBF_INCLUDE_KOHONEN_CONVEX

void RbfKohonenConvexInit(start_pattern, end_pattern, alpha_start,
	alpha_increment, learn_rate, count)
int	start_pattern, end_pattern, count;
float	alpha_start, alpha_increment, learn_rate;
{
	register float		scalar_prod;	/* act. scalar product	*/
	register float		maximum;	/* max scalar product	*/
	register struct Link	*link_ptr;	/* current Link		*/
	register struct Unit	*unit_ptr;	/* current Unit		*/
	register TopoPtrArray	topo_ptr;
	register Patterns	current_in_pattern;	/* in pattern	*/
	register int		pattern_no;
	register TopoPtrArray	topo_hidden_ptr;/* first hidden Unit	*/
	register float		alpha;		/* convex combination	*/
	register struct Unit	*winner;	/* Unit who's links	*/
						/* change		*/
	register float		norm_alpha;	/* convex combination	*/
	float			norm_init;	/* initialization value	*/
	
	/* search for the first hidden unit				*/
	topo_ptr = topo_ptr_array;
	while (*(++topo_ptr) != NULL);		/* empty loop!		*/

	/* reference to first hidden unit				*/
	topo_hidden_ptr = topo_ptr;
	topo_hidden_ptr++;

	/* initialize all weights leading to hidden units		*/
	norm_init = 1.0 / (float) sqrt((float) NoOfInputUnits);
	while ((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    FOR_ALL_LINKS(unit_ptr, link_ptr)
	    {
		link_ptr -> weight = norm_init;
	    }
	}

	/* do the kohonen training <count> times with increasing alpha	*/
	for (alpha = alpha_start; count > 0; alpha += alpha_increment, count--)
	{
	    /* precalculate the constant value for the convex		*/
	    /* combination method for the current alpha			*/
	    norm_alpha = (1.0 - alpha) * norm_init;

	    /* calculate index of first input pattern in Pattern array	*/
	    current_in_pattern = in_patterns + start_pattern * NoOfInputUnits;

	    /* present all input patterns and train the hidden layer	*/
	    for (pattern_no=start_pattern;pattern_no<=end_pattern;pattern_no++)
	    {
		/* activate input units with the pattern and calculate	*/
		/* their output value:					*/
		topo_ptr = topo_ptr_array;
		while ((unit_ptr = *(++topo_ptr)) != NULL)
		{
		    /* go through all input units, set activation and	*/
		    /* calculate output using the convex combination	*/
		    /* method						*/
		    unit_ptr -> act = *current_in_pattern++;
		    unit_ptr -> Out.output = unit_ptr->out_func==OUT_IDENTITY 
			? alpha * (unit_ptr -> act) + norm_alpha
			: alpha * ((*unit_ptr -> out_func) (unit_ptr -> act))
				+ norm_alpha;
		}

		/* determine the hidden unit with maximum scalar product*/
		/* between its weights and the output of the input layer*/
		winner = (struct Unit *) NULL;
		maximum = (float) -HUGE_VAL;	/* -oo, see math.h	*/
		topo_ptr = topo_hidden_ptr;
		while ((unit_ptr = *(topo_ptr++)) != NULL)
		{
		    /* calculate scalar product of current hidden unit	*/
		    scalar_prod = (float) 0.0;
		    FOR_ALL_LINKS(unit_ptr, link_ptr)
		    {
			scalar_prod += link_ptr -> weight * 
				link_ptr -> to -> Out.output;
		    }

		    /* change winner if s.p. is > than the current best	*/
		    if (scalar_prod > maximum)
		    {
			maximum = scalar_prod;
			winner = unit_ptr;
		    }
		}

		/* adjust weights of the hidden winner Unit		*/
		if (winner != NULL)
		{
		    FOR_ALL_LINKS(winner, link_ptr)
		    {
			link_ptr -> weight += learn_rate * 
			    (link_ptr->to->Out.output - link_ptr->weight);
		    }
		    printf("(%d,%d) ", winner -> unit_pos.x, 
			winner -> unit_pos.y); 
		}
		else
		{
		    fprintf(stderr,"Internal error in RbfKohonenConvexInit\n");
		}
	    }
	}
}

#endif

void RbfKohonenInit(start_pattern, end_pattern, learn_rate, count, shuffle)
int	start_pattern, end_pattern, count, shuffle;
float	learn_rate;
{
	register float		scalar_prod;	/* act. scalar product	*/
	register float		maximum;	/* max scalar product	*/
	register struct Link	*link_ptr;	/* current Link		*/
	register struct Unit	*unit_ptr;	/* current Unit		*/
	register TopoPtrArray	topo_ptr;
	register Patterns	current_in_pattern;	/* in pattern	*/
	register int		pattern_no;
	register TopoPtrArray	topo_hidden_ptr;/* first hidden Unit	*/
	register TopoPtrArray	help_topo_ptr;
	register struct Unit	*winner;	/* Unit who's links	*/
						/* change		*/
	float			norm_init;	/* initialization value	*/
	register struct Unit	*hidden_unit;	/* current hidden unit	*/
	register int		hidden_units;	/* number of hidden u.	*/
	register int		act_hidden_num;	/* number of current hu.*/
	int			reshuffle;	/* restore shuffled p.	*/

	fprintf(stderr, "RBF_Weights_Kohonen called, start initialization:\n");

	/* search for the first hidden unit				*/
	topo_ptr = topo_ptr_array;
	while (*(++topo_ptr) != NULL);		/* empty loop!		*/

	/* count hidden units and reference to first hidden unit	*/
	topo_hidden_ptr = topo_ptr;
	hidden_units = 0;
	while (*(++topo_hidden_ptr) != NULL)
		hidden_units++;
	topo_hidden_ptr = topo_ptr;
	topo_hidden_ptr++;

	if (shuffle)
	{
	    reshuffle = FALSE;
	    if (NoOfShuffledPatterns != NoOfPatternPairs)
	    {
		krui_shufflePatterns(TRUE);
		reshuffle = TRUE;
	    }
	}

	fprintf(stderr, "... init weights between input and hidden layer\n");

	/* initialize all weights leading to hidden units		*/
	norm_init = 1.0 / (float) sqrt((float) NoOfInputUnits);
	act_hidden_num = 0;
	while ((hidden_unit = *(++topo_ptr)) != NULL)
	{
	    if (shuffle && NoOfShuffledPatterns == NoOfPatternPairs)
		/* shuffle						*/
		pattern_no = pattern_numbers[start_pattern + 
			act_hidden_num % (end_pattern - start_pattern + 1)];
	    else
		/* do not shuffle					*/
		pattern_no = start_pattern + 
			((end_pattern - start_pattern) * act_hidden_num) /
			 (hidden_units - 1);

	    /* calculate index of first input pattern in Pattern array	*/
	    current_in_pattern = in_patterns + pattern_no * NoOfInputUnits;

	    /* activate input units with the pattern and calculate	*/
	    /* their output value:					*/
	    help_topo_ptr = topo_ptr_array;
	    while ((unit_ptr = *(++help_topo_ptr)) != NULL)
	    {
		/* go through all input units, set activation and	*/
		/* calculate output using the convex combination	*/
		/* method						*/
		unit_ptr -> act = *current_in_pattern++;
		unit_ptr -> Out.output = unit_ptr->out_func==OUT_IDENTITY 
			? (unit_ptr -> act)
			: ((*unit_ptr -> out_func) (unit_ptr -> act));
	    }

	    FOR_ALL_LINKS(hidden_unit, link_ptr)
	    {
		link_ptr -> weight = link_ptr -> to -> Out.output;
	    }

	    act_hidden_num++;
	}

	if (shuffle && reshuffle)
	{
	    krui_shufflePatterns(FALSE);
	}

	/* do the kohonen training <count> times			*/
	if (count > 0)
	{
	   fprintf(stderr, "... begin kohonen training\n");
	}
	for (; count > 0; count--)
	{
	    /* calculate index of first input pattern in Pattern array	*/
	    current_in_pattern = in_patterns + start_pattern * NoOfInputUnits;

	    /* present all input patterns and train the hidden layer	*/
	    for (pattern_no=start_pattern;pattern_no<=end_pattern;pattern_no++)
	    {
		/* activate input units with the pattern and calculate	*/
		/* their output value:					*/
		topo_ptr = topo_ptr_array;
		while ((unit_ptr = *(++topo_ptr)) != NULL)
		{
		    /* go through all input units, set activation and	*/
		    /* calculate output using the convex combination	*/
		    /* method						*/
		    unit_ptr -> act = *current_in_pattern++;
		    unit_ptr -> Out.output = unit_ptr->out_func==OUT_IDENTITY 
			? (unit_ptr -> act)
			: ((*unit_ptr -> out_func) (unit_ptr -> act));
		}

		/* determine the hidden unit with maximum scalar product*/
		/* between its weights and the output of the input layer*/
		winner = (struct Unit *) NULL;
		maximum = (float) -HUGE_VAL;	/* -oo, see math.h	*/
		topo_ptr = topo_hidden_ptr;
		while ((unit_ptr = *(topo_ptr++)) != NULL)
		{
		    /* calculate scalar product of current hidden unit	*/
		    scalar_prod = (float) 0.0;
		    FOR_ALL_LINKS(unit_ptr, link_ptr)
		    {
			scalar_prod += link_ptr -> weight * 
				link_ptr -> to -> Out.output;
		    }

		    /* change winner if s.p. is > than the current best	*/
		    if (scalar_prod > maximum)
		    {
			maximum = scalar_prod;
			winner = unit_ptr;
		    }
		}

		/* adjust weights of the hidden winner Unit		*/
		if (winner != NULL)
		{
		    FOR_ALL_LINKS(winner, link_ptr)
		    {
			link_ptr -> weight += learn_rate * 
			    (link_ptr->to->Out.output - link_ptr->weight);
		    }
/*
		    printf("(%d,%d) ", winner -> unit_pos.x, 
			winner -> unit_pos.y); 
*/
		}
		else
		{
		    fprintf(stderr,"Internal error in RbfKohonenConvexInit\n");
		}
	    }
	}
	fprintf(stderr, "Initialization done\n");
}

krui_err RbfStartInit(parameterArray, NoOfParams, init_type)
float parameterArray[];
int   NoOfParams;
int   init_type;
{
	krui_err	ret_code;	/* error return code		*/
	float		bias;		/* bias of hidden units		*/
	float		deviation;	/* deviation of centers		*/
	float		f_0_lin;	/* learning value for pattern==0*/
	float		f_1_lin;	/* learning value for pattern==1*/
	float		smoothness;	/* see documentation 		*/

	float		learn_rate;	/* kohonen training rate	*/
	int		count;		/* cycles for kohonen training  */

	/* for future use, now uncommented: 				*/
	/* check the number of input parameters				*/
/*
	if (NoOfParams != 5)
	    return( KRERR_PARAMETERS );
*/

	/* test if patterns available and valid:			*/
	if (NoOfPatternPairs == 0)  
	    return( KRERR_NO_PATTERNS );	/* no patterns defined  */
/*
	if (!PatternsValid)  
	    return( KRERR_INCORE_PATTERNS );	*//* (In-Core) patters	*/
						/* not valid		*/

	/* test net topology:						*/
	if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
	{
	    ret_code = RbfTopoCheck();	/* in learn_f.c			*/

	    if ((ret_code != KRERR_NO_ERROR) && (ret_code != KRERR_DEAD_UNITS))
		return( ret_code );

	    NetModified = FALSE;
	}

	switch(init_type)
	{
	    case RBF_INIT_FULL:
	    case RBF_INIT_REINIT:
		/* read input parameters:				*/
		bias = INIT_PARAM4(parameterArray);
		deviation = INIT_PARAM5(parameterArray);
		f_0_lin = INIT_PARAM1(parameterArray);
		f_1_lin = INIT_PARAM2(parameterArray);
		smoothness = INIT_PARAM3(parameterArray);

		/* call real initialization function:			*/
		ret_code = RbfInitNetwork(0, NoOfPatternPairs - 1, bias, 
			deviation, f_0_lin, f_1_lin, smoothness, init_type);
		break;

	    case RBF_INIT_KOHONEN:
		/* read input parameters:				*/
		count = (int) (INIT_PARAM1(parameterArray));
		learn_rate = INIT_PARAM2(parameterArray);
		RbfKohonenInit(0, NoOfPatternPairs - 1, learn_rate, count,
			INIT_PARAM3(parameterArray) != 0.0);
		ret_code = KRERR_NO_ERROR;
		break;
	}

	return ret_code;
}

/*
 * Use of initialization parameters:
 * RBF_Weights: (5)
 *   INIT_PARAM1: interpolation value for teaching pattern == 0.0
 *                (see documentation)
 *   INIT_PARAM2: interpolation value for teaching pattern == 1.0
 *                (see documentation)
 *   INIT_PARAM3: smoothness parameter
 *   INIT_PARAM4: initialization bias of hidden units
 *   INIT_PARAM5: deviation parameter for symmetry breaking
 * RBF_Weights_redo: (3)
 *   INIT_PARAM1: interpolation value for teaching pattern == 0.0
 *                (see documentation)
 *   INIT_PARAM2: interpolation value for teaching pattern == 1.0
 *                (see documentation)
 *   INIT_PARAM3: smoothness parameter
 * RBF_Weights_Kohonen: (3)
 *   INIT_PARAM1: number of cycles for the kohonen training
 *   INIT_PARAM2: learn_rate for kohonen training
 *   INIT_PARAM3: shuffle flag:
 *                0.0 = the patterns are normally distributed over the links
 *                      to the hidden units.
 *             != 0.0 = the patterns to use are randomly taken from all
 *                      available patterns.
 */

krui_err INIT_RBF_Weights(parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
	return RbfStartInit(parameterArray, NoOfParams, RBF_INIT_FULL);
}

krui_err INIT_RBF_Weights_redo(parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
	return RbfStartInit(parameterArray, NoOfParams, RBF_INIT_REINIT);
}

krui_err INIT_RBF_Weights_kohonen(parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
	return RbfStartInit(parameterArray, NoOfParams, RBF_INIT_KOHONEN);
}
