/*     @(#)kernel.h	1.2 7/21/92  */

/**********************************************************************
FILE   : kernel.h
PURPOSE: SNNS Kernel Function Prototypes
NOTES  : 
AUTHOR : Niels Mache
DATE   : 20.02.90
VERSION : 1.2  7/21/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#ifdef __STDC__


/*#################################################

GROUP:  Parallel kernel functions

#################################################*/

#ifdef  MASPAR_KERNEL
/*  Connects and Disconnects the MasPar.
    The mode switches are:  MASPAR_CONNECT and MASPAR_DISCONNECT.
*/
krui_err  kr_initMasPar( int  mode );

/* Returns the Status of the MasPar or an error code.
*/
krui_err  kr_getMasParStatus( void );
#endif

/*  Validate a network modifying operation according to
    the kernel mode.
*/
krui_err  kr_validateOperation( void );

/*  Sets the topologic type of the current network.
    Returns an error if the topologic type of the current network  
    doesn't fit to this type.
    Topologic types are:
      - NET_TYPE_GENERAL
        general purpose network type with no limitations
      - NET_TYPE_FF1
        feedforward network with fully connected units in
        neighbour layers
*/
krui_err  kr_setSpecialNetworkType( int  net_type );

/*  Returns the special topologic type of the current network, if set.
*/
int  kr_getSpecialNetworkType( void );

/*  Creates a user defined unit.
*/
int  kr_createUnit( char  *unit_name, char  *out_func_name, char  *act_func_name,
		    FlintTypeParam  i_act, FlintTypeParam  bias );


/* -------------------------------------------------------------------
*/


/*  Deletes all input links at current unit/site.
*/
krui_err  kr_deleteAllLinks( int  mode );


/*  Deletes the current link.

NOTE: To delete a link between the current unit/site and the source unit
      <source_unit_no>, call krui_isConnected( source_unit_no ) and
      krui_deleteLink().
*/
krui_err  kr_deleteLink( void );


/*  Creates a link between source unit and the current unit/site.
    Returns an error code:
     - if memory allocation fails
     - if source unit doesn't exist or
     - if there exists already a connection between current unit/site and
       the source unit
    0 otherwise.
    kr_createLink DO NOT set the current link.

NOTE: If you want to create a link and its unknown if there exists already a
      connection between the two units, use krui_createLink and test the return
      code, instead of the sequence kr_isConnected and kr_createLink.
*/
krui_err  kr_createLink( int  source_unit_no, FlintTypeParam  weight );


/*  Sets the link weight of the current link
*/
void  kr_setLinkWeight( FlintTypeParam	weight );


/*  Returns the link weight of the current link.
*/
FlintType  kr_getLinkWeight( void );


/*  True if there exists a connection between source unit <source_unit_no>
    and the current unit/site, otherwise false.

NOTE: If there exists a connection between the two units, the current link is set
      to the link between the two units.
*/
bool  kr_isConnected( int  source_unit_no, FlintType   *weight );


/*  True if there exists a connection between source unit <source_unit_no>
    and target unit <target_unit_no>, otherwise false. If there exist a
    connection between these units, kr_areConnected returns the connection
    strength also.
    Returns FALSE if unit doesn't exist.

IMPORTANT: If there exist a connection, the current unit and site will be
	   set to the target unit/site.

NOTE: This function is slow (Units are backward chained only).
*/
bool  kr_areConnected( int  source_unit_no, int  target_unit_no,
		       FlintType   *weight );

/*  Returns the no. of first or next succecessor unit of the
    given unit and the connection strenght.
    Sets the current unit/site.
*/
int  kr_getSuccessorUnit( int  mode, int  source_unit_no, FlintType  *weigth );



/*  initialize the first/next site or the named site at the current unit
    for access
*/
int  kr_setSite( int  mode, char  *site_name );

/*  initializes the given unit for access
*/
krui_err  kr_setCurrUnit( int  unit_no );

/*  returns the number of the first/next/current unit of the unit array
*/
int  kr_getUnit( int  mode );

/*  Returns the no. of first, next or current predecessor unit of the
    current unit/site and the connection weight.
*/
int  kr_getPredecessorUnit( int  mode, FlintType  *weight );

/*  Remove unit and all links from network.
*/
krui_err  kr_removeUnit( struct Unit  *unit_ptr );

/*  Returns the value of the specified unit component.
*/
FlintType   kr_getUnitValues( int  unit_no, int  component_selector );

/*  Sets the value of the specified unit component.
*/
krui_err  kr_setUnitValues( int  unit_no, int  component_selector,
                        FlintTypeParam  value );

/*  Sets all unit components of the specified unit.
*/
krui_err  kr_setAllUnitValues( int unit_no, FlintTypeParam out, FlintTypeParam act,
			       FlintTypeParam i_act, FlintTypeParam bias );


/*  delete all inputs at the given unit
*/
void kr_deleteAllInputs( struct Unit  *unit_ptr );

/*  Deletes all output links at <source_unit>
NOTE: This function is slow.
*/
void kr_deleteAllOutputLinks( struct Unit  *source_unit_ptr );

/*  search for a site at a unit 
*/
struct Site *kr_searchUnitSite( struct Unit  *unit_ptr, struct SiteTable  *stbl_ptr );

/*  search for a site at a unit and returns also the predecessor site ptr 
*/
void kr_searchUnitSiteWithPred( struct Unit  *unit_ptr, struct SiteTable  *stbl_ptr,
                                struct Site  * *Rsite_ptr, struct Site  * *Rsite_pred_ptr );

/*  searches for a site in the network  
*/
int kr_searchNetSite( struct SiteTable  *stbl_ptr );

/*  Creates a new site with default initialisation
*/
struct Site *kr_createDefaultSite( void );

/*  returns the pointer to the given unit, returns NULL if unit doesn't
    exist   
*/
struct Unit *kr_getUnitPtr( int  unit_no );

/*  Searches for a unit with the given symbol pointer.
    Returns the first unit no. if a unit with the given name was found,
    0 otherwise.
*/
int kr_unitNameSearch( int  min_unit_no, char  *unit_symbol_ptr );

/*  Creates a unit with default values.
*/
int  kr_makeDefaultUnit( void );


/*  Copy a given unit, according to the copy mode
        1. copy unit (with it sites, if available) and input/output links
        2. copy unit (with it sites, if available) and input links
        3. copy unit (with it sites, if available) and output links
        4. copy unit (with it sites, if available) but no input/output links

    Returns the unit number of the new unit or error message < 0 , if errors occured.
    Function has no effect on the current unit.

NOTE: Copying of output links is slow.
      If return code < 0, an error occured.
*/
krui_err  kr_copyUnit( int  copy_mode, int  source_unit );

/*  Sets the topologic type of the unit.
*/
krui_err  kr_unitSetTType( int  unit_no, int  UnitTType );


/*  changes all units in the network with the given functionality type
    to the new functions of the (new) functionality type
*/
void kr_changeFtypeUnits( struct  FtypeUnitStruct  *Ftype_entry );

/*   delete the functionality type of the units with the given type
*/
void kr_deleteUnitsFtype( struct  FtypeUnitStruct  *ftype_ptr );

/*  create a new unit with the given functionality type
*/
int  kr_makeFtypeUnit( char  *Ftype_symbol );

/*  returns TRUE, if there exists the given site at the given ftype entry
*/
bool kr_FtypeSiteSearch( struct  Site  *ftype_first_site, struct  SiteTable  *site_table_ptr );

/*  change the properties of the given unit to the properties of the
    given F-Type
*/
void kr_changeFtypeUnit( struct  Unit  *unit_ptr, struct  FtypeUnitStruct  *ftype_ptr );

/*  change a site at the F-Type
*/
void kr_changeFtypeSites( struct  FtypeUnitStruct  *Ftype_entry, 
                          struct  SiteTable  *old_site_table,
                          struct  SiteTable  *new_site_table );

/*  spell checker  (check identifiers for matching [A-Za-z]^[|, ]*)
*/
bool kr_symbolCheck( char    *symbol );

/*  translate unit flags to the topological type of the unit
*/
int kr_flags2TType( int  flags );

/*  translate the topological type to unit flags
*/
int kr_TType2Flags( int  ttype );

/*  update the outputs of all units in the network
*/
void kr_updateUnitOutputs( void );

/*  returns the no. of units of the specified topologic type
    (i.e. Input, Hidden, Output or Special units)
*/
int kr_getNoOfUnits( int  UnitTType );

/*  force unit array garbage collection
*/
void  kr_forceUnitGC( void );


/*#################################################

GROUP: Topological Sorting Functions

#################################################*/


/*  Sort units according to the given mode:
    TOPOLOGICAL:
      Sort units topological (general version) and stores the
      pointers to this units in the topologic array.
      NOTE: Units are not sorted by their topologic type (that's not
	    possible in general case).

    TOPOLOGICAL_FF:
      Sorts unit topological in feed-forward networks and stores the
      pointers to this units in the topologic array in the following order:
       - input,
       - hidden and
       - output units

      This function make following assumtions (like all learning functions for
      feed-forward networks):
       a) input units doesn't have input connections to other units and
       b) output units doesn't have outputs connections to other units.

    TOPOLOGIC_TYPE:
      Sort units by their topologic type, i.e. Input, Hidden, Output units and
      stores the pointers to this units in the topologic array.
*/
krui_err kr_topoSort( int sort_mode );


/*  Checks the topology of the network:
     a) counts the number of layers of the network and
     b) determines if the network has cycles.

    Returns the no. of layers of the network.
*/
int kr_topoCheck( void );

/*  Count the no. of input and output units and return an error code
    if the no. do not fit to the loaded patterns.
*/
krui_err  kr_IOCheck( void );


krui_err  kr_makeUnitPermutation( void );

/*#################################################

GROUP: Functions for pattern management

#################################################*/

/*  pattern operations
*/
krui_err  kr_pattern( int  mode, int  mode1, int  pattern_no );


/*  1. Create new pattern vector. Append the given input- and output patterns
       to the unit patterns by copying the contents of the pattern arrays.
    2. Allocates pattern arrays.
*/
krui_err  kr_vectorPatterns( int  mode,
			     float  *input_patterns, int  no_of_input_patterns,
			     float  *output_patterns, int  no_of_output_patterns,
			     int  no_of_pattern_pairs );

/*  Returns the startaddresses of the internal pattern arrays.

    NOTE: Adresses of the internal pattern arrays may change.
	  You have to call `krui_getPatternArrays' EVERYTIME to determine the
	  addresse of the pattern arrays before you access these arrays.
*/
krui_err  kr_getPatternArrays( float  * *input_patterns, int  *no_of_input_patterns,
				 float	* *output_patterns, int  *no_of_output_patterns );


/*#################################################

GROUP: Other functions

#################################################*/

/*  determines the network error for the given pattern and counts the output units
    with a higher error then the given delta max.

*/
krui_err kr_checkUnitsOutput( int  pattern_no,
			      float  parameterInArray[], int  NoOfInParams,
			      float * *parameterOutArray, int  *NoOfOutParams );


/*  Add random uniform distributed values to connection weights.
    <minus> must be less then <plus>.
*/
void kr_jogWeights( FlintTypeParam  minus, FlintTypeParam  plus );


/*  calls a network function
*/
krui_err  kr_callNetworkFunction( int  type, float  *parameterInArray, int  NoOfInParams,
				  float  * *parameterOutArray, int  *NoOfOutParams,
				  int  start_pattern, int  end_pattern );

/*  Returns information about the unit default settings.
*/
void kr_getUnitDefaults( FlintType  *act, FlintType  *bias, int	 *ttflags, int  *subnet_no,
                         int  *layer_no, 
                         char * *act_func, char * *out_func );

/*  Changes the unit default settings.
*/
krui_err kr_setUnitDefaults( FlintTypeParam  act, FlintTypeParam  bias, int  ttflags, 
                           int  subnet_no, int  layer_no, 
                           char  *act_func_ptr, char  *out_func_ptr );

#else

/* ------------------------------------------------------------------
*/
#ifdef  MASPAR_KERNEL
krui_err  kr_initMasPar();
krui_err  kr_getMasParStatus();
#endif

krui_err  kr_validateOperation();
krui_err  kr_setSpecialNetworkType();
int  kr_getSpecialNetworkType();

/* ------------------------------------------------------------------
*/

krui_err  krui_deleteAllLinks();
krui_err  kr_deleteLink();
krui_err  kr_createLink();
void  kr_setLinkWeight();
FlintType  kr_getLinkWeight();
bool  kr_isConnected();
bool  kr_areConnected();
int  kr_getSuccessorUnit();
int  kr_setSite();
krui_err  kr_setCurrUnit();
int  kr_getUnit();
int  kr_getPredecessorUnit();
int  kr_createUnit();
FlintType   kr_getUnitValues();
krui_err  kr_setUnitValues();
krui_err  kr_setAllUnitValues();
krui_err  kr_removeUnit();
void kr_deleteAllInputs();
void kr_deleteAllOutputLinks();
struct Site *kr_searchUnitSite();
void kr_searchUnitSiteWithPred();
int kr_searchNetSite();
struct Site *kr_createDefaultSite();
struct Unit *kr_getUnitPtr();
int kr_unitNameSearch();
int kr_makeDefaultUnit();
krui_err  kr_copyUnit();
krui_err  kr_unitSetTType();
void kr_changeFtypeUnits();
void kr_deleteUnitsFtype();
int  kr_makeFtypeUnit();
bool kr_FtypeSiteSearch();
void kr_changeFtypeUnit();
void kr_changeFtypeSites();
bool kr_symbolCheck();
int kr_flags2TType();
int kr_TType2Flags();
void kr_updateUnitOutputs();
int kr_getNoOfUnits();
void  kr_forceUnitGC();

krui_err  kr_topoSort();
int kr_topoCheck();
krui_err  kr_IOCheck();
krui_err  kr_makeUnitPermutation();

krui_err  kr_pattern();
krui_err  kr_vectorPatterns();
krui_err  kr_getPatternArrays();

krui_err kr_checkUnitsOutput();
void kr_jogWeights();

krui_err  kr_callNetworkFunction();

krui_err  kr_setUnitDefaults();
void kr_getUnitDefaults();

#endif
