/*     @(#)kr_funcs.c	1.2 7/21/92  */

/**********************************************************************
FILE   : kr_funcs.c
PURPOSE: SNNS-Kernel: Management routines for user functions
NOTES  : 
AUTHOR : Niels Mache
DATE   : 12.02.90
VERSION : 1.2  7/21/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <string.h>

#include "kr_typ.h"         /*  Kernel types and constants  */
#include "kr_const.h"       /*  Kernel constants  */
#include "kr_def.h"	    /*	Default values	*/
#include "kr_funcs.h"       /*  Function prototypes  */



/*#################################################

GROUP: Global Var's (as declared by the kernel)

#################################################*/

extern bool    NetInitialize,    /*  TRUE, if the network has been initialized  */
               LearnFuncHasChanged;  /*  TRUE, if the learning function has been changed  */

extern specialNetworkType;   /*  stores the topologic type of a network
                                 (needed for the parallel kernel)  */



/**********************************************************************
   This is the Function Table (Function Table may be modified by the user)
**********************************************************************/

extern struct FuncTable  kernel_func_table[];
extern int  NoOfKernelFuncs;

/**********************************************************************
   End of Function Table
**********************************************************************/



/*#################################################

GROUP: Local Vars

#################################################*/


#define  NO_OF_FUNC_TYPES  8
#define  NO_OF_KERNELS  2
char  CurrNetworkFunc[NO_OF_FUNC_TYPES * NO_OF_KERNELS][FUNCTION_NAME_MAX_LEN];

bool  netFuncInit[NO_OF_FUNC_TYPES * NO_OF_KERNELS] = { 
                         FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
                         FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE  };



/*#################################################

GROUP: MasPar functions

#################################################*/

#ifdef MASPAR_KERNEL
#ifndef MASPAR_KERNEL_EMULATION

extern  krmp_getMasParFuncInfo();

#endif
#endif


/*#################################################

GROUP: Functions

#################################################*/

static krui_err  krf_getInternalFuncInfo( mode, func_descr )
int  mode;
struct FuncInfoDescriptor  *func_descr;
{
  struct FuncTable  *ftbl_ptr;
  unsigned short  func_type;


  KernelErrorCode = KRERR_NO_ERROR;

  switch (mode)  {
    case  GET_NO_OF_FUNCS:
      func_descr->number = NoOfKernelFuncs;
      break;

    case  GET_FUNC_INFO:  /*  return all info about given function  */
      if ((func_descr->number < 0) || (func_descr->number >= NoOfKernelFuncs))  {
        KernelErrorCode = KRERR_PARAMETERS;
        return( KernelErrorCode );
      }

      ftbl_ptr = kernel_func_table + func_descr->number;

      strcpy( func_descr->func_name, ftbl_ptr->func_name );
      func_descr->func_type = ftbl_ptr->func_type & ~DEFAULT_FUNC;
      func_descr->no_of_input_parameters = ftbl_ptr->no_of_input_parameters;
      func_descr->no_of_output_parameters = ftbl_ptr->no_of_output_parameters;
      func_descr->function = ftbl_ptr->function;

      break;

    case  SEARCH_FUNC:  /*  search for the given function and return the
                            info about the function  */

      for (ftbl_ptr = kernel_func_table;
           ftbl_ptr < kernel_func_table + NoOfKernelFuncs;
           ftbl_ptr++)  {
        func_type = func_descr->func_type;
        if ((func_type == (ftbl_ptr->func_type & ~DEFAULT_FUNC)) &&
            (strcmp( func_descr->func_name, ftbl_ptr->func_name ) == 0))  {
          func_descr->no_of_input_parameters = ftbl_ptr->no_of_input_parameters;
          func_descr->no_of_output_parameters = ftbl_ptr->no_of_output_parameters;
          func_descr->function = ftbl_ptr->function;

          return( KRERR_NO_ERROR );
        }
      }

      func_descr->function = NULL;

      switch (func_descr->func_type)  {
        case  OUT_FUNC:
          KernelErrorCode = KRERR_OUTFUNC;
          break;
        case  ACT_FUNC:
          KernelErrorCode = KRERR_ACTFUNC;
          break;
        case  SITE_FUNC:
          KernelErrorCode = KRERR_SITEFUNC;
          break;
        case  LEARN_FUNC:
          KernelErrorCode = KRERR_LEARNING_FUNC;
          break;
        case  UPDATE_FUNC:
          KernelErrorCode = KRERR_UPDATE_FUNC;
          break;
        case  INIT_FUNC:
          KernelErrorCode = KRERR_INIT_FUNC;
          break;
        case  ACT_DERIV_FUNC:
          KernelErrorCode = KRERR_DERIV_FUNC;
          break;

        default:
          KernelErrorCode = KRERR_PARAMETERS;
      }

      break;

    case  GET_FUNC_NAME:   /*  search for the given function pointer and
                               returns the name of the function  */

      for (ftbl_ptr = kernel_func_table;
           ftbl_ptr < kernel_func_table + NoOfKernelFuncs;
           ftbl_ptr++)
        if (func_descr->function == ftbl_ptr->function)  {
          strcpy( func_descr->func_name, ftbl_ptr->func_name );
          func_descr->func_type = ftbl_ptr->func_type & ~DEFAULT_FUNC;
          func_descr->no_of_input_parameters = ftbl_ptr->no_of_input_parameters;
          func_descr->no_of_output_parameters = ftbl_ptr->no_of_output_parameters;

          return( KRERR_NO_ERROR );
        }

      func_descr->func_type = 0;
      break;

    case  GET_DEFAULT_FUNC:   /*  search for the given function type and
                                  returns the default function of this type  */

      func_type = func_descr->func_type | DEFAULT_FUNC;
      for (ftbl_ptr = kernel_func_table;
           ftbl_ptr < kernel_func_table + NoOfKernelFuncs;
           ftbl_ptr++)  {
        if (func_type == ftbl_ptr->func_type)  {
          func_descr->no_of_input_parameters = ftbl_ptr->no_of_input_parameters;
          func_descr->no_of_output_parameters = ftbl_ptr->no_of_output_parameters;
          func_descr->function = ftbl_ptr->function;
          strcpy( func_descr->func_name, ftbl_ptr->func_name );
          func_descr->number = ftbl_ptr - kernel_func_table;
          return( KRERR_NO_ERROR );
        }
      }

      func_descr->number = 0;
      KernelErrorCode = KRERR_MISSING_DEFAULT_FUNC;
      break;

    default:
      KernelErrorCode = KRERR_PARAMETERS;
  }

  return( KernelErrorCode );
}



krui_err  krf_getFuncInfo( mode, func_descr )
int  mode;
struct FuncInfoDescriptor  *func_descr;
{
  switch (specialNetworkType)  {
    case NET_TYPE_GENERAL:
      (void) krf_getInternalFuncInfo( mode, func_descr );
      break;

#ifdef MASPAR_KERNEL

    case NET_TYPE_FF1:

#ifndef MASPAR_KERNEL_EMULATION
      
  
      KernelErrorCode = callRequest( krmp_getMasParFuncInfo,
                                     sizeof (int) +
                                     sizeof (struct FuncInfoDescriptor *),
                                     mode, func_descr );
#else
      (void) krf_getInternalFuncInfo( mode, func_descr );
#endif

      break;
#endif

    default:
      KernelErrorCode = KRERR_PARAMETERS;
  }

  return( KernelErrorCode );
}


/*  Returns the number of functions in the function table
*/
int  krf_getNoOfFuncs()
{
  struct FuncInfoDescriptor  functionDescr;


  KernelErrorCode = krf_getFuncInfo( GET_NO_OF_FUNCS, &functionDescr );
  return( functionDescr.number );
}


/*  seaches for the given function (name and type) and returns a pointer to
    this function. krf_funcSearch(...) returns TRUE if the given function was
    found, FALSE otherwise.
*/
bool  krf_funcSearch( func_name, func_type, func_ptr)
char  *func_name;
int  func_type;
FunctionPtr  *func_ptr;
{
  struct FuncInfoDescriptor  functionDescr;

  functionDescr.func_type = func_type;
  strcpy( functionDescr.func_name, func_name );

  KernelErrorCode = krf_getFuncInfo( SEARCH_FUNC, &functionDescr );

  if (KernelErrorCode != KRERR_NO_ERROR)  return( FALSE );
  *func_ptr = functionDescr.function;
  return( TRUE );
}


/*  Returns the name of the given function
*/
char  *krf_getFuncName( func_ptr )
FunctionPtr  func_ptr;
{
  static struct FuncInfoDescriptor  functionDescr;

  functionDescr.function = func_ptr;

  KernelErrorCode = krf_getFuncInfo( GET_FUNC_NAME, &functionDescr );

  if (functionDescr.func_type == 0)  return( NULL );

  return( functionDescr.func_name );
}


/*#################################################

GROUP: Functions for managing current and
       default network functions

#################################################*/

/*  returns the name of the current network function
*/
char  *krf_getCurrentNetworkFunc( type )
int  type;
{
  int  sel;
  static struct FuncInfoDescriptor  func_descr;


  KernelErrorCode = KRERR_NO_ERROR;

  if (specialNetworkType == NET_TYPE_GENERAL)  sel = 0;
  else  sel = NO_OF_FUNC_TYPES;

  switch (type)  {
    case  UPDATE_FUNC:
      break;
    case  LEARN_FUNC:
      sel += 1;
      break;
    case  INIT_FUNC:
      sel += 2;
      break;
    case  OUT_FUNC:
      sel += 3;
      break;
    case  ACT_FUNC:
      sel += 4;
      break;
    case  SITE_FUNC:
      sel += 5;
      break;
    case  UPDATE_FUNC | BENCH_FUNC:
      sel += 6;
      break;
    case  LEARN_FUNC | BENCH_FUNC:
      sel += 7;
      break;

   default:
     KernelErrorCode = KRERR_PARAMETERS;
     return( NULL );
  }

  if (netFuncInit[sel])  return( &CurrNetworkFunc[sel][0] );

  func_descr.func_type = type;
  if (krf_getFuncInfo( GET_DEFAULT_FUNC, &func_descr ) == KRERR_NO_ERROR)
    return( func_descr.func_name );

  return( NULL );
}


static void  krf_storeCurrentNetworkFunc( function_name, type )
char  *function_name;
int  type;
{
  int  sel;

  if (specialNetworkType == NET_TYPE_GENERAL)  sel = 0;
  else  sel = 3;


  switch (type)  {
    case  UPDATE_FUNC:
      break;
    case  LEARN_FUNC:
      sel += 1;
      break;
    case  INIT_FUNC:
      sel += 2;
      break;
    case  OUT_FUNC:
      sel += 3;
      break;
    case  ACT_FUNC:
      sel += 4;
      break;
    case  SITE_FUNC:
      sel += 5;
      break;
    case  UPDATE_FUNC | BENCH_FUNC:
      sel += 6;
      break;
    case  LEARN_FUNC | BENCH_FUNC:
      sel += 7;
      break;

   default:
     KernelErrorCode = KRERR_PARAMETERS;
     return;
    }

  netFuncInit[sel] = TRUE;
  strcpy( &CurrNetworkFunc[sel][0], function_name );
  return;
}




/*  sets the current network function.
*/
krui_err  krf_setCurrentNetworkFunc( function_name, type )
char  *function_name;
int  type;
{
  FunctionPtr func_ptr;
  bool	func_has_changed;
  char  *current_func;

  KernelErrorCode = KRERR_NO_ERROR;

  if (!krf_funcSearch( function_name, type, &func_ptr ))
    return( KernelErrorCode );
  if ((current_func = krf_getCurrentNetworkFunc( type )) == NULL)
    return( KernelErrorCode );

  func_has_changed = strcmp( function_name, current_func ) != 0;

  switch (type)  {
    case  UPDATE_FUNC:
      if (func_has_changed)
	{  /*  update function has changed, initialize the network  */
        NetInitialize = TRUE;
        krf_storeCurrentNetworkFunc( function_name, type );
      }

      return( KernelErrorCode );

    case  LEARN_FUNC:
      if (func_has_changed)
	{  /*  Learning function has changed  */
	LearnFuncHasChanged = TRUE;
        krf_storeCurrentNetworkFunc( function_name, type );
      }

      return( KernelErrorCode );

    case  INIT_FUNC:
    case  OUT_FUNC:
    case  ACT_FUNC:
    case  SITE_FUNC:
      krf_storeCurrentNetworkFunc( function_name, type );
      break;

   default:
      KernelErrorCode = KRERR_PARAMETERS;
      return( KernelErrorCode );
  }

  return( KernelErrorCode );
}




