/*     @(#)kr_mem.c	1.2 7/21/92  */

/**********************************************************************
FILE   : kr_mem.c
PURPOSE: SNNS-Kernel Memory Manager
NOTES  : 
AUTHOR : Niels Mache
DATE   : 21.2.90
VERSION : 1.2  7/21/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <stdio.h>
#include <stdlib.h>
/*  #include <malloc.h>  */
#include <string.h>
#include <memory.h>
#include <values.h>


#include "kr_typ.h"	 /*  Kernel Types and Constants  */

#include "kr_const.h"	 /*  Constant Declarators for SNNS-Kernel  */
#include "kr_def.h"	 /*  Default Values  */

#include "kr_mem.h"	 /*  Function prototypes  */
#include "kr_mac.h"	 /*  Kernel Macros  */

#if defined (ultrix) || defined (__BORLANDC__)
/*  ULTRIX-32 Operating System	*/
#include "strdup.c"	/*  include strdup function because strdup is
			    missing in ULTRIX-32  */
#endif



/*#################################################

GROUP: Global Var's (as declared by the kernel)

#################################################*/


extern	bool NetModified;	  /*  TRUE, if the network topology was modified  */

extern	int  NoOfUnits, 	     /*  no. of units in the network  */
	     MinUnitNo, 	     /*  the first (lowest) used unit number in the network  */
	     MaxUnitNo, 	     /*  the last (highest) used unit number in the network  */
	     NoOfPatternPairs,	     /*  no. of defined pattern pairs */
	     NoOfShuffledPatterns,   /*  no. of entries in the pattern_number array  */
	     NoOfInputUnits,	     /*  no. of input units  */
	     NoOfOutputUnits,	     /*  no. of output units  */
             NoOfHiddenUnits,	     /*  no. of hidden units  */
	     NoOfInputPatterns,      /*  no. of input patterns	*/
	     NoOfOutputPatterns;     /*  no. of output patterns  */


extern	UnitArray	unit_array;  /*  the unit array  */

extern	TopoPtrArray	topo_ptr_array;  /*  stores pointers to topological sorted units
					     used by kr_topoSort()  */
extern	Patterns	in_patterns,	 /*  the input pattern array  */
			out_patterns;	 /*  the output pattern array  */
extern	PatternNumbers	pattern_numbers; /*  contains shuffled pattern numbers
					     used by kr_shufflePatterns()  */

extern int  specialNetworkType;  /*  stores the topologic type of a network
				     (needed for the parallel kernel)  */

#ifdef MASPAR_KERNEL
extern struct FFnetDescriptor  descrFFnet;  /*  stores the topologic description
						of a feedforward network  */
#endif

/*#################################################

GROUP: Global Var's (as declared by the Memory Manager)

#################################################*/

int  NoOfAllocPatternPairs = 0;  /*  no. of allocated pattern pairs  */


/*#################################################

GROUP: Local Var's

#################################################*/

static int  FreeUnitIndex  = 0,  /*  no. of the next free unit in the unit array  */
	    NoOfAllocUnits = 0,  /*  no. of allocated units  */

	    NoOfSites	   = 0,  /*  no. of used sites in the network and Ftype sites
				     (including sites for functionality type use only) */
	    NoOfNetSites   = 0,  /*  no. of used sites in the network (only unit sites)  */
	    NoOfAllocSites = 0,  /*  no. of allocated sites  */

	    NoOfLinks	   = 0,  /*  no. of links used for the network	*/
	    NoOfAllocLinks = 0,  /*  no. of allocated links  */

	    NoOfNTableEntries	    = 0,  /*  no. of name table entries  */
	    NoOfAllocNTableEntries  = 0,  /*  no. of allocated name table entries  */

	    NoOfSTableEntries	    = 0,  /*  no. of site table entries  */
	    NoOfAllocSTableEntries  = 0,  /*  no. of allocated site table entries  */

	    NoOfFTableEntries	    = 0,  /*  no. of functionality types  */

	    NoOfAllocPatternNumbers = 0;  /*  no. of allocated array elements for
					      shuffeling pattern numbers  */



static SiteArray       site_array	 = NULL,  /*  pointer to first site array  */
		       free_site_ptr	 = NULL,  /*  pointer to first free site  */
		       site_block_list	 = NULL;  /*  pointer to first free site block */

static LinkArray       link_array	 = NULL,  /*  pointer to first link array  */
		       free_link_ptr	 = NULL,  /*  pointer to first free link  */
		       link_block_list	 = NULL;  /*  pointer to first free link block */


static NTableArray     NTable_array	 = NULL,  /*  pointer to name table  */
		       free_NTable_entry = NULL,  /*  pointer to first free name table entry  */
		       NTable_block_list = NULL,  /*  pointer to first free name table block  */
		       curr_NTable_entry = NULL,  /*  pointer to current name table entry  */
		       curr_NTable_block = NULL;  /*  pointer to current name table block  */


static STableArray     STable_array	 = NULL,  /*  pointer to site table  */
		       free_STable_entry = NULL,  /*  pointer to first free site table entry  */
		       STable_block_list = NULL,  /*  pointer to first free site table block  */
		       curr_STable_entry = NULL,  /*  pointer to current name site entry  */
		       curr_STable_block = NULL;  /*  pointer to current name site block  */


static struct FtypeUnitStruct  *Ftype_list_root  = NULL,  /*  pointer to root of the Ftype list  */
			       *curr_Ftype_entry = NULL;  /*  pointer to current Ftype entry  */




/*#################################################

GROUP: Link Functions

#################################################*/

/*  allocate another link array (with size N entries)
*/
static  int     krm_allocLinks( N )
int     N;
{
  LinkArray     tmp_ptr;


  tmp_ptr = (LinkArray) calloc( N + 1, LINK_SIZE );
  if (tmp_ptr == NULL)  return( 1 );

  if (link_array == NULL)
    {
    tmp_ptr->next = NULL;           /*  free link/block sentinel
                                    */
    free_link_ptr = tmp_ptr;
    }
  else
    {
    tmp_ptr->next = link_block_list;
    }

  link_block_list = tmp_ptr;    /*  free link block sentinel
                                    */
  NoOfAllocLinks += N;
  link_array = tmp_ptr;             /*  link_array points to the first link entry
                                    */
  return( 0 );
}

/*  get one link structure
*/
struct Link  *krm_getLink()
{
  struct Link   *tmp_ptr;


  if ((NoOfLinks == NoOfAllocLinks) || (link_array == NULL))
    if (krm_allocLinks( LINK_BLOCK ) != 0)
      {  /*  memory allocation failed  */
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( NULL );
    }

  NoOfLinks++;

  if (free_link_ptr->next != NULL)
    {
    tmp_ptr = free_link_ptr;
    free_link_ptr = free_link_ptr->next;
  }
  else
    {
    tmp_ptr = ++link_array;
  }

/*
  tmp_ptr->weight = (FlintType) 0;
  tmp_ptr->value_a = (FlintType) 0;
  tmp_ptr->value_b = (FlintType) 0;
  tmp_ptr->value_c = (FlintType) 0;
*/

  return( tmp_ptr );
}


/*  release one link structure
*/
void    krm_releaseLink( link_ptr )
struct Link *link_ptr;
{
  --NoOfLinks;

  link_ptr->next = free_link_ptr;
  free_link_ptr  = link_ptr;
}


/*  release the link and all following links
*/
void    krm_releaseAllLinks( first_link_ptr )
struct Link *first_link_ptr;
{
  struct Link  *curr,
               *next,
               *free;


  free = free_link_ptr;
  curr = first_link_ptr;

  while (curr != NULL)  {
    --NoOfLinks;

    next = curr->next;
    curr->next = free;

    free = curr;
    curr = next;
  }

  free_link_ptr = free;
}

/*  free all link arrays
*/
void  krm_releaseLinkArrays()
{
  register struct Link  *tmp_ptr1, *tmp_ptr2;


  NoOfLinks = NoOfAllocLinks = 0;

  if (link_array != NULL)  {
    tmp_ptr2 = link_block_list;
    while (tmp_ptr2 != NULL)  {
      tmp_ptr1 = tmp_ptr2->next;
      free( (char *) tmp_ptr2 );
      tmp_ptr2 = tmp_ptr1;
    }

    free_link_ptr = link_array = NULL;
  }
}

/*#################################################

GROUP: Site Functions

#################################################*/

/*  allocate another site array (with size N entries)
*/
static  int     krm_allocSites( N )
int     N;
{
  SiteArray     tmp_ptr;


  tmp_ptr = (SiteArray) calloc( N + 1, SITE_SIZE );
  if (tmp_ptr == NULL)  return( 1 );

  if (site_array == NULL)  {
    tmp_ptr->next = NULL;           /*  free site/block sentinel
                                    */
    free_site_ptr = tmp_ptr;
  }
  else  {
    tmp_ptr->next = site_block_list;
  }

  site_block_list = tmp_ptr;    /*  free site block sentinel
                                    */
  NoOfAllocSites += N;
  site_array = tmp_ptr;             /*  site_array points to the sentinel
                                    */
  return( 0 );
}


/*  get one unit-site structure
*/
struct Site    *krm_getSite()
{
  struct Site   *tmp_ptr;


  if ((site_array == NULL) || (NoOfSites == NoOfAllocSites))
    if (krm_allocSites( SITE_BLOCK ) != 0)  {
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( NULL );
    }

  NoOfSites++;
  NoOfNetSites++;

  if (free_site_ptr->next != NULL)  {
    tmp_ptr = free_site_ptr;
    free_site_ptr = free_site_ptr->next;
  }
  else  {
    tmp_ptr = ++site_array;
  }

  return( tmp_ptr );
}


/*  get one site structure for functionality use only
*/
static struct Site  *krm_getFtypeSite()
{
  struct Site  *tmp_ptr;

  KernelErrorCode = KRERR_NO_ERROR;

  if ((site_array == NULL) || (NoOfSites == NoOfAllocSites))
    if (krm_allocSites( SITE_BLOCK ) != 0)  {
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( NULL );
    }

  NoOfSites++;

  if (free_site_ptr->next != NULL)  {
    tmp_ptr = free_site_ptr;
    free_site_ptr = free_site_ptr->next;
  }
  else  {
    tmp_ptr = ++site_array;
  }

  return( tmp_ptr );
}


/*  release one unit-site structure
*/
void    krm_releaseSite( site_ptr )
struct Site *site_ptr;
{
  --NoOfSites;
  --NoOfNetSites;

  site_ptr->next = free_site_ptr;
  free_site_ptr  = site_ptr;
}


/* Future Use: 
   release one Ftype-site structure
*/
/*
static void    krm_releaseFtypeSite( site_ptr )
struct Site *site_ptr;
{
  --NoOfSites;

  site_ptr->next = free_site_ptr;
  free_site_ptr  = site_ptr;
}
*/


/*  release the unit-site and all following sites (at this unit)
*/
void    krm_releaseAllSites( first_site_ptr )
struct Site *first_site_ptr;
{
  struct Site  *curr,
               *next,
               *free;


  free = free_site_ptr;
  curr = first_site_ptr;

  while (curr != NULL)  {
    --NoOfSites;
    --NoOfNetSites;

    next = curr->next;
    curr->next = free;

    free = curr;
    curr = next;
  }

  free_site_ptr = free;
}


/*  release the Ftype-site and all following sites
*/
static void    krm_releaseAllFtypeSites( first_site_ptr )
struct Site *first_site_ptr;
{
  struct Site  *curr,
               *next,
               *free;


  free = free_site_ptr;
  curr = first_site_ptr;

  while (curr != NULL)  {
    --NoOfSites;

    next = curr->next;
    curr->next = free;

    free = curr;
    curr = next;
  }

  free_site_ptr = free;
}


/*  free all site arrays
*/
static  void    krm_releaseSiteArrays()
{
  struct Site      *tmp_ptr;


  NoOfSites      = 0;
  NoOfNetSites   = 0;
  NoOfAllocSites = 0;

  if (site_array != NULL)  {
    while (site_block_list != NULL)  {
      tmp_ptr =  site_block_list->next;
      free( (char *) site_block_list );
      site_block_list = tmp_ptr;
    }

    free_site_ptr = NULL;
    site_array    = NULL;
  }
}



/*#################################################

GROUP: Unit Functions

#################################################*/

/*  garbage collection of unit array  
*/
void  krm_unitArrayGC()
{
  register struct Link   *link_ptr;
  register struct Site   *site_ptr;
  register struct Unit   *unit_ptr;
  register struct Unit   *new_unit_ptr;
  struct Unit   *dest_unit_ptr;


  /*  find first unused unit stucture  */
  dest_unit_ptr = NULL;
  FOR_ALL_UNITS( unit_ptr )
    if (!UNIT_IN_USE( unit_ptr ))
      {  /*  unit isn't in use  */
      dest_unit_ptr = unit_ptr;  /*  store the first unused unit stucture  */
      break;
    }

  if (dest_unit_ptr != NULL)  
    {  /*  do garbage collection  */
    NetModified = TRUE;

/*  store continous unit pointers in each unit struct  */
    new_unit_ptr = unit_array;
    FOR_ALL_UNITS( unit_ptr )
      if UNIT_IN_USE( unit_ptr )
        /*  unit is in use  */
        unit_ptr->Aux.ptr = (char *) ++new_unit_ptr;

/*  adjust the link pointers  */
    FOR_ALL_UNITS( unit_ptr )
      if UNIT_IN_USE( unit_ptr )  {
	if UNIT_HAS_SITES( unit_ptr )
	  FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
              link_ptr->to = (struct Unit *) link_ptr->to->Aux.ptr;
        else
	  if UNIT_HAS_DIRECT_INPUTS( unit_ptr )
	    FOR_ALL_LINKS( unit_ptr, link_ptr )
              link_ptr->to = (struct Unit *) link_ptr->to->Aux.ptr;
      }


/*  compress unit array  */
    for (unit_ptr = dest_unit_ptr + 1; unit_ptr <= unit_array + MaxUnitNo; unit_ptr++)
      if UNIT_IN_USE( unit_ptr )
        memcpy( (char *) dest_unit_ptr++, (char *) unit_ptr, UNIT_SIZE );

    MinUnitNo = 1;
    MaxUnitNo = NoOfUnits;
    FreeUnitIndex = 0;
  }

  /*  reduce size of unit array, if needed  */
  if ((NoOfAllocUnits - NoOfUnits) >= (2 * UNIT_BLOCK))  {
    unit_ptr = (UnitArray) realloc( (char *) unit_array, (unsigned)
                                   ((NoOfAllocUnits + 1 - UNIT_BLOCK) * UNIT_SIZE) );
    if (unit_ptr != NULL)  {
      unit_array = unit_ptr;
      NoOfAllocUnits -= UNIT_BLOCK;
    }
  }
}



/*  relocate the link pointers. (If the address of the unit array was modified
    because the memory was reallocated, the link pointers must be relocated)
*/
static  void    krm_relocateLinkPtrs( offset )
int     offset;
{
  register struct Link   *link_ptr;
  register struct Site   *site_ptr;
  register struct Unit   *unit_ptr;


  FOR_ALL_UNITS( unit_ptr )
    if UNIT_IN_USE( unit_ptr )  {
      if UNIT_HAS_SITES( unit_ptr )
	FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	  link_ptr->to = (struct Unit *) ((char *) link_ptr->to + offset);
      else
	if UNIT_HAS_DIRECT_INPUTS( unit_ptr )
	  FOR_ALL_LINKS( unit_ptr, link_ptr )
            link_ptr->to = (struct Unit *) ((char *) link_ptr->to + offset);
    }
}


/*  allocate the unit array
*/
krui_err  krm_allocUnits( N )
int  N;
{
  UnitArray  tmp_ptr;
  int  offset;

  if ((NoOfAllocUnits - NoOfUnits) < N)
    {  /*  alloc units	*/
    N = (N / UNIT_BLOCK + 1) * UNIT_BLOCK;
  }

  if (unit_array == NULL)  {
    tmp_ptr = (UnitArray) calloc( NoOfAllocUnits + N + 1, UNIT_SIZE );
    if (tmp_ptr == NULL)
      {  /*  mem alloc failed	 */
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( KernelErrorCode );
    }
    FreeUnitIndex = 0;
    tmp_ptr[0].Out.nextFreeUnit = 0;   /*  sentinel of free unit list
                                       */
  }
  else  {
    tmp_ptr = (UnitArray) realloc( (char *) unit_array, (unsigned)
                                   ((NoOfAllocUnits + N + 1 ) * UNIT_SIZE) );
    if (tmp_ptr == NULL)
      {  /*  mem alloc failed	 */
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( KernelErrorCode );
    }
    offset = (char *) tmp_ptr - (char *) unit_array;
    if (offset != 0)  krm_relocateLinkPtrs( offset );
  }

  NoOfAllocUnits += N;
  unit_array = tmp_ptr;

  KernelErrorCode = KRERR_NO_ERROR;
  return( KernelErrorCode );
}

/*  get one unit structure
*/
int  krm_getUnit()
{
  register int   unit_no;

  KernelErrorCode = KRERR_NO_ERROR;
  if ((unit_array == NULL) || (NoOfUnits == NoOfAllocUnits))
    if (krm_allocUnits( UNIT_BLOCK ) != 0)
      {  /*  Insufficient memory  */
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( 0 );
    }
  NoOfUnits++;

  if (FreeUnitIndex != 0)
    {  /*  reuse unit  */
    unit_no = FreeUnitIndex;
    FreeUnitIndex = unit_array[unit_no].Out.nextFreeUnit;
  }
  else
    unit_no = NoOfUnits;

  /*  this unit is ready to use  */
  unit_array[unit_no].flags = UFLAG_IN_USE;
  unit_array[unit_no].sites = NULL;

  if (NoOfUnits == 1)
    MinUnitNo = MaxUnitNo = unit_no;
  else  {
    /*	store the highest allocated unit number  */
    if (unit_no > MaxUnitNo)  MaxUnitNo = unit_no;
    /*	store the lowest allocated unit number	*/
    if (unit_no < MinUnitNo)  MinUnitNo = unit_no;
  }

  return( unit_no );
}


/*  searches for the last used unit in the unit array, returns the unit no. of
    this unit
*/
static  int     krm_highUsedUnit( UnitNo )
int     UnitNo;
{
  register int	 i;
  register struct Unit	 *unit_ptr;


  for (i = UnitNo - 1, unit_ptr = unit_array + (UnitNo - 1); i >= MinUnitNo; i--, unit_ptr--)
    if ((unit_ptr->flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      return( i );

  return( 0 );
}


/*  searches for the first used unit in the unit array, returns the unit no. of
    this unit
*/
static  int     krm_lowUsedUnit( UnitNo )
int     UnitNo;
{
  register int	 i;
  register struct Unit	 *unit_ptr;


  for (i = UnitNo + 1, unit_ptr = unit_array + (UnitNo + 1); i <= MaxUnitNo; i++, unit_ptr++)
    if ((unit_ptr->flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      return( i );

  return( 0 );
}

/*  release unit
*/
void    krm_releaseUnit( UnitNo )
int     UnitNo;
{
  if (unit_array[UnitNo].flags != UFLAG_FREE)
    {  /*  don't release units twice  */
    --NoOfUnits;

    if ((NoOfAllocUnits - NoOfUnits) >= (2 * UNIT_BLOCK))
      /*  do garbage collection  */
      krm_unitArrayGC();
      

    /*	find the highest used unit number  */
    if (UnitNo == MaxUnitNo)  MaxUnitNo = krm_highUsedUnit( UnitNo );

    /*	find the lowest used unit number  */
    if (UnitNo == MinUnitNo)  MinUnitNo = krm_lowUsedUnit( UnitNo );

    unit_array[UnitNo].flags = UFLAG_FREE;

    unit_array[UnitNo].Out.nextFreeUnit = FreeUnitIndex;
    FreeUnitIndex = UnitNo;
  }
}

/*  free the unit array
*/
static	void	krm_releaseUnitArrays()
{
  NoOfAllocUnits = FreeUnitIndex =
  NoOfUnits = NoOfInputUnits = NoOfOutputUnits =
  NoOfHiddenUnits = MaxUnitNo = 0;

  if (unit_array != NULL)  {
    free( (char *) unit_array );
    unit_array = NULL;
  }
}



/*#################################################

GROUP: General Purpose Functions

#################################################*/

/*  get information about memory usage
*/
void    krm_getMemoryManagerInfo( array_size, info_array )
int     *array_size,
        info_array[];
{
  info_array[ 0 ] = NoOfNetSites;

#ifdef MASPAR_KERNEL
  if (specialNetworkType == NET_TYPE_FF1)
    info_array[ 1 ] = descrFFnet.no_of_weights;
  else
    info_array[ 1 ] = NoOfLinks;
#else
  info_array[ 1 ] = NoOfLinks;
#endif

  info_array[ 2 ] = NoOfSTableEntries;
  info_array[ 3 ] = NoOfFTableEntries;

  info_array[ 4 ] = NoOfAllocUnits;
  info_array[ 5 ] = NoOfAllocSites;
  info_array[ 6 ] = NoOfAllocLinks;
  info_array[ 7 ] = NoOfAllocNTableEntries;
  info_array[ 8 ] = NoOfAllocSTableEntries;
  info_array[ 9 ] = NoOfFTableEntries;

  *array_size = 10;
}


/*  allocate the array for topological sorting of the units in the network
*/
krui_err  krm_allocUnitTopoArray( N )
int  N;
{
  KernelErrorCode = KRERR_NO_ERROR;  /*  reset return code  */

  if (topo_ptr_array != NULL)
    {  /*  reallocate array for topologic sorting  */
    topo_ptr_array = (TopoPtrArray) realloc( (char *) topo_ptr_array,
					     (unsigned) (N * TOPO_PTR_SIZE) );
  }
  else
    {  /*  allocate new array for topologic sorting  */
    topo_ptr_array = (TopoPtrArray) calloc( N, TOPO_PTR_SIZE);
  }

  if (topo_ptr_array == NULL)  KernelErrorCode = KRERR_INSUFFICIENT_MEM;

  return( KernelErrorCode );
}


/*  release the topolocic array
*/
void  krm_releaseUnitTopoArray()
{
  if (topo_ptr_array != NULL)  {
    free( (char *) topo_ptr_array );
    topo_ptr_array = NULL;
  }
}


/*#################################################

GROUP:	pattern functions

#################################################*/

/*  allocate arrays for storing pattern pairs
*/
krui_err  krm_allocPatternArrays( N )
int  N;
{
  Patterns  in_pat = NULL, 
            out_pat = NULL;

  if (N <= 0)  {
    KernelErrorCode = KRERR_PARAMETERS;
    return( KernelErrorCode );
  }

  KernelErrorCode = KRERR_NO_ERROR;

  if (NoOfAllocPatternPairs == 0)
    {  /*  alloc patterns (first time)	*/
    in_pat = (FlintType *) calloc( NoOfInputPatterns * (NoOfAllocPatternPairs + N),
				   PATTERN_SIZE );
    if (in_pat == NULL)  {
      NoOfInputPatterns = NoOfOutputPatterns = NoOfPatternPairs = 0;
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( KernelErrorCode );
    }

    if (NoOfOutputPatterns > 0)  {
      out_pat = (FlintType *) calloc( NoOfOutputPatterns * (NoOfAllocPatternPairs + N),
		        	      PATTERN_SIZE );
      if (out_pat == NULL)
        {  /*  mem alloc failed  */
        free( (char *) in_pat );
        NoOfInputPatterns = NoOfOutputPatterns = NoOfPatternPairs = 0;

#ifdef MASPAR_KERNEL
      descrFFnet.no_of_pattern_pairs = 0;
#endif

        KernelErrorCode = KRERR_INSUFFICIENT_MEM;
        return( KernelErrorCode );
      }
    }
  }
  else  {
    in_pat = (FlintType *) realloc( (char *) in_patterns, (unsigned)
				    (NoOfInputPatterns * (NoOfAllocPatternPairs + N) * PATTERN_SIZE) );
    if (in_pat == NULL)  {
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( KernelErrorCode );
    }

    if (NoOfOutputPatterns > 0)  {
      out_pat = (FlintType *) realloc( (char *) in_patterns, (unsigned)
				       (NoOfOutputPatterns * (NoOfAllocPatternPairs + N) * PATTERN_SIZE) );

      if (out_pat == NULL)
        {  /*  mem alloc failed, but keep old state  */
        KernelErrorCode = KRERR_INSUFFICIENT_MEM;
        return( KernelErrorCode );
      }
    }
  }

  in_patterns = in_pat;
  out_patterns = out_pat;

#ifdef MASPAR_KERNEL
  descrFFnet.in_patterns = in_pat;
  descrFFnet.out_patterns = out_pat;
#endif

  NoOfAllocPatternPairs += N;

  return( KernelErrorCode );
}


/*  allocate one pattern pair
*/
krui_err  krm_newPattern()
{
  if (NoOfPatternPairs == NoOfAllocPatternPairs)
    if (krm_allocPatternArrays( PATTERN_BLOCK ) != KRERR_NO_ERROR)
      return( KernelErrorCode );

  NoOfPatternPairs++;

#ifdef MASPAR_KERNEL
  descrFFnet.no_of_pattern_pairs = NoOfPatternPairs;
#endif
  return( KernelErrorCode );
}


/*  delete one pattern pair
*/
krui_err  krm_deletePatternPair( pat_no )
int  pat_no;
{
  int         patsize;
  Patterns    pat_adr, 
              in_pat = NULL, 
              out_pat = NULL;


  KernelErrorCode = KRERR_NO_ERROR;

  patsize = NoOfInputPatterns * PATTERN_SIZE;
  for (pat_adr = in_patterns + pat_no * NoOfInputPatterns;
       pat_adr < in_patterns + NoOfPatternPairs * NoOfInputPatterns;
       pat_adr += NoOfInputPatterns)
    memcpy( pat_adr, pat_adr + NoOfInputPatterns, patsize);

  if (NoOfOutputPatterns > 0)  {
    patsize = NoOfOutputPatterns * PATTERN_SIZE;
    for (pat_adr = out_patterns + pat_no * NoOfOutputPatterns;
         pat_adr < out_patterns + NoOfPatternPairs * NoOfOutputPatterns;
         pat_adr += NoOfOutputPatterns)
      memcpy( pat_adr, pat_adr + NoOfOutputPatterns, patsize);
  }

  --NoOfPatternPairs;  /*  decrement no. of pattern pairs  */

#ifdef MASPAR_KERNEL
  descrFFnet.no_of_pattern_pairs = NoOfPatternPairs;
#endif

  if ((NoOfAllocPatternPairs - NoOfPatternPairs) >= 2 * PATTERN_BLOCK)
    {  /*  reduce size of pattern arrays  */
    in_pat = (FlintType *) realloc( (char *) in_patterns, (unsigned)
				    (NoOfInputPatterns * (NoOfAllocPatternPairs - PATTERN_BLOCK) * PATTERN_SIZE) );
    if (in_pat == NULL)
      {  /* mem alloc failed (trouble with memory manager)  */
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( KernelErrorCode );
    }

    if (NoOfOutputPatterns > 0)  {
      out_pat = (FlintType *) realloc( (char *) out_patterns, (unsigned)
	  			     (NoOfOutputPatterns * (NoOfAllocPatternPairs - PATTERN_BLOCK) * PATTERN_SIZE) );
      if (out_pat == NULL)
        {  /*  mem alloc failed, but keep old state    */
        in_patterns = (FlintType *) realloc( (char *) in_pat, (unsigned)
					   (NoOfInputPatterns * NoOfAllocPatternPairs * PATTERN_SIZE) );
        KernelErrorCode = KRERR_INSUFFICIENT_MEM;
        return( KernelErrorCode );
      }
    }

    NoOfAllocPatternPairs -= PATTERN_BLOCK;    
    in_patterns = in_pat;
    out_patterns = out_pat;

#ifdef MASPAR_KERNEL
    descrFFnet.in_patterns = in_pat;
    descrFFnet.out_patterns = out_pat;
#endif
  }

  return( KernelErrorCode );
}


/*  release pattern arrays
*/
void  krm_releasePatternArrays()
{
  if (in_patterns != NULL)  {
    free( (char *) in_patterns );
    in_patterns = NULL;

#ifdef MASPAR_KERNEL
    descrFFnet.in_patterns = NULL;
#endif
  }

  if (out_patterns != NULL)  {
    free( (char *) out_patterns );
    out_patterns = NULL;
#ifdef MASPAR_KERNEL
    descrFFnet.out_patterns = NULL;
#endif
  }

  NoOfPatternPairs = NoOfAllocPatternPairs =
  NoOfInputPatterns = NoOfOutputPatterns = 0;

#ifdef MASPAR_KERNEL
  descrFFnet.no_of_pattern_pairs = 0;
#endif
}


/*  allocate array for shuffeling pattern numbers
*/
krui_err  krm_allocPatternNoArray()
{
  KernelErrorCode = KRERR_NO_ERROR;

  if (NoOfPatternPairs > NoOfAllocPatternNumbers)  {
    if (pattern_numbers != NULL)
      pattern_numbers = (PatternNumbers) realloc( (char *) pattern_numbers, 
                                                  (unsigned) (NoOfAllocPatternPairs * PATTERN_NO_SIZE) );
    else
      pattern_numbers = (PatternNumbers) calloc( NoOfAllocPatternPairs, PATTERN_NO_SIZE );

    if (pattern_numbers == NULL)  {
      NoOfAllocPatternNumbers = 0;
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( KernelErrorCode );
    }

    NoOfAllocPatternNumbers = NoOfAllocPatternPairs;
  }

  return( KernelErrorCode );
}


/*  release pattern number array
*/
void    krm_releasePatternNumbers()
{
  NoOfAllocPatternNumbers = 0;
  NoOfShuffledPatterns = 0;
  if (pattern_numbers != NULL)  {
    free( (char *) pattern_numbers );
    pattern_numbers = NULL;
  }
}



/*#################################################

GROUP: NameTable Functions

#################################################*/

/*  allocate one name-table block
*/
static  int     krm_allocNTableArray()
{
  NTableArray     tmp_ptr;


  tmp_ptr = (NTableArray) calloc( NTABLE_BLOCK + 1, NTABLE_SIZE );
  if (tmp_ptr == NULL)  return( 1 );

  if (NTable_array == NULL)  {
    tmp_ptr->Entry.next = NULL;     /*  free name-table block sentinel  */
    free_NTable_entry = tmp_ptr;    /*  free name-table entry sentinel  */
  }
  else  {
    tmp_ptr->Entry.next = NTable_block_list;    /*  append new name-table block
                                                    to block list   */
  }

  NTable_block_list = tmp_ptr;      /*  update block list ptr
                                    */
  NoOfAllocNTableEntries += NTABLE_BLOCK;
  NTable_array = tmp_ptr + 1;       /*  NTable_array points to the first entry
                                    */
  return( 0 );
}

/*  get one name-table entry
*/
static struct NameTable  *krm_getNTableEntry()
{
  struct NameTable  *tmp_ptr;


  if ((NTable_array == NULL) || (NoOfNTableEntries == NoOfAllocNTableEntries))
    if (krm_allocNTableArray() != 0)  {
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( NULL );
    }

  NoOfNTableEntries++;

  if (free_NTable_entry->Entry.next != NULL)
    {   /*  a previous released name-table entry is availabe    */
    tmp_ptr = free_NTable_entry;
    free_NTable_entry = free_NTable_entry->Entry.next;
  }
  else  {
    tmp_ptr = NTable_array++;
  }

  KernelErrorCode = KRERR_NO_ERROR;
  tmp_ptr->ref_count = 1;
  return( tmp_ptr );
}

/*  release one name-table entry
*/
void    krm_NTableReleaseEntry( NTable_ptr )
struct NameTable    *NTable_ptr;
{
  --NoOfNTableEntries;

  free( NTable_ptr->Entry.symbol );
  NTable_ptr->sym_type = UNUSED_SYM;
  NTable_ptr->Entry.next = free_NTable_entry;
  free_NTable_entry = NTable_ptr;
}


/*  release all name-table blocks
*/
static void  krm_releaseNTableArrays()
{
  struct NameTable     *tmp_ptr;


  NoOfNTableEntries      = 0;
  NoOfAllocNTableEntries = 0;

  if (NTable_array != NULL)  {
    for (tmp_ptr = NTable_array - 1; tmp_ptr > NTable_block_list; --tmp_ptr)  {
      if (tmp_ptr->sym_type != UNUSED_SYM)
        free( tmp_ptr->Entry.symbol );    /*  free symbols */
    }

    tmp_ptr = NTable_block_list->Entry.next;
    free( (char *) NTable_block_list );
    NTable_block_list = tmp_ptr;

    while (NTable_block_list != NULL)  {
      for (tmp_ptr = NTable_block_list + NTABLE_BLOCK;
           tmp_ptr > NTable_block_list; --tmp_ptr)  {
        if (tmp_ptr->sym_type != UNUSED_SYM)
          free( tmp_ptr->Entry.symbol );   /*  free symbols */
      }

      tmp_ptr =  NTable_block_list->Entry.next;
      free( (char *) NTable_block_list );
      NTable_block_list = tmp_ptr;
    }

    free_NTable_entry = NULL;
    NTable_array      = NULL;
  }
}

/*  Searches for a given symbol and symbol-type in the name table.
    Returns symbol ptr if symbol was found, NULL otherwise. 
*/
struct  NameTable   *krm_NTableSymbolSearch( symbol, sym_type )
char    *symbol;
int     sym_type;
{
  int   symbol_type;
  struct  NameTable   *n_ptr,
                      *block_list;


  if (NTable_array == NULL)  return( NULL );

  block_list = NTable_block_list;
  for (n_ptr = NTable_array - 1; n_ptr > block_list; n_ptr--)  {
    symbol_type = (int) n_ptr->sym_type;
    if ( (symbol_type != UNUSED_SYM) &&
         (symbol_type == sym_type) &&
         (strcmp( n_ptr->Entry.symbol, symbol ) == 0) )
      return( n_ptr );    /*  symbol was found    */
  }

  for (block_list = block_list->Entry.next;
       block_list != NULL;
       block_list = block_list->Entry.next)  {
    for (n_ptr = block_list + NTABLE_BLOCK; n_ptr > block_list; n_ptr--)  {
      symbol_type = (int) n_ptr->sym_type;
      if ( (symbol_type != UNUSED_SYM) &&
           (symbol_type == sym_type) &&
           (strcmp( n_ptr->Entry.symbol, symbol ) == 0) )
        return( n_ptr );    /*  symbol was found    */
    }
  }

  return( NULL );
}


/*  Creates a new symbol in the name-table.   Returns name-table ptr or NULL if
    memory alloc has failed.    
*/
struct NameTable  *krm_NTableCreateEntry( symbol_name, symbol_type)
char  *symbol_name;
int  symbol_type;
{
  char   *str_ptr;
  struct  NameTable     *n_ptr;


  KernelErrorCode = KRERR_NO_ERROR;

  if ( (n_ptr = krm_getNTableEntry() ) == NULL)
    return( NULL );     /*  memory alloc failed */

  if ((str_ptr = strdup( symbol_name ) ) == NULL)
    {  /*  memory alloc failed */
    KernelErrorCode = KRERR_INSUFFICIENT_MEM;
    return( NULL );
  }

  n_ptr->Entry.symbol = str_ptr;
  n_ptr->sym_type = (unsigned short) symbol_type;

  return( n_ptr );
}


/*  Inserts a symbol in the name-table. This function duplicates symbol ptrs
    if the symbol was found in the name-table.  Returns symbol ptr or NULL if
    memory alloc has failed.
*/
char  *krm_NTableInsertSymbol( symbol_name, symbol_type)
char  *symbol_name;
int   symbol_type;
{
  struct NameTable  *n_ptr;


  if ( (n_ptr = krm_NTableSymbolSearch( symbol_name, symbol_type ) ) != NULL)
    {   /*  symbol is already in the name table  */
    if (n_ptr->ref_count < MAXSHORT)
      n_ptr->ref_count++;

    return( n_ptr->Entry.symbol );
  }

  n_ptr = krm_NTableCreateEntry( symbol_name, symbol_type );
  return( n_ptr->Entry.symbol );
}


/*  release name-table entry if there is no other reference to this symbol
*/
void  krm_NTableReleaseSymbol( symbol_name, symbol_type)
char  *symbol_name;
int   symbol_type;
{
  struct NameTable  *n_ptr;


  if (symbol_name == NULL)  return;
  if ( (n_ptr = krm_NTableSymbolSearch( symbol_name, symbol_type ) ) != NULL)
    {   /*  symbol is in the name table  */
    if (n_ptr->ref_count < MAXSHORT)
      {   /*    No. of references to this symbol don't exceed the max. reference
                count. This means it is possible to delete the symbol if the
                reference count is zero.
          */
      if (--(n_ptr->ref_count) == 0)
        krm_NTableReleaseEntry( n_ptr );
    }
  }
}


/*  get the first name-table entry
*/
struct NameTable       *krm_getNTableFirstEntry()
{
  if (NTable_array == NULL)  return( NULL );

  curr_NTable_block = NTable_block_list;
  curr_NTable_entry = NTable_array - 1;
  return( curr_NTable_entry );
}


/*  get the next name-table entry
*/
struct NameTable       *krm_getNTableNextEntry()
{
  if ((NTable_array == NULL) || (curr_NTable_block == NULL))
    return( NULL );

  if (--curr_NTable_entry == curr_NTable_block)  {
    if ( (curr_NTable_block = curr_NTable_block->Entry.next) == NULL)
      return( NULL );

    curr_NTable_entry = curr_NTable_block + NTABLE_BLOCK;
  }

  return( curr_NTable_entry );
}





/*#################################################

GROUP: SiteTable Functions

#################################################*/

/*  allocate another site-table block
*/
static int  krm_allocSTableArray()
{
  STableArray     tmp_ptr;


  tmp_ptr = (STableArray) calloc( STABLE_BLOCK + 1, STABLE_SIZE );
  if (tmp_ptr == NULL)  return( 1 );

  if (STable_array == NULL)  {
    tmp_ptr->Entry.next = NULL;     /*  free site-table block sentinel  */
    free_STable_entry = tmp_ptr;    /*  free site-table entry sentinel  */
  }
  else  {
    tmp_ptr->Entry.next = STable_block_list;    /*  append new site-table block
                                                    to block list   */
  }

  STable_block_list = tmp_ptr;      /*  update block list ptr
                                    */
  NoOfAllocSTableEntries += STABLE_BLOCK;
  STable_array = tmp_ptr + 1;       /*  STable_array points to the first entry
                                    */
  return( 0 );
}


/*  get one site-table entry
*/
static  struct SiteTable    *krm_getSTableEntry()
{
  struct SiteTable  *tmp_ptr;


  KernelErrorCode = KRERR_NO_ERROR;
  if ((STable_array == NULL) || (NoOfSTableEntries == NoOfAllocSTableEntries))
    if (krm_allocSTableArray() != 0)  {
      KernelErrorCode = KRERR_INSUFFICIENT_MEM;
      return( NULL );
    }

  NoOfSTableEntries++;

  if (free_STable_entry->Entry.next != NULL)
    {   /*  a previous released site-table entry is availabe    */
    tmp_ptr = free_STable_entry;
    free_STable_entry = free_STable_entry->Entry.next;
  }
  else  {
    tmp_ptr = STable_array++;
  }

  return( tmp_ptr );
}

/*  release site table block
*/
static  void    krm_releaseSTableEntry( STable_ptr )
struct SiteTable    *STable_ptr;
{
  --NoOfSTableEntries;

  STable_ptr->site_func  = NULL;
  STable_ptr->Entry.next = free_STable_entry;
  free_STable_entry = STable_ptr;
}


/*  release all site-table blocks
*/
static  void    krm_releaseSTableArrays()
{
  struct SiteTable     *tmp_ptr;


  NoOfSTableEntries      = 0;
  NoOfAllocSTableEntries = 0;

  if (STable_array != NULL)  {
    while (STable_block_list != NULL)  {
      tmp_ptr =  STable_block_list->Entry.next;
      free( (char *) STable_block_list );
      STable_block_list = tmp_ptr;
    }

    free_STable_entry = NULL;
    STable_array      = NULL;
  }
}


/*  create new site-table entry
*/
struct  SiteTable   *krm_STableCreateEntry( site_symbol, site_func )
char    *site_symbol;
SiteFuncPtr  site_func;
{
  struct  NameTable     *n_ptr;
  struct  SiteTable     *s_ptr;


  if ( (s_ptr = krm_getSTableEntry() ) == NULL)
    return( NULL );     /*  memory alloc failed */

  if ((n_ptr = krm_NTableCreateEntry( site_symbol, SITE_SYM ) ) == NULL)  {
    krm_releaseSTableEntry( s_ptr );
    return( NULL );     /*  memory alloc failed */
  }

  s_ptr->Entry.site_name = n_ptr;
  s_ptr->site_func       = site_func;

  return( s_ptr );
}


/*  change the properties of the given site-table entry
*/
struct  SiteTable   *krm_STableChangeEntry( stbl_ptr , new_site_name , new_site_func )
struct  SiteTable       *stbl_ptr;
char    *new_site_name;
SiteFuncPtr     new_site_func;
{
  struct  NameTable     *n_ptr;


  if ((n_ptr = krm_NTableCreateEntry( new_site_name, SITE_SYM ) ) == NULL)
    return( NULL );     /*  memory alloc failed */

  krm_NTableReleaseEntry( stbl_ptr->Entry.site_name );

  stbl_ptr->Entry.site_name = n_ptr;
  stbl_ptr->site_func       = new_site_func;

  return( stbl_ptr );
}


/*  release a previosly defined site-table entry
*/
void    krm_STableRemoveEntry( STable_ptr )
struct SiteTable    *STable_ptr;
{
  krm_NTableReleaseEntry( STable_ptr->Entry.site_name );
  krm_releaseSTableEntry( STable_ptr );
}


/*  searches for a symbol in the site-table
*/
struct SiteTable    *krm_STableSymbolSearch( site_symbol )
char    *site_symbol;
{
  struct  NameTable   *n_ptr;
  struct  SiteTable   *s_ptr,
                      *block_list;


  if (STable_array == NULL)
    return( NULL );     /*  there are no site-table entries */

  if ( (n_ptr = krm_NTableSymbolSearch( site_symbol , SITE_SYM ) ) == NULL)
    return( NULL );     /*  symbol dosn't exist */


  block_list = STable_block_list;
  for (s_ptr = STable_array - 1; s_ptr > block_list; s_ptr--)
    if ( (s_ptr->site_func != NULL) &&
         (s_ptr->Entry.site_name == n_ptr ) )
      return( s_ptr );

  for (block_list = block_list->Entry.next;
       block_list != NULL;
       block_list = block_list->Entry.next)  {
    for (s_ptr = block_list + STABLE_BLOCK; s_ptr > block_list; s_ptr--)
      if ( (s_ptr->site_func != NULL) &&
           (s_ptr->Entry.site_name == n_ptr ) )
        return( s_ptr );
  }

  return( NULL );  /*  the site symbol is in the name table, but not in the site table: Error */
}


/*  returns a pointer to the next (used or unused) site-table entry
*/
static struct SiteTable       *krm_getSTableNextRawEntry()
{
  if ((STable_array == NULL) || (curr_STable_block == NULL))
    return( NULL );

  if (--curr_STable_entry == curr_STable_block)
    {  /*  get new site-table block  */
    if ( (curr_STable_block = curr_STable_block->Entry.next) == NULL)  {
      curr_STable_block = NULL;
      curr_STable_entry = NULL;

      return( NULL );
    }

    curr_STable_entry = curr_STable_block + STABLE_BLOCK;  /*  next site-table block  */
  }

  return( curr_STable_entry );
}


/*  returns a pointer to the next used site-table entry
*/
struct SiteTable       *krm_getSTableNextEntry()
{
  struct  SiteTable   *stbl_ptr;


  if ((stbl_ptr = krm_getSTableNextRawEntry()) == NULL)
    return( NULL );

  while ( stbl_ptr->site_func == NULL)	/*  return only used site-table entries  */
    if ((stbl_ptr = krm_getSTableNextRawEntry()) == NULL)
      return( NULL );
        
  return( stbl_ptr );
}



/*  returns a pointer to the first used site-table entry
*/
struct SiteTable       *krm_getSTableFirstEntry()
{
  struct  SiteTable   *stbl_ptr;
  

  if (STable_array == NULL)  return( NULL );

  curr_STable_block = STable_block_list;
  curr_STable_entry = STable_array - 1;
  stbl_ptr = curr_STable_entry;

  if (stbl_ptr->site_func == NULL)  /*	return only used site-table entries  */
    if ((stbl_ptr = krm_getSTableNextEntry()) == NULL)
      return( NULL );

  return( stbl_ptr );
}





/*#################################################

GROUP: Ftype entry functions

#################################################*/


/*  allocate a new Ftype entry
*/
struct FtypeUnitStruct  *krm_getFtypeEntry()
{
  struct FtypeUnitStruct  *Ftype_entry;


  KernelErrorCode = KRERR_NO_ERROR;
  if ((Ftype_entry = (struct FtypeUnitStruct *) malloc( FTYPE_UNIT_SIZE ) ) == NULL)
    {  /*  memory alloc failed */
    KernelErrorCode = KRERR_INSUFFICIENT_MEM;
    return( NULL );
  }

  if (Ftype_list_root != NULL)  {
    Ftype_list_root->prev = Ftype_entry;
    Ftype_entry->next = Ftype_list_root;
  }
  else
    Ftype_entry->next = NULL;

  Ftype_list_root   = Ftype_entry;
  Ftype_entry->prev = NULL;

  Ftype_entry->sites = NULL;
  Ftype_entry->Ftype_symbol = NULL;

  ++NoOfFTableEntries;

  return( Ftype_entry );
}


/*  free a previosly defined Ftype entry
*/
void    krm_releaseFtypeEntry( Ftype_entry )
struct FtypeUnitStruct  *Ftype_entry;
{
  struct  FtypeUnitStruct   *next_entry,
                            *prev_entry;


/*  release sites first */
  if ( Ftype_entry->sites != NULL)
    krm_releaseAllSites( Ftype_entry->sites );

/*  release Ftype symbol name */
  if ( Ftype_entry->Ftype_symbol != NULL )
    krm_NTableReleaseEntry( Ftype_entry->Ftype_symbol );

  next_entry = Ftype_entry->next;
  prev_entry = Ftype_entry->prev;

  if (prev_entry != NULL)
    prev_entry->next = next_entry;
  else
    Ftype_list_root = next_entry;

  if (next_entry != NULL)
    next_entry->prev = prev_entry;

/*  release unit entry  */
  free( (char *) Ftype_entry );       /*  release Ftype entry   */

  --NoOfFTableEntries;
}


/*  create and define a Ftype entry
*/
struct FtypeUnitStruct  *krm_FtypeCreateEntry( Ftype_symbol, out_func, act_func, act_deriv_func )
char        *Ftype_symbol;
OutFuncPtr  out_func;
ActFuncPtr  act_func;
ActDerivFuncPtr  act_deriv_func;
{
  struct  FtypeUnitStruct   *Ftype_entry;
  struct  NameTable         *n_ptr;


  if ( (Ftype_entry = krm_getFtypeEntry() ) == NULL)
    return( NULL );     /*  memory alloc failed */

  if ( (n_ptr = krm_NTableCreateEntry( Ftype_symbol, FTYPE_UNIT_SYM ) ) == NULL)  {
    krm_releaseFtypeEntry( Ftype_entry );
    return( NULL );     /*  memory alloc failed */
  }

  Ftype_entry->Ftype_symbol = n_ptr;
  Ftype_entry->out_func = out_func;
  Ftype_entry->act_func = act_func;
  Ftype_entry->act_deriv_func = act_deriv_func;
  Ftype_entry->sites = NULL;

  return( Ftype_entry );
}


/*  add a site to a previosly defined Ftype entry
*/
struct Site  *krm_FtypeAddSite( Ftype_entry , STable_entry )
struct  FtypeUnitStruct   *Ftype_entry;
struct  SiteTable         *STable_entry;
{
  struct  Site   *site_ptr;


  if ( (site_ptr = krm_getFtypeSite() ) == NULL)
    return( NULL );     /*  memory alloc failed */

  site_ptr->next = Ftype_entry->sites;
  Ftype_entry->sites = site_ptr;

  site_ptr->site_table = STable_entry;

  return( site_ptr );
}


/*  returns a pointer to first Ftype entry
*/
struct FtypeUnitStruct  *krm_getFtypeFirstEntry()
{
  curr_Ftype_entry = Ftype_list_root;
  return( Ftype_list_root );
}


/*  returns a pointer to next Ftype entry
*/
struct FtypeUnitStruct  *krm_getFtypeNextEntry()
{
  if (curr_Ftype_entry != NULL)  {
    if (curr_Ftype_entry->next != NULL)
      curr_Ftype_entry = curr_Ftype_entry->next;
    else
      return( NULL );
  }

  return( curr_Ftype_entry );
}


/*  searches for a Ftype entry with the given name
*/
struct  FtypeUnitStruct  *krm_FtypeSymbolSearch( Ftype_symbol )
char  *Ftype_symbol;
{
  struct  FtypeUnitStruct   *ftype_entry;


  if (Ftype_symbol == NULL)  return( NULL);

  ftype_entry = Ftype_list_root;
  while (ftype_entry != NULL)  {
    if (strcmp( Ftype_symbol, (ftype_entry->Ftype_symbol)->Entry.symbol ) == 0)
      return( ftype_entry );

    ftype_entry = ftype_entry->next;
  }

  return( NULL );
}


/*  releases all Ftype entries
*/
void    krm_releaseFtypeList()
{
  struct  FtypeUnitStruct   *Ftype_entry,
                            *ft_ptr;


  Ftype_entry = Ftype_list_root;

  while( Ftype_entry != NULL )  {
    /*  release sites first */
    if ( Ftype_entry->sites != NULL)
      krm_releaseAllFtypeSites( Ftype_entry->sites );

    /*  release Ftype symbol name */
    if ( Ftype_entry->Ftype_symbol != NULL )
      krm_NTableReleaseEntry( Ftype_entry->Ftype_symbol );

    ft_ptr = Ftype_entry;
    Ftype_entry = Ftype_entry->next;
    /*  release unit entry  */
    free( (char *) ft_ptr );       /*  release Ftype entry   */
  }

  Ftype_list_root = NULL;
  NoOfFTableEntries = 0;
}

#ifdef MASPAR_KERNEL
#ifdef MASPAR_KERNEL_EMULATION

/*#################################################

GROUP: Functions for the MasPar kernel

#################################################*/

/*  release weight arrays from memory
*/
void  krm_releaseWeightArrays()
{
  int  dest_layer, src_layer;
  FlintType  *weight_array;

  /*  get weight array pointers  */
  for (dest_layer = 1; dest_layer < descrFFnet.no_of_layers; dest_layer++)
    for (src_layer = 0; src_layer < dest_layer; src_layer++)  {
      weight_array = descrFFnet.layers[ dest_layer ].inputs[src_layer].weight_array;
      if (weight_array != NULL)  {
        free( (char *) weight_array );
	descrFFnet.layers[ dest_layer ].inputs[src_layer].weight_array = NULL;
	descrFFnet.layers[ dest_layer ].inputs[src_layer].no_of_inputs = 0;
      }
    }

  NoOfWeights = 0;
  specialNetworkType == NET_TYPE_GENERAL;
}


/*  creates arrays containing connection weights for feedforward networks
*/
krui_err  krm_createWeightArrays()
{
  int  dest_layer, src_layer, no_of_inputs;
  FlintType  *weight_array;

  KernelErrorCode = KRERR_NO_ERROR;
  NoOfWeights = 0;

  /*  allocate weight arrays  */
  for (dest_layer = 1; dest_layer < descrFFnet.no_of_layers; dest_layer++)
    for (src_layer = 0; src_layer < dest_layer; src_layer++)  {
      no_of_inputs = descrFFnet.layers[ dest_layer ].inputs[src_layer].no_of_inputs;
      if (no_of_inputs > 0)
        {  /*  there are <no_of_inputs> connections between layer <src_layer>
               and layer <dest_layer>.  */
	weight_array = (FlintType *) calloc( no_of_inputs, sizeof (FlintType) );
        if (weight_array == NULL)
          {  /*  insufficient memory  */
          KernelErrorCode = KRERR_INSUFFICIENT_MEM;
          krm_releaseWeightArrays();
          return( KernelErrorCode );
        }

	descrFFnet.layers[ dest_layer ].inputs[src_layer].weight_array = weight_array;
      }
    }

  NoOfWeights = descrFFnet.no_of_weights;
  return( KernelErrorCode );
}

#endif
#endif


/*#################################################

GROUP: Memory cleanup

#################################################*/

/*  frees all memory used for the internal representation of the network
*/
void    krm_releaseMem()
{
#ifdef MASPAR_KERNEL
#ifdef MASPAR_KERNEL_EMULATION

  if (specialNetworkType == NET_TYPE_FF1)
    krm_releaseWeightArrays();

#endif
#endif

  specialNetworkType = NET_TYPE_GENERAL;
  krm_releaseFtypeList();
  krm_releaseSTableArrays();
  krm_releaseNTableArrays();
  krm_releaseLinkArrays();
  krm_releaseSiteArrays();
  krm_releaseUnitArrays();
  krm_releasePatternArrays();
  krm_releasePatternNumbers();
  krm_releaseUnitTopoArray();
}


/*  ---------------------  End of Memory Management Functions  -------------  */
