/*     @(#)kr_ui.c	1.2 7/21/92	*/

/**********************************************************************
FILE   : kr_ui.c
PURPOSE: SNNS-Kernel User Interface
NOTES  : 
AUTHOR : Niels Mache
DATE   : 27.02.90
VERSION : 1.2  7/21/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <memory.h>

#include "random.h"	 /*  Randomize Library Function Prototypes  */
#include "kr_typ.h"	 /*  Kernel Types and Constants  */
#include "kr_const.h"	 /*  Constant Declarators for SNNS-Kernel  */
#include "kr_def.h"	 /*  Default Values  */

#include "kr_io.h"	 /*  Kernel I/O-Routines */
#include "kr_funcs.h"    /*  Function Prototypes */
#include "kernel.h"	 /*  Function Prototypes */
#include "kr_mem.h"      /*  Function Prototypes */
#include "kr_ui.h"	 /*  Interface function prototypes  */
#include "version.h"	 /*  Version and Patchlevel  */
#include "kr_mac.h"	 /*  Kernel Macros  */

#ifdef MASPAR_KERNEL

#include "kr_feedf.h"	 /*  Function Prototypes */

#endif

/*#################################################

GROUP: Global Var's (as declared by the kernel)

#################################################*/


extern bool NetModified;	 /*  TRUE, if the network topology was modified  */

extern int  NoOfUnits,		 /*  no. of units in the network  */
	    MinUnitNo,		 /*  the first (lowest) used unit number in the network  */
	    MaxUnitNo;		 /*  the last (highest) used unit number in the network  */

extern UnitArray  unit_array;	  /*  the unit array  */

/*  Pointers and numbers for storing the current unit, site or link.
    Used by unit/site/link searching routines.
*/
extern struct Unit  *unitPtr;
extern struct Site  *sitePtr, *prevSitePtr;
extern struct Link  *linkPtr, *prevLinkPtr;
extern int  unitNo;


extern int  specialNetworkType;  /*  stores the topologic type of a network
				     (needed for the parallel kernel)  */

/*  File I/O: Line number of the network file.
*/
extern  int  lineno;

/*  Stores the error codes and messages of the
    topologic sorting and network checking
    functions.
*/
extern struct TopologicMessages  topo_msg;


/*#################################################

GROUP: Local Var's

#################################################*/

static int  UICurrentNameSearchUnitNo = 0;
static struct Site  *UICurrentFtypeSite = NULL;
static struct FtypeUnitStruct  *UICurrentFtypeEntry = NULL;
static char  *UICurrentNameSearchUnitSymbolPtr = NULL;


/*#################################################

GROUP: Macros

#################################################*/

/*  Check operation for compatibility with MasPar parallel 
    network implementation
*/
#ifdef MASPAR_KERNEL

#define  MASPAR_FF1_VALIDATE_OP \
if (specialNetworkType == NET_TYPE_FF1)\
  { KernelErrorCode = KRERR_MODE_FF1_INVALID_OP;\
    return( KRERR_MODE_FF1_INVALID_OP ); }\

#endif


/*#################################################

GROUP: Unit Functions

#################################################*/


/*  Returns the number of units used by the network.
*/
int     krui_getNoOfUnits()
{
  return( NoOfUnits );
}


/*  Initializes the first available unit for access. If the unit has sites, the
    first site will be set to the current site.
    Returns the unit no. of the first unit or 0 if no units available.
*/
int  krui_getFirstUnit()
{
  return( kr_getUnit( FIRST ) );
}

/*  Initializes the next available unit for access. If the unit has sites, the
    first site will be set to the current site.
    Returns the unit no. of the next unit or 0 if no more units available.
*/
int  krui_getNextUnit()
{
  return( kr_getUnit( NEXT ) );
}


/*  Returns the no. of the current unit or 0 if no units available.
*/
int  krui_getCurrentUnit()
{
  return( kr_getUnit( CURRENT ) );
}


/*  Initializes a unit for access. If the unit has sites, the first site
    will be set to the current site.
    Returns error code if the given unit doesn't exist, 0 otherwise.
*/
krui_err  krui_setCurrentUnit( unit_no )
int     unit_no;
{
  return( kr_setCurrUnit( unit_no ) );
}


/*  Returns the name of the unit. (NULL if not available)
*/
char  *krui_getUnitName( UnitNo )
int  UnitNo;
{
  struct Unit   *unit_ptr;


  if ((unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return( NULL );    /*  invalid unit #  */

  return( unit_ptr->unit_name );
}


/*  Sets the name of the unit <UnitNo>.
    If the unit_name pointer is NULL, the unit's symbol will be deleted.
    Function has no effect on the current unit.
    Returns error code if memory allocation fails, 0 otherwise.
*/
krui_err  krui_setUnitName( unit_no, unit_name )
int     unit_no;
char    *unit_name;
{
  struct Unit   *unit_ptr;
  char    *str_ptr;


  KernelErrorCode = KRERR_NO_ERROR;
  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  if (unit_name == NULL)
    {
    krm_NTableReleaseSymbol( unit_ptr->unit_name, UNIT_SYM );
    unit_ptr->unit_name = NULL;
    return( KernelErrorCode );
  }

  if (!kr_symbolCheck( unit_name ))  return( KernelErrorCode );

  if ( (str_ptr = krm_NTableInsertSymbol( unit_name, UNIT_SYM ) ) == NULL)
    return( KernelErrorCode );

  unit_ptr->unit_name = str_ptr;

  return( KernelErrorCode );
}


/*  Searches for a unit with the given name.
    Returns the first unit no. if a unit with the given name was found,
    0 otherwise.

    Returns error code if no units defined.
*/
int  krui_searchUnitName( unit_name )
char  *unit_name;
{
  struct  NameTable  *n_ptr;

  if (NoOfUnits <= 0)  {
    UICurrentNameSearchUnitNo = 0;
    return( (int) KRERR_NO_UNITS );  /*  No Units defined  */
  }

  if ((n_ptr = krm_NTableSymbolSearch( unit_name, UNIT_SYM )) == NULL)  {
    UICurrentNameSearchUnitNo = 0;
    return( 0 );
  }

  UICurrentNameSearchUnitSymbolPtr = n_ptr->Entry.symbol;
  UICurrentNameSearchUnitNo = kr_unitNameSearch( MinUnitNo, UICurrentNameSearchUnitSymbolPtr );

  return( UICurrentNameSearchUnitNo );
}



/*  Searches for the next unit with the given name.
    Returns the next unit no. if a unit with the given name was found,
    0 otherwise.

NOTE: Call krui_searchUnitName( unit_name ) before calling krui_searchNextUnitName().

    Returns error code if no units defined.
*/
int  krui_searchNextUnitName()
{
  if (NoOfUnits <= 0)
    {
    UICurrentNameSearchUnitNo = 0;
    return( KRERR_NO_UNITS );  /*  No Units defined  */
  }

  if (UICurrentNameSearchUnitNo != 0)
    {
    UICurrentNameSearchUnitNo = kr_unitNameSearch( UICurrentNameSearchUnitNo + 1, UICurrentNameSearchUnitSymbolPtr );
    return( UICurrentNameSearchUnitNo );
  }
  else
    return( 0 );
}


/*  Returns the output function name of the unit.
*/
char  *krui_getUnitOutFuncName( UnitNo )
int  UnitNo;
{
  struct Unit   *unit_ptr;
  static char  out_func_name[FUNCTION_NAME_MAX_LEN];

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return( NULL );    /*  invalid unit #  */

  strcpy( out_func_name, krf_getFuncName( (FunctionPtr) unit_ptr->out_func ));

  return( out_func_name );
}


/*  The unit's FType will be set to 0, i.e. the
    unit's functionality type will be deleted.
    Function has no effect on the current unit.
NOTE: returns 0, if the function is a valid output function,
      error code otherwise.
*/
krui_err  krui_setUnitOutFunc( unit_no, unitOutFuncName )
int  unit_no;
char  *unitOutFuncName;
{
  struct Unit   *unit_ptr;
  FunctionPtr func_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  if ( !krf_funcSearch( unitOutFuncName, OUT_FUNC, &func_ptr ))
    return( KernelErrorCode );

  unit_ptr->out_func = (OutFuncPtr) func_ptr;
  unit_ptr->Ftype_entry = NULL;

  NetModified = TRUE;

  return( KRERR_NO_ERROR );
}


/*  Returns the activation function name of the unit.
*/
char  *krui_getUnitActFuncName( UnitNo )
int  UnitNo;
{
  struct Unit   *unit_ptr;
  static char  act_func_name[FUNCTION_NAME_MAX_LEN];


  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return( NULL );    /*  invalid unit #  */

  strcpy( act_func_name, krf_getFuncName( (FunctionPtr) unit_ptr->act_func ));

  return( act_func_name );
}


/*  The unit's FType will be set to 0, i.e. the
    unit's functionality type will be deleted.
    Function has no effect on the current unit.
NOTE: returns 0, if the function is a valid activation function,
      error code otherwise.
*/
krui_err  krui_setUnitActFunc( unit_no, unitActFuncName )
int  unit_no;
char  *unitActFuncName;
{
  struct Unit   *unit_ptr;
  FunctionPtr   act_func_ptr, act_deriv_func_ptr;

#ifdef MASPAR_KERNE
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );
  if ( !krf_funcSearch( unitActFuncName, ACT_FUNC, &act_func_ptr ))
    return( KernelErrorCode );
  /*  set the derivation function of the activation function  */ 
  if ( !krf_funcSearch( unitActFuncName, ACT_DERIV_FUNC, &act_deriv_func_ptr ))
    return( KernelErrorCode );


  unit_ptr->act_func = (ActFuncPtr) act_func_ptr;
  unit_ptr->act_deriv_func = (ActDerivFuncPtr) act_deriv_func_ptr;
  unit_ptr->Ftype_entry = NULL;

  NetModified = TRUE;

  return( KRERR_NO_ERROR );
}


/*  Returns the functionality type name of the unit.
    Function has no effect on the current unit.
    Returns NULL if unit has no FType.
*/
char  *krui_getUnitFTypeName( UnitNo )
int  UnitNo;
{
  struct FtypeUnitStruct    *Ftype_entry;
  struct Unit   *unit_ptr;


  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return( NULL );    /*  invalid unit #  */

  if ( (Ftype_entry = unit_ptr->Ftype_entry) == NULL)
    return( NULL );

  return( Ftype_entry->Ftype_symbol->Entry.symbol );
}





/*  Returns the activation value of the unit.
    Function has no effect on the current unit.
*/
FlintType  krui_getUnitActivation( UnitNo )
int  UnitNo;
{
  if KERNEL_STANDARD {
    return( kr_getUnitValues( UnitNo, SEL_UNIT_ACT ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getUnitValues( UnitNo, SEL_UNIT_ACT ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( 0.0 );

#endif
  }
}



/*  Sets the activation value of the unit.
    Function has no effect on the current unit.
*/
void  krui_setUnitActivation( UnitNo, unit_activation )
int  UnitNo;
FlintTypeParam  unit_activation;
{
  if KERNEL_STANDARD  {
    (void) kr_setUnitValues( UnitNo, SEL_UNIT_ACT, unit_activation );
  }
  else  {

#ifdef MASPAR_KERNEL

    (void) krff_setUnitValues( UnitNo, SEL_UNIT_ACT, unit_activation );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;

#endif
  }
}



/*  Returns the initial activation value of the unit.
    Function has no effect on the current unit.
*/
FlintType  krui_getUnitInitialActivation( UnitNo )
int  UnitNo;
{
  if KERNEL_STANDARD
    return( kr_getUnitValues( UnitNo, SEL_UNIT_IACT ) );
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getUnitValues( UnitNo, SEL_UNIT_IACT ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( 0.0 );

#endif
  }
}


/*  Sets the initial activation value of the unit.
    Function has no effect on the current unit.
*/
void  krui_setUnitInitialActivation( UnitNo, unit_i_activation )
int  UnitNo;
FlintTypeParam  unit_i_activation;
{
  if KERNEL_STANDARD
    kr_setUnitValues( UnitNo, SEL_UNIT_IACT, unit_i_activation );
  else  {

#ifdef MASPAR_KERNEL

    krff_setUnitValues( UnitNo, SEL_UNIT_IACT, unit_i_activation );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;

#endif
  }
}



/*  Returns the output value of the unit.
    Function has no effect on the current unit.
*/
FlintType  krui_getUnitOutput( UnitNo )
int  UnitNo;
{
  if KERNEL_STANDARD
    return( kr_getUnitValues( UnitNo, SEL_UNIT_OUT ) );
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getUnitValues( UnitNo, SEL_UNIT_OUT ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( 0.0 );

#endif
  }
}


/*  Sets the output value of the unit.
    Function has no effect on the current unit.
*/
krui_err  krui_setUnitOutput( unit_no, unit_output )
int  unit_no;
FlintTypeParam  unit_output;
{
  if KERNEL_STANDARD  {
    kr_setUnitValues( unit_no, SEL_UNIT_OUT, unit_output );
  }
  else  {

#ifdef MASPAR_KERNEL

    krff_setUnitValues( unit_no, SEL_UNIT_OUT, unit_output );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;

#endif
  }

  return( KernelErrorCode );
}



/*  Returns the bias value of the unit.
    Function has no effect on the current unit.
*/
FlintType  krui_getUnitBias( UnitNo )
int  UnitNo;
{
  if KERNEL_STANDARD  {
    return( kr_getUnitValues( UnitNo, SEL_UNIT_BIAS ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getUnitValues( UnitNo, SEL_UNIT_BIAS ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( 0.0 );

#endif
  }
}



/*  Sets the bias value of the unit.
    Function has no effect on the current unit.
*/
void  krui_setUnitBias( UnitNo, unit_bias )
int  UnitNo;
FlintTypeParam  unit_bias;
{
  if KERNEL_STANDARD  {
    kr_setUnitValues( UnitNo, SEL_UNIT_BIAS, unit_bias );
  }
  else  {

#ifdef MASPAR_KERNEL

    krff_setUnitValues( UnitNo, SEL_UNIT_BIAS, unit_bias );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;

#endif
  }
}



/*  Returns the subnet number of the unit.
    Function has no effect on the current unit.
NOTE: The range of the subnet no. is -32736...+32735
*/
int  krui_getUnitSubnetNo( UnitNo )
int  UnitNo;
{
  struct Unit   *unit_ptr;

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return( 0 );    /*  invalid unit #  */

  return( (int) unit_ptr->subnet_no );
}

/*  Sets the subnet number of the unit.
    Function has no effect on the current unit.
NOTE: The range of the subnet no. is -32736...+32735
*/
void  krui_setUnitSubnetNo( UnitNo, subnet_no)
int   UnitNo,
      subnet_no;
{
  struct Unit   *unit_ptr;

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return;    /*  invalid unit #  */

  NetModified = TRUE;

  unit_ptr->subnet_no = (short) subnet_no;
}


/*  Returns the layer number of the unit.
    Function has no effect on the current unit.
NOTE: The range of the layer no. is -32736...+32735
*/
unsigned short	krui_getUnitLayerNo( UnitNo )
int     UnitNo;
{
  struct Unit   *unit_ptr;

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return( 0 );    /*  invalid unit #  */

  return( (unsigned short) unit_ptr->layer_no );
}

/*  Sets the layer number of the unit.
    Function has no effect on the current unit.
NOTE: The range of the layer no. is -32736...+32735
*/
void    krui_setUnitLayerNo( UnitNo, layer_no)
int     UnitNo;
int     layer_no;
{
  struct Unit   *unit_ptr;

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return;    /*  invalid unit #  */

  NetModified = TRUE;

  unit_ptr->layer_no = (unsigned short) layer_no;
}



/*  Returns the position of the unit.
    Function has no effect on the current unit.
*/
void    krui_getUnitPosition( UnitNo, position )
int     UnitNo;
struct  PosType *position;
{
  struct Unit   *unit_ptr;

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return;    /*  invalid unit #  */

  position->x = unit_ptr->unit_pos.x;
  position->y = unit_ptr->unit_pos.y;
#ifdef KERNEL3D
  position->z = unit_ptr->unit_pos.z;
#endif
}

/*  Sets the position of the unit.
    Function has no effect on the current unit.
*/
void    krui_setUnitPosition( UnitNo, position )
int     UnitNo;
struct PosType  *position;
{
  struct Unit   *unit_ptr;

  if ( (unit_ptr = kr_getUnitPtr( UnitNo ) ) == NULL)
    return;    /*  invalid unit #  */

  unit_ptr->unit_pos.x = position->x;
  unit_ptr->unit_pos.y = position->y;
#ifdef KERNEL3D
  unit_ptr->unit_pos.z = position->z;
#endif
}


/*  Returns the unit no. at the given position and subnet number or 0, if
    no unit exists at this position.
    Function has no effect on the current unit.
NOTE: This function is slow because it uses linear search to
      find the unit with the given position.

REMARK: getUnitNoAtPosition is for downward compatibility only.
        Do not use this function in future applications.
*/
int   krui_getUnitNoAtPosition( position, subnet_no )
struct PosType  *position;
int  subnet_no;
{
  register int       i;
  register short     x, y, net_no;
  struct Unit     *unit_ptr;


  x = position->x;
  y = position->y;

  net_no = (short) subnet_no;

  for (i = 1, unit_ptr = unit_array + 1;
       i <= MaxUnitNo;
       i++, unit_ptr++)
    {
    if ( UNIT_IN_USE( unit_ptr ) &&
         (unit_ptr->subnet_no == net_no) &&
         (unit_ptr->unit_pos.x == x) && (unit_ptr->unit_pos.y == y) )

      return( i );
    }

  return( 0 );
}


/*  Returns the unit no. near the given position and the specified
    subnet no or 0, if no unit exists at this position.
    Function has no effect on the current unit.
               
    NOTE: This function is slow because it uses linear search to
          find the unit with the given position.
*/
int  krui_getUnitNoNearPosition( position, subnet_no, range, gridWidth )
struct PosType  *position;
int   subnet_no, range, gridWidth;
{
  register int       i, devit, width;
  register short     x, y, net_no;
  struct Unit        *unit_ptr;


  x = position->x;
  y = position->y;
  net_no = (short) subnet_no;
  devit = range;
  width = gridWidth;

  for (i = 1, unit_ptr = unit_array + 1; i <= MaxUnitNo;
       i++, unit_ptr++)
    {
    if ( UNIT_IN_USE( unit_ptr ) &&
         (unit_ptr->subnet_no == net_no) &&
         (abs( unit_ptr->unit_pos.x - x) * width <= devit) && 
         (abs( unit_ptr->unit_pos.y - y) * width <= devit) )
      return( i );
    }

  return( 0 );
}



/*  Returns the topologic type, i.e. input, output, hidden
    Function has no effect on the current unit.
*/
int  krui_getUnitTType( unit_no )
int  unit_no;
{
  struct Unit   *unit_ptr;

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  return( kr_flags2TType( (int) unit_ptr->flags & UFLAG_TTYP_PAT ) );
}


/*  Sets the topologic type of the unit.
    Function has no effect on the current unit.
    Returns error code if topologic type or unit number is invalid.
*/
krui_err  krui_setUnitTType( unit_no, UnitTType )
int  unit_no,
     UnitTType;
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  return( kr_unitSetTType( unit_no, UnitTType ) );
}


/*  Freezes the output of a unit, i.e. the unit is disabled.
    Function has no effect on the current unit.
*/
krui_err  krui_freezeUnit( unit_no )
int  unit_no;
{
  struct Unit   *unit_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  unit_ptr->flags &= (~UFLAG_ENABLED);
  return( KRERR_NO_ERROR );
}


/*  Enables a previosly freezed unit.
    Function has no effect on the current unit.
*/
krui_err  krui_unfreezeUnit( unit_no )
int  unit_no;
{
  struct Unit   *unit_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );
 
  unit_ptr->flags |= UFLAG_ENABLED;
  return( KRERR_NO_ERROR );
}


/*  Returns true, if unit is frozen
*/
bool  krui_isUnitFrozen( unit_no )
int  unit_no;
{
  struct Unit   *unit_ptr;

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  return( (unit_ptr->flags & UFLAG_ENABLED) == 0 );
}



/*  Returns the input type of the unit:
     NO_INPUTS    : if the unit has not inputs (at least not now)
     SITES        : if the unit has one or more sites (and no direct input links !)
     DIRECT_LINKS : if the unit has direct input links (and no sites !)

    Function has no effect on the current unit.
*/
int  krui_getUnitInputType( unit_no )
int  unit_no;
{
  struct Unit  *unit_ptr;

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  switch ((int) unit_ptr->flags & UFLAG_INPUT_PAT)
    {
    case UFLAG_NO_INP :  return( NO_INPUTS    );
    case UFLAG_SITES  :  return( SITES        );
    case UFLAG_DLINKS :  return( DIRECT_LINKS );

    default :  return( NO_INPUTS );
  }
}


/*  *************************************   */

/*  creates a unit with default properties:
    1. default activation and output functions
    2. default activation and bias
    3. default position, subnet and layer number
    4. no functionality type
    5. no sites
    6. no inputs and outputs
    7. no unit_name

    Returns an (negative) error code, if memory allocation fails or
    invalid functions occured. Returns (positive) unit number otherwise.
    Function has no effect on the current unit.

NOTE: See file "kr_def.h" for more details about default presettings.
*/
int  krui_createDefaultUnit()
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  NetModified = TRUE;
  return( kr_makeDefaultUnit() );
}


/*  Creates a user defined unit.
    Returns an (negative) error code, if memory allocation fails or
    invalid functions occured. Returns (positive) unit number otherwise.
    Function has no effect on the current unit.

    Unit has additional default settings:
    1. default position, subnet and layer number
    2. no functionality type
    3. no sites
    4. no inputs and outputs

*/
int  krui_createUnit( unit_name, out_func_name, act_func_name,
		      i_act, bias)
char  *unit_name, *out_func_name, *act_func_name;
FlintTypeParam   i_act, bias;
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  return( kr_createUnit( unit_name, out_func_name, act_func_name, i_act, bias) );
}


/*  Deletes a unit from network. Removes all links to other
    units. Returns error code if unit doesn't exist.
*/
krui_err  krui_deleteUnit( unit_no )
int unit_no;
{
  struct  Unit      *unit_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  (void) kr_removeUnit( unit_ptr );
  NetModified = TRUE;
  return( KernelErrorCode );
}


/*  creates a unit with FType properties, but:
    1. no inputs and outputs
    2. default position, subnet and layer

    Returns the unit number or (negative) error code if memory allocation
    fails or functionality type isn't defined.
    Function has no effect on the current unit.
*/
int  krui_createFTypeUnit( Ftype_symbol )
char  *Ftype_symbol;
{
  int     unit_no;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  unit_no = kr_makeFtypeUnit( Ftype_symbol );
  if (KernelErrorCode != KRERR_NO_ERROR)
    return( KernelErrorCode );

  NetModified = TRUE;
  return( unit_no );
}



/*  Changes the properties of unit <UnitNo> to FType properties.
    Changes: activation/output function and site name/functions.

    Returns an error code if
     - FType name doesn't exist or
     - unit doesn't exist or
     - memory allocation fails
     otherwise 0.

    Function has no effect on the current unit.
*/
krui_err  krui_setUnitFType( unit_no, Ftype_symbol )
int   unit_no;
char  *Ftype_symbol;
{
  struct  Unit      *unit_ptr;
  struct  FtypeUnitStruct   *ftype_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );
  if (!kr_symbolCheck( Ftype_symbol ))
    return( KernelErrorCode );

  if ( (ftype_ptr = krm_FtypeSymbolSearch( Ftype_symbol ) ) == NULL)
    {  /*  Ftype name isn't defined    */
    KernelErrorCode = KRERR_FTYPE_SYMBOL;
    return( KernelErrorCode );
  }

  NetModified = TRUE;
  kr_changeFtypeUnit( unit_ptr, ftype_ptr );
  return( KRERR_NO_ERROR );
}



/*  Copy a given unit, according to the copy mode
        1. copy unit (with it sites, if available) and input/output links
        2. copy unit (with it sites, if available) and input links
        3. copy unit (with it sites, if available) and output links
        4. copy unit (with it sites, if available) but no input/output links

    Returns the unit number of the new unit or error message < 0 , if errors occured.
    Function has no effect on the current unit.

NOTE: Copying of output links is slow.
      If return code < 0, an error occured.
*/
int   krui_copyUnit( unit_no, copy_mode )
int   unit_no, copy_mode;
{
  int   new_unit_no;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  new_unit_no = kr_copyUnit( copy_mode, unit_no );
  if (KernelErrorCode != KRERR_NO_ERROR)
    return( KernelErrorCode );

  return( new_unit_no );
}



/*#################################################

GROUP: Functions for manipulation of the Unit-Functionality-List
REMEMBER: The Unit-Functionality-List stores:
            1. unit activation and output functions
            2. if sites:  3. all site functions
                          4. all site names


#################################################*/

/*  Initializes the first FType entry.
    Returns true, if an entry is available
*/
bool  krui_setFirstFTypeEntry()
{
  UICurrentFtypeEntry = krm_getFtypeFirstEntry();
  UICurrentFtypeSite = NULL;
  return( UICurrentFtypeEntry != NULL );
}

/*  Initializes the next FType entry.
    Returns true, if an entry is available
*/
bool  krui_setNextFTypeEntry()
{
  struct  FtypeUnitStruct   *ftype_ptr;


  UICurrentFtypeSite = NULL;
  if ( (ftype_ptr = krm_getFtypeNextEntry() ) != NULL)
    {
    UICurrentFtypeEntry = ftype_ptr;
    return( TRUE );
    }

  return( FALSE );
}


/*  Initializes the FType entry with the given name.
    Returns true, if an entry with this name is available.
*/
bool  krui_setFTypeEntry( Ftype_symbol )
char  *Ftype_symbol;
{
  struct  FtypeUnitStruct   *ftype_ptr;


  if ( (ftype_ptr = krm_FtypeSymbolSearch( Ftype_symbol ) ) != NULL)
    {
    UICurrentFtypeEntry = ftype_ptr;
    return( TRUE );
    }

  return( FALSE );
}


/*  Returns the name of the current FType entry.
NOTE: The FType name is definite and will never be
      a NULL pointer.
*/
char  *krui_getFTypeName()
{
  if (UICurrentFtypeEntry == NULL)
    return( NULL );

  return( UICurrentFtypeEntry->Ftype_symbol->Entry.symbol );
}


/*  Sets the name of the current FType entry.
    Returns error code if memory allocation fails or Ftype name
    isn't definite, 0 otherwise.
NOTE: The new FType name have to be definite and must not be
      a NULL pointer.
*/
krui_err  krui_setFTypeName( Ftype_symbol )
char  *Ftype_symbol;
{
  struct NameTable    *NTable_ptr;


  if (UICurrentFtypeEntry == NULL)
    return( KRERR_FTYPE_ENTRY );   /*  Current Ftype entry isn't defined   */

  if (Ftype_symbol == NULL)
    return( KRERR_FTYPE_NAME );    /*  Ftype name isn't definite   */

  if (!kr_symbolCheck( Ftype_symbol ))
    return( KRERR_SYMBOL );  /*  Symbol pattern invalid (must match [A-Za-z]^[|, ]*)  */

  if (krm_NTableSymbolSearch( Ftype_symbol, FTYPE_UNIT_SYM ) != NULL)
    return( KRERR_FTYPE_NAME );    /*  Ftype name isn't definite   */

  if ( (NTable_ptr = krm_NTableCreateEntry( Ftype_symbol, FTYPE_UNIT_SYM ) ) == NULL)
    return( KRERR_INSUFFICIENT_MEM );  /*  Insufficient memory  */

  krm_NTableReleaseEntry( UICurrentFtypeEntry->Ftype_symbol );

  UICurrentFtypeEntry->Ftype_symbol = NTable_ptr;

  return( KRERR_NO_ERROR );
}


/*  Returns the name of the activation function in the current
    FType entry.
*/
char  *krui_getFTypeActFuncName()
{
  static char  act_func_name[FUNCTION_NAME_MAX_LEN];


  if (UICurrentFtypeEntry == NULL)
    return( NULL );

  strcpy( act_func_name, krf_getFuncName( (FunctionPtr) UICurrentFtypeEntry->act_func ) );

  return( act_func_name );
}


/*  Sets the activation function in the current FType entry
    returns 0, if the function is a valid activation function,
    error code otherwise.
    All units (in the existing network) of the current Ftype changes
    their activation function.
*/
krui_err  krui_setFTypeActFunc( act_func_name )
char   *act_func_name;
{
  FunctionPtr  act_func_ptr, act_deriv_func_ptr;


#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  KernelErrorCode = KRERR_NO_ERROR;
  if (UICurrentFtypeEntry == NULL)
    {  /*  Current Ftype entry isn't defined   */
    KernelErrorCode = KRERR_FTYPE_ENTRY;
    return( KernelErrorCode );
  }

  if ( !krf_funcSearch( act_func_name, ACT_FUNC, &act_func_ptr ) )
    return( KernelErrorCode );
  /*  set the derivation function of the activation function  */ 
  if ( !krf_funcSearch( act_func_name, ACT_DERIV_FUNC, &act_deriv_func_ptr ))
    return( KernelErrorCode );

  UICurrentFtypeEntry->act_func = (ActFuncPtr) act_func_ptr;
  UICurrentFtypeEntry->act_deriv_func = (ActDerivFuncPtr) act_deriv_func_ptr;

  kr_changeFtypeUnits( UICurrentFtypeEntry );
  return( KernelErrorCode );
}


/*  Returns the name of the output function in the current
    FType entry.
*/
char  *krui_getFTypeOutFuncName()
{
  static char  out_func_name[FUNCTION_NAME_MAX_LEN];

  if (UICurrentFtypeEntry == NULL)
    return( NULL );

  strcpy( out_func_name, krf_getFuncName( (FunctionPtr) UICurrentFtypeEntry->out_func ) );

  return( out_func_name );
}




/*  Sets the output function in the current FType entry
    Returns 0, if the function is a valid output function,
    error code otherwise.
    All units (in the existing network) of the current Ftype changes
    their output function.
*/
krui_err   krui_setFTypeOutFunc( out_func_name )
char   *out_func_name;
{
  FunctionPtr	 out_func_ptr;


  KernelErrorCode = KRERR_NO_ERROR;
  if (UICurrentFtypeEntry == NULL)
    {  /*  Current Ftype entry isn't defined   */
    KernelErrorCode = KRERR_FTYPE_ENTRY;
    return( KernelErrorCode );
  }

  if ( !krf_funcSearch( out_func_name, OUT_FUNC, &out_func_ptr ) )
    return( KernelErrorCode );

  UICurrentFtypeEntry->out_func = (OutFuncPtr) out_func_ptr;

  kr_changeFtypeUnits( UICurrentFtypeEntry );
  return( KernelErrorCode );
}



/*  FType sites
*/

/*  Initializes the first site of the current FType.
    Returns FALSE, if no sites are available in the current
    FType entry.
*/
bool  krui_setFirstFTypeSite()
{
  if (UICurrentFtypeEntry == NULL)  {
    UICurrentFtypeSite = NULL;
    return( FALSE );
  }

  if (UICurrentFtypeEntry->sites == NULL)  {
    UICurrentFtypeSite = NULL;
    return( FALSE );
  }
  else  {
    UICurrentFtypeSite = UICurrentFtypeEntry->sites;
    return( TRUE );
  }
}


/*  Initializes the next FType site.
    Returns FALSE, if no more sites are available in the current
    FType entry.
*/
bool  krui_setNextFTypeSite()
{
  if (UICurrentFtypeSite == NULL)
    return( FALSE );

  if (UICurrentFtypeSite->next == NULL)
    return( FALSE );

  UICurrentFtypeSite = UICurrentFtypeSite->next;
  return( TRUE );
}


/*  Returns the name of the current FType site (in the current
    Ftype entry).
*/
char  *krui_getFTypeSiteName()
{
  if (UICurrentFtypeSite == NULL)
    return( NULL );

  return( ((UICurrentFtypeSite->site_table)->Entry.site_name)->Entry.symbol );
}



/*  Sets the name and function of the current FType site (in the
    current FType entry).
    All sites (in the existing network) of the current Ftype and the
    same (old) name changes their names and site functions.

    Returns an error code if
      - current Ftype site isn't defined
      - site name does not exist in the site name table
      0 otherwise.
*/
krui_err  krui_setFTypeSiteName( FType_site_name )
char  *FType_site_name;
{
  struct  SiteTable     *stbl_ptr;


#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  KernelErrorCode = KRERR_NO_ERROR;
  if (UICurrentFtypeEntry == NULL)
    {  /*  Current Ftype entry isn't defined   */
    KernelErrorCode = KRERR_FTYPE_ENTRY;
    return( KernelErrorCode );
  }

  if (!kr_symbolCheck( FType_site_name ))
    return( KernelErrorCode );

  if ( (stbl_ptr = krm_STableSymbolSearch( FType_site_name )) == NULL)
    {  /*   site name isn't defined */
    KernelErrorCode = KRERR_UNDEF_SITE_NAME;
    return( KernelErrorCode );
  }

  kr_changeFtypeSites( UICurrentFtypeEntry, UICurrentFtypeSite->site_table, stbl_ptr);
  UICurrentFtypeSite->site_table = stbl_ptr;

  return( KernelErrorCode );
}



/*  Create a new functionality type, needs a definite FType symbol,
    the unit output and activation function
    and the number of sites provided for this unit FType.
    An additional array with N elements of pointers to site names
    is required for the definition of the sites.

    Returns error code if:
     - memory allocation fails
     - FType name isn't definite (symbol is used for another FType
       or symbol is a NULL pointer)
     - one or more site names are undefined

     0 otherwise.

NOTE: The number of Ftype entries and the number of sites per Ftype is
      only limited by the size of system memory.
*/
krui_err  krui_createFTypeEntry( Ftype_symbol, act_func_name, out_func_name,
				 no_of_sites, array_of_site_names )
char  *Ftype_symbol,
      *out_func_name,
      *act_func_name;
int   no_of_sites;
char  * *array_of_site_names;
{
  FunctionPtr	act_func, out_func, act_deriv_func;
  struct  FtypeUnitStruct   *Ftype_entry;
  struct  SiteTable         *STable_entry;
  int       i;
  char      *Ftype_site_name;


  KernelErrorCode = KRERR_NO_ERROR;

  if (Ftype_symbol == NULL)
    {  /*  Ftype name isn't definite   */
    KernelErrorCode = KRERR_FTYPE_NAME;
    return( KernelErrorCode );
  }

  if (!kr_symbolCheck( Ftype_symbol ))
    return( KernelErrorCode );
  if (krm_NTableSymbolSearch( Ftype_symbol, FTYPE_UNIT_SYM ) != NULL)
    return( KernelErrorCode );
  if ( !krf_funcSearch( out_func_name, OUT_FUNC, &out_func ) )
    return( KernelErrorCode );
  if ( !krf_funcSearch( act_func_name, ACT_FUNC, &act_func ) )
    return( KernelErrorCode );
  /*  set the derivation function of the activation function  */ 
  if ( !krf_funcSearch( act_func_name, ACT_DERIV_FUNC, &act_deriv_func ))
    return( KernelErrorCode );

  if ((Ftype_entry = krm_FtypeCreateEntry( Ftype_symbol, (OutFuncPtr) out_func, (ActFuncPtr) act_func,
					   (ActDerivFuncPtr) act_deriv_func )) == NULL)
    return( KernelErrorCode );

  /*  create sites at the Ftype  */
  for (i = 0; i < no_of_sites; i++)  {
    if ( (Ftype_site_name = array_of_site_names[ i ]) == NULL)
      {  /*   site name isn't defined because it is a NULL pointer  */
      KernelErrorCode = KRERR_UNDEF_SITE_NAME;
      return( KernelErrorCode );
    }

    if ((STable_entry = krm_STableSymbolSearch( Ftype_site_name )) == NULL)
      {  /*  site isn't defined  */
      krm_releaseFtypeEntry( Ftype_entry );
      KernelErrorCode = KRERR_UNDEF_SITE_NAME;
      return( KernelErrorCode );
    }

    if (krm_FtypeAddSite( Ftype_entry , STable_entry ) == NULL)
      {  /*  memory alloc failed */
      krm_releaseFtypeEntry( Ftype_entry );
      return( KernelErrorCode );
    }
  }

  return( KernelErrorCode );
}


/*  Deletes the specified FType entry. If there exists units in the
    network with this FType, all these units will lost their FType
    but the functionality of the units will not be changed.
    Returns error code if FType symbol dosn't exist, 0 otherwise.
*/
krui_err    krui_deleteFTypeEntry( Ftype_symbol )
char   *Ftype_symbol;
{
  struct  FtypeUnitStruct   *ftype_ptr;


  if ( (ftype_ptr = krm_FtypeSymbolSearch( Ftype_symbol ) ) == NULL)
    return( KRERR_FTYPE_SYMBOL );   /*  FType symbol isn't defined  */

  kr_deleteUnitsFtype( ftype_ptr );
  krm_releaseFtypeEntry( ftype_ptr );
  return( KRERR_NO_ERROR );
}



/*#################################################

GROUP: Functions for reading of the function table

#################################################*/

/*  Returns the number of available functions
*/
int  krui_getNoOfFunctions()
{
  return( krf_getNoOfFuncs() );
}


/*  Returns the name of the function and the function type (Output,
    Activation, Site).
*/
void  krui_getFuncInfo( func_no, func_name, func_type )
int  func_no;
char  * *func_name;
int  *func_type;
{
  static struct FuncInfoDescriptor  functionDescr;

  functionDescr.number = func_no - 1;
  KernelErrorCode = krf_getFuncInfo( GET_FUNC_INFO, &functionDescr );

  *func_type = functionDescr.func_type;
  *func_name = functionDescr.func_name;


}


/*  Returns true if the given function name and type exists.
*/
bool  krui_isFunction( func_name, func_type )
char  *func_name;
int   func_type;
{
  FunctionPtr  dummy_func_ptr;
  bool	is_func;

  is_func = krf_funcSearch( func_name, func_type, &dummy_func_ptr );
  KernelErrorCode = KRERR_NO_ERROR;
  return( is_func );
}


/*  Returns the no. of input and output parameters of the given
    function (only relevant for learning, update and initialisation
    functions).

    Returns TRUE if the given function exists, FALSE otherwise.
*/
bool  krui_getFuncParamInfo( func_name, func_type,
                             no_of_input_params,
                             no_of_output_params )
char *func_name;
int  func_type;
int *no_of_input_params, *no_of_output_params;
{
  static struct FuncInfoDescriptor  functionDescr;

  functionDescr.func_type = func_type;
  strcpy( functionDescr.func_name, func_name );

  KernelErrorCode = krf_getFuncInfo( SEARCH_FUNC, &functionDescr );

  if (KernelErrorCode != KRERR_NO_ERROR)  return( FALSE );

  *no_of_input_params = functionDescr.no_of_input_parameters;
  *no_of_output_params = functionDescr.no_of_output_parameters;
  return( TRUE );
}



/*#################################################

GROUP: Site Table Functions

#################################################*/


/*  Returns the first site name/function pair in the site table.
    Returns FALSE and NULL, if not available.
*/
bool   krui_getFirstSiteTableEntry( site_name, site_func )
char   * *site_name,
       * *site_func;
{
  struct  SiteTable   *s_ptr;


  if ( (s_ptr = krm_getSTableFirstEntry() ) == NULL)
    {
    *site_name = NULL;
    *site_func = NULL;
    return( FALSE );
    }
  else
    {
    *site_name = (s_ptr->Entry.site_name)->Entry.symbol;
    *site_func = krf_getFuncName( (FunctionPtr) s_ptr->site_func );
    return( TRUE );
    }
}



/*  Returns the next site name/function pair in the site table.
    Returns FALSE and NULL, if not available.
*/
bool  krui_getNextSiteTableEntry( site_name, site_func )
char  * *site_name,
      * *site_func;
{
  struct  SiteTable   *s_ptr;


  if ( (s_ptr = krm_getSTableNextEntry() ) == NULL)
    {
    *site_name = NULL;
    *site_func = NULL;
    return( FALSE );
    }
  else
    {
    *site_name = (s_ptr->Entry.site_name)->Entry.symbol;
    *site_func = krf_getFuncName( (FunctionPtr) s_ptr->site_func );
    return( TRUE );
    }
}



/*  Returns the name of the site function that is associated with
    the site name.
    If the site name do not exist, function returns NULL.
*/
char  *krui_getSiteTableFuncName( site_name )
char    *site_name;
{
  struct SiteTable  *s_ptr;


  if ( (s_ptr = krm_STableSymbolSearch( site_name )) == NULL)
    return( NULL );

  return( krf_getFuncName( (FunctionPtr) s_ptr->site_func ) );
}



/*  Creates a new site name and associate this name with a site
    function.
    Returns error code if:
     - site name already exists or
     - site function is invalid or
     - memory allocation has failed
    0 otherwise.
*/
krui_err  krui_createSiteTableEntry( site_name, site_func )
char    *site_name,
        *site_func;
{
  FunctionPtr   func_ptr;

  KernelErrorCode = KRERR_NO_ERROR;
  if ( !krf_funcSearch( site_func, SITE_FUNC, &func_ptr ) )
    return( KernelErrorCode );
  if (!kr_symbolCheck( site_name ))
    return( KernelErrorCode );
  if (krm_STableSymbolSearch( site_name ) != NULL)
    {  /*  symbol is already in the site table */
    KernelErrorCode = KRERR_REDEF_SITE_NAME;
    return( KernelErrorCode );
  }

  (void) krm_STableCreateEntry( site_name, (SiteFuncPtr) func_ptr );

  return( KernelErrorCode );
}


/*  Changes the site function of a previously defined site name.
    Returns error code if <old_site_name> or <new_site_func>
    isn't defined, 0 otherwise.

NOTE: All sites in the network with the name <old_site_name>
      changes their names and functions.
*/
krui_err  krui_changeSiteTableEntry( old_site_name, new_site_name, new_site_func )
char  *old_site_name,
      *new_site_name,
      *new_site_func;
{
  FunctionPtr   func_ptr;
  SiteFuncPtr	    site_func_ptr;
  struct SiteTable  *stbl_ptr1,
                    *stbl_ptr2;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  KernelErrorCode = KRERR_NO_ERROR;
  if (!kr_symbolCheck( new_site_name ))
    return( KernelErrorCode );
  if ( !krf_funcSearch( new_site_func, SITE_FUNC, &func_ptr ) )
    return( KernelErrorCode );

  site_func_ptr = (SiteFuncPtr) func_ptr;

  if ( (stbl_ptr1 = krm_STableSymbolSearch( old_site_name )) == NULL)
    {  /*  old site name isn't defined */
    KernelErrorCode = KRERR_UNDEF_SITE_NAME;
    return( KernelErrorCode );
  }

  stbl_ptr2 = krm_STableSymbolSearch( new_site_name );
  if ( (stbl_ptr2 != NULL) && (stbl_ptr2 != stbl_ptr1) )
    {  /*  new symbol is already in the site table (and new_site_name and
       old_site_name are not identical) */
    KernelErrorCode = KRERR_REDEF_SITE_NAME;
    return( KernelErrorCode );
  }

  (void) krm_STableChangeEntry( stbl_ptr1 , new_site_name , site_func_ptr );
  return( KernelErrorCode );
}


/*  Removes the current site name entry from the site table.

    Returns an error code if
     - there exists sites with the given name in the network or
     - <site_name> isn't defined
     0 otherwise.
*/
krui_err  krui_deleteSiteTableEntry( site_name )
char   *site_name;
{
  struct SiteTable    *st_ptr;


  if ((st_ptr = krm_STableSymbolSearch( site_name )) == NULL)
    return( KRERR_UNDEF_SITE_NAME );   /*  Site name isn't defined */

  if (kr_searchNetSite( st_ptr ) != 0)
    return( KRERR_INUSE_SITE );   /*  Site is in use and must not be deleted  */

  krm_STableRemoveEntry( st_ptr );
  return( KRERR_NO_ERROR );
}


/*#################################################

GROUP: Site Functions

#################################################*/

/*  Initializes the first site at the current unit.
    Returns false if no site available or if no sites permitted at this unit.
*/
bool  krui_setFirstSite()
{
  if KERNEL_STANDARD
    return( kr_setSite( FIRST, NULL ) );

  KernelErrorCode = KRERR_SITES_NO_SUPPORT;
  return( FALSE );
}


/*  Initializes the next site at the current unit.
    Returns false if no more sites available.
*/
bool  krui_setNextSite()
{
  if KERNEL_STANDARD
    return( kr_setSite( NEXT, NULL ) );

  KernelErrorCode = KRERR_SITES_NO_SUPPORT;
  return( FALSE );
}


/*  Initializes the given site at the current unit.
    Returns error code if
     - unit dosn't exist
     - site name doesn't exist
     - unit don't has sites
     - unit don't has a site with this name
     0 otherwise.
*/

krui_err  krui_setSite( site_name )
char    *site_name;
{
  if KERNEL_STANDARD
    return( kr_setSite( NAME, site_name ) );

  KernelErrorCode = KRERR_SITES_NO_SUPPORT;
  return( KernelErrorCode );
}



/*  Returns the actual value of the current site
*/
FlintType   krui_getSiteValue()
{
  if KERNEL_STANDARD
    {
    if (sitePtr == NULL)
      KernelErrorCode = KRERR_NO_SITES;
    else
      return( (*sitePtr->site_table->site_func) (sitePtr) );
    }
  else
    KernelErrorCode = KRERR_SITES_NO_SUPPORT;

  return( (FlintType) 0 );
}



/*  returns the name of the current unit/site,
    NULL if not available.
*/
char  *krui_getSiteName()
{
  if KERNEL_STANDARD
    {
    if (sitePtr == NULL)
      KernelErrorCode = KRERR_NO_SITES;
    else
     return( ((sitePtr->site_table)->Entry.site_name)->Entry.symbol );
    }
  else
    KernelErrorCode = KRERR_SITES_NO_SUPPORT;

  return( NULL );
}

/*  Sets the name/function of the current unit/site.
    Current Unit will loose the functionality type.
    Returns error code if site name isn't defined.
*/
krui_err  krui_setSiteName( site_name )
char  *site_name;
{
  struct  SiteTable     *stbl_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if (sitePtr == NULL)
    {
    KernelErrorCode = KRERR_FTYPE_SITE;
    return( KernelErrorCode );
  }

  if (!kr_symbolCheck( site_name ))
    return( KernelErrorCode );
  if ((stbl_ptr = krm_STableSymbolSearch( site_name )) == NULL)
    {  /*   site name isn't defined */
    KernelErrorCode = KRERR_UNDEF_SITE_NAME;
    return( KernelErrorCode );
  }

  KernelErrorCode = KRERR_NO_ERROR;
  unitPtr->Ftype_entry = NULL;
  sitePtr->site_table  = stbl_ptr;

  NetModified = TRUE;

  return( KRERR_NO_ERROR );
}


/*  Returns the name of the current unit/site function
*/
char  *krui_getSiteFuncName()
{
  static char  site_func_name[FUNCTION_NAME_MAX_LEN];

  if (sitePtr == NULL)
    {
    KernelErrorCode = KRERR_FTYPE_SITE;
    return( NULL );
  }

  strcpy( site_func_name, krf_getFuncName( (FunctionPtr) sitePtr->site_table->site_func ) );

  return( site_func_name );
}




/*  Adds a site at the current unit. If the unit has already sites, this new
    site will be inserted above all other sites, i.e. the new created site will
    be the first site at this unit.
    If the unit has direct input links, i.e the unit has input links
    but no sites, the creation of sites is not permitted (krui_addSite will
    return an error code).
    If there exists already a site with the given name, the creation of the
    new site is prohibited and krui_addSite returns an error code.

    krui_addSite has no effect on the current site. To change the current site
    to this new site, call krui_setFirstSite().
    The unit's FType will be deleted.

    Returns error code if
      - memory allocation fails or
      - unit has direct input links or
      - site name isn't defined or
      - site with the given name exists already at this unit
      0 otherwise.

NOTE: The number of sites per unit is nearly unlimited (2^32).
*/
krui_err  krui_addSite( site_name )
char  *site_name;
{
  FlagWord    flags;
  struct Site       *site_ptr,
                    *site_ptr1;
  struct SiteTable  *stbl_ptr;


#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  KernelErrorCode = KRERR_NO_ERROR;
  if ( (stbl_ptr = krm_STableSymbolSearch( site_name )) == NULL)
    {  /*  site name isn't defined */
    KernelErrorCode = KRERR_UNDEF_SITE_NAME;
    return( KernelErrorCode );
  }

  flags = unitPtr->flags & UFLAG_INPUT_PAT;

  switch (flags)
    {
    case UFLAG_NO_INP :
        /*  Unit has no inputs  */
	if ((site_ptr = kr_createDefaultSite() ) == NULL)
	  return( KernelErrorCode );

	unitPtr->sites = site_ptr;     /*  Connect site    */
	unitPtr->flags |= UFLAG_SITES; /*  Set site flag   */
	unitPtr->Ftype_entry = NULL;   /*  Delete Ftype    */

        break;

    case UFLAG_SITES :
        /*  Unit has already sites  */
	if (kr_searchUnitSite( unitPtr, stbl_ptr ) != NULL)
	  {  /*  there exists already a site with this name at this unit   */
	  KernelErrorCode = KRERR_DUPLICATED_SITE;
	  return( KernelErrorCode );
	}

        if ( (site_ptr = kr_createDefaultSite() ) == NULL)
	  return( KernelErrorCode );

	site_ptr1 = unitPtr->sites;
	unitPtr->sites = site_ptr;     /*  Connect site    */
        site_ptr->next = site_ptr1;
	unitPtr->Ftype_entry = NULL;   /*  Delete Ftype    */

        break;

    case UFLAG_DLINKS :
        /*  Unit has direct input links and can't have sites */
	KernelErrorCode = KRERR_CREATE_SITE;
	return( KernelErrorCode );

    default :
	KernelErrorCode = KRERR_CREATE_SITE;
	return( KernelErrorCode );
    }

  site_ptr->site_table = stbl_ptr;

  NetModified = TRUE;
  return( KernelErrorCode );
}


/*  Removes the current site at the current unit and removes all
    links from predecessor units to this site.
    krui_setFirstSite (krui_setNextSite) must be called at least once
    before using this function.
    The current site will be set to the next available site, if no more
    sites available, krui_deleteSite returns 0 otherwise 1.
    Returns an error code if ther was a problem.
    The unit's FType will be set to 0, i.e. the unit's functionality
    type will be deleted.

NOTE: To delete all sites at a unit:
        if ( krui_setFirstSite() )
          while ( krui_deleteSite() > 0) { }
*/
bool  krui_deleteSite()
{
  struct Site   *next_site_ptr;
  struct Unit   *unit_ptr;


#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  unit_ptr = unitPtr;

  if ( UNIT_HAS_SITES( unit_ptr ) &&
       (unit_ptr->sites != NULL) &&
       (sitePtr != NULL)   )
    {   /*  Unit has sites  */
    NetModified = TRUE;

    next_site_ptr = sitePtr->next;

    krm_releaseAllLinks( sitePtr->links );  /*	Remove links	*/
    krm_releaseSite( sitePtr ); 	    /*	Remove site	*/

    if (prevSitePtr == NULL)
      {     /*  This site is the first site at the current unit */

      if (next_site_ptr == NULL)
        {   /*  Unit has only this site */
        unit_ptr->sites = NULL;                 /*  Clear site pointer  */
        unit_ptr->flags &= (~UFLAG_INPUT_PAT);  /*  Clear input flags   */

	sitePtr     = NULL;	       /*  No more sites available  */
	prevSitePtr = NULL;
        }
      else
        {   /*  It is the first site at the unit but not the only one   */
        unit_ptr->sites  = next_site_ptr;       /*  Connect the other sites */
	sitePtr = next_site_ptr;       /*  Current site ptr points to the next site    */
        }
      }
    else
      {     /*  This is not the first site at the unit  */
      prevSitePtr->next = next_site_ptr;   /*  Connect the previous site
                                                        with the next site  */
      sitePtr = next_site_ptr; /*  Current site ptr points to the next site    */
      }

    unit_ptr->Ftype_entry = NULL;   /*  Delete unit's Ftype */
    }

  if (sitePtr != NULL)
    return( TRUE );  /*  Returns TRUE if more sites available  */
  else
    return( FALSE );
}



/*#################################################

GROUP: Link Functions

#################################################*/

/*  Returns the no. of first predecessor unit of the current unit/site
    and the connection strenght.
    Returns 0 if no predecessor unit available, i.e. if the current unit
    and/or site has no inputs.

NOTE: If a predecessor unit exists, the current link is set to the link
      between the two units.
*/
int  krui_getFirstPredUnit( strength )
FlintType  *strength;
{
  if KERNEL_STANDARD  {
    return( kr_getPredecessorUnit( FIRST, strength ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getPredecessorUnit( FIRST, strength ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( KernelErrorCode );

#endif
  }
}


/*  Returns the no. of the next predecessor unit of the current
    unit/site and the connection strenght.
    Returns 0 if no more predecessor units exists.

NOTE: If another predecessor unit exists, the current link is set to the link
      between the two units.
*/
int  krui_getNextPredUnit( strength )
FlintType   *strength;
{
  if KERNEL_STANDARD  {
    return( kr_getPredecessorUnit( NEXT, strength ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getPredecessorUnit( NEXT, strength ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( KernelErrorCode );

#endif
  }
}



/*  Returns the no. of the current predecessor unit (of the current
    unit/site) and the connection strenght.
    Returns 0 if no predecessor unit available, i.e. if the current unit
    and/or site has no inputs.
*/
int  krui_getCurrentPredUnit( strength )
FlintType   *strength;
{
  if KERNEL_STANDARD  {
    return( kr_getPredecessorUnit( CURRENT, strength ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getPredecessorUnit( CURRENT, strength ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( KernelErrorCode );

#endif
  }
}


/*  Returns the no. of the first successor unit of the unit <source_unit_no>
    and the connection strenght.
    Returns (negative) error code if unit doesn't exist.
    Returns 0 if no successor unit available, i.e. if the given unit
    has no output connection.

IMPORTANT: If a successor unit exists, the current unit and site will be
           set to this successor unit and the attached site.

NOTE: This function is slow (Units are backward chained only).
*/
int  krui_getFirstSuccUnit( source_unit_no, weight )
int  source_unit_no;
FlintType   *weight;
{
  if KERNEL_STANDARD  {
    return( kr_getSuccessorUnit( FIRST, source_unit_no, weight ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getSuccessorUnit( FIRST, source_unit_no, weight ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( KernelErrorCode );

#endif
  }
}


/*  Returns the no. of the next successor unit and the connection strenght.

IMPORTANT: If a successor unit exists, the current unit and site will be
           set to this successor unit and the attached site.

NOTE: This function is slow  (Units are backward chained only)
*/
int  krui_getNextSuccUnit( weight )
FlintType  *weight;
{
  if KERNEL_STANDARD  {
    return( kr_getSuccessorUnit( NEXT, 0, weight ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getSuccessorUnit( NEXT, 0, weight ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( KernelErrorCode );

#endif
  }
}


/*  True if there exists a connection between source unit <source_unit_no>
    and target unit <target_unit_no>, otherwise false. If there exist a
    connection between these units, krui_areConnected returns the connection
    strength also.
    Returns FALSE if unit doesn't exist.

IMPORTANT: If there exist a connection, the current unit and site will be
	   set to the target unit/site.

NOTE: This function is slow (Units are backward chained only).
*/
bool  krui_areConnected( source_unit_no, target_unit_no, weight)
int  source_unit_no, target_unit_no;
FlintType   *weight;
{
  if KERNEL_STANDARD  {
    return( kr_areConnected( source_unit_no, target_unit_no, weight ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_areConnected( source_unit_no, target_unit_no, weight ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( FALSE );

#endif
  }
}



/*  True if there exists a connection between source unit <source_unit_no>
    and the current unit/site, otherwise false.

NOTE: If there exists a connection between the two units, the current link is set
      to the link between the two units. (alter the link weight with krui_setLinkWeight)
*/
bool  krui_isConnected( source_unit_no )
int  source_unit_no;
{
  FlintType  weight;

  if KERNEL_STANDARD  {
    return( kr_isConnected( source_unit_no, &weight ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_isConnected( source_unit_no, &weight ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( FALSE );

#endif
  }
}



/*  Returns the link weight of the current link.
*/
FlintType  krui_getLinkWeight()
{
  if KERNEL_STANDARD  {
    return( kr_getLinkWeight() );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_getLinkWeight() );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( 0.0 );

#endif
  }
}

/*  Sets the link weight of the current link
*/
void  krui_setLinkWeight( weight )
FlintTypeParam	weight;
{
  if KERNEL_STANDARD  {
    kr_setLinkWeight( weight );
  }
  else  {

#ifdef MASPAR_KERNEL

    krff_setLinkWeight( weight );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;

#endif
  }
}


/*  Creates a link between source unit and the current unit/site.
    Returns an error code:
     - if memory allocation fails
     - if source unit doesn't exist or
     - if there exists already a connection between current unit/site and
       the source unit
    0 otherwise.
    krui_createLink DO NOT set the current link.

NOTE: If you want to create a link and its unknown if there exists already a
      connection between the two units, use krui_createLink and test the return
      code, instead of the sequence krui_isConnected and krui_createLink.
*/
krui_err  krui_createLink( source_unit_no, weight )
int  source_unit_no;
FlintTypeParam	weight;
{
  if KERNEL_STANDARD  {
    return( kr_createLink( source_unit_no, weight ) );
  }
  else  {

#ifdef MASPAR_KERNEL

    return( krff_createLink( source_unit_no, weight ) );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;
    return( KernelErrorCode );

#endif
  }
}



/*  Deletes the current link.

NOTE: To delete a link between the current unit/site and the source unit
      <source_unit_no>, call krui_isConnected( source_unit_no ) and
      krui_deleteLink().
*/
krui_err  krui_deleteLink()
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  return( kr_deleteLink() );
}


/*  Deletes all input links at current unit/site.
*/
krui_err  krui_deleteAllInputLinks()
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  return( kr_deleteAllLinks( INPUTS ) );
}


/*  Deletes all output links at current unit.
    NOTE: This function is slow.
*/
krui_err  krui_deleteAllOutputLinks()
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  return( kr_deleteAllLinks( OUTPUTS ) );
}


/*  Add uniform distributed random values to connection weights.
    <minus> must be less then <plus>.
*/
void  krui_jogWeights( minus, plus )
FlintTypeParam   minus, plus;
{
  if (minus >= plus)  {
    KernelErrorCode = KRERR_PARAMETERS;
    return;  /*  wrong bounds    */
  }

  if KERNEL_STANDARD  {
    kr_jogWeights( minus, plus );
  }
  else  {

#ifdef MASPAR_KERNEL

    krff_jogWeights( minus, plus );

#else

    KernelErrorCode = KRERR_NO_MASPAR_KERNEL;

#endif
  }
}



/*#################################################

GROUP: Functions for network propagation

#################################################*/

/*  Updates a single unit. Returns error code if unit doesn't exist,
    0 otherwise.

    NOTE: Updates also frozen Units.
*/
krui_err   krui_updateSingleUnit( unit_no )
int     unit_no;
{
  register struct Unit   *unit_ptr;

#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  if (unit_ptr->out_func == NULL)
    /*  Identity Function   */
    unit_ptr->Out.output = unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);
  else
    {
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);
    unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
  }

  return( KRERR_NO_ERROR );
}



/*  Returns the current update function. The default update function is
    SerialOrder() (see also kr_def.h).
*/
char  *krui_getUpdateFunc()
{
  static char  updateFunc[FUNCTION_NAME_MAX_LEN];

  strcpy( updateFunc, krf_getCurrentNetworkFunc( UPDATE_FUNC ) );

  return( updateFunc );
}


/*  Changes the current update function.  Returns error code if update
    function is invalid.
*/
krui_err   krui_setUpdateFunc( update_func )
char  *update_func;
{
  return( krf_setCurrentNetworkFunc( update_func, UPDATE_FUNC ) );
}


/*  Updates the network according to update function:
NOTE: The network should be feedforward in topological mode,
      otherwise function will return a warning message.

      See also krui_setSeedNo for initializing the pseudo
      random generator.

Returns error code if an error occured, 0 othrwise.
*/
krui_err  krui_updateNet( parameterInArray, NoOfInParams )
float	parameterInArray[];
int  NoOfInParams;
{
  return( kr_callNetworkFunction( UPDATE_FUNC,
				  parameterInArray, NoOfInParams,
				  NULL, NULL,
				  0, 0 ) );
}



/*  Calculates the network error whith the given pattern. 
    Uses the current update function to propagate the network.

    parameterArray contains the parameters of the update function.
    NoOfParams contains the number of parameters of the update function.

    parameterInArray[0] contains the max. devitation.
    Set NoOfInParameters to 1.

    parameterOutArray[0] contains the error of the network/pattern,
    parameterOutArray[0] contains the number of output units with a higher error value than 
    the given max. devitation.
    NoOfOutParams will be set to 2.

NOTE: Patterns must be loaded before calling this function.

Returns error code if an error occured, 0 otherwise.
*/
krui_err   krui_testNet( pattern_no,
			 updateParameterInArray, updateNoOfInParams,
                         parameterInArray, NoOfInParams,
                         parameterOutArray, NoOfOutParams )
int  pattern_no;
float  updateParameterInArray[], parameterInArray[], * *parameterOutArray;
int  updateNoOfInParams, NoOfInParams, *NoOfOutParams;
{
#ifdef MASPAR_KERNEL
  MASPAR_FF1_VALIDATE_OP;
#endif

  KernelErrorCode = KRERR_NO_ERROR;

  if (kr_pattern( PATTERN_SHOW, OUTPUT_NOTHING, pattern_no - 1) != KRERR_NO_ERROR)
    return( KernelErrorCode );

  if (kr_callNetworkFunction( UPDATE_FUNC,
			      updateParameterInArray, updateNoOfInParams,
			      NULL, NULL,
                              0, 0 ) != KRERR_NO_ERROR)
    return( KernelErrorCode );

  (void) kr_checkUnitsOutput( pattern_no - 1,
			      parameterInArray, NoOfInParams,
			      parameterOutArray, NoOfOutParams );
  return( KernelErrorCode );
}




/*#################################################

GROUP: Initialisation Functions

#################################################*/


/*  Returns the current initialisation function. The default initialisation function is
    RandomizeWeights (see also kr_def.h).
*/
char  *krui_getInitialisationFunc()
{
  return( krf_getCurrentNetworkFunc( INIT_FUNC ) );
}


/*  Changes the current learning function.  Returns error code if learning
    function is invalid.
*/
krui_err   krui_setInitialisationFunc( initialisation_func )
char  *initialisation_func;
{
  return( krf_setCurrentNetworkFunc( initialisation_func, INIT_FUNC ) );
}


/*  Initializes the network
*/
krui_err  krui_initializeNet( parameterInArray, NoOfInParams )
float	parameterInArray[];
int  NoOfInParams;
{
  return( kr_callNetworkFunction( INIT_FUNC,
				  parameterInArray, NoOfInParams,
				  NULL, NULL,
				  0, 0 ) );
}


/*#################################################

GROUP: Learning Functions

#################################################*/


/*  Returns the current learning function. The default learning function is
    Backpropagation (see also kr_def.h).
*/
char  *krui_getLearnFunc()
{
  static char  learnFunc[FUNCTION_NAME_MAX_LEN];

  strcpy( learnFunc, krf_getCurrentNetworkFunc( LEARN_FUNC ) );

  return( learnFunc );
}



/*  Changes the current learning function.  Returns error code if learning
    function is invalid.
*/
krui_err   krui_setLearnFunc( learning_func )
char  *learning_func;
{

  return( krf_setCurrentNetworkFunc( learning_func, LEARN_FUNC ) );
}


/*  Learn all pattern pairs using current learning method.
    parameterInArray contains the learning
    parameter(s). NoOfInParams stores the number of learning parameters.
    parameterOutArray returns the results from the learning function.
    this array is a static array defined in the learning function.
    *NoOfOutParams points to a integer value that contains the number
    of output parameters from the current learning function.
    Returns an error code if memory allocation has failed or if
    the parameters are invalid.

    Returns error code of the learning function.
REMEMBER:  The backpropagation learning function takes the learning
           parameter from parameterInArray[ 0 ]. parameterOutArray[ 0 ]
           returns the current net error.

NOTE: Patterns must be loaded before calling this function.
*/
krui_err   krui_learnAllPatterns( parameterInArray, NoOfInParams,
				  parameterOutArray, NoOfOutParams )
/*  REMEMBER:  parameterOutArray[ 0 ] returns the current net error
               parameterInArray[ 0 ] contains the 1st learning parameter
*/
float   parameterInArray[], * *parameterOutArray;
int  NoOfInParams, *NoOfOutParams;
{
  /*  learn all patterns  */
  return( kr_callNetworkFunction( LEARN_FUNC,
				  parameterInArray, NoOfInParams,
				  parameterOutArray, NoOfOutParams,
				  0, kr_pattern( PATTERN_GET_NUMBER, 0, 0 ) - 1 ) );
}


/*  Same as krui_learnAllPatterns( ... ) but learns only the current
    pattern pair.

NOTE: Patterns must be loaded before calling this function.
*/
krui_err   krui_learnSinglePattern( pattern_no,
				    parameterInArray, NoOfInParams,
				    parameterOutArray, NoOfOutParams)
int     pattern_no;
float   parameterInArray[],
        * *parameterOutArray;
int     NoOfInParams,
        *NoOfOutParams;
{
  KernelErrorCode = KRERR_NO_ERROR;
  return( kr_callNetworkFunction( LEARN_FUNC,
				  parameterInArray, NoOfInParams,
				  parameterOutArray, NoOfOutParams,
				  pattern_no - 1, pattern_no - 1 ) );
}



/*#################################################

GROUP: Pattern Management

#################################################*/

/*  Sets the current pattern.
    Returns a error code if pattern number is invalid.

NOTE: Patterns must be loaded before calling this function.
*/
krui_err  krui_setPatternNo( pattern_no )
int  pattern_no;
{
  return( kr_pattern( PATTERN_SET, 0, pattern_no ) );
}


/*  delete the current pattern pair
*/
krui_err  krui_deletePattern()
{
  return( kr_pattern( PATTERN_DELETE, 0 , 0 ) );
}


/*  modify the current pattern pair
*/
krui_err  krui_modifyPattern()
{
  return( kr_pattern( PATTERN_MODIFY, 0 , 0 ) );
}


/*  According to the mode krui_showPattern stores the current
    Pattern into the units activation (and/or output) values.
    The modes are:
      - OUTPUT_NOTHING
         store input pattern into input units activations
      - OUTPUT_ACT
         store input pattern into input units activations and
         store output pattern into output units activations
      - OUTPUT_OUT
         store input pattern into input units activations,
         store output pattern into output units activations and
         update output units output

NOTE: See include file glob_typ.h for mode constants.
*/
krui_err  krui_showPattern( mode )
int  mode;
{
  return( kr_pattern( PATTERN_SHOW, mode, 0 ) );
}


/*  Creates a new pattern pair.
    A pattern pair can be created by modifying the activation
    value of the input/output units.
    Returns error code if memory is insufficent or no. of
    input/output units is incompatible, 0 otherwise.

NOTE: krui_newPattern switches pattern shuffeling off.
      For shuffeling the new pattern pairs call
        krui_newPattern(...)
        krui_shufflePattern( TRUE )
*/
krui_err  krui_newPattern()
{
  return( kr_pattern( PATTERN_NEW, 0, 0 ) );
}


/*  Create new pattern vector. Append the given input- and output patterns
    to the unit patterns by copying the contents of the pattern arrays.
*/
krui_err  krui_newVectorPatternPair( input_patterns, no_of_input_patterns,
				     output_patterns, no_of_output_patterns )
float  input_patterns[], output_patterns[];
int  no_of_input_patterns, no_of_output_patterns;
{
  return( kr_vectorPatterns( PATTERN_VECTOR_NEW,
			     input_patterns, no_of_input_patterns,
			     output_patterns, no_of_output_patterns,
			     0 ) );
}

/*  Allocate pattern pairs.
*/
krui_err  krui_allocatePatterns( no_of_input_patterns, no_of_output_patterns,
				 no_of_pattern_pairs )
int  no_of_input_patterns, no_of_output_patterns;
int  no_of_pattern_pairs;
{
  return( kr_vectorPatterns( PATTERN_ARRAY_ALLOC,
			     NULL, no_of_input_patterns,
			     NULL, no_of_output_patterns,
			     no_of_pattern_pairs ) );
}

/*  Returns the startaddresses of the internal pattern arrays.

    NOTE: Adresses of the internal pattern arrays may change.
	  You have to call `krui_getPatternArrays' EVERYTIME to determine the
	  addresse of the pattern arrays before you access these arrays.
*/
krui_err  krui_getPatternArrays( input_patterns, no_of_input_patterns,
				 output_patterns, no_of_output_patterns )
float  * *input_patterns, * *output_patterns;
int  *no_of_input_patterns, *no_of_output_patterns;
{
  return( kr_getPatternArrays( input_patterns, no_of_input_patterns,
			       output_patterns, no_of_output_patterns ) );
}

/*  Set the number of available pattern pairs
*/
krui_err  krui_setNoOfPatterns( no_of_pattern_pairs )
int  no_of_pattern_pairs;
{
  return( kr_pattern( PATTERN_SET_NUMBER, 0, no_of_pattern_pairs ) );
}


/*  Returns the no. of available pattern pairs.
*/
int  krui_getNoOfPatterns()
{
  return( kr_pattern( PATTERN_GET_NUMBER, 0, 0 ) );
}



/*  Release previously defined patterns from memory.
    Call krui_releasePatterns() if you want to create totally
    new patterns with krui_newPattern().
*/
void  krui_deleteAllPatterns()
{
  (void) kr_pattern( PATTERN_DELETE_ALL, 0, 0 );
}


/*  Shuffle pattern pairs by using pseudo random generator.
    Returns error code if memory allocation fails.
    Shuffeling of patterns is used by krui_learnAllPatterns(...).
    krui_shufflePatterns( TRUE ) switches shuffeling of patterns
    on, krui_shufflePatterns( FALSE ) switches shuffeling of
    patterns off.
    The default presetting is krui_shufflePatterns( FALSE ).

NOTE:  See also krui_setSeedNo( seed )
*/
krui_err  krui_shufflePatterns( on_or_off )
bool    on_or_off;
{
  if (on_or_off)
    return( kr_pattern( PATTERN_SHUFFLE_ON, 0, 0 ) );
  else
    return( kr_pattern( PATTERN_SHUFFLE_OFF, 0, 0 ) );
}



/*#################################################

GROUP: I/O Functions

#################################################*/

/*  Save a network.
    If netname is a NULL pointer, the net will get the name "UNTITLED"
    Returns error code if an error occured, or 0 otherwise.
*/
krui_err  krui_saveNet( filename, netname )
char  *filename, *netname;
{
  return( krio_saveNet( filename, netname ) );
}


/*  Load a network file and create a network.
    Returns the name of the net or "UNTITLED" if unknown.
    Returns error code if an error occured during
    loading/memory allocation, or 0 otherwise.
*/
krui_err  krui_loadNet( filename, netname )
char    *filename,
        **netname;
{
  char	*netfile_version;  /*  isn't used now  */
              
  KernelErrorCode = KRERR_NO_ERROR;

  KernelErrorCode = krio_loadNet( filename, netname, &netfile_version );
  if (KernelErrorCode != KRERR_NO_ERROR)
    return( KernelErrorCode );

  kr_updateUnitOutputs();

  if (kr_pattern( PATTERN_GET_NUMBER, 0, 0 ) > 0)
    KernelErrorCode = kr_IOCheck();

  return( KernelErrorCode );
}


/*  Load file containing pattern pairs.
    Returns error code if an error occured during
    loading/memory allocation, or 0 otherwise.

*/
krui_err  krui_loadPatterns( filename )
char  *filename;
{
  return( krio_loadPatterns( filename ) );
}

/*  Save all pattern pairs.
    Returns error code if an error occured during file I/O,
    0 otherwise.
*/
krui_err   krui_savePatterns( filename )
char  *filename;
{
  return( krio_savePatterns( filename ) );
}


/*  Save the network result which depends on the loaded patterns.
    If create is false, the new file will be appended to an existing file.
    startpattern and endpattern determine the range of patterns to use.
    The input patterns and the teaching output patterns can be included.
*/
krui_err  krui_saveResult(filename, create, startpattern,
	endpattern, includeinput, includeoutput)
char	*filename;
bool	create;
int	startpattern, endpattern;
bool	includeinput, includeoutput;
{
  return( krio_saveResult(filename, create, startpattern, endpattern,
	includeinput, includeoutput));
}



/*#################################################

GROUP: Functions for memory management

#################################################*/

/*  Allocates a given number of units, additional units
    may allocated by calling this function again.
    This function is called automatically if the user
    construct more units than have been allocated before, but
    it is recommended to use this function if a large amount
    of units is needed (the UNIX System can manage system resources
    much better, if the amount of memory used for the network is
    approximately known before the network is created).

    Returns error code if memory allocation fails, 0 otherwise.

NOTE: If krui_create_unit has been called before using this
      function, at least <UNIT_BLOCK> numbers of units have been
      allocated.
      See "kr_def.h" for more information about memory allocation
      block sizes.
*/
krui_err  krui_allocateUnits( no_of_units )
int  no_of_units;
{
  return( krm_allocUnits( no_of_units ) );
}


/*  Delete network, names and unit types.
    Frees all memory used for the network.

NOTE: If krui_loadNet is called more then once,  krui_deleteNet will
      be called by krui_loadNet, because the kernel have to free the
      memory used for the old network.
      It is recommended (but not neccessary) to call krui_deleteNet
      before terminating the SNNS-Kernel.
*/
void  krui_deleteNet()
{
  krm_releaseMem();
}


/*#################################################

GROUP: Functions for reading/searching the symbol table

#################################################*/

/*  Returns the first symbol/symbol type entry in the
    symbol table. Returns true if this entry is available,
    false otherwise.
*/
bool  krui_getFirstSymbolTableEntry( symbol_name, symbol_type )
char * *symbol_name;
int  *symbol_type;
{
  struct NameTable      *n_tbl;


  n_tbl = krm_getNTableFirstEntry();

  while ( (n_tbl != NULL) && ((int) n_tbl->sym_type == UNUSED_SYM) )
    n_tbl = krm_getNTableNextEntry();

  if (n_tbl == NULL)
    {
    *symbol_name = NULL;
    *symbol_type = 0;
    return( FALSE );
    }

  *symbol_name = n_tbl->Entry.symbol;
  *symbol_type = (int) n_tbl->sym_type;
  return( TRUE );
}



/*  Returns the next symbol/symbol type entry in the
    symbol table. Returns true if another entry is available,
    false otherwise.
*/
bool  krui_getNextSymbolTableEntry( symbol_name, symbol_type )
char * *symbol_name;
int  *symbol_type;
{
  struct NameTable      *n_tbl;


  n_tbl = krm_getNTableNextEntry();

  while ( (n_tbl != NULL) && ((int) n_tbl->sym_type == UNUSED_SYM) )
    n_tbl = krm_getNTableNextEntry();

  if (n_tbl == NULL)
    {
    *symbol_name = NULL;
    *symbol_type = 0;
    return( FALSE );
    }

  *symbol_name = n_tbl->Entry.symbol;
  *symbol_type = (int) n_tbl->sym_type;
  return( TRUE );
}





/*  Searches the symbol table for a given symbol and
    symbol type (unit name symbol, site name symbol,
    functionality unit name symbol)
    Returns true, if the symbol exists.
*/
bool  krui_symbolSearch( symbol, symbol_type)
char  *symbol;
int  symbol_type;
{
  return( krm_NTableSymbolSearch( symbol, symbol_type ) != NULL );
}






/*#################################################

GROUP: Miscellanous

#################################################*/


/*  Returns the current Version of the SNNS-Kernel.
*/
char  *krui_getVersion()
{
  static char	snns_version[128];


  strcpy( snns_version, SNNS_VERSION );
  strcat( snns_version, KERNEL_PATCH );
  return( snns_version );
}


/*  Returns miscellanous information about the current network.
*/
void  krui_getNetInfo( no_of_sites, no_of_links, no_of_STable_entries,
                       no_of_FTable_entries )
int  *no_of_sites,
     *no_of_links,
     *no_of_STable_entries,
     *no_of_FTable_entries;
{
  int   array_size,
        info_array[ 10 ];


  krm_getMemoryManagerInfo( &array_size, info_array );

  *no_of_sites          = info_array[ 0 ];
  *no_of_links          = info_array[ 1 ];
  *no_of_STable_entries = info_array[ 2 ];
  *no_of_FTable_entries = info_array[ 3 ];
}


/*  Returns the number of ALLOCATED (not the number of USED) bytes
    per entry.
*/
void  krui_getMemoryManagerInfo( unit_bytes, site_bytes, link_bytes,
                                 NTable_bytes, STable_bytes, FTable_bytes )
int  *unit_bytes,
     *site_bytes,
     *link_bytes,
     *NTable_bytes,
     *STable_bytes,
     *FTable_bytes;
{
  int   array_size,
        info_array[ 10 ];


  krm_getMemoryManagerInfo( &array_size, info_array );

  *unit_bytes   = info_array[ 4 ] * UNIT_SIZE;
  *site_bytes   = info_array[ 5 ] * SITE_SIZE;
  *link_bytes   = info_array[ 6 ] * LINK_SIZE;
  *NTable_bytes = info_array[ 7 ] * NTABLE_SIZE;
  *STable_bytes = info_array[ 8 ] * STABLE_SIZE;
  *FTable_bytes = info_array[ 9 ] * FTYPE_UNIT_SIZE;
}


/*  Returns Information about the unit default settings.
    For more information about default settings see krui_createDefaultUnit() and
    krui_createFTypeUnit( .. ).
*/

void  krui_getUnitDefaults( act, bias, st, subnet_no, layer_no,
                            act_func, out_func )
FlintType   *act, *bias;
int         *st;
int         *subnet_no, *layer_no;
char        * *act_func, * *out_func;
{
  int	ttflags;

  kr_getUnitDefaults( act, bias, &ttflags, subnet_no, layer_no,
		      act_func, out_func );

  *st = kr_flags2TType( ttflags );
}


/*  Changes the unit default settings.
    For more information about default settings see krui_createDefaultUnit() and
    krui_createFTypeUnit( .. ).

    Returns error code if
     - activation/output function is invalid
     - Topologic type is invalid
    0 otherwise.
*/
krui_err  krui_setUnitDefaults( act, bias, st, subnet_no, layer_no,
                                act_func, out_func )
FlintTypeParam  act, bias;
int         st;
int         subnet_no, layer_no;
char        *act_func, *out_func;
{
  int	       ttflags;

  ttflags = kr_TType2Flags( st );
  if (KernelErrorCode != KRERR_NO_ERROR)
    return( KernelErrorCode );

  kr_setUnitDefaults( act, bias, ttflags, subnet_no, layer_no,
		      act_func, out_func );
  return( KernelErrorCode );
}



/*  Reset the network by changeing the unit activation
    to the initial activation value.
*/
void  krui_resetNet()
{
  register int   i;
  register struct Unit   *unit_ptr;


  if ( (unit_array == NULL) || (NoOfUnits == 0) )
    return;


  for (i = MinUnitNo, unit_ptr = unit_array + MinUnitNo; i <= MaxUnitNo; i++, unit_ptr++)
    if UNIT_IN_USE( unit_ptr )  {
      /*  unit is in use  */
      unit_ptr->act = unit_ptr->i_act;

      if (unit_ptr->out_func == OUT_IDENTITY)
	/*  identity output function: there is no need to call the output function  */
	unit_ptr->Out.output = unit_ptr->act;
      else
	/*  no identity output function: calculate unit's output also  */
	unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
    }
}



/*  Initialize the pseudo random generator.
    0 as argument reinitializes the random generator.
*/
void  krui_setSeedNo( seed )
long  seed;
{
  if (seed != 0)
    srand48( seed );   /*  initialize random generator with seed value	*/
  else
    srand48( (long) time( (long *) 0) );  /*  initialize random generator random value	*/
}


/*  Returns the no. of input units
*/
int  krui_getNoOfInputUnits()
{
  return( kr_getNoOfUnits( INPUT ) );
}

/*  returns the no. of output units
*/
int  krui_getNoOfOutputUnits()
{
  return( kr_getNoOfUnits( OUTPUT ) );
}

/*  returns the no. of units of the specified topologic type
    (i.e. Input, Hidden, Output or Special units)
*/
int  krui_getNoOfTTypeUnits( UnitTType )
int  UnitTType;
{
  return( kr_getNoOfUnits( UnitTType ) );
}



#ifdef KERNEL3D

/*  returns the base address of the XY-Translation table
*/
krui_err  krui_getXYTransTable( xy_trans_tbl_ptr )
struct TransTable  * *xy_trans_tbl_ptr;
{
  KernelErrorCode = KRERR_NO_ERROR;

  *xy_trans_tbl_ptr = krio_getXYTransTablePtr();

  return( KRERR_NO_ERROR );
}

/*  Returns the 3D transform center of the specified unit and center number
*/
krui_err  krui_getUnitCenters( unit_no, center_no, unit_center )
int  unit_no, center_no;
struct PositionVector  * *unit_center;
{
  struct Unit   *unit_ptr;

  KernelErrorCode = KRERR_NO_ERROR;

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  if ((center_no < 0) || (center_no >= NO_OF_UNIT_CENTER_POS))  {
    KernelErrorCode = KRERR_PARAMETERS;
    return( KRERR_PARAMETERS );
  }

  *unit_center = unit_ptr->unit_center_pos + center_no;
  return( KRERR_NO_ERROR );
}


/*  Sets the 3D transform center of the specified unit and center number
*/
krui_err  krui_setUnitCenters( unit_no, center_no, unit_center )
int  unit_no, center_no;
struct PositionVector  *unit_center;
{ 
  struct Unit   *unit_ptr;
  struct PositionVector    *unit_posvec_ptr;


  KernelErrorCode = KRERR_NO_ERROR;

  if ((unit_ptr = kr_getUnitPtr( unit_no )) == NULL)
    return( KernelErrorCode );

  if ((center_no < 0) || (center_no >= NO_OF_UNIT_CENTER_POS))  {
    KernelErrorCode = KRERR_PARAMETERS;
    return( KRERR_PARAMETERS );
  }

  unit_posvec_ptr = unit_ptr->unit_center_pos + center_no;

  memcpy( (char *) unit_posvec_ptr, (char *) unit_center, sizeof( struct PositionVector ) );

  return( KRERR_NO_ERROR );
}


#else

/*
krui_err  krui_getXYTransTable( dummy )
int dummy;
{
  KernelErrorCode = KRERR_NOT_3D;

  return( KRERR_NOT_3D );
}


krui_err  krui_getUnitCenters( unit_no, center_no, dummy )
int  unit_no, center_no, dummy;
{ 
  KernelErrorCode = KRERR_NOT_3D;

  return( KRERR_NOT_3D );
}


krui_err  krui_setUnitCenters( unit_no, center_no, dummy )
int  unit_no, center_no, dummy;
{ 
  KernelErrorCode = KRERR_NOT_3D;

  return( KRERR_NOT_3D );
}
*/

#endif



static char  *krui_topo_err_msg()
{
  char  *dest_unit_name,  *src_unit_name;
  char  msg1[512], msg2[512];


  if (topo_msg.dest_error_unit > 0)
    dest_unit_name = krui_getUnitName( topo_msg.dest_error_unit );

  if (topo_msg.src_error_unit > 0)
    src_unit_name = krui_getUnitName( topo_msg.src_error_unit );

  if (topo_msg.dest_error_unit > 0)
    if (dest_unit_name == NULL)
      sprintf( msg1, "Unit #%d is the destination unit. ", topo_msg.dest_error_unit );
    else
      sprintf( msg1, "Unit #%d (%s) is the destination unit. ", topo_msg.dest_error_unit,
               dest_unit_name );

  if (topo_msg.src_error_unit > 0)
    if (src_unit_name == NULL)
      sprintf( msg2, "Unit #%d is the source unit. ", topo_msg.src_error_unit );
    else
      sprintf( msg2, "Unit #%d (%s) is the source unit. ", topo_msg.src_error_unit,
               src_unit_name );

  if (topo_msg.dest_error_unit == 0)
    return( msg2 );

  if (topo_msg.src_error_unit > 0)
    strcat( msg1, msg2 );

  return( msg1 );
}



/*  Returns an error message, depending on the error code.
    If a function returns an error code use krui_error to
    get the message.
*/
char  *krui_error( error_code )
int  error_code;
{
  static char	*err_message[] = {
      "Invalid error code",
      "Insufficient memory",
      "Invalid unit number",
      "Invalid unit output function",
      "Invalid unit activation function",
      "Invalid site function",
      "Creation of sites isn't permitted because unit has direct input links",
      "Creation of a link isn't permitted because there exists already a link between these units",
      "Memory allocation failed during critical operation. Have lost some pointers, but consistency of the network is guaranteed",
      "Ftype name isn't definite",
/*10*/"Current Ftype entry isn't defined",
      "Invalid copy mode",
      "Current unit doesn't have sites",
      "Can't update unit because unit is frozen",
      "Redefinition of site name isn't permitted (site name already exists)",
      "Site name isn't defined",
      "This is not a 3D-Kernel",
      "This unit has already a site with this name",
      "Can't delete site table entry because site is in use",
      "Current Ftype site isn't defined",
/*20*/"Ftype symbol isn't defined",
      "I/O error: ",
      "Creation of output file failed (line length limit exceeded)",
      "The network has not enough layers: ",
      "No Units defined",
      "Unexpected EOF",
      "Line length exceeded",
      "Incompatible file format",
      "Can't open file",
      "Syntax error",
/*30*/"Memory allocation error #01",
      "Topologic type invalid",
      "Symbol pattern invalid (must match [A-Za-z]^[|, ]*)",
      "Current unit doesn't have a site with this name",
      "No hidden units defined",
      "Network contains cycle(s): ",
      "Network contains dead unit(s): ",
      "Pattern file contains not the same no. of input units as the network",
      "Pattern file contains not the same no. of output units as the network",
      "No. of input units have changed",
/*40*/"No. of output units have changed",
      "No input units defined",
      "No output units defined",
      "No patterns defined",
      "In-Core patterns incompatible with current network (remove In-Core patterns before loading a new network)",
      "Invalid pattern number",
      "Invalid learning function",
      "Invalid parameters",
      "Invalid update function",
      "Invalid initialisation function",
/*50*/"Derivation function of the activation function doesn't exist",
      "Input unit(s) with input connection(s) to other units: ",
      "Output unit(s) with output connection(s) to other units: ",
      "Invalid topological sorting mode",
      "Learning function doesn't support sites",
      "Sites are not supported",
      "This isn't a MasPar Kernel",
      "Connection(s) between unit(s) in non-neighbour layers are not supported: ",
      "The network has too much layers: ",
      "The network layers aren't fully connected",
/*60*/"This operation isn't allowed in the parallel kernel mode",
      "Change of network type isn't possible in parallel kernel mode",
      "No current link defined",
      "No current unit defined",
      "Current unit doesn't have any inputs",
      "Invalid parameter in topologic definition section",
      "Creation of link between these units isn't permitted",
      "MasPar don't respond",
      "This function isn't implemented yet",
      "Kernel isn't in parallel mode",
/*70*/"MasPar ran out of memory",
      "MasPar communication error",
      "MasPar ran out of processors",
      "Missing default function (check function table)",
      "MasPar kernel doesn't support multiple unit output functions",
      "MasPar kernel doesn't support multiple unit activation functions",
      "The depth of the network doesn't fit to the learning function"

    };  /*  76 error messages */

  static char	*ext_messages[] = {
      "SNNS-Kernel No Errors",
      "SNNS-Kernel Error: ",
      "SNNS-Kernel Network Topologic Error: "
      };

  int  NoOfMessages = (sizeof err_message) / (sizeof err_message[0]);
  static char  mesg[512], aux[512];


  if (error_code == 0)	return( ext_messages[0] );  /*	No errors  */

  error_code = abs( error_code );
  if ( error_code >= NoOfMessages )  error_code = 0;  /*  invalid error code  */

  switch (-error_code)
    {
    case KRERR_CYCLES:
    case KRERR_DEAD_UNITS:
    case KRERR_I_UNITS_CONNECT:
    case KRERR_O_UNITS_CONNECT:
    case KRERR_NOT_NEIGHBOUR_LAYER:
      strcpy( mesg, ext_messages[2] );
      strcat( mesg, err_message[ error_code ] );

      switch (-error_code)
        {
        case KRERR_CYCLES:
          sprintf( aux, "%d cycle(s) in the network. ", topo_msg.no_of_cycles );
          strcat( mesg, aux );
          break;
        case KRERR_DEAD_UNITS:
          sprintf( aux, "%d dead unit(s) in the network. ", topo_msg.no_of_dead_units );
          strcat( mesg, aux );
          break;

        default: break;
      }

      strcat( mesg, krui_topo_err_msg() );
      return( mesg );


    case KRERR_FEW_LAYERS:
    case KRERR_MUCH_LAYERS:
    case KRERR_NOT_FULLY_CONNECTED:
      strcpy( mesg, ext_messages[2] );
      strcat( mesg, err_message[ error_code ] );

      switch (-error_code)
        {
        case KRERR_FEW_LAYERS:
          sprintf( aux, "Only %d layers found.", topo_msg.no_of_layers );
          break;
        case KRERR_MUCH_LAYERS:
          sprintf( aux, "%d layers found.", topo_msg.no_of_layers );
          break;

        default: break;
        }

      strcat( mesg, aux );
      return( mesg );


    case KRERR_FILE_OPEN:
      lineno = 0;

    default:
      if (lineno != 0)
	{  /*  file I/O error  */
	strcpy( mesg, ext_messages[1] );
	sprintf( aux, "Loading file failed at line %d : ", lineno );
	strcat( mesg, aux );
	strcat( mesg, err_message[ error_code ] );

	lineno = 0;
	return( mesg );
      }
      else
	{  /*  standard error  */
	strcpy( mesg, ext_messages[1] );
	strcat( mesg, err_message[ error_code ] );
	return( mesg );
      }
    }
}


/* #############################################################
 
  Functions for the extern kernels

############################################################# */

/*  Sets the topologic type of the current network.
    Returns an error if the topologic type of the current network  
    doesn't fit to this type.
    Topologic types are:
      - NET_TYPE_GENERAL
        general purpose network type with no limitations
      - NET_TYPE_FF1
        feedforward network with fully connected units in
        neighbour layers
*/
krui_err  krui_setSpecialNetworkType( net_type )
int  net_type;
{
  (void) kr_setSpecialNetworkType( net_type );

  return( KernelErrorCode );
}

/*  Returns the special topologic type of the current network, if set.
*/
int  krui_getSpecialNetworkType()
{
  return( kr_getSpecialNetworkType() );
}


/* #############################################################
 
  Functions for the MasPar kernel

############################################################# */

#ifdef MASPAR_KERNEL

/*  Connects and disconnects the MasPar.
    The mode switches are:  MASPAR_CONNECT and MASPAR_DISCONNECT.
*/
krui_err  krui_MasPar( mode )
int  mode;
{
  return( kr_initMasPar( mode ) );
}

/*  Returns the status of the MasPar.
*/
krui_err  krui_getMasParStatus()
{
  return( kr_getMasParStatus() );
}


/*  Perform benchmark tests
*/
krui_err  krui_MasParBenchmark( func_type, cycles, result )
int  func_type, cycles;
float  *result;
{
  float  parameterInArray[NO_OF_LEARN_PARAMS];
  float  *parameterOutArray;
  int  NoOfOutParams;


  parameterInArray[0] = (float) cycles;
  (void) kr_callNetworkFunction( func_type | BENCH_FUNC,
		    	         parameterInArray, 1,
				 &parameterOutArray, &NoOfOutParams,
				 0, 0 );

  *result = parameterOutArray[0];
  return( KernelErrorCode );
}


#endif
