/*     @(#)learn_f.c	1.4 9/21/92	*/

/**********************************************************************
FILE   : learn_f.c
PURPOSE: SNNS-Kernel Learning Functions
NOTES  : with following learning functions:
	   - Backpropagation
	   - Backpropagation with momentum term
           - Quickprop
	   - Counterpropagation
	   - BackPercolation

AUTHOR : Niels Mache
DATE   : 01.10.90
VERSION : 1.4  9/21/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <stdio.h>
#include <math.h>
#include <values.h>

#include "kr_typ.h"	 /*  Kernel Types and Constants  */
#include "kr_const.h"	 /*  Constant Declarators for SNNS-Kernel  */
#include "kr_def.h"	 /*  Default Values  */
#include "kernel.h"	 /*  kernel function prototypes  */
#include "kr_mac.h"	 /*  Kernel Macros   */


#define  NET_ERROR( param )      param[ 0 ]  /*    returns the net error   */
#define  LEARN_PARAM1( param )   param[ 0 ]  /*    contains the 1st learning parameter  */
#define  LEARN_PARAM2( param )   param[ 1 ]  /*    contains the 2nd learning parameter  */
#define  LEARN_PARAM3( param )   param[ 2 ]  /*    contains the 3rd learning parameter  */
#define  LEARN_PARAM4( param )   param[ 3 ]  /*    contains the 4th learning parameter  */
#define  LEARN_PARAM5( param )   param[ 4 ]  /*    contains the 5th learning parameter  */

#define  SIGMOID_PRIME_OFFSET    0.1         /*    for modificated sigmoid function    */
#define SQR(x)  (x) * (x)


/*#################################################

GROUP: Global Var's

#################################################*/

extern bool    NetModified,	 /*  TRUE, if the network topology was modified  */
               NetInitialize,    /*  TRUE, if the network has been initialized  */
               LearnFuncHasChanged;  /*  TRUE, if the learning function has been changed  */

extern int  NoOfUnits,		 /*  no. of units in the network  */
	    MinUnitNo,		 /*  the first (lowest) used unit number in the network  */
	    MaxUnitNo,		 /*  the last (highest) used unit number in the network  */
	    NoOfPatternPairs,	 /*  no. of defined pattern pairs */
	    NoOfShuffledPatterns,  /*  no. of entries in the pattern_number array  */

	    NoOfInputPatterns,	 /*  no. of input patterns  */
	    NoOfOutputPatterns,  /*  no. of output patterns  */

	    NoOfInputUnits,	 /*  no. of input units  */
            NoOfOutputUnits,     /*  no. of output units  */
	    NoOfHiddenUnits,	 /*  no. of hidden units  */
	    TopoSortID; 	 /*  topologic mode identifier	*/

extern UnitArray       unit_array;  /*	the unit array	*/

extern TopoPtrArray    topo_ptr_array;	/*  stores pointers to topological sorted units
					    used by kr_topoSort()  */

extern Patterns        in_patterns,	/*  the input pattern array  */
		       out_patterns;	/*  the output pattern array  */
extern PatternNumbers  pattern_numbers;   /*  contains shuffled pattern numbers
					      used by kr_shufflePatterns()  */

extern int     no_of_topo_units;	/*  no. of unit pointers in the topo_ptr_array	*/




/*#################################################

GROUP: Local Vars

#################################################*/

static int  NoOfLearnedPatterns;



/*  --------------   Learning  Functions  ----------------------    */



/*#################################################

GROUP: backpropagation learning function

#################################################*/



/*  topological forward propagation
*/
static	void  propagateNetForward( pattern_no )
int     pattern_no;
{
  register struct Unit   *unit_ptr;
  register Patterns  in_pat;
  register TopoPtrArray     topo_ptr;


  /*  calculate startaddress for input pattern array  */
  in_pat = in_patterns + pattern_no * NoOfInputPatterns;

  topo_ptr = topo_ptr_array;

  /*  copy pattern into input unit's activation and
      calculate output of the input units
  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture (input units first)  */
    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act = *in_pat++;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act = *in_pat++);
    }

  /*  popagate hidden units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture */
    /*  clear error values  */
    unit_ptr->Aux.flint_no = 0.0;

    /*  calculate the activation value of the unit: call the activation function if needed  */
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
    }

  /*  popagate output units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture */
    /*  clear error values  */
    unit_ptr->Aux.flint_no = 0.0;

    /*  calculate the activation value of the unit: call the activation function if needed  */
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
    }
}


/*  Backward error propagation (topological)
*/
static	float propagateNetBackward2( pattern_no, learn_parameter, delta_max )
int     pattern_no;
float   learn_parameter, delta_max;
{
  register struct Link   *link_ptr;
  register struct Site   *site_ptr;
  register struct Unit   *unit_ptr;
  register Patterns      out_pat;
  register float         error,  sum_error,  eta,  devit, learn_error;
  register TopoPtrArray     topo_ptr;


  sum_error = 0.0;    /*  reset network error  */
  eta = learn_parameter;   /*  store learn_parameter in CPU register  */
  /*  calculate address of the output pattern (with number pattern_no + 1)  */
  out_pat = out_patterns + (pattern_no + 1) * NoOfOutputPatterns;

  /*  add 3 to no_of_topo_units because the topologic array contains
      4 NULL pointers  */
  topo_ptr = topo_ptr_array + (no_of_topo_units + 3);

  /*  calculate output units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    devit = *(--out_pat) - unit_ptr->Out.output;  /*  calc. devitation	*/
    if ( (float) fabs( devit ) <= delta_max )  continue;

    sum_error += devit * devit;  /*  sum up the error of the network  */

    /*	calc. error for output units	 */
    error = devit * (unit_ptr->act_deriv_func) ( unit_ptr );
    /*	calc. the error for adjusting weights and bias of the predecessor units  */
    learn_error = eta * error;
    /*	adjust bias value  */
    unit_ptr->bias += learn_error;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direkt links  */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units  */
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        link_ptr->weight += learn_error * link_ptr->to->Out.output;
        }
      }
    else
      {  /*	the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
        {  /*  adjust link weights and calc. sum of errors of the predecessor units */
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        link_ptr->weight += learn_error * link_ptr->to->Out.output;
        }
      }
    }

  /*  calculate hidden units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    /*	calc. the error of the (hidden) unit  */
    error = (unit_ptr->act_deriv_func) ( unit_ptr ) * unit_ptr->Aux.flint_no;
    /*	calc. the error for adjusting weights and bias of the predecessor units  */
    learn_error = eta * error;
    /*	adjust bias value  */
    unit_ptr->bias += learn_error;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direkt links	*/
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units  */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->weight += learn_error * link_ptr->to->Out.output;
	}
      }
    else
      {  /*  the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->weight += learn_error * link_ptr->to->Out.output;
	}
      }
    }

/*  sum_error *= 0.5;    deleted for version 2.1  Guenter Mamier 21.09.92 */
  return( sum_error );  /*  return the error of the network */
}


/*  Backpropagation Learning Function
    Input Parameters:   1 : learning parameter
                        2 : delta max

    Output Parameters:  1 : error of the network (sum of all cycles)

*/
krui_err  LEARN_backprop( start_pattern, end_pattern, parameterInArray, NoOfInParams, parameterOutArray, NoOfOutParams )
int     start_pattern, end_pattern;
float   parameterInArray[],
        * *parameterOutArray;
int  NoOfInParams,
     *NoOfOutParams;
{
  static  float  OutParameter[1];   /*	OutParameter[0] stores the learning error  */
  int	i, pattern_no, no_of_layers;


  KernelErrorCode = KRERR_NO_ERROR;  /*  reset return code  */

  /*  ####  have to be changed (must be 2)  #### */
  if (NoOfInParams < 1)
    {  /*  Not enough input parameters	*/
    KernelErrorCode = KRERR_PARAMETERS;
    return( KernelErrorCode );
  }

  *NoOfOutParams = 1;   /*  One return value is available (the learning error)  */
  *parameterOutArray = OutParameter;   /*  set the output parameter reference  */ 

  if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
    {  /*  Net has been modified or topologic array isn't initialized */
    /*  check the topology of the network  */
    no_of_layers = kr_topoCheck();
    if (KernelErrorCode != KRERR_NO_ERROR)
      /*  an error has occured	*/
      return( KernelErrorCode );

    if (no_of_layers < 2)
      {  /*  the network has less then 2 layers  */
      KernelErrorCode = KRERR_FEW_LAYERS;
      return( KernelErrorCode );
    }

    /*	count the no. of I/O units and check the patterns  */
    if (kr_IOCheck() != KRERR_NO_ERROR)
      return( KernelErrorCode );

    /*	sort units by topology and by topologic type  */
    (void) kr_topoSort( TOPOLOGICAL_FF );
    if ((KernelErrorCode != KRERR_NO_ERROR) && (KernelErrorCode != KRERR_DEAD_UNITS))
      return( KernelErrorCode );

    NetModified = FALSE;
  }


  NET_ERROR( OutParameter ) = 0.0;   /*  reset network error value  */
  for (i = start_pattern; i <= end_pattern; i++)
    {
    if (NoOfShuffledPatterns == NoOfPatternPairs)
      pattern_no = pattern_numbers[i];  /*  shuffle patterns   */
    else
      pattern_no = i;  /*  do not schuffle patterns  */

    propagateNetForward( pattern_no );	/*  Forward propagation */

    /*  Backward propagation  */
    /*  1st parameter is the learning parameter
	2nd parameter is the max. devitation between output pattern and
	the output of the output unit (delta max)
    */
    NET_ERROR( OutParameter ) +=
      propagateNetBackward2( pattern_no,
			     LEARN_PARAM1( parameterInArray ),
			     LEARN_PARAM2( parameterInArray ) );
  }

  return( KernelErrorCode );
}


/*  Backward error propagation (topological)
*/
static	float propagateNetBackwardBatch( pattern_no, delta_max )
int     pattern_no;
float   delta_max;
{
  register struct Link   *link_ptr;
  register struct Site   *site_ptr;
  register struct Unit   *unit_ptr;
  register Patterns      out_pat;
  register float         error,  sum_error,  devit;
  register TopoPtrArray     topo_ptr;


  sum_error = 0.0;    /*  reset network error  */
  /*  calculate address of the output pattern (with number pattern_no + 1)  */
  out_pat = out_patterns + (pattern_no + 1) * NoOfOutputPatterns;

  /*  add 3 to no_of_topo_units because the topologic array contains
      4 NULL pointers  */
  topo_ptr = topo_ptr_array + (no_of_topo_units + 3);

  /*  calculate output units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    devit = *(--out_pat) - unit_ptr->Out.output;  /*  calc. devitation	*/
    if ( (float) fabs( devit ) <= delta_max )  continue;

    sum_error += devit * devit;  /*  sum up the error of the network  */

    /*	calc. error for output units	 */
    error = devit * (unit_ptr->act_deriv_func) ( unit_ptr );
    /*	calc. the error for adjusting weights and bias of the predecessor units  */
    /*	adjust bias value  */
    unit_ptr->value_a += error;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direkt links  */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units  */
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        link_ptr->value_a += error * link_ptr->to->Out.output;
        }
      }
    else
      {  /*	the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
        {  /*  adjust link weights and calc. sum of errors of the predecessor units */
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        link_ptr->value_a += error * link_ptr->to->Out.output;
        }
      }
    }

  /*  calculate hidden units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    /*	calc. the error of the (hidden) unit  */
    error = (unit_ptr->act_deriv_func) ( unit_ptr ) * unit_ptr->Aux.flint_no;
    /*	calc. the error for adjusting weights and bias of the predecessor units  */
    /*	adjust bias value  */
    unit_ptr->value_a += error;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direkt links	*/
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units  */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->value_a += error * link_ptr->to->Out.output;
	}
      }
    else
      {  /*  the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->value_a += error * link_ptr->to->Out.output;
	}
      }
    }

/*  sum_error *= 0.5;    deleted for version 2.1  Guenter Mamier 21.09.92 */
  return( sum_error );  /*  return the error of the network */
}


static krui_err clearDeltas()
{
  register FlagWord      flags;
  register struct Link   *link_ptr;
  register struct Unit   *unit_ptr;
  register struct Site   *site_ptr;


  FOR_ALL_UNITS( unit_ptr )
    {
    flags = unit_ptr->flags;

    if ( (flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      {  /*  unit is in use  */
      unit_ptr->value_a = (FlintType) 0;

      if (flags & UFLAG_SITES)
	{  /*  unit has sites  */
        FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
          link_ptr->value_a = (FlintType) 0;
        }
      else
	{  /*  unit has no sites   */
	if (flags & UFLAG_DLINKS)
	  {  /*  unit has direct links	 */
          FOR_ALL_LINKS( unit_ptr, link_ptr )
            link_ptr->value_a = (FlintType) 0;
          }
        }
      }
    }

  return( KRERR_NO_ERROR );
}


static krui_err updateWeights( eta )
float  eta;
{
  register FlagWord      flags;
  register struct Link   *link_ptr;
  register struct Unit   *unit_ptr;
  register struct Site   *site_ptr;


  FOR_ALL_UNITS( unit_ptr )
    {
    flags = unit_ptr->flags;

    if ( (flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      {  /*  unit is in use  */
      unit_ptr->bias += unit_ptr->value_a * eta;

      if (flags & UFLAG_SITES)
	{  /*  unit has sites  */
        FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
          link_ptr->weight += link_ptr->value_a * eta;
        }
      else
	{  /*  unit has no sites   */
	if (flags & UFLAG_DLINKS)
	  {  /*  unit has direct links	 */
          FOR_ALL_LINKS( unit_ptr, link_ptr )
            link_ptr->weight += link_ptr->value_a * eta;
          }
        }
      }
    }

  return( KRERR_NO_ERROR );
}




/*  Backpropagation Learning Function
    Input Parameters:   1 : learning parameter
                        2 : delta max

    Output Parameters:  1 : error of the network (sum of all cycles)

*/
krui_err  LEARN_backpropBatch( start_pattern, end_pattern, parameterInArray, NoOfInParams, parameterOutArray, NoOfOutParams )
int     start_pattern, end_pattern;
float   parameterInArray[],
        * *parameterOutArray;
int  NoOfInParams,
     *NoOfOutParams;
{
  static  float  OutParameter[1];   /*	OutParameter[0] stores the learning error  */
  int	i, pattern_no, no_of_layers;


  KernelErrorCode = KRERR_NO_ERROR;  /*  reset return code  */

  /*  ####  have to be changed (must be 2)  #### */
  if (NoOfInParams < 1)
    {  /*  Not enough input parameters	*/
    KernelErrorCode = KRERR_PARAMETERS;
    return( KernelErrorCode );
  }

  *NoOfOutParams = 1;   /*  One return value is available (the learning error)  */
  *parameterOutArray = OutParameter;   /*  set the output parameter reference  */ 

  if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
    {  /*  Net has been modified or topologic array isn't initialized */
    /*  check the topology of the network  */
    no_of_layers = kr_topoCheck();
    if (KernelErrorCode != KRERR_NO_ERROR)
      /*  an error has occured	*/
      return( KernelErrorCode );

    if (no_of_layers < 2)
      {  /*  the network has less then 2 layers  */
      KernelErrorCode = KRERR_FEW_LAYERS;
      return( KernelErrorCode );
    }

    /*	count the no. of I/O units and check the patterns  */
    if (kr_IOCheck() != KRERR_NO_ERROR)
      return( KernelErrorCode );

    /*	sort units by topology and by topologic type  */
    (void) kr_topoSort( TOPOLOGICAL_FF );
    if ((KernelErrorCode != KRERR_NO_ERROR) && (KernelErrorCode != KRERR_DEAD_UNITS))
      return( KernelErrorCode );

    NetModified = FALSE;
  }

  clearDeltas();

  NET_ERROR( OutParameter ) = 0.0;   /*  reset network error value  */
  for (i = start_pattern; i <= end_pattern; i++)
    {
    if (NoOfShuffledPatterns == NoOfPatternPairs)
      pattern_no = pattern_numbers[i];  /*  shuffle patterns   */
    else
      pattern_no = i;  /*  do not schuffle patterns  */

    propagateNetForward( pattern_no );	/*  Forward propagation */

    /*  Backward propagation  */
    /*  1st parameter is the learning parameter
	2nd parameter is the max. devitation between output pattern and
	the output of the output unit (delta max)
    */
    NET_ERROR( OutParameter ) +=
      propagateNetBackwardBatch( pattern_no,
			         LEARN_PARAM2( parameterInArray ) );
  }

  updateWeights( LEARN_PARAM1( parameterInArray ) );


  return( KernelErrorCode );
}


/*  ----------  End of backpropagation learning function ----------  */


/*  -----------------  Begin of new learning function -----------------  */


/*#################################################

GROUP: backpropagation learning function with momentum term

#################################################*/

/*  backprop-momentum initialisation
*/
static krui_err initializeBackpropMomentum()
{
  register FlagWord      flags;
  register struct Link   *link_ptr;
  register struct Unit   *unit_ptr;
  register struct Site   *site_ptr;


  FOR_ALL_UNITS( unit_ptr )
    {
    flags = unit_ptr->flags;

    if ( (flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      {  /*  unit is in use  */
      unit_ptr->value_a = (FlintType) 0;

      if (flags & UFLAG_SITES)
	{  /*  unit has sites  */
        FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
          link_ptr->value_b = (FlintType) 0;
        }
      else
	{  /*  unit has no sites   */
	if (flags & UFLAG_DLINKS)
	  {  /*  unit has direct links	 */
          FOR_ALL_LINKS( unit_ptr, link_ptr )
            link_ptr->value_b = (FlintType) 0;
          }
        }
      }
    }

  return( KRERR_NO_ERROR );
}



/*  Backward error propagation (topological) of backpropagation learnig function
    with momentum term and flat spot elimination
*/
static	float Backprop_momentum_FSE( pattern_no, learn_parameter, mu, FSE_term, delta_max )
int     pattern_no;
float   learn_parameter, mu, FSE_term, delta_max;
{
  register struct Link   *link_ptr;
  register struct Site   *site_ptr;
  register struct Unit   *unit_ptr;
  register Patterns      out_pat;
  register float         error,  sum_error,  eta,  devit, learn_error;
  register TopoPtrArray     topo_ptr;


  sum_error = 0.0;    /*  reset network error  */
  eta = learn_parameter;   /*  store learn_parameter in CPU register  */
  /*  calculate address of the output pattern (with number pattern_no + 1)  */
  out_pat = out_patterns + (pattern_no + 1) * NoOfOutputPatterns;


  /*  add 3 to no_of_topo_units because the topologic array contains
      4 NULL pointers  */
  topo_ptr = topo_ptr_array + (no_of_topo_units + 3);

  /*  calculate output units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    devit = *(--out_pat) - unit_ptr->Out.output;  /*  calc. devitation	*/
    if ( (float) fabs( devit ) <= delta_max )  continue;

    sum_error += devit * devit;  /*  sum up the error of the network  */
    /*	calc. error for output units	 */
    error = devit * ((unit_ptr->act_deriv_func) ( unit_ptr ) + FSE_term);

    /*	calc. the error for adjusting weights and bias of the predecessor units  */
    learn_error = eta * error;

    unit_ptr->value_a = learn_error + mu * unit_ptr->value_a;
    /*  adjust bias value  */
    unit_ptr->bias += unit_ptr->value_a;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direkt links  */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units  */
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        link_ptr->weight += learn_error * link_ptr->to->Out.output;
        }
      }
    else
      {  /*	the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
        {  /*  adjust link weights and calc. sum of errors of the predecessor units */
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        link_ptr->weight += learn_error * link_ptr->to->Out.output;
        }
      }
    }


  /*  calculate hidden units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    /*	calc. the error of the (hidden) unit  */
    error = unit_ptr->Aux.flint_no * ((unit_ptr->act_deriv_func) ( unit_ptr ) + FSE_term);
   
    /*  calc. the error for adjusting weights and bias of the predecessor units	*/
    learn_error = eta * error;
    unit_ptr->value_a = learn_error + mu * unit_ptr->value_a;
    /*  adjust bias value  */
    unit_ptr->bias += unit_ptr->value_a;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direkt links	*/
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  adjust link weights and calc. sum of errors of the predecessor units  */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->value_b = learn_error * link_ptr->to->Out.output + mu * link_ptr->value_b;
	link_ptr->weight += link_ptr->value_b;
	}
      }
    else
      {  /*  the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	  {  /*  adjust link weights and calc. sum of errors of the predecessor units */
          if IS_HIDDEN_UNIT( link_ptr->to )
	    /*	this link points to a hidden unit: sum up the error's from previos units  */
	    link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	  link_ptr->value_b = learn_error * link_ptr->to->Out.output + mu * link_ptr->value_b;
	  link_ptr->weight += link_ptr->value_b;
	  }
      }
    }

/*  sum_error *= 0.5;    deleted for version 2.1  Guenter Mamier 21.09.92 */
  return( sum_error );  /*  return the error of the network */
}


/*  Backpropagation Learning Function
    Input Parameters:   1 : learning parameter
                        2 : momentum term
                        3 : flat-spot-elimination value
                        4 : delta max

    Output Parameters:  1 : error of the network (sum of all cycles)

*/
krui_err  LEARN_backpropMomentum( start_pattern, end_pattern, 
                                   parameterInArray, NoOfInParams, 
                                   parameterOutArray, NoOfOutParams )
int     start_pattern, end_pattern;
float   parameterInArray[],
        * *parameterOutArray;
int  NoOfInParams,
     *NoOfOutParams;
{
  static  float  OutParameter[1];   /*	OutParameter[0] stores the learning error  */
  int   i, ret_code, pattern_no;


  if (NoOfInParams < 1)          /*  ####  have to be changed (must be 2)  #### */
    return( KRERR_PARAMETERS );  /*  Not enough input parameters  */

  *NoOfOutParams = 1;   /*  One return value is available (the learning error)  */
  *parameterOutArray = OutParameter;   /*  set the output parameter reference  */ 
  ret_code = KRERR_NO_ERROR;  /*  reset return code  */

  if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
    {  /*  Net has been modified or topologic array isn't initialized */
    /*  check the topology of the network  */
    ret_code = kr_topoCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );  /*  an error has occured  */
    if (ret_code < 2)  return( KRERR_FEW_LAYERS );  /*  the network has less then 2 layers  */

    /*	count the no. of I/O units and check the patterns  */
    ret_code = kr_IOCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );

    /*	sort units by topology and by topologic type  */
    ret_code = kr_topoSort( TOPOLOGICAL_FF );
    if ((ret_code != KRERR_NO_ERROR) && (ret_code != KRERR_DEAD_UNITS))
      return( ret_code );

    NetModified = FALSE;
    }

  if (NetInitialize || LearnFuncHasChanged)
    {  /*  Net has been modified or initialized, initialize backprop now  */
    ret_code = initializeBackpropMomentum();
    if (ret_code != KRERR_NO_ERROR)  return( ret_code );
    }

  NET_ERROR( OutParameter ) = 0.0;   /*  reset network error value  */
  for (i = start_pattern; i <= end_pattern; i++)
    {
    if (NoOfShuffledPatterns == NoOfPatternPairs)
      pattern_no = pattern_numbers[i];  /*  shuffle patterns   */
    else
      pattern_no = i;  /*  do not schuffle patterns  */

    propagateNetForward( pattern_no );	/*  Forward propagation */

    /*  Backward propagation  */
    /*  1st parameter is the learning parameter
        2nd parameter is the momentum term
        3rd parameter is the flat-spot-elimination value
        4th parameter is the max. devitation between output pattern and the output of the output unit
        (delta max)
    */
    NET_ERROR( OutParameter ) +=
      Backprop_momentum_FSE( pattern_no,
                             LEARN_PARAM1( parameterInArray ),
                             LEARN_PARAM2( parameterInArray ),
                             LEARN_PARAM3( parameterInArray ),
                             LEARN_PARAM4( parameterInArray ) );
    }

  return( ret_code );
}





/*#################################################

GROUP: quickpropagation learning function
       by Peter Zimmerer, modified by Niels Mache

#################################################*/



/*  quickprop initialisation
*/
static krui_err initializeQuickprop()
{
  register unsigned short    flags;
  register struct Link   *link_ptr;
  register struct Unit   *unit_ptr;
  register struct Site   *site_ptr;


  FOR_ALL_UNITS( unit_ptr )
    {
    flags = unit_ptr->flags;

    if ( (flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      {  /*  unit is in use  */
      unit_ptr->value_a = unit_ptr->value_b =
      unit_ptr->value_c = (FlintType) 0;

      if (flags & UFLAG_SITES)
	{  /*  unit has sites  */
	FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
          link_ptr->value_a = link_ptr->value_b = link_ptr->value_c = (FlintType) 0;
        }
      else
	{  /*  unit has no sites   */
	if (flags & UFLAG_DLINKS)
	  {  /*  unit has direct links	 */
	  FOR_ALL_LINKS( unit_ptr, link_ptr )
            link_ptr->value_a = link_ptr->value_b = link_ptr->value_c = (FlintType) 0;
          }
        }
      }
    }

  return( KRERR_NO_ERROR );
}



/*  quickprop backward error propagation
    (topological) for quickprop with SIGMOID_PRIME_OFFSET
    batch-modus: without adaption of links and bias
*/
static float  propagateNetBackwardQuickprop( pattern_no, delta_max )
int pattern_no;    /*  number of actual pattern  */
float  delta_max;  /*  delta max  */
{
  register struct Link *link_ptr;
  register struct Site *site_ptr;
  register struct Unit *unit_ptr;
  register Patterns out_pat;
  register float error,                    /*  error  */
                 sum_error,                /*  sum of the error  */
                 devit;                    /*  deviation  */
  TopoPtrArray topo_ptr;


  sum_error = 0.0;  /*  reset network error  */

  /*  calculate address of the output pattern (with number pattern_no + 1)  */
  out_pat = out_patterns + (pattern_no + 1) * NoOfOutputPatterns;

  /*  add 3 to no_of_topo_units because the topologic array contains
      4 NULL pointers  */
  topo_ptr = topo_ptr_array + (no_of_topo_units + 3);

  /*  calculate output units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    devit = *(--out_pat) - unit_ptr->Out.output;
       /*= o * (1.0 - o) in [0.0,0.25],*/
       /*for asymmetric logistic function*/

    if ( (float) fabs( devit ) <= delta_max )  continue;

    sum_error += devit * devit;  /*  sum up the error of the network  */

    /*	calc. error for output units	 */
    error = devit * ((unit_ptr->act_deriv_func) ( unit_ptr ) + SIGMOID_PRIME_OFFSET);

    unit_ptr->value_c += -error /* * 1 */;   /*  calculate the bias slopes  */
                                            /*  learn bias like a weight  */
    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direct links  */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
        {  /*	calculate the slopes  */
        link_ptr->value_c += - error * link_ptr->to->Out.output;
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        }
      }
    else
      {  /*  the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	{  /*  calculate the value_cs  */
	link_ptr->value_c += - error * link_ptr->to->Out.output;
	link_ptr->to->Aux.flint_no += link_ptr->weight * error;
	}
      }
    }


  /*  calculate hidden units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    error = ((unit_ptr->act_deriv_func) (unit_ptr) + SIGMOID_PRIME_OFFSET) * unit_ptr->Aux.flint_no;

    unit_ptr->value_c += - error /* * 1 */;   /*  calculate the bias slopes  */
                                            /*  learn bias like a weight  */
    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direct links  */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{  /*  calculate the slopes  */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->value_c += - error * link_ptr->to->Out.output;
	}
      }
    else
      {  /*  the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	{  /*  calculate the slopes  */
        if IS_HIDDEN_UNIT( link_ptr->to )
	  /*  this link points to a hidden unit: sum up the error's from previos units  */
	  link_ptr->to->Aux.flint_no += link_ptr->weight * error;

	link_ptr->value_c += - error * link_ptr->to->Out.output;
	}
      }
    }

/*  sum_error *= 0.5;    deleted for version 2.1  Guenter Mamier 21.09.92 */
 return( sum_error );  /*  return the error of the network  */
}



static void MODI_quickprop( learn_parameter, /*modificate all weights and bias*/
                            max_factor,      /*of the net after every epoch*/
                            decay )
float learn_parameter,                    /*learning parameter*/
      max_factor,                         /*maximal grow factor of weights*/
      decay;                              /*decay factor*/

{
  double deltaw;			   /*actual weight (bias) change*/
  float shfac;				   /*shrink factor*/
  register struct Link *link_ptr;
  register struct Site *site_ptr;
  register struct Unit *unit_ptr;
  TopoPtrArray topo_ptr;
  bool hidden_units;


  /*  maximal grow factor of weights is max_factor  */
  shfac = max_factor / (1.0 + max_factor);

  topo_ptr = topo_ptr_array + (NoOfInputPatterns + 1);
  hidden_units = TRUE;  

  /*  calculate hidden and output units only  */
  do
    {
    if ((unit_ptr = *++topo_ptr) == NULL)
      {
      if (!hidden_units)  break;  /*  end of topologic pointer array reached  */
      unit_ptr = *++topo_ptr;  /*  skip NULL pointer  */
      hidden_units = FALSE;
      }

    deltaw = 0.0;  /*  adjust bias like a weight  */
    if (unit_ptr->value_a > 0.0)
      {  /*  previous step was positive  */
      if (unit_ptr->value_c < 0.0)
	/*  same direction,i.e. slope, value_b have same sign  */
	deltaw += learn_parameter * (- unit_ptr->value_c);

      if (unit_ptr->value_c <= shfac * unit_ptr->value_b)
	/*  maximal positive step  */
	deltaw += max_factor * unit_ptr->value_a;
      else
	/*  littler positive step squared approximation  */
	deltaw += unit_ptr->value_c / (unit_ptr->value_b - unit_ptr->value_c)
		  * unit_ptr->value_a;
      }
    else
      if (unit_ptr->value_a < 0.0)
	{  /*  previous step was negative  */
	if (unit_ptr->value_c > 0.0)
	   /*  same direction,i.e. slope, prevslope have same sign  */
	  deltaw += learn_parameter * (- unit_ptr->value_c);

	if (unit_ptr->value_c >= shfac * unit_ptr->value_b)
	  /*  maximal negative step  */
	  deltaw += max_factor * unit_ptr->value_a;
	else
	  /* littler negative step squared approximation */
	  deltaw += unit_ptr->value_c / (unit_ptr->value_b - unit_ptr->value_c)
		    * unit_ptr->value_a;
	}
      else
	/*  previous step was 0.0  */
	/*  start of learning process with BP  */
	deltaw += learn_parameter * (- unit_ptr->value_c);

    unit_ptr->bias += deltaw;		    /*new bias*/
    unit_ptr->value_a = deltaw;	    /*bias change*/
    unit_ptr->value_b = unit_ptr->value_c;  /*previous slope*/
    unit_ptr->value_c = decay * unit_ptr->bias; /*  set new slope  */

    /*adjust links*/
    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /*  the unit has direct links	*/
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{
	deltaw = 0.0;

	if (link_ptr->value_a > 0.0)  /*  previous step was positive*/
	  {
	  if (link_ptr->value_c < 0.0)
	    /*	same direction,i.e. slope, prevslope have same sign  */
	    deltaw += learn_parameter * (- link_ptr->value_c);

	  if (link_ptr->value_c <= shfac * link_ptr->value_b)
	    /*	maximal positive step  */
	    deltaw += max_factor * link_ptr->value_a;
	  else
	    deltaw += link_ptr->value_c / (link_ptr->value_b - link_ptr->value_c)
		    * link_ptr->value_a;
	  }
	else
	  if (link_ptr->value_a < 0.0)  /*	previous step was negative*/
	    {
	    if (link_ptr->value_c > 0.0)
	      /*  same direction,i.e. slope, prevslope have same sign */
	      deltaw += learn_parameter * (- link_ptr->value_c);

	    if (link_ptr->value_c >= shfac * link_ptr->value_b)
	      /*  maximal negative step  */
	      deltaw += max_factor * link_ptr->value_a;
	    else
	      deltaw += link_ptr->value_c / (link_ptr->value_b - link_ptr->value_c)
			* link_ptr->value_a;
	    }
	  else	/*  previous step was 0.0  */
	    /*	start of learning process with BP  */
	    deltaw += learn_parameter * (- link_ptr->value_c);

	link_ptr->weight += deltaw;	      /*new weight*/
	link_ptr->value_a = deltaw;       /*weight change*/
	link_ptr->value_b = link_ptr->value_c;	/*previous slope*/
	/*  set new slope  */
	link_ptr->value_c = decay * link_ptr->weight;
	}  /*  for links  */
      }
    else
      {  /*  the unit has sites  */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	{
	deltaw = 0.0;
	if (link_ptr->value_a > 0.0)
	  {  /*  previous step was positive  */
	  if (link_ptr->value_c < 0.0)
	    /*	same direction,i.e. slope, prevslope have same sign  */
	    deltaw += learn_parameter * (- link_ptr->value_c);

	  if (link_ptr->value_c <= shfac * link_ptr->value_b)
	    /*	maximal positive step  */
	    deltaw += max_factor * link_ptr->value_a;
	  else
	    /*	littler positive step squared approximation  */
	    deltaw += link_ptr->value_c / (link_ptr->value_b - link_ptr->value_c)
		      * link_ptr->value_a;
	  }
	else
	  if (link_ptr->value_a < 0.0)
	    {  /*  previous step was negative  */
	    if (link_ptr->value_c > 0.0)
	      /*  same direction,i.e. slope, prevslope have same sign  */
	      deltaw += learn_parameter * (- link_ptr->value_c);

	    if (link_ptr->value_c >= shfac * link_ptr->value_b)
	      /*  maximal negative step  */
	      deltaw += max_factor * link_ptr->value_a;
	    else
	      deltaw += link_ptr->value_c / (link_ptr->value_b - link_ptr->value_c)
			* link_ptr->value_a;
	    }
	  else	/*  previous step was 0.0  */
	    /*	start of learning process with BP  */
	    deltaw += learn_parameter * (- link_ptr->value_c);

	link_ptr->weight += deltaw;	     /*new weight*/
	link_ptr->value_a = deltaw;      /*weight change*/
	link_ptr->value_b = link_ptr->value_c;	/*previous slope*/
	/*set new slope*/
	link_ptr->value_c = decay * link_ptr->weight;
	}
      }
    }  /*  for units  */
  while( TRUE );

}


/*  Quickprop learning function
    Input Parameters:   1 : learning parameter
                        2 : max factor (of the net after every epoch)
                        3 : decay
                        4 : delta max

    Output Parameters:  1 : error of the network (sum of all cycles)
*/
krui_err LEARN_quickprop( start_pattern, end_pattern,
			 parameterInArray, NoOfInParams,
			 parameterOutArray, NoOfOutParams )
int  start_pattern, end_pattern;
float  parameterInArray[], * *parameterOutArray;
int  NoOfInParams, *NoOfOutParams;
{
  static float OutParameter[1];           /*OutParameter[0] stores the*/
                                          /*learning error*/
  int  i, pattern_no, ret_code;


  if (NoOfInParams < 1)    /* #########   have to be changed  (must be 3)  ######### */
    return( KRERR_PARAMETERS );            /*  not enough input parameters  */

  *NoOfOutParams = 1;                      /*  one return value is available (the learning error) */

  *parameterOutArray = OutParameter;       /*  set output parameter reference  */
  ret_code = KRERR_NO_ERROR;               /*  reset return code  */

  if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
    {  /*  Net has been modified or topologic array isn't initialized */
    /*  check the topology of the network  */
    ret_code = kr_topoCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );  /*  an error has occured  */
    if (ret_code < 2)  return( KRERR_FEW_LAYERS );  /*  the network has less then 2 layers  */

    /*	count the no. of I/O units and check the patterns  */
    ret_code = kr_IOCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );

    /*	sort units by topology and by topologic type  */
    ret_code = kr_topoSort( TOPOLOGICAL_FF );
    if ((ret_code != KRERR_NO_ERROR) && (ret_code != KRERR_DEAD_UNITS))
      return( ret_code );

    NetModified = FALSE;
    }

  if (NetInitialize || LearnFuncHasChanged)
    {  /*  Net has been modified or initialized, initialize backprop now  */
    ret_code = initializeQuickprop();
    if (ret_code != KRERR_NO_ERROR)  return( ret_code );
    }


  NET_ERROR( OutParameter ) = 0.0;   /*  reset network error value  */
  for (i = start_pattern; i <= end_pattern; i++)
    {
    if (NoOfShuffledPatterns == NoOfPatternPairs)
      pattern_no = pattern_numbers[i];  /*  shuffle patterns   */
    else
      pattern_no = i;  /*  do not schuffle patterns  */

    propagateNetForward( pattern_no );         /*  forward propagation  */

    /*  backward propagation and summation of gradient  */
    NET_ERROR(OutParameter) +=
      propagateNetBackwardQuickprop( pattern_no,
                                     LEARN_PARAM4( parameterInArray ) );
    }

  /*  modificate links and bias  */
  MODI_quickprop( LEARN_PARAM1( parameterInArray ),
                  LEARN_PARAM2( parameterInArray ),
                  LEARN_PARAM3( parameterInArray ) );

  return( ret_code );
}



/*#################################################

GROUP: Counterpropagation learning function

#################################################*/


/*  Counterpropagation initialisation
*/
static krui_err  initializeCPN()
{
  register struct Unit   *unit_ptr;


  NoOfLearnedPatterns = 0;

  /*  set unit's bias to zero  */
  FOR_ALL_UNITS( unit_ptr )
    if ( (unit_ptr->flags & UFLAG_IN_USE) == UFLAG_IN_USE)
      /*  unit is in use  */
      unit_ptr->bias = (FlintType) 0;

  return( KRERR_NO_ERROR );
}


static void normalize_weight( winner_ptr, sum )
struct Unit *winner_ptr;
float       sum;
{
  register struct Site *site_ptr;
  register struct Link *link_ptr;
  register float amount;


  amount = 1.0 / sqrt( sum );

  if (winner_ptr->flags & UFLAG_SITES)
    /* the unit has sites */
    FOR_ALL_SITES_AND_LINKS( winner_ptr, site_ptr, link_ptr )
      link_ptr->weight = link_ptr->weight * amount;
  else 
    /* the unit has direct links */
    FOR_ALL_LINKS( winner_ptr, link_ptr )
      link_ptr->weight = link_ptr->weight * amount;
}


static void normalize_inputvector( sum )
float sum;
{
  register struct Unit *unit_ptr;
  register float amount;


  amount = 1.0 / sqrt( sum );

  FOR_ALL_UNITS( unit_ptr )
    if (IS_INPUT_UNIT( unit_ptr ) && UNIT_IN_USE( unit_ptr ))
      /* this is a input unit */
      unit_ptr->Out.output = unit_ptr->Out.output * amount;
}  


/************************************************************************
 *                   function propagateNet_CPN                          *
 ************************************************************************
 *                                                                      *
 ************************************************************************/

static float propagateNet_CPN( pattern_no, alpha, beta, threshold )
int    pattern_no;
float  alpha;
float  beta;
float  threshold;
{
  register struct Link    *link_ptr;
  register struct Site    *site_ptr;
  register struct Unit    *unit_ptr;
  register struct Unit    *winner_ptr;
  register Patterns       in_pat, out_pat;
  float          maximum, sum_error, devit, learn_error, sum;
  float          unit_ptr_net;
  float          noOfPatterns_mul_NoHiddenUnits;
  register TopoPtrArray     topo_ptr;


  /***************************************************************/
  /*      calculate the activation and the output values         */
  /*      of the input units (Input Layer)                       */
  /***************************************************************/ 

  noOfPatterns_mul_NoHiddenUnits = (float) NoOfLearnedPatterns * (float) NoOfHiddenUnits;

  sum = 0.0;

  /*  calculate startaddress for input pattern array  */
  in_pat = in_patterns + pattern_no * NoOfInputPatterns;

  topo_ptr = topo_ptr_array;

  /*  copy pattern into input unit's activation and
      calculate output of the input units
  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to the unit stuctures
       (sorted by: input-, hidden- and output-units, separated with NULL pointers)
       */
    sum += *in_pat * *in_pat;

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act = *in_pat++;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act = *in_pat++);
    }

   if (sum != 0.0)
     /* normalize the inputvector */
     normalize_inputvector( sum );


  /***************************************************************/
  /*              propagate Kohonen Layer                        */
  /***************************************************************/

  /***************************************************************/
  /*      calculate the activation and the output values         */
  /*      of the hidden units (Kohonen Layer)                    */
  /***************************************************************/

  winner_ptr = NULL;
  maximum = -1.0e30;  /* contains the maximum of the activations */

  /*  popagate hidden units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture */
    unit_ptr_net = 0.0;
    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /* the unit has direct links */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
        unit_ptr_net += (link_ptr->weight * link_ptr->to->Out.output);
      }
    else
      { /*  the unit has sites	*/
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	unit_ptr_net += (link_ptr->weight * link_ptr->to->Out.output);
      }

    if (unit_ptr->bias >= noOfPatterns_mul_NoHiddenUnits)
      unit_ptr_net -= threshold;

    if (maximum < unit_ptr_net)
      {  /*  determine winner unit  */
      winner_ptr = unit_ptr;
      maximum = unit_ptr_net;
      }

    /*  reset output and activation of hidden units  */
    unit_ptr->Out.output = unit_ptr->act = (FlintType) 0;
    }

  /***************************************************************/
  /*             the competitive winner is chosen                */
  /***************************************************************/
  winner_ptr->Out.output = winner_ptr->act = (FlintType) 1;
  winner_ptr->bias++;


  /***************************************************************/
  /*                Training the Kohonen Layer                   */
  /*                                                             */
  /* Only the weights of links that go to the winning unit are   */
  /* adjusted, the others remain the same.                       */
  /* The incoming weights to the competitive unit are adapted as */
  /* follows:                                                    */
  /*                                                             */
  /* weight(new) = weight(old) + alpha * (output - weight(old))  */
  /*                                                             */
  /* where eta is the learning constant (0<eta<=1.0)             */
  /* and output is the output of the input unit                  */
  /*                                                             */
  /***************************************************************/


  sum = 0.0;
  if (winner_ptr->flags & UFLAG_DLINKS)
    {  /*  the winner unit has direct links  */
    FOR_ALL_LINKS( winner_ptr, link_ptr )
      {
      devit             = link_ptr->to->Out.output - link_ptr->weight;
      learn_error       = alpha * devit;
      link_ptr->weight += learn_error;        
      /* this is needed for the normalization of the weight_vector */
      sum              += link_ptr->weight * link_ptr->weight;
      }
    }
  else
    {  /*  the winner unit has sites  */
    FOR_ALL_SITES_AND_LINKS( winner_ptr, site_ptr, link_ptr )
      {
      devit		= link_ptr->to->Out.output - link_ptr->weight;
      learn_error	= alpha * devit;
      link_ptr->weight += learn_error;
      /* this is needed for the normalization of the weight_vector */
      sum	       += link_ptr->weight * link_ptr->weight;
      }
    }

  if (sum != 0.0)
     normalize_weight( winner_ptr, sum );

  /***************************************************************/
  /*              propagate Grossberg Layer                      */
  /***************************************************************/
  /*                    AND                                      */
  /***************************************************************/
  /*              Training the Grossberg Layer                   */
  /*                                                             */
  /* Adaptation of the Grossberg Layer weights is done by the    */
  /* Widrow-Hoff rule:                                           */
  /*                                                             */
  /* weight(new) = weight(old) + beta * (target output - output) */
  /*                                                             */
  /* for all weights connected with the winning unit of the      */
  /* Kohonen Layers                                              */
  /*                                                             */
  /***************************************************************/




  /* calculate address of the output pattern */
  out_pat = out_patterns + pattern_no * NoOfOutputPatterns;

  sum_error = 0.0;

  /*  popagate output units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture */

    /***************************************************************/
    /*      calculate the activation and the output values         */
    /*      of the output units (Grossberg Layer)                  */
    /***************************************************************/
    /* the activation function is the identity function (weighted sum)
       and identity output function
    */
    unit_ptr->Out.output = unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    devit       = *out_pat++ - unit_ptr->Out.output; /* calculate devitation */
    sum_error  += devit * devit;
    learn_error = beta * devit ;

    if (UNIT_HAS_DIRECT_INPUTS( unit_ptr ))
      {  /* the unit has direct links */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
        if ( link_ptr->to == winner_ptr )
          {
           /* link to the winning unit of the Kohonen Layer */
           link_ptr->weight += learn_error;
           break;
	  }
      }
    else
      {  /* the unit has sites */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	if ( link_ptr->to == winner_ptr )
	  {
	  /* link to the winning unit of the Kohonen Layer */
	  link_ptr->weight  += learn_error;
	  break;
          }
      }

    }

/*  sum_error *= 0.5;    deleted for version 2.1  Guenter Mamier 21.09.92 */
  return( sum_error );
}



krui_err  LEARN_CPN( start_pattern, end_pattern, parameterInArray, NoOfInParams, parameterOutArray, NoOfOutParams )
int     start_pattern, end_pattern;
float   parameterInArray[],
        * *parameterOutArray;
int  NoOfInParams,
     *NoOfOutParams;
{
  static  float  OutParameter[1];   /*  OutParameter[0] stores the learning error  */
  int     ret_code, i, pattern_no;


  if (NoOfInParams < 1)          /*  have to be changed (must be 3) */
    return( KRERR_PARAMETERS );  /*  Not enough input parameters  */

  *NoOfOutParams = 1;                   /*  one return value is available (the learning error) */
  *parameterOutArray = OutParameter;    /*  set output parameter reference  */
  ret_code = KRERR_NO_ERROR;            /*  clear return code  */


  if (NetModified || (TopoSortID != TOPOLOGIC_TYPE))
    {  /*  Net has been modified or topologic array isn't initialized */
    /*  check the topology of the network  */
    ret_code = kr_topoCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );  /*  an error has occured  */
    if (ret_code != 3)  return( KRERR_FEW_LAYERS );  /*  the network has less then 2 layers  */

    /*	count the no. of I/O units and check the patterns  */
    ret_code = kr_IOCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );

    /*	sort units by topology and by topologic type  */
    ret_code = kr_topoSort( TOPOLOGIC_TYPE );
    if ((ret_code != KRERR_NO_ERROR) && (ret_code != KRERR_DEAD_UNITS))
      return( ret_code );

    NetModified = FALSE;
    }

  if (NetInitialize || LearnFuncHasChanged)
    {  /*  Net has been modified or initialized, initialize backprop now  */
    ret_code = initializeCPN();
    if (ret_code != KRERR_NO_ERROR)  return( ret_code );
    }


  NET_ERROR( OutParameter ) = 0.0;   /*  reset network error value  */
  for (i = start_pattern; i <= end_pattern; i++)
    {
    if (NoOfShuffledPatterns == NoOfPatternPairs)
      pattern_no = pattern_numbers[i];  /*  shuffle patterns   */
    else
      pattern_no = i;  /*  do not schuffle patterns  */

    NoOfLearnedPatterns++;
    NET_ERROR( OutParameter ) +=
      propagateNet_CPN( pattern_no,
                        LEARN_PARAM1( parameterInArray ),
                        LEARN_PARAM2( parameterInArray ),
                        LEARN_PARAM3( parameterInArray ) );
    }

  return( ret_code );
}




/* *******************************************************

  Back-Percolation Learning Function
  by Artemis Hatzigeorgiou
  Algorithm by Mark Jurik

*/

/*  topological forward propagation
*/
static	void  propagateNetForward_perc( pattern_no )
int     pattern_no;
{
  register struct Unit   *unit_ptr;
  register Patterns  in_pat;
  register TopoPtrArray     topo_ptr;


  /*  calculate startaddress for input pattern array  */
  in_pat = in_patterns + pattern_no * NoOfInputPatterns;

  topo_ptr = topo_ptr_array;

  /*  copy pattern into input unit's activation and
      calculate output of the input units
  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture (input units first)  */
    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act = *in_pat++;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act = *in_pat++);
    }

  /*  popagate hidden units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture */
    /*  clear values  */
    unit_ptr->Aux.flint_no = 0.0;
    unit_ptr->value_a = 0.0;
    unit_ptr->value_b = 0.000001;

    /*  calculate the activation value of the unit: call the activation function if needed  */
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
    }

  /*  popagate output units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  topo_ptr points to a (topological sorted) unit stucture */
    /*  clear values  */
    unit_ptr->Aux.flint_no = 0.0;
    unit_ptr->value_a = 0.0;
    unit_ptr->value_b = 0.000001;

    /*  calculate the activation value of the unit: call the activation function if needed  */
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
    }
}


/*  Backward error propagation (topological)
*/
static	float propagateNetBackward_perc( pattern_no, learn_parameter, delta_max ,perc_error)
int     pattern_no;
float   learn_parameter, delta_max,  *perc_error;
{
  register struct Link   *link_ptr;
  register struct Unit   *unit_ptr;
  register Patterns      out_pat;
  register float	 error,  sum_error,  eta,  devit;
  register TopoPtrArray     topo_ptr;
  register  float     norm, delta_sig_normaliser, message_weight;
  register  float     act_err, normalised_error, scaled_error, delta_weight_normaliser;
  register  float     der = 0.0;

  sum_error = 0.0;    /*  reset network error  */
  eta = learn_parameter;   /*  store learn_parameter in CPU register  */
  /*  calculate address of the output pattern (with number pattern_no + 1)  */
  out_pat = out_patterns + (pattern_no + 1) * NoOfOutputPatterns;

  /*  add 3 to no_of_topo_units because the topologic array contains
      4 NULL pointers  */
  topo_ptr = topo_ptr_array + (no_of_topo_units + 3);

  /*  calculate output units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    devit = *(--out_pat) - unit_ptr->Out.output;  /*  calc. devitation	*/

    if ( fabs( devit ) > delta_max )
      {  /*  calc. error for output units     */
      *perc_error += fabs(devit);
      error = -2.0 * devit * (unit_ptr->act_deriv_func) ( unit_ptr );
      act_err = devit*eta; 
      sum_error += devit * devit;  /*  sum up the error of the network	*/
      }
    else
      {  /*  set error of output units to zero	*/
      error = 0.0;
      act_err = devit*eta; 
      }

    /*	calc. the error for adjusting weights and bias of the predecessor units  */
  

        norm = 0.0;
        delta_sig_normaliser = 0.000001;     
        FOR_ALL_LINKS( unit_ptr, link_ptr )
	  {  /*  adjust link weights and calc. sum of errors of the predecessor units  */
          link_ptr->to->Aux.flint_no += link_ptr->weight * error;
          if IS_INPUT_UNIT( link_ptr->to )
            norm += fabs( link_ptr->weight );
          delta_sig_normaliser += SQR(link_ptr->to->Out.output);
	  }
        delta_weight_normaliser =   delta_sig_normaliser + 1;
        norm += delta_sig_normaliser;
        normalised_error = act_err/norm; 
        scaled_error = act_err/delta_weight_normaliser;
        FOR_ALL_LINKS( unit_ptr, link_ptr )    
          {         
          message_weight = (error * link_ptr->weight) * (error * link_ptr->weight);
          link_ptr->to->value_a += link_ptr->weight * normalised_error * message_weight;
          link_ptr->to->value_b += message_weight;
          link_ptr->weight += link_ptr->to->Out.output * scaled_error;
	  }


    /*	adjust bias value  */
    unit_ptr->bias += scaled_error;
    }

  /*  calculate hidden units only  */
  while ((unit_ptr = *--topo_ptr) != NULL)
    {
    der =  (unit_ptr->act_deriv_func) ( unit_ptr );
    error = der * unit_ptr->Aux.flint_no;
    act_err = (unit_ptr->value_a / unit_ptr->value_b) * der ;  
    
    /*	calc. the error for adjusting weights and bias of the predecessor units  */
 

      norm = 0.0;
      delta_sig_normaliser = 0.000001;    
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	{ 
        link_ptr->to->Aux.flint_no += link_ptr->weight * error;
        if IS_HIDDEN_UNIT( link_ptr->to )
            norm += fabs( link_ptr->weight );

        delta_sig_normaliser += SQR(link_ptr->to->Out.output);
        }
      delta_weight_normaliser =   delta_sig_normaliser + 1;
      norm += delta_sig_normaliser;
      normalised_error = act_err/norm; 
      scaled_error = act_err/delta_weight_normaliser;
      FOR_ALL_LINKS( unit_ptr, link_ptr ) 
        {
	message_weight = (error * link_ptr->weight) * (error * link_ptr->weight);
        link_ptr->to->value_a += link_ptr->weight * normalised_error * message_weight;
        link_ptr->to->value_b += message_weight;
        link_ptr->weight += link_ptr->to->Out.output * scaled_error;
        }


    /*	adjust bias value  */
    unit_ptr->bias += scaled_error;
   }


/*  sum_error *= 0.5;    deleted for version 2.1  Guenter Mamier 21.09.92 */
  return( sum_error );  /*  return the error of the network */
}


/*  BackPercolation Learning Function
    Input Parameters:   1 : learning parameter
                        2 : delta max

    Output Parameters:  1 : error of the network (sum of all cycles)

*/
krui_err  LEARN_perc( start_pattern, end_pattern, parameterInArray, NoOfInParams, parameterOutArray, NoOfOutParams )
int     start_pattern, end_pattern;
float   parameterInArray[],
        * *parameterOutArray;
int  NoOfInParams,
     *NoOfOutParams;
{
  static  float  OutParameter[1];   /*	OutParameter[0] stores the learning error  */
  int	  i, ret_code, pattern_no;
  float   p_error,l_error;
  register struct Unit   *unit_ptr;

  if (NoOfInParams < 1)          /*  ####  have to be changed (must be 2)  #### */
    return( KRERR_PARAMETERS );  /*  Not enough input parameters  */

  *NoOfOutParams = 1;   /*  One return value is available (the learning error)  */
  *parameterOutArray = OutParameter;   /*  set the output parameter reference  */ 
  ret_code = KRERR_NO_ERROR;  /*  reset return code  */

  if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
    {  /*  Net has been modified or topologic array isn't initialized */
       /*  check the topology of the network  */
    FOR_ALL_UNITS( unit_ptr )
      if UNIT_HAS_SITES( unit_ptr )
        return( KRERR_SITES_NO_SUPPORT );
    
    ret_code = kr_topoCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );  /*  an error has occured  */
    if (ret_code < 2)  return( KRERR_FEW_LAYERS );  /*  the network has less then 2 layers  */

    /*	count the no. of I/O units and check the patterns  */
    ret_code = kr_IOCheck();
    if (ret_code < KRERR_NO_ERROR)  return( ret_code );

    /*	sort units by topology and by topologic type  */
    ret_code = kr_topoSort( TOPOLOGICAL_FF );
    if ((ret_code != KRERR_NO_ERROR) && (ret_code != KRERR_DEAD_UNITS))
      return( ret_code );

    NetModified = FALSE;
    }

  if (NetInitialize || LearnFuncHasChanged)
    {  /*  Net has been modified or initialized, initialize backprop now  */
    if (ret_code != KRERR_NO_ERROR)  return( ret_code );
    parameterInArray[1] = 1.0;
    }

  NET_ERROR( OutParameter ) = 0.0;   /*  reset network error value  */
  p_error = 0.0;
  for (i = start_pattern; i <= end_pattern; i++)
    {
    if (NoOfShuffledPatterns == NoOfPatternPairs)
      pattern_no = pattern_numbers[i];  /*  shuffle patterns   */
    else
      pattern_no = i;  /*  do not schuffle patterns  */

    propagateNetForward_perc( pattern_no );	/*  Forward propagation */

    /*  Backward propagation  */
    /*  1st parameter is the learning parameter
        2nd parameter is the max. devitation between output pattern and the output of the output unit
        (delta max)
    */

      NET_ERROR( OutParameter ) +=
	propagateNetBackward_perc( pattern_no,
			      LEARN_PARAM1( parameterInArray ),
                              LEARN_PARAM5( parameterInArray ), &p_error ); 
    }

     p_error = p_error /( NoOfPatternPairs * NoOfOutputUnits ) ;
    l_error = exp( ( parameterInArray[1] - p_error ) / (parameterInArray[1] + p_error ) ); 
    parameterInArray[1] = p_error;

    if( l_error <= 0.5)    l_error = 0.5;
    else  if( l_error >= 1.05)   l_error = 1.05; 
 
  
    if( ( l_error = parameterInArray[0] * l_error ) < 0.8)
          parameterInArray[0] = l_error;
   return( ret_code );
}

/*#################################################

GROUP: Radial Basis Functions Learning

AUTHOR: Michael Vogt

#################################################*/

/*
 * Use of special entries in links and units:
 *
 * for Units in hidden layer:
 * Unit value_a: |X - L| ^ 2  == norm ^ 2 == square of euklidean distance
 *               between all links and all input units to this unit.
 * Unit value_b: delta_BIAS == sum of all deltas to BIAS during learning
 * Unit value_c: Backpropagated weighted sum of errors in output layer
 *
 * for Units in output layer:
 * Unit value_a: error (y_learn - y_net) during learning current pattern
 * Unit value_b: delta_BIAS == sum of all deltas to BIAS during learning
 *
 * for links between input and hidden layer:
 * Link value_b: delta for this link during learning (link treated as vector)
 * Link value_a: Momentum term for this link (last change)
 *
 * for links between hidden and output layer:
 * Link value_b: delta for weight of this link during learning.
 * Link value_a: Momentum term for this link (last change)
 *
 * for links between input and output layer:
 * Link value_b: delta for weight of this link during learning. 
 * Link value_a: Momentum term for this link (last change)
 */

/*
 * Use of Learning Parameters:
 * LEARN_PARAM1: learning parameter for adjusting centers (links between
 *               input and hidden layer, treated as vectors)
 * LEARN_PARAM2: learning parameter for adjusting RBF-parameter (BIAS of
 *               units in hidden layer)
 * LEARN_PARAM3: learning parameter for adjusting weights (all links to
 *               output layer + bias of output units)
 * LEARN_PARAM4: maximum difference between output value and teaching
 *               input which is treated as error 0.0 (delta_max)
 * LEARN_PARAM5: factor for momentum term
 *
 */


/*
 * Clean all deltas, so that learning can start.
 * Called every time LEARN_RBF is called to be sure that there is no stuff
 * inside the value_b fields of links and units
 */
 
krui_err  RbfLearnClean()
{
	register struct Unit	*unit_ptr;
	register struct Link	*link_ptr;

	FOR_ALL_UNITS(unit_ptr)
	{
	    unit_ptr -> value_b = 0.0;
	    FOR_ALL_LINKS(unit_ptr, link_ptr)
	    {
		link_ptr -> value_b = 0.0;
	    }
	}

	return KRERR_NO_ERROR;
}

/*
 * Forward propagation of current pattern. Calculation of different 
 * value_a fields. value_c of hidden units is set to 0.0
 */

krui_err  RbfLearnForward(pattern_no)
int	pattern_no;
{
	register struct Unit	*unit_ptr;
	register Patterns	current_in_pattern;
	register Patterns	current_out_pattern;
	register TopoPtrArray	topo_ptr;

	/* calculate index of current input pattern in Pattern array:	*/

	current_in_pattern = in_patterns + pattern_no * NoOfInputUnits;

	/* activate input units with current patterns and calculate	*/
	/* their output value:						*/

	topo_ptr = topo_ptr_array;
	while ((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    /* go through all input units, set activation and calculate */
	    /* output:							*/

	    unit_ptr -> act = *current_in_pattern++;
	    unit_ptr -> Out.output = unit_ptr -> out_func == OUT_IDENTITY 
		? unit_ptr -> act
		: (*unit_ptr -> out_func) (unit_ptr -> act);
	}

	/* activate hidden units, by calling the activation function	*/
	/* (has to be a RBF activation function which places norm ^ 2	*/
	/* into value_a of the unit: see trans_f.c: RbfUnitGetNormsqr). */
	/* The output function is supposed to be OUT_IDENTITY !		*/
	/* (so the output function is never called !)			*/

	while ((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    unit_ptr -> act = unit_ptr -> Out.output =
		(*unit_ptr -> act_func) (unit_ptr);
 
	    unit_ptr -> value_c = 0.0;
	}

	/* activate output units. Again, the output function is supposed*/
	/* to be OUT_IDENTITY. The calculated output is compared to the */
	/* current pattern, the error (difference) is calculated and    */
	/* stored in value_a of the current output unit.		*/

	current_out_pattern = out_patterns + pattern_no * NoOfOutputUnits;

	while ((unit_ptr = *(++topo_ptr)) != NULL)
	{
	    unit_ptr -> act = unit_ptr -> Out.output =
		(*unit_ptr -> act_func) (unit_ptr);
	    unit_ptr -> value_a = *current_out_pattern++ - unit_ptr -> act;
	}

	return KRERR_NO_ERROR;
}

/*
 * Adjusting of all deltas (value_b fields) by using the current input 
 * pattern (activation of input units) and the stored error of the output 
 * units (value_a). value_c of hidden units is used too!
 */

#define	RBF_LEARN_CENTER	0x1
#define RBF_LEARN_BIAS		0x2
#define RBF_LEARN_WEIGHT	0x4
#define RBF_LEARN_PAIN		0x8

float RbfLearnAdjustDelta(para_center, para_bias, para_weight, para_pain,
	para_momentum, para_delta_max, learn_mask)
float	para_center, para_bias, para_weight, para_pain, para_momentum,
	para_delta_max;
int	learn_mask;
{
	register struct Unit	*curr_unit;	/* current unit		*/
	register struct Link	*curr_link;	/* current link		*/
	register struct Unit	*source_unit;	/* input unit to link	*/
	register TopoPtrArray	topo_ptr;
	register float		center_delta;	/* delta of centers	*/
	register float		w_error;	/* weighted error of 	*/
						/* output unit		*/
	register float		learn_error;

	/* start with last unit in output layer:			*/
	topo_ptr = topo_ptr_array + no_of_topo_units + 3;

	learn_error = 0.0;

	/* work on the output layer and all links leading to it:	*/

	while ((curr_unit = *(--topo_ptr)) != NULL)
	{
	    /* go on with next unit if |error| <= delta_max		*/
	    if ((float) fabs(curr_unit -> value_a) <= para_delta_max)
		continue;

	    /* error, weighted by the deviation of the activation:	*/
	    w_error = (curr_unit -> value_a) * 
		(*curr_unit -> act_deriv_func) (curr_unit);

	    /* sum up the learning error:				*/
	    learn_error += (curr_unit -> value_a) * (curr_unit -> value_a);

	  if (learn_mask & RBF_LEARN_WEIGHT)
	  {
	    /* sum up all deltas for change of bias:			*/
#ifdef RBF_INCR_LEARNING
	    curr_unit -> bias += para_weight * w_error;
#else
	    curr_unit -> value_b += w_error;
#endif
	  }
	  if (learn_mask)
	  {
	    FOR_ALL_LINKS(curr_unit, curr_link)
	    {
		source_unit = curr_link -> to;

		/* sum up deltas for change of link weight:		*/
#ifdef RBF_INCR_LEARNING
		curr_link -> weight += para_weight * w_error *
		    source_unit -> Out.output;
#else
		curr_link -> value_b += w_error * source_unit -> Out.output;
#endif

		/* if comming from hidden unit: sum up delta for change */
		/* of bias of hidden unit:				*/
		if (IS_HIDDEN_UNIT(source_unit))
		    source_unit -> value_c += w_error * curr_link -> weight;
	    }
	  }
	}

	/* work on the hidden layer and all links leading to it:	*/

	if (learn_mask & (RBF_LEARN_CENTER | RBF_LEARN_BIAS))
	{
	  while ((curr_unit = *(--topo_ptr)) != NULL)
	  {
	    /* now calculate delta for weights of links (centers of the */
	    /* RBF function)						*/
	    curr_unit -> Aux.int_no = 2;	/* derivated to norm ^2 */
	    center_delta = curr_unit -> value_c *
		(*curr_unit -> act_deriv_func) (curr_unit);

	   if (learn_mask & RBF_LEARN_CENTER)
	   {
	    FOR_ALL_LINKS(curr_unit, curr_link)
	    {
#ifdef RBF_INCR_LEARNING
		curr_link -> weight += para_center * center_delta *
		    ((curr_link -> to -> Out.output) - (curr_link -> weight));
#else
		curr_link -> value_b += center_delta *
		    ((curr_link -> to -> Out.output) - (curr_link -> weight));
#endif
	    }
	   }

	    /* calculate delta for bias (parameter of RBF function):	*/
	    curr_unit -> Aux.int_no = 3;	/* derivation to bias!  */
#ifdef RBF_INCR_LEARNING
	    curr_unit -> bias += para_bias * curr_unit -> value_c *
		(*curr_unit -> act_deriv_func) (curr_unit);
#else
	    curr_unit -> value_b += curr_unit -> value_c *
		(*curr_unit -> act_deriv_func) (curr_unit);
#endif
	  }
	}

	return learn_error;
}

/*
 * Adjusting of all learnable parameters, depending on collected deltas
 * and on actual learning parameters.
 */

void RbfLearnAdjustWeights(para_center, para_bias, para_weight, para_momentum)
float	para_center, para_bias, para_weight, para_momentum;
{
	register struct Unit	*curr_unit;	/* current unit		*/
	register struct Link	*curr_link;	/* current link		*/
	register TopoPtrArray	topo_ptr;

#ifdef RBF_DELTA_PROT
	static int		step = 0;	/* current learning step*/
	char			filename[20];	/* Name of prot file	*/
	FILE			*protfile;	/* filepointer		*/
#endif

#ifdef RBF_DELTA_PROT
	step++;
	sprintf(filename,"rbf_%04d.prot", step);
	protfile = fopen(filename, "w");
	if (protfile == NULL)
		fprintf(stderr,"RbfLearnAdjustWeights: Can't open protfile\n");
#endif
	
	/* start with last unit in output layer:			*/
	topo_ptr = topo_ptr_array + no_of_topo_units + 3;

#ifdef RBF_DELTA_PROT
	fprintf(protfile, "%s\t\t\n","h -> o");
#endif
	while ((curr_unit = *(--topo_ptr)) != NULL)
	{
	    /* adjust bias of output unit:				*/
	    curr_unit -> bias += para_weight * (curr_unit -> value_b);

#ifdef RBF_DELTA_PROT
	    fprintf(protfile, "%13s:\t\n", curr_unit -> unit_name);
#endif
	    /* adjust weights of links leading to this unit:		*/
	    FOR_ALL_LINKS(curr_unit, curr_link)
	    {
#ifdef RBF_DELTA_PROT
		fprintf(protfile, "%-10.2e\t\n", 
			para_weight * (curr_link -> value_b));
#endif
		curr_link -> weight +=
		(curr_link -> value_a = para_weight * (curr_link -> value_b)
			+ para_momentum * curr_link -> value_a);
	    }    
	}

	/* now adjust weights of hidden layer:				*/

#ifdef RBF_DELTA_PROT
	fprintf(protfile, "%s\t\t\n","i -> h");
#endif
	while ((curr_unit = *(--topo_ptr)) != NULL)
	{
	    /* adjust bias of hidden unit (parameter of RBF function):	*/
	    curr_unit -> bias += para_bias * (curr_unit -> value_b);
	    if (curr_unit -> bias <= 0.0)
		fprintf(stderr,"Hidden unit bias %f !\n", curr_unit -> bias);

#ifdef RBF_DELTA_PROT
	    fprintf(protfile, "%13s:\t\n", curr_unit -> unit_name);
#endif
	    /* adjust weights of links (centers of RBF functions):	*/
	    FOR_ALL_LINKS(curr_unit, curr_link)
	    {
#ifdef RBF_DELTA_PROT
		fprintf(protfile, "%-10.2e\t\n", 
			para_center * (curr_link -> value_b));
#endif
		curr_link -> weight +=
		(curr_link -> value_a = para_center * (curr_link -> value_b)
			+ para_momentum * curr_link -> value_a);
	    }
	}

#ifdef RBF_DELTA_PROT
	fclose(protfile);
#endif
}

/*
 * Topological Check for Radial Basis Functions.
 * Also the number of output units is compared to the patternss.
 */

krui_err RbfTopoCheck()
{
	krui_err	ret_code;	/* error return code		*/

	/*  Net has been modified or topologic array isn't		*/
	/* initialized. check the topology of the network.		*/
	ret_code = kr_topoCheck();
	if (ret_code < KRERR_NO_ERROR)
	    return( ret_code );			/* an error has occured */
    	if (ret_code < 2)
	    return( KRERR_NET_DEPTH );		/* the network has less */
						/* then 2 layers	*/

	/* count the no. of I/O units and check the patterns		*/
	ret_code = kr_IOCheck();
	if (ret_code < KRERR_NO_ERROR)
	    return( ret_code );

	/* sort units by topology and by topologic type			*/
	ret_code = kr_topoSort( TOPOLOGICAL_FF );

	return ret_code;
}

/*
 * Learning function for RBF (GRBF) called from kernel.
 */

krui_err  LEARN_RBF(start_pattern, end_pattern, parameterInArray, 
		NoOfInParams, parameterOutArray, NoOfOutParams )
int	start_pattern, end_pattern;
float	parameterInArray[],
	**parameterOutArray;
int	NoOfInParams,
	*NoOfOutParams;
{
	static float	OutParameter[1];    /* OutParameter[0] stores	*/
					    /* the learning error	*/
	int	i, ret_code, pattern_no, learn_mask;
	float	para_bias, para_center, para_weight, para_pain,
		para_momentum, para_delta_max;

	register struct Unit	*unit_ptr;
	register struct Link	*link_ptr;

#ifdef RBF_LEARN_PROT
	static	int	schritt = 1;
	int		fehler_zaehler = 0;
	float		temp_fehler;
	FILE		*protfile;
#endif

	if (NoOfUnits == 0)
	    return( KRERR_NO_UNITS );  	/* No Units defined		*/
	if (NoOfInParams < 1)          	/* #### has to be changed	*/
					/* (must be 4) ####		*/
	    return( KRERR_PARAMETERS ); /* Not enough input parameters  */

	*NoOfOutParams = 1;   		/* One return value is available*/
					/* (the learning error)		*/
	*parameterOutArray = OutParameter;	/* set the reference to */
						/* the output parameter */

	ret_code = KRERR_NO_ERROR;	/* default return code		*/

	if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
	{
	    ret_code = RbfTopoCheck();

	    if ((ret_code != KRERR_NO_ERROR) && (ret_code != KRERR_DEAD_UNITS))
		return( ret_code );

	    NetModified = FALSE;
	}

	if (NetInitialize || LearnFuncHasChanged)
	{
	    fprintf(stderr, "Initialization RBF_Weights should be called!\n");
	    /* initialize fields for momentum term			*/
	    FOR_ALL_UNITS(unit_ptr)
	    {
	    	FOR_ALL_LINKS(unit_ptr, link_ptr)
	    	{
		    link_ptr -> value_a = 0.0;
	    	}
	    }
	}

	NET_ERROR(OutParameter) = 0.0;
	para_center = - LEARN_PARAM1(parameterInArray);
	para_bias = LEARN_PARAM2(parameterInArray);
	para_weight = LEARN_PARAM3(parameterInArray);
	para_momentum = LEARN_PARAM5(parameterInArray);
	para_delta_max = LEARN_PARAM4(parameterInArray);
	para_pain = 0.0;	/* not used now				*/

	/* set learn mask in condition of the learning parameters:	*/
	learn_mask = 0;
	if (para_center != 0.0)
		learn_mask |= RBF_LEARN_CENTER;
	if (para_bias != 0.0)
		learn_mask |= RBF_LEARN_BIAS;
	if (para_weight != 0.0)
		learn_mask |= RBF_LEARN_WEIGHT;
	if (para_pain != 0.0)
		learn_mask |= RBF_LEARN_PAIN;

#ifndef RBF_INCR_LEARNING
	ret_code = RbfLearnClean();
	if (ret_code != KRERR_NO_ERROR)
	    return ret_code;
#endif

	for (i = start_pattern; i <= end_pattern; i++)
	{
	    if (NoOfShuffledPatterns == NoOfPatternPairs)
		pattern_no = pattern_numbers[i];	/* shuffle	*/
	    else
		pattern_no = i;  		/* do not schuffle	*/

	    RbfLearnForward( pattern_no );	/* forward propagation	*/

	    /* backward propagation 					*/
#ifdef RBF_LEARN_PROT
	    temp_fehler = RbfLearnAdjustDelta(para_center,
		para_bias, para_weight, para_pain, para_momentum,
		para_delta_max, learn_mask);
	    NET_ERROR(OutParameter) += temp_fehler;
	    if (temp_fehler > 0.0)
		fehler_zaehler++;
#else
	    NET_ERROR(OutParameter) += RbfLearnAdjustDelta(para_center,
		para_bias, para_weight, para_pain, para_momentum,
		para_delta_max, learn_mask);
#endif
	}

#ifndef RBF_INCR_LEARNING
	RbfLearnAdjustWeights(para_center, para_bias, para_weight, 
		para_momentum);
#endif

#ifdef RBF_LEARN_PROT
	protfile = fopen("rbf_learn_prot_file", "a");
	if (schritt == 1)
	{
		fprintf(protfile, "# Neues Lernprotokoll: \n");
	}
	fprintf(protfile, "%d %f %d\n", schritt, NET_ERROR(OutParameter), 
		fehler_zaehler);
	fclose(protfile);
	schritt++;
#endif

	return( ret_code );
}

