/*     @(#)snnsbat.c	1.6 9/3/92  */

/**********************************************************************
FILE   : snnsbat.c
PURPOSE: SNNS Batchlearning Program
NOTES  : 
AUTHOR : Guenter Mamier, Niels Mache
DATE   : 24.07.92
VERSION : 1.6  9/3/92

 Copyright (c) 1990,1991,1992 by Guenter Mamier and the SNNS Group

**********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <ctype.h>
#include <limits.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/times.h>
#include <sys/utsname.h>

#include "glob_typ.h"   /* SNNS-Kernel constants and data type definitions  */
#include "kr_ui.h"      /* SNNS-Kernel User-Interface Function Prototypes   */



#ifdef __STDC__
extern int unlink(char const *);
#endif

/*  save temporary networkfile every 1800 seconds of cpu time */
#define SAVE_PERIOD 1800

/*  display not more than 100 cycles  (reduce size of the logfile)  */
#define NO_OF_CYCLES_DISPLAYED  100

/*  longest path name  */
#ifndef PATH_MAX
#define PATH_MAX 512
#endif

#define ABNORMAL  TRUE
#define NORMAL	  FALSE



/*  globals  */
FILE  *file_out = NULL;
char  *tmp_file1, *tmp_file2, *netname;
char  log_file[PATH_MAX+1];
long  starttime, HZ_value;
int   no_of_sites, no_of_links, no_of_units, cycle,
      no_of_patterns, no_of_return_values, tmp_switch;
float *return_values;
struct tms  tp;




/**************************************************************************
*
*  Function: skipComments
*  Purpose : skip comments 
*  Author  : Niels Mache
*
**************************************************************************/
static	bool skipComments(stream)
register FILE  *stream;
{
    register int   c;

    while(TRUE){
	do{ c = getc(stream);}while(isspace(c));

	if(c != '#')break;

	do{ c = getc(stream);}while(c != EOF && c != '\n');
    }

    if(c == EOF)  
	return(FALSE);

    ungetc(c,stream);
    return( TRUE );
}



/**************************************************************************
*
*  Function: errChk
*  Purpose : Check for errors
*  Author  : Niels Mache
*
**************************************************************************/
static  void  errChk(errNo)
int  errNo;
{
    if(errNo != KRERR_NO_ERROR){
	fprintf(stderr, "Batchlearning: %s\n", krui_error(errNo));
	if(file_out != NULL)
	    fprintf(file_out,"Batchlearning: %s\n",krui_error(errNo));
	exit(1);
    }
}



/**************************************************************************
*
*  Function: writeLearnFuncValues
*  Purpose : write the learning function values to the log file
*  Author  : Niels Mache
*
**************************************************************************/
static void  writeLearnFuncValues()
{
    int  j;

    fprintf(file_out,"\nCycle: %d\nLearning function value(s): ",cycle);
    for(j=0; j<no_of_return_values; j++)
	fprintf(file_out,"[%d]: %g  ",j+1,return_values[j]);
    fprintf(file_out,"\n");
}



/**************************************************************************
*
*  Function: writeInfo
*  Purpose : write system information
*  Author  : Niels Mache
*
**************************************************************************/
static void  writeInfo(stream)
FILE  *stream;
{
    long	   clock;
    struct utsname name;

    if(stream != NULL){
	clock = 1;
	(void) time(&clock);

	fprintf(stream, "\n%s Batchlearning terminated at %s",
		krui_getVersion(),ctime(&clock));
	if(uname(&name) == 0)
	    fprintf(stream, "System: %s  Node: %s  Machine: %s\n",
		    name.sysname, name.nodename, name.machine );
    }
}



/**************************************************************************
*
*  Function: writeStatistics
*  Purpose : write Statistics
*  Author  : Niels Mache
*
**************************************************************************/
static void  writeStatistics(status)
bool  status;
{
    long  clock;
    double temp;
    float cpu_time;


    if(file_out != NULL){
	clock = 1;
	(void)time(&clock);
	(void)times(&tp);
	cpu_time = (float)tp.tms_utime / (float)HZ_value;

	writeInfo(file_out);
	fprintf(file_out, 
		"********************************************************\n");
	fprintf(file_out,"\n----   STATISTICS  ----\n");
	fprintf(file_out,"No. of learned cycles: %d\n",cycle);
	if(status)  
	    writeLearnFuncValues();
	if((temp = (float)no_of_units*(float)cycle*(float)no_of_patterns) 
	         > 1000000)
	    fprintf(file_out,"No. of units updated : %lf\n",temp);
	else
	    fprintf(file_out,"No. of units updated : %d\n",(int)temp);
	if((temp = (float)no_of_sites*(float)cycle*(float)no_of_patterns) 
	         > 1000000)
	    fprintf(file_out,"No. of sites updated : %lf\n",temp);
	else
	    fprintf(file_out,"No. of sites updated : %d\n",(int)temp);
	if((temp = (float)no_of_links*(float)cycle*(float)no_of_patterns) 
	         > 1000000)
	    fprintf(file_out,"No. of links updated : %lf\n",temp);
	else
	    fprintf(file_out,"No. of links updated : %d\n",(int)temp);

	fprintf(file_out,"CPU Time used: %.2f seconds\n",cpu_time);
	fprintf(file_out,"User time: %d seconds\n",clock-starttime);
	fprintf(file_out,"No. of connection updates per second (CUPS): %e\n\n",
		((double)no_of_links * (double)cycle * 
		 (double)no_of_patterns) / ((double)cpu_time));

    }
}




/**************************************************************************
*
*  Function: writeCaughtMsg
*  Purpose : log the number of a caught signal
*  Author  : Niels Mache
*
**************************************************************************/
static void  writeCaughtMsg()
{
    fprintf( stdout, "\nSignal caught, %s Batchlearning terminated.\n",
	    krui_getVersion() );
    writeInfo( stdout );

    if(file_out != NULL){
	fprintf(file_out, 
		"\n#####  Signal caught, %s Batchlearning terminated.\n",
		krui_getVersion() );
	writeInfo( file_out );
    }
}



/**************************************************************************
*
*  Function: emergency_exit
*  Purpose : exit after signal
*
**************************************************************************/
static void  emergency_exit()
{
    writeCaughtMsg();

    fprintf(stdout,"\nFatal Error in SNNS-Kernel. Networkfile is lost.\n");

    if(file_out != NULL) {
	fprintf( stdout, 
		"\n#####  Fatal Error in SNNS-Kernel. Networkfile is lost.\n");
	writeStatistics(ABNORMAL);
	fprintf(stdout, "Logfile '%s' written.\n",log_file);
	fclose(file_out);
    }

    exit(-3);
}




/**************************************************************************
*
*  Function: exit_batchlearn
*  Purpose : handle signals
*  Author  : Niels Mache
*
**************************************************************************/
static void  exit_batchlearn()
{
    char *tmp_file;


    writeCaughtMsg();

    if(krui_getNoOfUnits() > 0){ /*  save network  */
	if(tmp_switch & 1)
	    tmp_file = tmp_file1;
	else
	    tmp_file = tmp_file2;

	(void) krui_saveNet(tmp_file,netname);
	fprintf( stdout, "Networkfile '%s' saved.\n",tmp_file);
	if(file_out != NULL)
	    fprintf(file_out,"Networkfile '%s' saved.\n",tmp_file);
    }

    if(file_out != NULL){
	writeStatistics(ABNORMAL);
	fprintf(stdout,"Logfile '%s' written.\n",log_file);
	fclose(file_out);
    }

    exit(-2);
}



/**************************************************************************
*
*  Function: catchSignals
*  Purpose : catch these signals  
*  Author  : Niels Mache
*
**************************************************************************/
static void  catchSignals()
{
  signal( SIGHUP,  exit_batchlearn);  signal( SIGINT,  exit_batchlearn);
  signal( SIGQUIT, exit_batchlearn);  signal( SIGALRM, exit_batchlearn);
  signal( SIGTERM, exit_batchlearn);

#ifdef SIGPWR
  signal( SIGPWR,  exit_batchlearn);
#endif

  signal( SIGILL,  emergency_exit );
  signal( SIGTRAP, emergency_exit );  signal( SIGIOT,  emergency_exit );
  signal( SIGEMT,  emergency_exit );  signal( SIGFPE,  emergency_exit );
  signal( SIGBUS,  emergency_exit );  signal( SIGSEGV, emergency_exit );
}



/**************************************************************************
*
*  Function: getYesNoSwitch
*  Purpose : read a YES/NO switch
*  Author  : Niels Mache
*
**************************************************************************/
static bool  getYesNoSwitch(file_in)
FILE  *file_in;
{
    int  i, ret;
    char do_shuffleing[80];


    skipComments(file_in);
    ret = fscanf(file_in,"%s",do_shuffleing);
    if(ret != 1){
	fprintf(stderr,"Missing switch value (Expecting YES or NO).\n");
	exit(1);
    }
    for(i=0; i<strlen(do_shuffleing); i++)
	do_shuffleing[i] = toupper(do_shuffleing[i]);

    if(strcmp(do_shuffleing,"YES") == 0)
	return(TRUE);
    if(strcmp(do_shuffleing,"NO") != 0){
	fprintf( stderr, "Wrong switch value (Expecting YES or NO).\n" );
	exit(1);
    }
    return(FALSE);
}



/**************************************************************************
*
*  Function: main
*  Purpose : perform batch processing with the SNNS kernel
*  Author  : Guenter Mamier
*
**************************************************************************/
int main( argc, argv )
int  argc;
char  *argv[];
{
    int	        ret, i, step,
                no_of_cycles, 
                no_of_learn_param,
                no_of_init_param,
                start_pattern,
                end_pattern;
    float       learn_param[20],
                init_param[20],
                best_net_error;
    char        key[80],type[80],unknown[80],
        	config_file[PATH_MAX+1],
                net_file[PATH_MAX+1],
                trained_net_file[PATH_MAX+1],
                learn_pat_file[PATH_MAX+1],
                test_pat_file[PATH_MAX+1],
                res_file[PATH_MAX+1],
                init_func[80],
                dummy[PATH_MAX+1],
                init_param_string[PATH_MAX+1],
                learn_param_string[PATH_MAX+1];
    bool	shuffle,
                input_included,
                output_included;
    long	clock;
    time_t      lasttime;
    FILE	*file_in;





    /* set configuration and log file names */

    switch (argc)  {
      case 1:
	strcpy( config_file, "snnsbat.cfg" );
	strcpy( log_file, "snnsbat.log" );
	break;
      case 2:
	strcpy( config_file, argv[1] );
	strcpy( log_file, "snnsbat.log" );
	break;
      case 3:
	strcpy( config_file, argv[1] );
	strcpy( log_file, argv[2] );
	break;
      default:
	fprintf( stdout, "Usage: batch [config_file] [log_file]\n" );
	exit( 1 );
    }


    
    /* initialize keyword values */

    net_file[0]           = '\0';
    trained_net_file[0]   = '\0';
    learn_pat_file[0]     = '\0';
    test_pat_file[0]      = '\0';
    init_func[0]          = '\0';
    init_param_string[0]  = '\0';
    learn_param_string[0] = '\0';
    start_pattern         = 0;
    end_pattern           = 0;
    best_net_error        = 0.0;
    tmp_switch            = 0;
    cycle                 = 0;
    no_of_cycles          = 0;
    no_of_learn_param     = 0;
    no_of_init_param      = 0;
    for(i=0; i<5; i++) 
	learn_param[i] = init_param[i] = 0.0;
    shuffle             = FALSE;
    input_included      = FALSE;
    output_included     = TRUE;



    /* get HZ value */

    if((getenv("HZ") == NULL) || (sscanf(getenv("HZ"),"%d",&HZ_value)!=1)){
#ifndef HZ
	fprintf( stderr, "****  WARNING: no HZ defined!\n" );
	HZ_value = 100;
#else
	HZ_value = HZ;
#endif
    }


    /* open Log-file */

    if((file_out = fopen(log_file,"w")) == NULL){
	fprintf(stderr,"Can't open LOG file\n");
	exit(1);
    }
    fprintf(file_out,"%s Batchlearning Program\n",krui_getVersion());


    /*  open configuration file  */

    if((file_in = fopen(config_file,"r")) == NULL){
	fprintf(stderr,"Can't open configuration file '%s'\n",config_file);
	exit(1);
    }


    /* begin loggin */

    fprintf(stdout,"\n%s\n", krui_getVersion() );
    fprintf(stdout,"-----  Batchlearning  -----\n\n" );
    fprintf(file_out,"Configuration file: '%s'\n",config_file);
    fprintf(file_out,"Log file          : '%s'\n\n",log_file);


    /* do configuration file scan */

    skipComments(file_in);
    ret = fscanf(file_in,"%s",key);
    if(ret != 1 || strcmp(key,"Type:")){
	fprintf( stderr, "Configuration file must first disclose type\n" );
	exit( 1 );
    }
    ret = fscanf(file_in,"%s",type);
    skipComments(file_in);
    if(strcmp(type,"SNNSBATCH_1")){
	fprintf(stderr,"This program handles only filetype %s not %s\n",
		"SNNSBATCH_1",type);
	exit(1);
    }
    skipComments(file_in);

    while((ret = fscanf(file_in,"%s",key)) != EOF){

	if(strcmp(key,"NoOfLearnParam:") == 0){
	    ret = fscanf(file_in,"%d",&no_of_learn_param);
	    if(learn_param_string[0] != '\0'){
		for(i=0; i<no_of_learn_param; i++){
		    ret = sscanf(learn_param_string,"%f %[^\n]",
				 &learn_param[i],dummy);
		    strcpy(learn_param_string,dummy);
		    if(ret < 1){
			fprintf(stderr,
				"Wrong number of learn params given !!\n");
			fprintf(stderr,"Given %d instead of %d!!\n",
				i,no_of_learn_param);
			exit(1);
		    }
		}
		learn_param_string[0] = '\0';
	    }
	}else if(strcmp(key,"LearnParam:") == 0){
	    if(!no_of_learn_param){ /* value not read yet */
		fscanf(file_in,"%[^\n]",learn_param_string);
	    }else{
		for(i=0; i<no_of_learn_param; i++){
		    ret = fscanf(file_in,"%f",&learn_param[i]);
		    if(ret != 1){
			fprintf(stderr,
				"Wrong number of learn params given !!\n");
			fprintf(stderr,"Given %d instead of %d!!\n",
				i,no_of_learn_param);
			exit(1);
		    }
		}
	    }
	}else if(strcmp(key,"InitParam:") ==0){
	    if(!no_of_init_param){ /* value not read yet */
		fscanf(file_in,"%[^\n]",init_param_string);
	    }else{
		for(i=0; i<no_of_init_param; i++){
		    ret = fscanf(file_in,"%f",&init_param[i]);
		    if(ret != 1){
			fprintf(stderr,
				"Wrong number of init params given !!\n");
			fprintf(stderr,"Given %d instead of %d!!\n",
				i,no_of_init_param);
			exit(1);
		    }
		}
	    }
	}else if(strcmp(key,"NoOfInitParam:") == 0){
	    ret = fscanf(file_in,"%d",&no_of_init_param);
	    if(init_param_string[0] != '\0'){
		for(i=0; i<no_of_init_param; i++){
		    ret = sscanf(init_param_string,"%f %[^\n]",
				 &init_param[i],dummy);
		    strcpy(init_param_string,dummy);
		    if(ret < 1){
			fprintf(stderr,
				"Wrong number of init params given !!\n");
			fprintf(stderr,"Given %d instead of %d!!\n",
				i,no_of_init_param);
			exit(1);
		    }
		}
		init_param_string[0] = '\0';
	    }
	}else if(strcmp(key,"NetworkFile:") == 0)
	    ret = fscanf(file_in,"%s",net_file);
	else if(strcmp(key,"TrainedNetworkFile:") == 0)
	    ret = fscanf(file_in,"%s",trained_net_file);
	else if(strcmp(key,"LearnPatternFile:") == 0)
	    ret = fscanf(file_in,"%s",learn_pat_file);
	else if(strcmp(key,"TestPatternFile:") == 0)
	    ret = fscanf(file_in,"%s",test_pat_file);
	else if(strcmp(key,"ResultFile:") == 0)
	    ret = fscanf(file_in,"%s",res_file);
	else if(strcmp(key,"InitFunction:") == 0)
	    ret = fscanf(file_in,"%s",init_func);
	else if(strcmp(key,"MaxLearnCycles:") == 0)
	    ret = fscanf(file_in,"%d",&no_of_cycles);
	else if(strcmp(key,"MaxErrorToStop:") == 0)
	    ret = fscanf(file_in,"%f",&best_net_error);
	else if(strcmp(key,"Shuffle:") == 0)
	    shuffle = getYesNoSwitch(file_in);
	else if(strcmp(key,"ResultMinMaxPattern:") == 0){
	    ret = fscanf(file_in,"%d",&start_pattern);
	    ret = fscanf(file_in,"%d",&end_pattern);
	}else if(strcmp(key,"ResultIncludeInput:") == 0)
	    input_included = getYesNoSwitch(file_in);
	else if(strcmp(key,"ResultIncludeOutput:") == 0)
	    output_included = getYesNoSwitch(file_in);
	else{
	    ret = fscanf(file_in,"%s",unknown);
	    fprintf(stderr,"Found unknown key in configuration file!!\n");
	    fprintf(stderr,"Ignoring unknown key '%s' with value '%s' !!\n",
		    key,unknown);
	}
	skipComments(file_in);
    }
    fclose(file_in);

    if(no_of_cycles && learn_pat_file[0] == '\0'){
	fprintf(stderr,"If learning is to be performed, a pattern file");
	fprintf(stderr," has to be specified !! \n");
	exit(1);
    }


    /* load the network */

    ret = krui_loadNet(net_file,&netname);
    errChk(ret);
    fprintf(file_out,"Networkfile '%s' loaded.\n",net_file);

    
    /* log network info */

    krui_getNetInfo(&no_of_sites,&no_of_links,&ret,&ret);
    no_of_units = krui_getNoOfUnits();
    fprintf( file_out,"Network name: %s\n", netname );
    fprintf( file_out,"No. of units: %d\n", no_of_units );
    fprintf( file_out,"No. of sites: %d\n", no_of_sites );
    fprintf( file_out,"No. of links: %d\n", no_of_links );
    fprintf( file_out,"Learning Function: %s\n",krui_getLearnFunc());
    fprintf( file_out,"%d Learning Parameters\n\n",no_of_learn_param);
    for(i=0; i<no_of_learn_param; i++)
	fprintf(file_out,"Learning Parameter #%d: %f\n",i+1,
		learn_param[i]);
    fprintf(file_out,"\n");


    /* load learn pattern file */

    if(learn_pat_file[0] != '\0'){
	ret = krui_loadPatterns(learn_pat_file);
	errChk(ret);
	fprintf(file_out,"Patternfile '%s' loaded.\n",learn_pat_file);
	no_of_patterns = krui_getNoOfPatterns();
	fprintf(file_out,"No. of patterns: %d\n\n",no_of_patterns);
	fprintf(file_out,"No. of cycles: %d\n\n", no_of_cycles);
	fprintf(file_out,"Max. network error to stop: %f\n\n",best_net_error);
	if(shuffle)
	    fprintf(file_out,"Patterns are shuffled\n");
	else
	    fprintf(file_out,"Patterns are in order\n");

	fprintf(file_out,"\n");
    }


    /* log test pattern file */

    if(test_pat_file[0] != '\0')
	    fprintf(file_out,"\nTest pattern file : '%s'\n",test_pat_file);


    /* log initialization settings */

    if(init_func[0] == '\0'){
	fprintf( file_out, "No Initialization \n");
    }else{
	fprintf( file_out,"Initialization Function: %s\n",init_func);
	fprintf( file_out,"%d Initialization Parameters\n\n",no_of_init_param);
	for(i=0; i<no_of_init_param; i++)
	    fprintf(file_out,"Initialization Parameter #%d: %f\n",i+1,
		    init_param[i]);
    }
    fprintf(file_out,"\n");


    /* log result file settings */

    if(res_file[0] == '\0'){
	fprintf(file_out,"No Result File \n");
    }else{
	fprintf(file_out,"\nResult File              : '%s'\n",res_file);
	fprintf(file_out,"Result File Start Pattern: %d\n",start_pattern);
	fprintf(file_out,"Result File End Pattern  : %d\n",end_pattern);
	if(input_included)
	    fprintf(file_out,"Result File Input Pattern included\n");
	if(output_included)
	    fprintf(file_out,"Result File Output Pattern included\n");
	fprintf(file_out, "\n" );
    }
    fprintf(file_out, 
	    "\n********************************************************\n");


    /*  init cpu time  */

    (void)times(&tp);
    lasttime = tp.tms_utime;


    /* start timer */

    clock = 1;
    (void)time(&clock);
    starttime = clock;
    if(no_of_cycles)
	fprintf(file_out,"\n%s Batchlearning started at %s\n",
		krui_getVersion(),ctime(&clock));
    else
	fprintf(file_out,"\n%s Result file computation started at %s\n",
		krui_getVersion(),ctime(&clock));



    /*  create names for temporary files */

    tmp_file1 = tempnam("./","SNNS_");
    tmp_file2 = tempnam("./","SNNS_");


    /*  catch signals  */

    catchSignals();


    /* initialize net */

    if(init_func[0] != '\0'){
	ret = krui_setInitialisationFunc(init_func);
	errChk(ret);
	ret = krui_initializeNet(init_param,no_of_init_param);
	errChk(ret);
	fprintf(file_out, "\nNetwork initialized with\n%s ",init_func);
	for(i=0; i<no_of_init_param; i++)
	    fprintf(file_out,"%4.2f ",init_param[i]);
	fprintf(file_out,"\n");
    }


    /* train the net */

    step = ((no_of_cycles-1) / NO_OF_CYCLES_DISPLAYED)+1;
    while((cycle < no_of_cycles) && 
	  (cycle && (return_values[0] > best_net_error) || !cycle)){
	if(shuffle){		/*  shuffle patterns every cycle  */
	    ret = krui_shufflePatterns(TRUE);
	    errChk(ret);
	}

	/**************************************************************
	*  REMEMBER:  return_values[0] returns the current net error  *
	*  learn_param[0] contains the 1st learning parameter         *
	**************************************************************/
	ret = krui_learnAllPatterns(learn_param, NO_OF_LEARN_PARAMS, 
				    &return_values, &no_of_return_values );
	errChk(ret);

	/* print the return values of the learning function  */
	if ((cycle % step == 0) || cycle == (no_of_cycles - 1))
	    writeLearnFuncValues();

	/* get user cpu time */
	(void) times(&tp);

	if(((tp.tms_utime-lasttime) / HZ_value) >= SAVE_PERIOD){
	    /*  save temporary network  */
	    if(++tmp_switch & 1)
		ret = krui_saveNet(tmp_file1,netname);
	    else
		ret = krui_saveNet(tmp_file2,netname);
	    errChk(ret);

	    if(tmp_switch == 1)
		fprintf(file_out,
		    "\n#####  Temporary network file '%s' created.  #####\n\n",
			tmp_file1);
	    if(tmp_switch == 2)
		fprintf( file_out,
		    "\n#####  Temporary network file '%s' created.  #####\n\n",
			tmp_file2);

	    lasttime = tp.tms_utime;
	}
	cycle++;
    }


    /* perform result file computation */

    if(res_file[0] !='\0'){

	if(test_pat_file[0] == '\0' && learn_pat_file[0] == '\0'){
	    fprintf(stderr,"No pattern file present !!\n");
	    exit(0);
	}

	/* load test pattern file */
	if(test_pat_file[0] != '\0' && 
	   strcmp(test_pat_file,learn_pat_file) != 0){ 
	    ret = krui_loadPatterns(test_pat_file);
	    errChk(ret);
	    fprintf(file_out,"Test Pattern File '%s' loaded.\n",test_pat_file);
	    no_of_patterns = krui_getNoOfPatterns();
	    fprintf(file_out,"No. of test patterns: %d\n\n",no_of_patterns);
	}

	/* create result file */
	ret = krui_saveResult(res_file,TRUE,start_pattern,end_pattern,
			      input_included,output_included);
	errChk(ret);
	fprintf(file_out,"\nResult file saved.\n");
    }


    /*  save the network */

    if(trained_net_file[0] == '\0'){
	fprintf(file_out, "\nTrained Network was not saved \n");
    }else{
	ret = krui_saveNet(trained_net_file,netname);
	errChk(ret);
	fprintf(file_out,"\nNetwork saved to %s.\n",trained_net_file);
    }


    /* remove temporary files */

    if(tmp_switch >= 1){
	if(unlink(tmp_file1) == 0)
	    fprintf(file_out, 
		    "\n#####  Temporary network file '%s' removed.  #####\n", 
		    tmp_file1);
	if(tmp_switch > 1){
	    if(unlink(tmp_file2) == 0)
		fprintf(file_out, 
			"#####  Temporary network file '%s' removed.  #####\n",
			tmp_file2);
	}
    }


    /* write statistics */

    writeStatistics(NORMAL);
    fclose(file_out);

    fprintf(stdout,"\n%s Batchlearning has terminated successfully.\n",
	    krui_getVersion());
    fprintf(stdout,"Logfile '%s' written.\n",log_file);

    return(0);
}
