/*     @(#)trans_f.c	1.5 9/23/92  */

/**********************************************************************
FILE   : trans_f.c
PURPOSE: SNNS-Kernel transfer functions
NOTES  : 
AUTHOR : Niels Mache
DATE   : 25.06.90
VERSION : 1.5  9/23/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#include "kr_typ.h"	    /*	Kernel types and constants  */
#include "kr_const.h"       /*  Constant Declarators for SNNS-Kernel  */
#include "func_mac.h"	    /*	Transfer function macros  */

#ifdef  __BORLANDC__
#pragma option -w-
#endif

/*#################################################

GROUP: Unit Output Functions

#################################################*/

/*  Linear Output Function
    This function isn't used now, because the identity output function is
    the NULL pointer.
*/
FlintType  OUTP_Identity( activation )
register FlintType  activation;
{
  return( activation );
}


/*  Clipping [0,1] function
*/
FlintType  OUT_Clip_01( activation )
register FlintType  activation;
{
  if (activation < 0.0)  return( (FlintType) 0.0 );
  if (activation > 1.0)  return( (FlintType) 1.0 );
  return( activation );
}


/*  Clipping [-1,1] function
*/
FlintType  OUT_Clip_11( activation )
register FlintType  activation;
{
  if (activation < -1.0)  return( (FlintType) -1.0 );
  if (activation > 1.0)  return( (FlintType) 1.0 );
  return( activation );
}

/*  Threshold 0.5 Output Function
*/
FlintType  OUT_Threshold05( activation )
register FlintType  activation;
{
  if (activation < 0.5)  return( (FlintType) 0.0 );
  return( (FlintType) 1.0 );
}


/*#################################################

GROUP: Unit Activation Functions

#################################################*/


/*  Linear Activation Function
*/
FlintType   ACT_Linear( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  return( sum );
}

/*  Brain-State-in-a-Box Function
*/
FlintType   ACT_BSBFunction( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  return( sum * GET_UNIT_BIAS( unit_ptr ));
}

/*  Minimum Function (Unit's output and weight)
*/
FlintType   ACT_MinOutPlusWeight( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  min1, min2;


  min1 = 0.0;

  if (GET_FIRST_UNIT_LINK( unit_ptr ))  {
    min1 = GET_OUTPUT + GET_WEIGHT;
    while (GET_NEXT_LINK)
      if ((min2 = GET_OUTPUT + GET_WEIGHT) < min1)
	min1 = min2;
  }
  else
    if (GET_FIRST_SITE( unit_ptr ))  {
      min1 = GET_SITE_VALUE;
      while (GET_NEXT_SITE)
       if ((min2 = GET_SITE_VALUE) < min1)
	    min1 = min2;
    }

  return( min1 );
}


/*  Hyperbolic Tangent Function
*/
FlintType   ACT_TanHFunction( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  return( tanh( sum + GET_UNIT_BIAS( unit_ptr )));
}



/*  Sigmoid Function
*/
FlintType   ACT_Logistic( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  return( (FlintType) (1.0 / (1.0 + exp( -sum - GET_UNIT_BIAS( unit_ptr )))) );
}


/*  Perceptron Function
*/
FlintType   ACT_Perceptron( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  if (sum > 0.0)
    return( sum );

  return( (FlintType) 0.0 );
}

/*  Signum Function
*/
FlintType   ACT_Signum( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  if (sum > 0.0)
    return( (FlintType) 1.0 );

  return( (FlintType) -1.0 );
}


/*  Signum0 Function
*/
FlintType   ACT_Signum0( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  if (sum > 0.0)  return( (FlintType) 1.0 );
  if (sum < 0.0)  return( (FlintType) -1.0 );
  return( (FlintType) 0.0 );
}


/*  Step Function
*/
FlintType   ACT_StepFunction( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  if (sum > 0.0)  return( (FlintType) 1.0 );
  return( (FlintType) 0.0 );
}


/*  Bi-Directional Associative Memory
*/
FlintType   ACT_BAMFunction( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
	sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  if (sum > 0.0)  return( (FlintType) 1.0 );
  if (sum < 0.0)  return( (FlintType) -1.0 );
  return( GET_UNIT_ACT( unit_ptr ) );
}


/*  demonstation function: this function act like the Logistic function,
    but the site with the name "Inhibit" will skipped.
*/
FlintType   ACT_LogisticI( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum = 0.0;
  if (GET_FIRST_SITE( unit_ptr ))
    /*	Do not calculate the 'Inhibit' site */
    do
      if (strcmp( "Inhibit", GET_SITE_NAME ))
        sum += GET_SITE_VALUE;
    while (GET_NEXT_SITE);
  else
    if (GET_FIRST_UNIT_LINK( unit_ptr ))
      do
        sum += GET_WEIGHTED_OUTPUT;
      while (GET_NEXT_LINK);

  return( (FlintType) (1.0 / (1.0 + exp( -sum - GET_UNIT_BIAS( unit_ptr )))) );
}

/* help function for all Radial Basis Activation, Derivation and Learn
 * functions. Computes the square of the L2-Norm of (T - X), where T is the
 * vector of all weights from links leading to <unit_ptr> and X is the
 * vector of output units the links are connected from.
 * Store calculated value into value_a field of the current unit.
 * ALL FUTURE RBF ACTIVATION FUNCTIONS HAVE TO CALL THIS FUNCTION !!!!!!!!!!
 */

FlintType RbfUnitGetNormsqr(unit_ptr)
UNIT_PTR        unit_ptr;
{
        ACT_FUNC_DEFS
        register FlintType      norm_2 = 0.0;   /* |X - T|^2            */
        register FlintType      diff;           /* difference           */


        if (!GET_FIRST_UNIT_LINK(unit_ptr))
        {
                fprintf(stderr,"No input links!\n");
                return norm_2;
        }

        do
        {
                diff = GET_OUTPUT - GET_WEIGHT;
                norm_2 += diff * diff;
        }
        while (GET_NEXT_LINK);

      return unit_ptr -> value_a = norm_2;
}

/*
 * Gaussian RBF Activation function: h(L2, s) = exp(-s*L2^2)
 * where L2 is the L2 Norm (see RbfUnitGetNormsqr), and s is the bias 
 * of <unit_ptr>.
 */

FlintType   ACT_RBF_Gaussian( unit_ptr )
UNIT_PTR    unit_ptr;
{
        register FlintType      norm_2;

        norm_2 = RbfUnitGetNormsqr(unit_ptr);
        return (FlintType) exp(- GET_UNIT_BIAS(unit_ptr)*norm_2);
}

/*
 * Multiquadratic Activation function: h(L2, s) = sqrt(s^2 + L2^2)
 */

FlintType ACT_RBF_Multiquadratic( unit_ptr )
UNIT_PTR    unit_ptr;
{
      register FlintType      norm_2;

      norm_2 = RbfUnitGetNormsqr(unit_ptr);
      return (FlintType) sqrt(norm_2 + GET_UNIT_BIAS(unit_ptr));
}

/*
 * Thin plate splines Activation function: h(L2, s) = (L2*s)^2*ln(L2*s)
 */

FlintType ACT_RBF_Thinplatespline( unit_ptr )
UNIT_PTR    unit_ptr;
{
      register FlintType      norm_2;
      register FlintType      bias;

      norm_2 = RbfUnitGetNormsqr(unit_ptr);
      bias = GET_UNIT_BIAS(unit_ptr);

      if (norm_2 == (FlintType) 0.0)
          return (FlintType) 0.0;
      else
          return (FlintType) bias*bias*norm_2*(0.5*log(norm_2) + log(bias));
}

/*  Linear Activation Function + BIAS
*/
FlintType   ACT_Linear_bias( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  register FlintType  sum;


  sum =  0.0;
  if (GET_FIRST_UNIT_LINK( unit_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);
  else
    if (GET_FIRST_SITE( unit_ptr ))
      do
      sum += GET_SITE_VALUE;
      while (GET_NEXT_SITE);

  return( sum + GET_UNIT_BIAS(unit_ptr));
}


/*#################################################

GROUP: Derivation Functions of the Activation Functions

#################################################*/

/*  Sigmoid Derivation Function
*/
FlintType   ACT_DERIV_Logistic( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  
  return( GET_UNIT_ACT( unit_ptr ) * (1.0 - GET_UNIT_ACT( unit_ptr )) );
}


/*  Identity Derivation Function
*/
FlintType   ACT_DERIV_Identity( unit_ptr )
UNIT_PTR    unit_ptr;
{
  return( (FlintType) 1.0 );
}

/*  Brain-State-in-a-Box Derivation Function
*/
FlintType   ACT_DERIV_BSBFunction( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS

  return( GET_UNIT_BIAS( unit_ptr ));
}

/* TanH Derivation Function
*/
FlintType   ACT_DERIV_TanHFunction( unit_ptr )
UNIT_PTR    unit_ptr;
{
  ACT_FUNC_DEFS
  
  return(1.0-GET_UNIT_ACT( unit_ptr ) * (GET_UNIT_ACT( unit_ptr )) );

}

/*  Dummy function for the derivation functions. Returns always the value 1.0.
    This function is used for activation functions that can't have a derivation
    function.

    NOTE: All activation functions have to provide a derivation function.
*/
FlintType   ACT_DERIV_Dummy( unit_ptr )
UNIT_PTR    unit_ptr;
{
  return( (FlintType) 1.0 );
}

/* Gaussian Radial Basis Derivation functionS
 * depending on Aux: 0 derivated to T
 *                   1 derivated to s (BIAS)
 *                 2 derivated to T if value_a holds norm ^ 2;
 *                 3 derivated to s if value_a holds norm ^ 2;
 *                 others: const 1;
 */

FlintType   ACT_DERIV_RBF_Gaussian(unit_ptr)
UNIT_PTR    unit_ptr;
{
      register FlintType      rc;             /* return value         */
      register FlintType      norm_2;         /* norm ^ 2             */

      switch (unit_ptr -> Aux.int_no)
      {
          case 0:
              /* derivated to norm_2:                                 */
              norm_2 = RbfUnitGetNormsqr(unit_ptr);
              rc =  (FlintType) -GET_UNIT_BIAS(unit_ptr)
                      * exp(- GET_UNIT_BIAS(unit_ptr)*norm_2);
              break;
          case 1:
              /* derivated to BIAS:                                   */
              norm_2 = RbfUnitGetNormsqr(unit_ptr);
              rc = (FlintType) -norm_2 
                      * exp(- GET_UNIT_BIAS(unit_ptr)*norm_2);
              break;
          case 2:
              /* derivated to norm_2: (norm ^ 2 = value_a)            */
              rc =  (FlintType) -GET_UNIT_BIAS(unit_ptr)
                      * exp(- GET_UNIT_BIAS(unit_ptr)*unit_ptr -> value_a);
              break;
          case 3:
              /* derivated to BIAS: (norm ^ 2 = value_a)              */
              rc = (FlintType) -unit_ptr -> value_a 
                      * exp(- GET_UNIT_BIAS(unit_ptr)*unit_ptr -> value_a);
              break;
          default:
              rc = (FlintType) 1.0;
      }

  return rc;
}

/* Multiquadratic Radial Basis Derivation functionS
 * depending on Aux: 0 derivated to T
 *                   1 derivated to s (BIAS)
 *                 2 derivated to T if value_a holds norm ^ 2;
 *                 3 derivated to s if value_a holds norm ^ 2;
 *                 others: const 1;
 */

FlintType   ACT_DERIV_RBF_Multiquadratic(unit_ptr)
UNIT_PTR    unit_ptr;
{
      register FlintType      rc;             /* return value         */
      register FlintType      norm_2;         /* norm ^ 2             */
      register FlintType      bias;           /* s                    */

      bias = (FlintType) GET_UNIT_BIAS(unit_ptr);
      switch (unit_ptr -> Aux.int_no)
      {
          case 0:
          case 1:
              /* derivated to BIAS:                                   */
              /* derivated to norm_2:                                 */
              norm_2 = RbfUnitGetNormsqr(unit_ptr);
              rc =  (FlintType) 1.0/(2.0 * sqrt(bias + norm_2));
              break;
          case 2:
          case 3:
              /* derivated to BIAS: (norm ^ 2 = value_a)              */
              /* derivated to norm_2: (norm ^ 2 = value_a)            */
              rc =  (FlintType) 1.0/(2.0 * sqrt(bias + unit_ptr -> value_a));
              break;
          default:
              rc = (FlintType) 1.0;
      }

  return rc;
}

/* Thin Plate Spline Radial Basis Derivation functionS
 * depending on Aux: 0 derivated to T
 *                   1 derivated to s (BIAS)
 *                 2 derivated to T if value_a holds norm ^ 2;
 *                 3 derivated to s if value_a holds norm ^ 2;
 *                 others: const 1;
 */

FlintType   ACT_DERIV_RBF_Thinplatespline(unit_ptr)
UNIT_PTR    unit_ptr;
{
      register FlintType      rc;             /* return value         */
      register FlintType      norm_2;         /* norm ^ 2             */
      register FlintType      bias;           /* s                    */

      bias = (FlintType) GET_UNIT_BIAS(unit_ptr);
      switch (unit_ptr -> Aux.int_no)
      {
          case 0:
              /* derivated to norm_2:                                 */
              norm_2 = RbfUnitGetNormsqr(unit_ptr);
              if (norm_2 == (FlintType) 0.0)
                  rc = (FlintType) 0.0;
              else
                  rc =  (FlintType) bias * bias *
                      (log(norm_2) + 2.0*log(bias) + 1.0) / 2.0;
              break;
          case 1:
              /* derivated to BIAS:                                   */
              norm_2 = RbfUnitGetNormsqr(unit_ptr);
              if (norm_2 == (FlintType) 0.0)
                  rc = (FlintType) 0.0;
              else
                  rc = (FlintType) bias * norm_2 *
                      (log(norm_2) + 2.0*log(bias) + 1.0);
              break;
          case 2:
              /* derivated to norm_2: (norm ^ 2 = value_a)            */
              if (unit_ptr -> value_a == (FlintType) 0.0)
                  rc = (FlintType) 0.0;
              else
                  rc =  (FlintType) bias * bias *
                      (log(unit_ptr -> value_a) + 2.0*log(bias) + 1.0) / 2.0;
              break;
          case 3:
              /* derivated to BIAS: (norm ^ 2 = value_a)              */
              if (unit_ptr -> value_a == (FlintType) 0.0)
                  rc = (FlintType) 0.0;
              else
                  rc = (FlintType) bias * unit_ptr -> value_a *
                      (log(unit_ptr -> value_a) + 2.0*log(bias) + 1.0);
              break;
          default:
              rc = (FlintType) 1.0;
      }

  return rc;
}

/*#################################################

GROUP: Site functions

#################################################*/

/*  Linear Site Function
*/
FlintType  SITE_WeightedSum( site_ptr )
SITE_PTR    site_ptr;
{
  SITE_FUNC_DEFS
  register FlintType  sum;


  sum = 0.0;
  if (GET_FIRST_SITE_LINK( site_ptr ))
    do
      sum += GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);

  return( sum );
}


/*  Product of all predecessor outputs and input link weights
*/
FlintType  SITE_Product( site_ptr )
SITE_PTR    site_ptr;
{
  SITE_FUNC_DEFS
  register FlintType  prod;


  if (GET_FIRST_SITE_LINK( site_ptr ))  {
    prod = 1.0;
    do
      prod *= GET_WEIGHTED_OUTPUT;
    while (GET_NEXT_LINK);

    return( prod );
  }
  else
    return( (FlintType) 0.0 );
}


/*  Like SITE_Product() but no weighting of the unit's output
*/
FlintType  SITE_ProductA( site_ptr )
SITE_PTR    site_ptr;
{
  SITE_FUNC_DEFS
  register FlintType  prod;


  if (GET_FIRST_SITE_LINK( site_ptr ))  {
    prod = 1.0;
    do
      prod *= GET_OUTPUT;
    while (GET_NEXT_LINK);

/*  Future Application (in SNNS-Kernel V2.1 the sites don't have weights).
    So the return value is only the product.
*/
    return( GET_SITE_WEIGHT * prod );
  }
  else
    return( (FlintType) 0.0 );
}



/*  Get the highest weighted output
*/
FlintType  SITE_Max( site_ptr )
SITE_PTR    site_ptr;
{
  SITE_FUNC_DEFS
  register FlintType  max, out;


  if (GET_FIRST_SITE_LINK( site_ptr ))  {
    max = GET_WEIGHTED_OUTPUT;

    while (GET_NEXT_LINK)  {
      out = GET_WEIGHTED_OUTPUT;
      if (max < out)  max = out;
    }

    return( max );
  }
  else
    return( (FlintType) 0.0 );
}


/*  Get the lowest weighted output
*/
FlintType  SITE_Min( site_ptr )
SITE_PTR    site_ptr;
{
  SITE_FUNC_DEFS
  register FlintType  min, out;


  if (GET_FIRST_SITE_LINK( site_ptr ))  {
    min = GET_WEIGHTED_OUTPUT;

    while (GET_NEXT_LINK)  {
      out = GET_WEIGHTED_OUTPUT;
      if (min > out)  min = out;
    }

    return( min );
  }
  else
    return( (FlintType) 0.0 );
}

#ifdef  __BORLANDC__
#pragma option -w+.
#endif
