/*     @(#)update_f.c	1.2 7/21/92  */

/**********************************************************************
FILE   : update_f.c
PURPOSE: SNNS-Kernel Network Update Functions
NOTES  : 
AUTHOR : Niels Mache
DATE   : 18.03.91
VERSION : 1.2  7/21/92

 Copyright (c) 1990,1991,1992 by Niels Mache and the SNNS Group

**********************************************************************/

#include <math.h>
#include <values.h>

#include "kr_typ.h"	 /*  Kernel Types and Constants  */
#include "kr_const.h"	 /*  Constant Declarators for SNNS-Kernel  */
#include "kr_def.h"	 /*  Default Values  */
#include "kernel.h"	 /*  Kernel Functions  */
#include "kr_mem.h"	 /*  Memory Manager Functions  */
#include "random.h"	 /*  Randomize Library Function Prototypes  */
#include "kr_mac.h"	 /*  Kernel Macros   */


#define  UPDATE_PARAM1( param )   param[ 0 ]  /*    contains the 1st initialisation parameter  */
#define  UPDATE_PARAM2( param )   param[ 1 ]  /*    contains the 2nd initialisation parameter  */
#define  UPDATE_PARAM3( param )   param[ 2 ]  /*    contains the 3rd initialisation parameter  */
#define  UPDATE_PARAM4( param )   param[ 3 ]  /*    contains the 4th initialisation parameter  */


/*#################################################

GROUP: Global Var's

#################################################*/


extern bool    NetModified,	 /*  TRUE, if the network topology was modified  */
               NetInitialize;    /*  TRUE, if the network has been initialized  */

extern int  NoOfUnits,		 /*  no. of units in the network  */
	    MinUnitNo,		 /*  the first (lowest) used unit number in the network  */
	    MaxUnitNo,		 /*  the last (highest) used unit number in the network  */
	    NoOfPatternPairs,	 /*  no. of defined pattern pairs */
	    NoOfShuffledPatterns,  /*  no. of entries in the pattern_number array  */
	    NoOfInputUnits,	 /*  no. of input units  */
	    NoOfOutputUnits,	 /*  no. of output units  */
	    TopoSortID; 	 /*  topologic mode identifier	*/

extern UnitArray       unit_array;  /*	the unit array	*/

extern TopoPtrArray    topo_ptr_array;	/*  stores pointers to topological sorted units
					    used by kr_topoSort()  */

extern Patterns        in_patterns,	/*  the input pattern array  */
		       out_patterns;	/*  the output pattern array  */
extern PatternNumbers  pattern_numbers;   /*  contains shuffled pattern numbers
					      used by kr_shufflePatterns()  */

extern int     no_of_topo_units;	/*  no. of unit pointers in the topo_ptr_array	*/





/*#################################################

GROUP: Update Functions

#################################################*/

/*  synchronous propagation
*/
krui_err  UPDATE_syncPropagate( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit   *unit_ptr;


  /*    update unit activations first
  */
  FOR_ALL_UNITS( unit_ptr )
    if ( !IS_INPUT_UNIT( unit_ptr) && UNIT_IN_USE( unit_ptr ))
      /*  unit isn't an input unit and is in use and enabled  */
      unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

  /*    update unit outputs
  */
  FOR_ALL_UNITS( unit_ptr )
    if UNIT_IN_USE( unit_ptr )
      /*  unit is in use and enabled  */
      if (unit_ptr->out_func == OUT_IDENTITY)
	/*  identity output function: there is no need to call the output function  */
	unit_ptr->Out.output = unit_ptr->act;
      else
	/*  no identity output function: calculate unit's output also  */
	unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);

  return( KRERR_NO_ERROR );
}



/*  serial propagation
*/
krui_err  UPDATE_serialPropagate( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit   *unit_ptr;


  /*    update unit activations and outputs
  */
  FOR_ALL_UNITS( unit_ptr )
    if UNIT_IN_USE( unit_ptr )
      {     /*  unit is in use and enabled  */
      if (!IS_INPUT_UNIT( unit_ptr ))
	/*  this isn't a input unit: calculate the activation of the unit by
	    calling the activation function
	*/
        unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

      if (unit_ptr->out_func == OUT_IDENTITY)
	/*  identity output function: there is no need to call the output function  */
	unit_ptr->Out.output = unit_ptr->act;
      else
	/*  no identity output function: calculate unit's output also  */
	unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
    }

  return( KRERR_NO_ERROR );
}



/*  random order propagation
*/
krui_err  UPDATE_randomOrderPropagate( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit   *unit_ptr, *u_array;
  register int   no_of_units;
  int   n;


  u_array = unit_array;
  no_of_units = NoOfUnits;

  for (n = 0; n < no_of_units; n++)
    {
    /*	  choose unit  */
    unit_ptr = u_array + (1 + lrand48() % no_of_units);

    if (!IS_INPUT_UNIT( unit_ptr ))
      /*  this isn't a input unit: calculate the activation of the unit by
	  calling the activation function
      */
      unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
  }

  return( KRERR_NO_ERROR );
}



/*  random permutation propagation
*/
krui_err  UPDATE_randomPermutPropagate( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit	 *unit_ptr;
  register TopoPtrArray  topo_ptr;
  int  ret_code;


  if (NetModified || (TopoSortID != PERMUTATION))
    {  /*  networt was modified or permutation array isn't initialized  */
    ret_code = kr_makeUnitPermutation();
    if (ret_code != KRERR_NO_ERROR)
      return( ret_code );
    }

  topo_ptr = topo_ptr_array;

  /*  propagate net  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {
    if (!IS_INPUT_UNIT( unit_ptr ))
      /*  this isn't a input unit: calculate the activation of the unit by
	  calling the activation function
      */
      unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
  }

  return( KRERR_NO_ERROR );
}



/*  Propagate Units in topological order
*/
krui_err  UPDATE_topologicalPropagate( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit  *unit_ptr;
  register TopoPtrArray  topo_ptr;
  int  ret_code;


  if (NetModified || (TopoSortID != TOPOLOGICAL_FF))
    {  /*  networt was modified or topologic array isn't initialized  */
    ret_code = kr_topoSort( TOPOLOGICAL_FF );
    if (ret_code != KRERR_NO_ERROR)
      return( ret_code );

    NetModified = FALSE;
    }


  topo_ptr = topo_ptr_array + 1;

  /*  propagate input units only  */
  while ((unit_ptr = *topo_ptr++) != NULL)
    {
    /*  input units doesn't have inputs, so don't call the activation function  */

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
  }

  /*  propagate hidden units only  */
  while ((unit_ptr = *topo_ptr++) != NULL)
    {
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
  }

  /*  propagate output units only  */
  while ((unit_ptr = *topo_ptr++) != NULL)
    {
    unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);

    if (unit_ptr->out_func == OUT_IDENTITY)
      /*  identity output function: there is no need to call the output function  */
      unit_ptr->Out.output = unit_ptr->act;
    else
      /*  no identity output function: calculate unit's output also  */
      unit_ptr->Out.output = (*unit_ptr->out_func) (unit_ptr->act);
  }

  return( KRERR_NO_ERROR );
}


/*#################################################

GROUP: User Defined Update Functions

#################################################*/


/*   Counterpropagation Update Function
*/

static void normalize_inputvector( sum )
float sum;
{
  register struct Unit *unit_ptr;
  register float amount;


  amount = 1.0 / sqrt( sum );

  FOR_ALL_UNITS( unit_ptr )
    if (IS_INPUT_UNIT( unit_ptr ) && UNIT_IN_USE( unit_ptr ))
      /* this is a input unit */
      unit_ptr->Out.output = unit_ptr->Out.output * amount;
}  


krui_err  UPDATE_CPNPropagate( parameterArray, NoOfParams )
float parameterArray[];
int   NoOfParams;
{
  register struct Unit   *unit_ptr, *winner_ptr;
  register struct Site   *site_ptr;
  register struct Link   *link_ptr;
  register TopoPtrArray  topo_ptr;

  float maximum, unit_ptr_net, sum; 
  int  ret_code;


  if (NetModified || (TopoSortID != TOPOLOGIC_TYPE))
    {  /*  networt was modified or topologic array isn't initialized  */
    ret_code = kr_topoSort( TOPOLOGIC_TYPE );
    if (ret_code != KRERR_NO_ERROR)
      return( ret_code );

    NetModified = FALSE;
    }

  topo_ptr = topo_ptr_array;
  sum = 0.0;

  /*  propagagate all input units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {  /*  this is a input unit     */
    unit_ptr->Out.output = unit_ptr->act;
    sum += unit_ptr->Out.output * unit_ptr->Out.output;
    }
  
   if (sum != 0.0)
     /* normalize the inputvector */
     normalize_inputvector( sum );


  /***************************************************************/
  /*              propagate Kohonen Layer                        */
  /***************************************************************/

  /***************************************************************/
  /*      calculate the activation and the output values         */
  /*      of the hidden units (Kohonen Layer)                    */
  /***************************************************************/

  winner_ptr = NULL;
  maximum = -1.0e30;  /* contains the maximum of the activations */

  /*  propagagate all hidden units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    {	/* this is a hidden unit */
    unit_ptr_net = 0.0;

    if (unit_ptr->flags & UFLAG_SITES)
      { /* the unit has sites */
      FOR_ALL_SITES_AND_LINKS( unit_ptr, site_ptr, link_ptr )
	  unit_ptr_net += (link_ptr->weight * link_ptr->to->Out.output);
      }
    else
      {  /* the unit has direct links */
      FOR_ALL_LINKS( unit_ptr, link_ptr )
	unit_ptr_net += (link_ptr->weight * link_ptr->to->Out.output);
      }

    if (maximum < unit_ptr_net)
      {  /*  determine winner unit  */
      winner_ptr = unit_ptr;
      maximum = unit_ptr_net;
      }

    /*	reset output and activation of hidden units  */
    unit_ptr->Out.output = unit_ptr->act = (FlintType) 0;
    }

  /***************************************************************/
  /*             the competitive winner is chosen                */
  /***************************************************************/
  winner_ptr->Out.output = winner_ptr->act = (FlintType) 1;


  /* propagate the Grossberg Layer */

  /*  propagagate all output units  */
  while ((unit_ptr = *++topo_ptr) != NULL)
    { /* this is a output unit */
    /* the activation function is the identity function ( weighted sum) */
    unit_ptr->Out.output = unit_ptr->act = (*unit_ptr->act_func) (unit_ptr);
    }

  return( KRERR_NO_ERROR );
}
