/******************************************************************************/
/*     FILE    :  analyze.c                                                   */
/*     TYPE    :  program module                                              */
/*     AUTHORS :  Stefan Broeckel, Tobias Soyez                               */
/*     DATE    :  30.07.92                                                    */
/*     VERSION :  1.0                                                         */
/******************************************************************************/


/******************************************************************************/
/* included headers                                                           */
/******************************************************************************/

#include <stdio.h>


/******************************************************************************/
/* constants                                                                  */
/******************************************************************************/

#define  WRONG   1
#define  RIGHT   2
#define  UNKNOWN 4

#define  ON  1
#define  OFF 2

#define  R402040  1
#define  WTA      2


/******************************************************************************/
/* type definitions                                                           */
/******************************************************************************/

typedef struct 
{
  int  no_of_patterns           ;
  int  no_of_input_units        ;
  int  no_of_output_units       ;
  int  startpattern             ;
  int  endpattern               ;
  int  input_pattern_included   ;
  int  teaching_output_included ;
} FileHeaderInfo ;


/******************************************************************************/
/* global variables                                                           */
/******************************************************************************/

FILE  *in_file  ;
FILE  *out_file ;



/******************************************************************************/
/* function Error                                                             */
/******************************************************************************/

float Error (output, teaching_output, no_of_units)

  float  *output          ;
  float  *teaching_output ;
  int    no_of_units      ;

  {   
    int    i ;
    float  e, diff ;

    e = 0.0 ;
    for (i = 0 ; i < no_of_units ; i++)
    { diff = teaching_output[i] - output[i]  ;
      e    = e + diff * diff                 ;
    }
    
    return e;
  }



/******************************************************************************/
/* The function F_402040 returns :                                            */
/*                                                                            */
/*    RIGHT   : the output of exactly one unit is >= high               and   */
/*              this unit has the greatest teaching output              and   */
/*              the output of all the other units is <= low                   */
/*                                                                            */   
/*    WRONG   : the output of exactly one unit is >= high               and   */
/*              this unit has NOT the greatest teaching output          and   */
/*              the output of all the other units is <= low                   */
/*                                                                            */    
/*    UNKNOWN : in any other case                                             */
/*                                                                            */
/* high, low  : parameters of F_402040                                        */   
/*              default values: low = 0.4, high = 0.6                         */  
/******************************************************************************/

int F_402040 (output, teaching_output, no_of_units, high, low)

  float  *output          ;
  float  *teaching_output ;
  int    no_of_units      ;
  float  high             ;
  float  low              ;


  {
    int    o_pos, t_pos ;
    int    on, off, i   ;
    float  t_max        ;

    on    = 0 ;
    off   = 0 ;
    o_pos = 0 ;
    t_pos = 0 ;
    t_max = teaching_output[t_pos] ;

    for (i = 0 ; i < no_of_units ; i++)
    {
      if      (output[i] >= high) { on++  ; o_pos = i ; } 
      else if (output[i] <= low)    off++ ;
    
      if (teaching_output[i] > t_max)
      {  t_max = teaching_output[i] ;
         t_pos = i ; 
      } 
    }
 
    if ((on + off == no_of_units) && (on == 1))
    {  if (o_pos == t_pos)
            return (RIGHT) ;
       else return (WRONG) ;
    }
    else return (UNKNOWN) ;
  }



/******************************************************************************/
/* The function F_WTA returns :                                               */
/*                                                                            */
/*    RIGHT   :  there is exactly one unit j with maximal output a       and  */
/*               a > high                                                and  */
/*               the output of all the other units is < a - low          and  */
/*               the unit j has the greatest teaching output                  */
/*                                                                            */
/*    WRONG   :  there is exactly one unit j with maximal output a       and  */
/*               a > high                                                and  */
/*               the output of all the other units is < a - low          and  */
/*               the unit j has NOT the greatest teaching output              */
/*                                                                            */   
/*    UNKNOWN :  in any other case                                            */
/*                                                                            */
/* high, low  : parameters of F_WTA                                           */ 
/*              default values: low = 0.0, high = 0.0                         */ 
/******************************************************************************/

int F_WTA (output, teaching_output, no_of_units, high, low)

  float  *output          ;
  float  *teaching_output ;
  int    no_of_units      ;
  float  high             ;
  float  low              ;


  {
    int    i, o_pos, t_pos, no_of_max ;
    float  min, max, max2 , t_max     ;


    o_pos     = 0                      ;
    t_pos     = 0                      ;
    t_max     = teaching_output[t_pos] ;
    max       = output[o_pos]          ;
    min       = max                    ;
    no_of_max = 1                      ;

    for (i = 1 ; i < no_of_units ; i++)
    { if (output[i] > max)
      {  max       = output[i] ; 
         o_pos     = i         ;
         no_of_max = 1         ;
      }
      else if (output[i] == max) no_of_max++     ;
      else if (output[i] <  min) min = output[i] ;

      if (teaching_output[i] > t_max)
      {  t_max = teaching_output[i] ;
         t_pos = i ; 
      } 
    }
   
    max2 = min  ;
    for (i = 1 ; i < no_of_units ; i++) 
      if ((output[i] > max2) && (output[i] < max)) max2 = output[i] ;

    if ((no_of_max == 1) && (output[o_pos] > high) && (max2  < max - low))
    {  if (o_pos == t_pos)
            return (RIGHT) ; 
       else return (WRONG) ;
    }
    else return (UNKNOWN) ;
  }



/******************************************************************************/
/*  function get_options                                                      */
/******************************************************************************/

int get_options (argc, argv, function, sel_cond, output_text, high, low, statistics)

  int    argc         ;
  char   *argv[]      ;
  int    *function    ;
  int    *sel_cond    ;
  int    *output_text ;
  float  *high        ;
  float  *low         ;
  int    *statistics  ;

  {
    char         c       ;
    extern char  *optarg ;
    int          error   ;
    int          hl_flag ;

    *function    = R402040       ;
    *sel_cond    = 0             ;
    *output_text = OFF           ;
    *statistics  = OFF           ;
    in_file      = (FILE *) NULL ;
    out_file     = (FILE *) NULL ;
    hl_flag      = 0             ;

    error = 0 ;
   
    while ((c = getopt (argc, argv, "awruve:i:o:h:l:s")) != -1)
      switch (c)
      {  
        case 'l' :
          sscanf (optarg, "%f", low) ;
          hl_flag = hl_flag | 1 ;
          break ;
        case 'h' :
          sscanf (optarg, "%f", high) ;
          hl_flag = hl_flag | 2 ;
          break ;
        case 'e' :
          if      (strcmp (optarg, "402040") == 0) 
                  *function = R402040 ;
          else if (strcmp (optarg, "WTA"   ) == 0)
                  *function = WTA    ;
          else error++ ; 
          break ;
        case 'r' :
          *sel_cond = *sel_cond | RIGHT   ;
          break ;
        case 'w' :
          *sel_cond = *sel_cond | WRONG   ;
          break ;
        case 'u' :
          *sel_cond = *sel_cond | UNKNOWN ;
          break ;
        case 'a' :
          *sel_cond = WRONG | RIGHT | UNKNOWN ;
          break ;
        case 'v' :
          *output_text = ON ;
          break ;
        case 's' :
          *statistics  = ON ;
          break ;
        case 'i' :
          if ((in_file = fopen(optarg, "r")) == (FILE *) NULL)
          {  
             fprintf (stderr, "error:  can't read file %s \n", optarg) ;
             error++ ;
          } 
          break ;
        case 'o' :
          if ((out_file = fopen(optarg, "w")) == (FILE *) NULL)
          {  
             fprintf (stderr, "error:  can't create file %s\n", optarg) ;
             error++ ;
          } 
          break ;
        default  : error++ ;
      }

    if (*sel_cond == 0) *sel_cond = WRONG  ;

    if ((hl_flag & 1) == 0) 
    {  switch (*function)
       {
         case R402040 : *low  = 0.4 ; break ;
         case WTA     : *low  = 0.0 ; break ;
       }
    }

    if ((hl_flag & 2) == 0)
    {  switch (*function)
       {
         case R402040 : *high = 0.6 ; break ;
         case WTA     : *high = 0.0 ; break ;
       }
    }
       
    if (in_file  == (FILE *) NULL) in_file  = stdin  ;
    if (out_file == (FILE *) NULL) out_file = stdout ;

    return (error) ;  
  }



/******************************************************************************/
/* function read_file_header                                                  */
/*                                                                            */
/* reads from the input file :                                                */
/*     no. of patterns                                                        */
/*     no. of input units                                                     */
/*     no. of output units                                                    */
/*     startpattern                                                           */
/*     endpattern                                                             */
/******************************************************************************/

int read_file_header (file_header_info)

  FileHeaderInfo  *file_header_info ;

  {
    char  str1[80], str2[80], str3[80] ;
    int   error ;

    error = 0 ;

    fscanf (in_file, "%s %s %s", str1, str2, str3) ;
    if ((strcmp (str1, "SNNS"  ) != 0) ||   
        (strcmp (str2, "result") != 0) ||  
        (strcmp (str3, "file"  ) != 0))   error++ ;   
      
    if (error == 0)
    {
       fscanf (in_file, "%*s %*s %*s %*s %*s %*s %*s %*s") ;

       fscanf (in_file, "%*s %*s %*s %*s     %d", &(file_header_info -> no_of_patterns)    ) ;
       fscanf (in_file, "%*s %*s %*s %*s %*s %d", &(file_header_info -> no_of_input_units) ) ;
       fscanf (in_file, "%*s %*s %*s %*s %*s %d", &(file_header_info -> no_of_output_units)) ;
       fscanf (in_file, "%*s %*s             %d", &(file_header_info -> startpattern      )) ;
       fscanf (in_file, "%*s %*s             %d", &(file_header_info -> endpattern        )) ;
          
       fscanf (in_file, "%s", str1) ;

       if (strcmp (str1, "input") == 0)
       {   
         fscanf (in_file, "%s %s %s", str2, str3, str1) ;
         if ((strcmp (str2, "patterns") == 0) &&  
             (strcmp (str3, "included") == 0)) 
              file_header_info -> input_pattern_included = 1 ;   
         else file_header_info -> input_pattern_included = 0 ;
       }
       else file_header_info -> input_pattern_included = 0 ;

       if (strcmp (str1, "teaching") == 0) 
       {
         fscanf (in_file, "%s %s %s", str2, str3, str1) ; 
         if ((strcmp (str2, "output"  ) == 0) &&  
             (strcmp (str3, "included") == 0)) 
              file_header_info -> teaching_output_included = 1 ;
         else
         {    file_header_info -> teaching_output_included = 0 ;
              fprintf (stderr, "error:  missing teaching_output \n") ;
              error++ ;
         }   
       }
       else
       {    file_header_info -> teaching_output_included = 0 ;
            fprintf (stderr, "error:  missing teaching_output \n") ;
            error++ ;
       }   

     }
     else 
     {
       fprintf (stderr, "error:  no SNNS result file\n") ;
     }
     return (error) ;
  }



/******************************************************************************/
/* main program                                                               */
/******************************************************************************/

main (argc, argv)

int   argc   ;
char  *argv[] ;

{
  int             pat_no           ;
  int             i, result        ;
  int             function         ;
  int             sel_cond         ;
  int             output_text      ;
  float           low, high        ;
  int             statistics       ;
  int             right, wrong, 
                  unknown          ;
  float           error            ;
  float           *output          ;
  float           *teaching_output ;
  FileHeaderInfo  file_header_info ;


  if (get_options (argc, argv, &function, &sel_cond, &output_text,
                   &high, &low, &statistics) != 0)
  {
    fprintf (stderr, "usage: %s [options]        \n", argv[0]) ;
    fprintf (stderr, "options: -w                \n") ;
    fprintf (stderr, "options: -r                \n") ;
    fprintf (stderr, "options: -u                \n") ;
    fprintf (stderr, "options: -a                \n") ;
    fprintf (stderr, "options: -t                \n") ;
    fprintf (stderr, "options: -s                \n") ;
    fprintf (stderr, "options: -l <float>        \n") ;
    fprintf (stderr, "options: -h <float>        \n") ;
    fprintf (stderr, "options: -e <function>     \n") ;
    fprintf (stderr, "options: -i <input file>   \n") ;
    fprintf (stderr, "options: -o <output file   \n") ;
    fprintf (stderr, "<function> = [402040 | WTA]\n") ;
  }
  else
  {
    if (read_file_header (&file_header_info) == 0) 
    {
      output          = (float *) malloc (file_header_info.no_of_output_units * sizeof(float)) ;
      teaching_output = (float *) malloc (file_header_info.no_of_output_units * sizeof(float)) ;

      wrong   = 0   ;
      right   = 0   ;
      unknown = 0   ;
      error   = 0.0 ;

      for (pat_no = file_header_info.startpattern ; 
          pat_no <= file_header_info.endpattern   ; pat_no++)
      {                
        if (file_header_info.input_pattern_included != 0)
	{ for (i = 1 ; i <= file_header_info.no_of_input_units ; i++)
          fscanf (in_file, "%*f") ;
        }

        for (i = 0 ; i < file_header_info.no_of_output_units ; i++)
          fscanf (in_file, "%f", &teaching_output[i]) ;

        for (i = 0 ; i < file_header_info.no_of_output_units ; i++)
          fscanf (in_file, "%f", &output[i]) ;
        
        switch (function)
	{ 
          case R402040 :
            result = F_402040 (output, teaching_output, file_header_info.no_of_output_units, high, low) ;
            break ;
    	  case WTA    :
            result = F_WTA    (output, teaching_output, file_header_info.no_of_output_units, high, low) ;
            break ;
        }
        
        switch (result)
	{
	  case WRONG   :  wrong++   ; break ;
          case RIGHT   :  right++   ; break ;
          case UNKNOWN :  unknown++ ; break ;
	}

        if (statistics == OFF)
        {     
     	  if (output_text == OFF)
	  {
             if      ((sel_cond & WRONG)   == result) fprintf (out_file, "%d\n", pat_no) ;
             else if ((sel_cond & RIGHT)   == result) fprintf (out_file, "%d\n", pat_no) ;
             else if ((sel_cond & UNKNOWN) == result) fprintf (out_file, "%d\n", pat_no) ;
	  }
          else
	  {
             if      ((sel_cond & WRONG)   == result) fprintf (out_file, "wrong   : %d\n", pat_no) ;
             else if ((sel_cond & RIGHT)   == result) fprintf (out_file, "right   : %d\n", pat_no) ;
             else if ((sel_cond & UNKNOWN) == result) fprintf (out_file, "unknown : %d\n", pat_no) ;
	  }
        }
        else error = error + Error (output, teaching_output, file_header_info. no_of_output_units) ;
	
        if (pat_no < file_header_info.endpattern) fscanf (in_file, "%*s") ;    
      }

      free (output)          ;
      free (teaching_output) ;

      if (statistics == ON)
      { 
         fprintf (out_file, "STATISTICS ( %d patterns )\n", file_header_info.no_of_patterns) ;
         fprintf (out_file, "wrong   : %5.2f %%  ( %d pattern(s) )\n", 
                  100.0*wrong  /file_header_info.no_of_patterns, wrong) ;
         fprintf (out_file, "right   : %5.2f %%  ( %d pattern(s) )\n",
                  100.0*right  /file_header_info.no_of_patterns, right) ;
         fprintf (out_file, "unknown : %5.2f %%  ( %d pattern(s) )\n",
                  100.0*unknown/file_header_info.no_of_patterns, unknown) ;
         fprintf (out_file, "error   : %f\n", error)  ;
      } 
    }
    else
    {
      fprintf (stderr, "error:  invalid file header\n") ;
    }
  } 
  if (in_file  != stdin)  fclose (in_file)  ;
  if (out_file != stdout) fclose (out_file) ;
}  



/******************************************************************************/
/* end of file                                                                */
/******************************************************************************/
