/*****************************************************************************
  FILE     : d3_graph.c
  SHORTNAME: graph.c

  PURPOSE  : routines for polygons and colors
  FUNCTIONS: 
  NOTES    :

  AUTHOR   : Ralf Huebner
  DATE     : 1.12.1991

             (c) 1992 by Ralf Huebner and the SNNS-Group
             
******************************************************************************/



#include <stdio.h>
#include <values.h>
#include <math.h>

#include <X11/Xlib.h>
#include <X11/X.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>


#include "d3_global.h"
#include "d3_graph.h"


#define MAXCOLSTEPS 15
#define POLY_MASK(elem)  (1 << (&(*(*poly_dummy+elem)) - (float *)poly_dummy))


extern Display      *ui_display;
extern Display      *d3_display;
extern Window        d3_window;
extern GC            d3_gc;
extern int           d3_screen;


extern unsigned long ui_col_mainCells[10][3]; 
extern unsigned long ui_col_rangePixels[2 * MAXCOLSTEPS + 1]; 



extern d3_state_type d3_state;



typedef struct {                 /* WINDOW */
     int x0, y0;                 /* xmin and ymin */
     int x1, y1;                 /* xmax and ymax */
} windowType;

windowType clipWindow;




static vector *poly_dummy;




/*****************************************************************************
  FUNCTION : incrementalize_y

  PURPOSE  : increment y
  RETURNS  : 
  NOTES    :

******************************************************************************/



static void incrementalize_y (p1, p2, p, dp, y, mask)

    register float *p1, *p2, *p, *dp;
    register int mask;
    int y;

{
    float dy, frac;

    dy = *(*(vector *)p2+1) - *(*(vector *)p1+1);
    if (dy==0.0) dy = 1.0;
    frac = y + 0.5 - *(*(vector *)p1+1);

    for (; mask!=0; mask>>=1, p1++, p2++, p++, dp++)
        if (mask&1) {
            *dp = (*p2-*p1) / dy;
            *p = *p1 + *dp * frac;
        }
}



/*****************************************************************************
  FUNCTION : incrementalize_x

  PURPOSE  : increment x
  RETURNS  : 
  NOTES    :

******************************************************************************/



static void incrementalize_x (p1, p2, p, dp, x, mask)

    register float *p1, *p2, *p, *dp;
    register int mask;
    int x;

{
    float dx, frac;

    dx = **(vector *)p2 - **(vector *)p1;
    if (dx==0.0) dx = 1.0;
    frac = x + 0.5 - **(vector *)p1;

    for (; mask!=0; mask>>=1, p1++, p2++, p++, dp++)
        if (mask&1) {
            *dp = (*p2-*p1) / dx;
            *p = *p1 + *dp * frac;
        }
}



/*****************************************************************************
  FUNCTION : increment

  PURPOSE  : increment a masked value
  RETURNS  : 
  NOTES    :

******************************************************************************/


static void increment (p, dp, mask)

    register float *p, *dp;
    register int mask;

{
    for (; mask!=0; mask>>=1, p++, dp++)
        if (mask&1)
            *p += *dp;
}




/*****************************************************************************
  FUNCTION : put_pixel

  PURPOSE  : draw a z-buffered pixel
  RETURNS  : 
  NOTES    :

******************************************************************************/



static void put_pixel (px, py, point)

    int px, py;
    vector *point;

{
    float zp;

    d3_readZbuffer (px, py, &zp);
    if (*(*point+2) < zp) {
        d3_putColPixel (px, py);
        d3_writeZbuffer (px, py, *(*point+2));
    }
}




/*****************************************************************************
  FUNCTION : scanline

  PURPOSE  : scan a horizontal line in a polygon
  RETURNS  : 
  NOTES    :

******************************************************************************/



static void scanline (y, l, r, mask)


    int y, mask;
    vector *l, *r;


{
    int x, lx, rx;
    vector p, dp;

    mask &= ~POLY_MASK(0);
    lx = ceil ((**l) - 0.5);
    if (lx<clipWindow.x0) lx = clipWindow.x0;
    rx = floor ((**r) - 0.5);
    if (rx>clipWindow.x1) rx = clipWindow.x1;
    if (lx>rx) return;
    incrementalize_x (l, r, p, dp, lx, mask);
    for (x=lx; x<=rx; x++) {
        put_pixel (x, y, p);
        increment (p, dp, mask);
    }
}




/*****************************************************************************
  FUNCTION : d3_drawPoly

  PURPOSE  : draw a zbuffered polygon
  RETURNS  : 
  NOTES    :

******************************************************************************/



void d3_drawPoly (p)

    register d3_polygon_type *p;

{
    register int i, li, ri, y, ly, ry, top, rem, mask;
    float ymin;
    vector l, r, dl, dr;

    p->mask = POLY_MASK(0) | POLY_MASK(1) | POLY_MASK(2);

    ymin = MAXFLOAT;
    for (i=0; i<p->n; i++) {            /* find top vertex */
        if (p->vert[i][1] < ymin) {
            ymin = (p->vert[i])[1];
            top = i;
        }
    }
    li = ri = top;                     /* left and right vertex indices */
    rem = p->n;                        /* number of vertices remaining */
    y = ceil (ymin - 0.5);             /* current scanline */
    ly = ry = y-1;                     /* lower end of left & right edges */
    mask = p->mask & ~POLY_MASK(1);   /* stop interpolating screen y */

    while (rem > 0) {
         while (ly<=y && rem>0) {
             rem--;
             i = li-1;
             if (i<0) i = p->n-1;
             incrementalize_y (p->vert[li], p->vert[i], l, dl, y, mask);
             ly = floor (p->vert[i][1] + 0.5);
             li = i;
         }
         while (ry<=y && rem>0) {
             rem--;
             i = ri+1;
             if (i>=p->n) i = 0;
             incrementalize_y (p->vert[ri], p->vert[i], r, dr, y, mask);
             ry = floor (p->vert[i][1] + 0.5);
             ri = i;
         }


         while (y<ly && y<ry) {
             if (y>=clipWindow.y0 && y<=clipWindow.y1)
               if (*l<=*r)
                 scanline (y, l, r, mask);
               else
                 scanline (y, r, l, mask);
             y++;
             increment (l, dl, mask);
             increment (r, dr, mask);
         }
    }
}





/*****************************************************************************
  FUNCTION : d3_setClipWindow

  PURPOSE  : sets the clipping window for the polygon routines
  RETURNS  : 
  NOTES    :

******************************************************************************/



void d3_setClipWindow (x0, y0, x1, y1)

    int x0, y0, x1, y1; 

{
    clipWindow.x0 = x0;
    clipWindow.x1 = x1;
    clipWindow.y0 = y0;
    clipWindow.y1 = y1;
}




/*****************************************************************************
  FUNCTION : d3_drawLine

  PURPOSE  : draw a line
  RETURNS  : 
  NOTES    :

******************************************************************************/




void d3_drawLine (x1, y1, x2, y2)

    int x1, y1, x2, y2;

{
    XDrawLine(d3_display, d3_window, d3_gc, x1, y1, x2 ,y2);
}




/*****************************************************************************
  FUNCTION : d3_intens_to_grayval

  PURPOSE  : converts the polygon light intensity in a palette gray value
  RETURNS  : palette index
  NOTES    :

******************************************************************************/



int d3_intens_to_grayval (intens)

    float intens;

{
    return (floor ((1.0 - intens) * (float) MAXCOLSTEPS));
}





/*****************************************************************************
  FUNCTION : d3_value_to_color

  PURPOSE  : converts the polygon light intensity in a palette color value
  RETURNS  : palette index
  NOTES    :

******************************************************************************/


int d3_value_to_color (value)

    float value;

{
    return ((int) ((1.0 + value) * (float) MAXCOLSTEPS));
}




/*****************************************************************************
  FUNCTION : d3_setColor

  PURPOSE  : sets the drawing color
  RETURNS  : 
  NOTES    :

******************************************************************************/


void d3_setColor (color)

    unsigned long color;

{
    XSetForeground(d3_display, d3_gc, 
                   ui_col_rangePixels[color]); 
}



/*****************************************************************************
  FUNCTION : d3_setBlackColor

  PURPOSE  : sets the drawing color to black
  RETURNS  : 
  NOTES    :

******************************************************************************/



void d3_setBlackColor ()

{
    XSetForeground(d3_display, d3_gc, 
                   BlackPixel (d3_display, d3_screen)); 
}



/*****************************************************************************
  FUNCTION : d3_setLinkColor

  PURPOSE  : calculates the link color from a value
  RETURNS  : 
  NOTES    :

******************************************************************************/



void d3_setLinkColor (weight)

    float *weight;

{
    int procent_value;

    if (*weight >= 0.0) {
         procent_value = abs ((int)(100.0 * (*weight) / d3_state.link_scale));
	 if (procent_value > 100)
            procent_value = 100;
         XSetForeground (d3_display, d3_gc, ui_col_rangePixels[MAXCOLSTEPS + 
			procent_value * MAXCOLSTEPS / 100]);
     } else {
         procent_value = abs ((int)(100.0 * (*weight) / d3_state.link_scale));
	 if (procent_value > 100)
            procent_value = 100;
	 XSetForeground (d3_display, d3_gc, ui_col_rangePixels[MAXCOLSTEPS - 
			 procent_value * MAXCOLSTEPS / 100]);
     }
}	    




/*****************************************************************************
  FUNCTION : d3_clearDispaly

  PURPOSE  : clears the display window
  RETURNS  : 
  NOTES    :

******************************************************************************/


void d3_clearDispaly ()

{
    XClearWindow (d3_display, d3_window);
}



/*****************************************************************************
  FUNCTION : d3_getRootSizes

  PURPOSE  : gets the dimensions of the root window
  RETURNS  : width and height
  NOTES    :

******************************************************************************/


void d3_getRootSizes (xsize, ysize)

    unsigned int *xsize, *ysize;

{
    Window root;
    int x, y;
    unsigned int width, height, border, depth;
    Status status;

    status = XGetGeometry (ui_display, DefaultRootWindow (ui_display), &root, 
                           &x, &y, &width, &height, &border, &depth);
    *xsize = width;
    *ysize = height;
}




/* end of file */
/* lines: 512 */
