/*****************************************************************************
  FILE     : ui_colEdit.c
  SHORTNAME: colEdit.c

  PURPOSE  : functions to manipulate the text, background and selection color
  FUNCTIONS: 
  NOTES    : uses the fixed palette defined in ui_color.c

  AUTHOR   : Ralf Huebner
  DATE     : 27.5.1992

             (c) 1992 by Ralf Huebner and the SNNS-Group
             
******************************************************************************/


#include <stdio.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Simple.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Toggle.h>

#include "ui.h"
#include "ui_xWidgets.h"

#ifndef ZERO
#define ZERO 0
#endif


#define noOfColToggles     3

extern XtAppContext ui_appContext;      /* the application context */

extern Widget ui_toplevel;              /* the toplevel widget */
extern Display *ui_display;             /* the X11 display */
extern XFontStruct *ui_fontStruct;      /* font for xgui display */ 
extern GC ui_gc;                        /* graphic context */

extern struct Ui_DisplayType  *ui_set_displayPtr; /* display to setup */

extern unsigned long ui_editColor[UI_MAX_EDIT_COLS]; /* all possible colors */
extern unsigned long ui_col_rangePixels[/*2 * UI_MAXCOLORSTEPS + 1*/];

static Widget ui_colWidget[UI_MAX_EDIT_COLS]; /* widgets for the edit colors */
static Widget colorToggle[noOfColToggles];    /* widgets for text,         */
                                              /* background and selection */

static int colWidgetXsize = 32;   /* X size of a color edit widget */
static int colWidgetYsize = 16;   /* Y size of a color edit widget */

static Window colWindow;          /* id of the test window */ 

static int ui_currentBackCol;   /* currently selected background color index */
static int ui_currentSelCol;    /* currently selected selection color index */
static int ui_currentTextCol;   /* currently selected background color index */

static int currentSelection = UI_SELECT_BACK_COLOR; /* default selection */

static GC colGC;               /* graphic context for the test window */



/*****************************************************************************
  FUNCTION : ui_cancelColorEditPannel

  PURPOSE  : callback if the cancel button is pressed
  NOTES    : closes the pannel
******************************************************************************/

static void
ui_cancelColorEditPannel (w, pannel, call_data)

    Widget w;
    Widget pannel;
    caddr_t call_data;

{
    XtDestroyWidget (pannel);
}



/*****************************************************************************
  FUNCTION : ui_closeColorEditPannel

  PURPOSE  : sets the selected colors in the current display
  NOTES    :
******************************************************************************/


static void
ui_closeColorEditPannel (w, pannel, call_data)

    Widget w;
    Widget pannel;
    caddr_t call_data;

{
    (ui_set_displayPtr->setup).backgroundColorIndex = ui_currentBackCol;
    (ui_set_displayPtr->setup).selectionColorIndex = ui_currentSelCol;
    (ui_set_displayPtr->setup).textColorIndex = ui_currentTextCol;
    XtDestroyWidget (pannel);
}




/*****************************************************************************
  FUNCTION : ui_getDisplayColors

  PURPOSE  : reads the color indices out of the display
  NOTES    :
******************************************************************************/


static void
ui_getDisplayColors()

{
    ui_currentBackCol = (ui_set_displayPtr->setup).backgroundColorIndex;
    ui_currentSelCol = (ui_set_displayPtr->setup).selectionColorIndex;
    ui_currentTextCol = (ui_set_displayPtr->setup).textColorIndex;
}




/*****************************************************************************
  FUNCTION : ui_xCreateColButtonItem

  PURPOSE  : creates a widget for a color button
  NOTES    : the color is defined in pixel
******************************************************************************/


static Widget 
ui_xCreateColButtonItem (parent, pixel, xsize, ysize, left, top)

    Widget parent;
    unsigned long pixel;
    int xsize, ysize;
    Widget left, top;

{
    Cardinal n;
    Widget   w;
    Arg	     args[15];

    n = 0;
    XtSetArg(args[n], XtNinternalHeight, 1); n++;
    XtSetArg(args[n], XtNinternalWidth , 1); n++;
    XtSetArg(args[n], XtNfromVert , top);  n++;
    XtSetArg(args[n], XtNfromHoriz, left);  n++;
    XtSetArg(args[n], XtNleft  , XtChainLeft); n++;
    XtSetArg(args[n], XtNright , XtChainLeft); n++;
    XtSetArg(args[n], XtNtop   , XtChainTop); n++;
    XtSetArg(args[n], XtNbottom, XtChainTop); n++;
    XtSetArg(args[n], XtNwidth, xsize); n++;
    XtSetArg(args[n], XtNheight, ysize); n++;
    XtSetArg(args[n], XtNbackground, pixel); n++;
    w = XtCreateManagedWidget("", commandWidgetClass, parent, args, n);
    return(w);
}





/*****************************************************************************
  FUNCTION : ui_createColWidgets

  PURPOSE  : creates the widgets for all colors
  NOTES    : 
******************************************************************************/


static void 
ui_createColWidgets (parent)

    Widget parent;

{
    Widget top, left;
    unsigned long pixel;
    int i, ix, iy;
    Widget last = NULL;

    for (i=0; i<UI_MAX_EDIT_COLS; i++) {
        ix = i MOD UI_VERT_COLS;
        iy = i DIV UI_VERT_COLS;
        if (ix-1 < 0)
           left = NULL;
        else
           left = ui_colWidget[iy * UI_VERT_COLS + ix - 1];
           
        if (iy-1 < 0)
           top = NULL;
        else
           top = ui_colWidget[iy * UI_VERT_COLS - 1];
        pixel = ui_editColor[i];
        ui_colWidget[i] = ui_xCreateColButtonItem (parent, pixel,
                         colWidgetXsize, colWidgetYsize, left, top);
    }
}





/*****************************************************************************
  FUNCTION : ui_createTestPannel

  PURPOSE  : creates the test pannel for the selected colors
  NOTES    : the sizes are absolute and depend on the
             vars in ui_redrawColorWindow
******************************************************************************/


static Widget
ui_createTestPannel(parent, left, top)

    Widget parent;
    Widget left, top;

{
    Cardinal n;
    Widget   w, frame;
    Arg	     args[15];

    n = 0;
    XtSetArg(args[n], XtNheight, 72); n++; /* 58 */
    XtSetArg(args[n], XtNwidth, 234); n++;
    XtSetArg(args[n], XtNfromVert , top);  n++;
    XtSetArg(args[n], XtNfromHoriz, left);  n++;
    XtSetArg(args[n], XtNleft  , XtChainLeft); n++;
    XtSetArg(args[n], XtNright , XtChainLeft); n++;
    XtSetArg(args[n], XtNtop   , XtChainTop); n++;
    XtSetArg(args[n], XtNbottom, XtChainTop); n++;
    w = XtCreateManagedWidget("testPanel", boxWidgetClass, 
                                        parent, args, n);
    return(w);

}



/*****************************************************************************
  FUNCTION : ui_redrawColorWindow

  PURPOSE  : draws the test pannel
  NOTES    : in the first line is the colorscale from red to green
             in the second are units in the selection color
             the background and text color are also shown
******************************************************************************/


static void
ui_redrawColorWindow()

{
    static int xsize = 16;
    static int ysize = 16;
    static int xspace = 32;
    static int yspace = 32;
    static int xoffset = 14;
    static int yoffset = 14;

    char buf[10];
    int i;

    static int ui_stepColor[] = {0, 5, 10, 15, 20, 25, 30};

    XSetWindowBackground(ui_display, colWindow,
               ui_editColor[ui_currentBackCol]);
    XClearWindow (ui_display, colWindow);
    for (i=0; i<7; i++) {
        XSetForeground(ui_display, colGC,
                       ui_col_rangePixels[i*5]);
        XFillRectangle(ui_display, colWindow, colGC,
                       xoffset + i*xspace, yoffset, xsize, ysize);
        XSetForeground(ui_display, colGC, ui_editColor[ui_currentTextCol]);
        sprintf (buf, "u%d", i); 
        XDrawString(ui_display, colWindow, colGC,
		       xoffset + i*xspace, yoffset-2, buf, strlen(buf));
    }
    for (i=0; i<7; i++) {
        XSetForeground(ui_display, colGC, ui_editColor[ui_currentSelCol]);
        XFillRectangle(ui_display, colWindow, colGC,
                       xoffset + i*xspace, yoffset+yspace, xsize, ysize);
        XSetForeground(ui_display, colGC, ui_editColor[ui_currentTextCol]);
        sprintf (buf, "u%d", i+7); 
        XDrawString(ui_display, colWindow, colGC,
		       xoffset + i*xspace, yoffset+yspace-2, buf, strlen(buf));
    }

}



/*****************************************************************************
  FUNCTION : ui_colorUpdateProc

  PURPOSE  : selects which color to edit
  NOTES    :
******************************************************************************/


static void
ui_colorUpdateProc (w, color, call_data)

    Widget w;
    int color;
    caddr_t call_data;

{
    switch (currentSelection) {
        case UI_SELECT_BACK_COLOR: ui_currentBackCol = color;
                                   break;
        case UI_SELECT_SEL_COLOR : ui_currentSelCol = color;
                                   break;
        case UI_SELECT_TEXT_COLOR: ui_currentTextCol = color;
                                   break;
    }
    ui_redrawColorWindow();
}




/*****************************************************************************
  FUNCTION : ui_timerProc

  PURPOSE  : redraws the test window
  NOTES    : callback from a timer. called 1 sec after the edit pannel is
             popped up. this is nessecary, because a redraw function can't
             called directly
******************************************************************************/


static void
ui_timerProc(client_data, timerId)

    caddr_t client_data;
    XtIntervalId *timerId;

{
    ui_redrawColorWindow();
}



/*****************************************************************************
  FUNCTION : ui_colorWindowEventProc

  PURPOSE  : redraws the test window
  NOTES    : called by the event loop if redraw is nessecary
******************************************************************************/

static void
ui_colorWindowEventProc(w, display, event)

    Widget w;
    Display *display;
    XEvent *event;

{
    if ((event->type == Expose) AND (event->xexpose.count == 0))
        ui_redrawColorWindow();        
}




/*****************************************************************************
  FUNCTION : ui_setToggles

  PURPOSE  : swiches the radio buttons
  NOTES    :
******************************************************************************/


static void
ui_setToggles(index, new, max, widgets)

   int index;
   int *new;
   int max;
   Widget widgets[];

{
   int i;

   for(i=0; i<max; i++)
       ui_xSetToggleState(widgets[i], FALSE);
   ui_xSetToggleState(widgets[index], TRUE);
   *new = index;
}



/*****************************************************************************
  FUNCTION : ui_selectColorItemProc

  PURPOSE  : toggles the specified button
  NOTES    :
******************************************************************************/


void
ui_selectColorItemProc (w, value, call_data)

    Widget w;
    int value;
    caddr_t call_data;

{
    ui_setToggles(value, &currentSelection, noOfColToggles, colorToggle);
}



/*****************************************************************************
  FUNCTION : ui_createColorEditPannel

  PURPOSE  : creates the color edit pannel
  NOTES    :
******************************************************************************/


void 
ui_createColorEditPannel (w, button, call_data)

    Widget w;
    Widget button;
    caddr_t call_data;

{
    int n, i;
    Arg arg[10];
    Arg args[20];
    char buf[20];
    Position xPos, yPos;
    Dimension width, height;
    Widget colEditPannel, pannel, border, label, done;
    Widget cancel, colframe, testPannel, frame;

    n = 0;
    XtSetArg (arg[0], XtNwidth, &width); n++;
    XtSetArg (arg[1], XtNheight, &height); n++;
    XtGetValues (button, arg, n);

    XtTranslateCoords (button, (Position) (width / 2), (Position) (height / 2), &xPos, &yPos);

    n = 0;
    XtSetArg(arg[n], XtNx, xPos); n++;
    XtSetArg(arg[n], XtNy, yPos); n++;

    colEditPannel = XtCreatePopupShell ("color edit", transientShellWidgetClass, button, arg, n);

    border = XtCreateManagedWidget("border", boxWidgetClass,
                                   colEditPannel, NULL, ZERO);
    pannel = XtCreateManagedWidget("pannel", formWidgetClass, 
                                   border, NULL, ZERO);
    colframe = XtCreateManagedWidget("colframe", formWidgetClass, 
                                   pannel, NULL, ZERO);

    ui_createColWidgets (colframe);

    done = ui_xCreateButtonItem ("done", border, NULL, NULL);
    cancel = ui_xCreateButtonItem ("cancel", border, NULL, done);

    colorToggle[0] = ui_xCreateToggleItem ("text", pannel, NULL,
                     NULL, colframe);
    colorToggle[1] = ui_xCreateToggleItem ("background", pannel, NULL,
                     NULL, colorToggle[0]);
    
    colorToggle[2] = ui_xCreateToggleItem ("selection", pannel, NULL,
                     NULL, colorToggle[1]);
     
    ui_getDisplayColors();
 
    testPannel = ui_createTestPannel(pannel, colorToggle[0], colframe);


 
    XtAddCallback (done, XtNcallback, ui_closeColorEditPannel,
                   (Widget) colEditPannel);
    XtAddCallback (cancel, XtNcallback, ui_cancelColorEditPannel,
                   (Widget) colEditPannel);

    XtAddCallback (colorToggle[0], XtNcallback, ui_selectColorItemProc,
                   (int) UI_SELECT_TEXT_COLOR);
    XtAddCallback (colorToggle[1], XtNcallback, ui_selectColorItemProc,
                   (int) UI_SELECT_BACK_COLOR);
    XtAddCallback (colorToggle[2], XtNcallback, ui_selectColorItemProc,
                   (int) UI_SELECT_SEL_COLOR);

    for (i=0; i<UI_MAX_EDIT_COLS; i++)
        XtAddCallback (ui_colWidget[i], XtNcallback,
                       ui_colorUpdateProc, (int) i);

    XtAddEventHandler (testPannel, ExposureMask, 
                       FALSE, ui_colorWindowEventProc, ui_display);


    XtPopup (colEditPannel, XtGrabNone);

    ui_setToggles(currentSelection, &currentSelection,
                  noOfColToggles, colorToggle);

    colWindow = XtWindow(testPannel);

    colGC = XCreateGC (ui_display, colWindow, 0, 0);
    XSetFont (ui_display, colGC, ui_fontStruct->fid);
    
    XtAppAddTimeOut (ui_appContext, 1000L, ui_timerProc, NULL);

}
