/*****************************************************************************
* FILE     : ui_displwght.c
*
* PURPOSE  : contains all new routines to display the weights as Hinton or
*            WV- diagrams
*
* FUNCTIONS: -- ui_displWeights
*               Purpose : Main program to display the weights of the network
*               Calls   : krui_getNoOfUnits()
*                         ui_confirmOk()
*                         DisplayPlanes()
*                         ui_xCreateButtonItem()
*                         ui_createMyScreen()
*            -- ui_displWeightsBW
*               Purpose : Display the weights of the network in black and white
*               Calls   : krui_getNoOfUnits()
*                         krui_getFirstUnit()
*                         krui_setCurrentUnit()
*                         krui_getFirstPredUnit()
*                         krui_getNextPredUnit()
*                         krui_getNextUnit()
*            -- ui_displWeightsCOL
*               Purpose : Display the weights of the network in color
*               Calls   : krui_getNoOfUnits()
*                         krui_getFirstUnit()
*                         krui_setCurrentUnit()
*                         krui_getFirstPredUnit()
*                         krui_getNextPredUnit()
*                         krui_getNextUnit()
*            -- ui_displWeightsFromUpdate
*               Purpose : Interface program, used if net was updated and has 
*                         to be redrawn.
*               Calls   : krui_getNoOfUnits()
*                         ui_displWeights()
*                         ui_displWeightsBW()
*                         ui_displWeightsCOL()
*            -- ui_uncreate
*               Purpose : notify xgui about the destruction of the window
*               Calls   : XtDestroyWidget()
*                         
*
* AUTHOR   : Guenter Mamier
* Version  : 2.1
* DATE     : 29.01.92
*
*            (c) 1992 by Guenter Mamier and the SNNS-Group
*            
******************************************************************************/

#include <stdio.h>			/* For the Syntax message */

#include "ui.h"

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/TextSrc.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Viewport.h>

#include "ui_displwght.h"
#include "ui_xWidgets.h"


extern XtAppContext  ui_appContext;     /* X application context */
extern Bool           ui_col_monochromeMode;

/*****************************************************************************
  FUNCTION : ui_displWeights

  PURPOSE  : Main program to display the weights of the network
  NOTES    : Initializes widgets and switches for color or b&w terminal
  UPDATE   : 29.01.92
******************************************************************************/

void ui_displWeights(w,call_data)
Widget w;
caddr_t call_data;

{
    Widget donebutton,setbutton,framewidget,subframe;
    Widget view,grid_lab,grid_dia,scale;
    int maxunits;
    Arg		   args[25];   
    Cardinal	   n = 0;
    char           buf[40];


    /* do nothing, if no network is defined */

    if( (maxunits = krui_getNoOfUnits()) == 0){
      ui_confirmOk("No network loaded !");
      return;
    }


    /* If a weight window already exists, just put it on top */

    if(WEIGHT_WINDOW_created == 1){
       Window root;
       int    x, y;
       unsigned int borderWidth, depth, width, height;
       XWindowChanges xChange;

       (void) XGetGeometry(ui_display, XtWindow(WEIGHT_GRAPH_mainwidget),
			   &root,&x,&y,&width,&height,&borderWidth,&depth);
       xChange.stack_mode = (int) Above;
       XConfigureWindow(ui_display, XtWindow(WEIGHT_GRAPH_mainwidget), 
			CWStackMode, &xChange);
       return;
    }


    /* set the maximum size of each square and the size of the whole window */

    if(ui_col_monochromeMode){
       /* B&W terminal => variable square size needed */
       WEIGHT_squaresize = (maxunits > WEIGHT_viewsize/16)? 
	                   16: (int)((float)WEIGHT_viewsize/(float)maxunits);
       WEIGHT_windowsize = WEIGHT_squaresize*maxunits+40;
    }else{
       /* color terminal => fixed square size */
       WEIGHT_squaresize = 
              (maxunits > WEIGHT_viewsize/5)?5:(int)(WEIGHT_viewsize/maxunits);
       WEIGHT_windowsize = WEIGHT_squaresize*maxunits+40;
    }


    /* Create all necessary widgets */

    n = 0;
    sprintf(buf,"Weight Visualization\n");
    WEIGHT_GRAPH_mainwidget = 
           XtCreatePopupShell(buf,topLevelShellWidgetClass,ui_toplevel,args,n);
    n = 0;
    framewidget = XtCreateManagedWidget("form",formWidgetClass,
					WEIGHT_GRAPH_mainwidget,args,n);
    donebutton  = ui_xCreateButtonItem("done", framewidget, NULL, NULL);
    grid_lab    = ui_xCreateLabelItem("    grid width",framewidget,110,
				      donebutton,NULL);
    WEIGHT_gridWidth = ((int)(0.1*maxunits)<1)? 1: 0.1*maxunits;
    sprintf(buf,"%d",WEIGHT_gridWidth);
    grid_dia = ui_xCreateDialogItem("grid",framewidget,buf,32,
				    grid_lab,NULL);
    setbutton   = ui_xCreateButtonItem("set",framewidget,grid_dia,NULL);

    n = 0;
    XtSetArg(args[n], XtNx, 0);n++;
    XtSetArg(args[n], XtNy, 0);n++;
    XtSetArg(args[n], XtNborderWidth,  1); n++;
    XtSetArg(args[n], XtNwidth,  WEIGHT_windowsize); n++;
    XtSetArg(args[n], XtNheight, WEIGHT_windowsize); n++;
    XtSetArg(args[n], XtNfromHoriz, NULL);      n++;
    XtSetArg(args[n], XtNfromVert, donebutton); n++;
    XtSetArg(args[n], XtNleft  , XtChainLeft);  n++;
    XtSetArg(args[n], XtNright , XtChainRight); n++;
    XtSetArg(args[n], XtNtop   , XtChainTop);   n++;
    XtSetArg(args[n], XtNbottom, XtChainBottom);n++;
    subframe = XtCreateManagedWidget("sub",formWidgetClass,framewidget,args,n);
    scale = ui_xCreateLabelItem(" ",framewidget,200,setbutton,NULL);
    n = 0;
    XtSetArg(args[n], XtNx, 0);n++;
    XtSetArg(args[n], XtNy, 0);n++;
    XtSetArg(args[n], XtNborderWidth,  1); n++;
    XtSetArg(args[n], XtNwidth,  WEIGHT_viewsize+20); n++;
    XtSetArg(args[n], XtNheight, WEIGHT_viewsize+20); n++;
    XtSetArg(args[n], XtNallowHoriz, TRUE);n++;
    XtSetArg(args[n], XtNallowVert, TRUE);n++;
    view = XtCreateManagedWidget("view",viewportWidgetClass,subframe,args,n);
    WEIGHT_GRAPH_displaywidget = 
           ui_createMyScreen("screen",view,WEIGHT_windowsize,WEIGHT_windowsize,
			     NULL,donebutton);
    WEIGHT_WINDOW_created = 1;

 
    /* define the routines to be called for widget events */

    XtAddCallback(donebutton, XtNcallback,ui_uncreate,WEIGHT_GRAPH_mainwidget);
    XtAddCallback(setbutton, XtNcallback,ui_setGrid,grid_dia);
    XtAddEventHandler(WEIGHT_GRAPH_displaywidget,ButtonPressMask|
		            ButtonReleaseMask,FALSE, ui_weightInfo,NULL);
    XtAddEventHandler(WEIGHT_GRAPH_displaywidget, 
		      StructureNotifyMask | ExposureMask, 
		      GraphicsExpose, ui_displWghtEvent,ui_display);
    XtAddEventHandler(WEIGHT_GRAPH_mainwidget, 
		      StructureNotifyMask |  SubstructureNotifyMask, 
		      GraphicsExpose, ui_displWghtEvent,ui_display);
    if(!ui_col_monochromeMode)
       XtAddEventHandler(scale, ExposureMask, 
			 GraphicsExpose, ui_displScaleCOL,NULL);

    /* Realize the window */

    XtPopup(WEIGHT_GRAPH_mainwidget, XtGrabNone);

    ui_display        = XtDisplay(WEIGHT_GRAPH_mainwidget);
    WEIGHT_GRAPH_screen = DefaultScreen(ui_display);
    WEIGHT_GRAPH_win    = XtWindow(WEIGHT_GRAPH_displaywidget);

      
}




/*****************************************************************************
  FUNCTION : ui_displWghtEvent

  PURPOSE  : event handler for graphic window
  NOTES    : 
  UPDATE   : 20.02.92
*****************************************************************************/
void ui_displWghtEvent(w, display, event)
Widget w;
Display *display;
XEvent *event;
{
    switch (event->type){
        case Expose:
               if (event->xexpose.count == 0)

		   if(ui_col_monochromeMode)

		      /* this is a  B&W terminals, so use ui_displWeightsBW */
		       ui_displWeightsBW();
		   else
		       ui_displWeightsCOL();
	       break;
	default:
	       break;
    }
}




/*****************************************************************************
  FUNCTION : ui_displWeightsBW

  PURPOSE  : Program to display the weights of the network in black and white
  NOTES    : 
  UPDATE   : 29.01.92
******************************************************************************/

void ui_displWeightsBW()
{
   char outString[40];
   int  maxunits,step;
   int  n,next,pred;
   FlintType strength;
   float maxStrength;
   float reference;
   int xx,yy,dx,dy;
   XPoint p1,p2,p3,p4;
   static char dash[]={6,6};
   XtWidgetGeometry geo_info, geo_data;
   XtGeometryResult geo_result;


   /* do nothing, if no network defined */

   if( (maxunits = krui_getNoOfUnits()) == 0)return;


   /* set the necessary xlib environment variables */

   maxStrength = (float)ui_maxWeight;
   WEIGHT_GRAPH_gc[0]  = XCreateGC(ui_display,WEIGHT_GRAPH_win,0,0);

   /* get widget geometry */

   geo_result = XtQueryGeometry(WEIGHT_GRAPH_mainwidget,&geo_data,&geo_info);
   if(geo_result != XtGeometryYes){
      geo_info.height = 400;
      geo_info.width  = 400;
   }


   /* clear the window */

   XSetForeground(ui_display, WEIGHT_GRAPH_gc[0], 
		  WhitePixel(ui_display,WEIGHT_GRAPH_screen));
   XFillRectangle(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],0,0,
		  geo_info.width,geo_info.height);


   XSetForeground(ui_display, WEIGHT_GRAPH_gc[0], 
		  BlackPixel(ui_display,WEIGHT_GRAPH_screen));
   XSetBackground(ui_display, WEIGHT_GRAPH_gc[0], 
		  WhitePixel(ui_display,WEIGHT_GRAPH_screen));
   XClearWindow(ui_display,WEIGHT_GRAPH_win);


   /* set output grid of units */

   dash[0] = dash[1] = WEIGHT_squaresize/2;
   XSetDashes(ui_display,WEIGHT_GRAPH_gc[0],WEIGHT_squaresize,dash,2);
   XSetLineAttributes(ui_display,WEIGHT_GRAPH_gc[0],0,
		      LineOnOffDash,CapButt,JoinMiter);
   next = krui_getFirstUnit();
   step = WEIGHT_gridWidth;
   for(n=1; n<=maxunits; n+=step){
      sprintf(outString,"%d",next);
      p1.x = (int)(WEIGHT_squaresize*n-WEIGHT_squaresize*0.5)+20;
      p1.y = 10;
      p2.x = (maxunits)*WEIGHT_squaresize+30;
      p2.y = (int)(WEIGHT_squaresize*n-WEIGHT_squaresize*0.5)+20;
      p3.x = 0;
      p3.y = p2.y;
      p4.x = p1.x;
      p4.y = (maxunits)*WEIGHT_squaresize+20;
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],p1.x,p1.y,
		  outString,strlen(outString));
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],p2.x,p2.y,
		  outString,strlen(outString));
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],p3.x,p3.y,
		  outString,strlen(outString));
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],p4.x,p4.y+20,
		  outString,strlen(outString));
      XDrawLine(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],
		p1.x,p1.y,p4.x,p4.y);
      XDrawLine(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[0],
		p2.x,p2.y,p3.x+10,p3.y);
      next += step;
   }
   XSetLineAttributes(ui_display,WEIGHT_GRAPH_gc[0],0,
		      LineSolid,CapButt,JoinMiter);


   n = 1;
   next = krui_getFirstUnit();
   krui_setCurrentUnit(next);
   reference = (float)WEIGHT_squaresize / maxStrength; 
   do{
      if((pred=krui_getFirstPredUnit(&strength)) !=0){
         do{
	    dx = dy = abs((int)(reference * (float)strength));
	    xx = (int)(pred*WEIGHT_squaresize-WEIGHT_squaresize*0.5-dx*0.5)+20;
	    yy = (int)(next*WEIGHT_squaresize-WEIGHT_squaresize*0.5-dy*0.5)+20;

	    if(strength > 0.0){
	       XFillRectangle(ui_display,WEIGHT_GRAPH_win,
			      WEIGHT_GRAPH_gc[0],xx,yy,dx,dy);
	    }else{
	       XDrawRectangle(ui_display,WEIGHT_GRAPH_win,
			      WEIGHT_GRAPH_gc[0],xx,yy,dx,dy);
	    }
	 }while( (pred=krui_getNextPredUnit(&strength)) != 0);
      }
    }while((next=krui_getNextUnit()) != 0);

  
   /* get everything on the screen */

   XFlush(ui_display);

} 



/*****************************************************************************
  FUNCTION : ui_displWeightsCOL

  PURPOSE  : Program to display the weights of the network in color
  NOTES    : 
  UPDATE   : 29.01.92
******************************************************************************/

void ui_displWeightsCOL()

{
   char outString[40];
   int  maxunits,step;
   int  n,next,pred;
   FlintType strength;
   float maxStrength;
   int xx,yy,dx,dy;
   XPoint p1,p2,p3,p4;
   static char dash[]={6,6,6};
   XtWidgetGeometry geo_info, geo_data;
   XtGeometryResult geo_result;


   /* do nothing, if no network defined */

   if( (maxunits = krui_getNoOfUnits()) == 0)return;


   /* set the necessary xlib environment variables */

   maxStrength  = (float)ui_maxWeight;
   WEIGHT_GRAPH_screen = DefaultScreen(ui_display);
   WEIGHT_GRAPH_win    = XtWindow(WEIGHT_GRAPH_displaywidget);
   for(n=0;n<=20;n++)
      WEIGHT_GRAPH_gc[n] = XCreateGC(ui_display,WEIGHT_GRAPH_win,0,0); 


   /* Allocate color table */

   WEIGHT_GRAPH_col = DefaultColormap(ui_display,WEIGHT_GRAPH_screen);
   for(n=0; n<10;n++){
      WEIGHT_GRAPH_color_def[n].red   = 65535;
      WEIGHT_GRAPH_color_def[n].green = n*6553;
      WEIGHT_GRAPH_color_def[n].blue  = n*6553;
      XAllocColor(ui_display, WEIGHT_GRAPH_col, &WEIGHT_GRAPH_color_def[n]);
      WEIGHT_GRAPH_color1[n] = WEIGHT_GRAPH_color_def[n].pixel;
      XSetForeground(ui_display,WEIGHT_GRAPH_gc[n], WEIGHT_GRAPH_color1[n]);
      XSetBackground(ui_display,WEIGHT_GRAPH_gc[n], 
		     WhitePixel(ui_display,WEIGHT_GRAPH_screen));
      WEIGHT_GRAPH_color_def[10+n].red   = 65535-n*6553;
      WEIGHT_GRAPH_color_def[10+n].green = 65535;
      WEIGHT_GRAPH_color_def[10+n].blue  = 65535-n*6553;
      XAllocColor(ui_display,WEIGHT_GRAPH_col,&WEIGHT_GRAPH_color_def[10+n]);
      WEIGHT_GRAPH_color2[n] = WEIGHT_GRAPH_color_def[10+n].pixel;
      XSetForeground(ui_display,WEIGHT_GRAPH_gc[10+n],
		     WEIGHT_GRAPH_color2[n]);
      XSetBackground(ui_display,WEIGHT_GRAPH_gc[10+n],
		     WhitePixel(ui_display,WEIGHT_GRAPH_screen));
   }


   /* get widget geometry */

   geo_result = XtQueryGeometry(WEIGHT_GRAPH_mainwidget,&geo_data,&geo_info);
   if(geo_result != XtGeometryYes){
      geo_info.height = 400;
      geo_info.width  = 400;
   }


   /* adjust squaresize */

   if(geo_info.height < geo_info.width)
       WEIGHT_viewsize = geo_info.height-62;
   else
       WEIGHT_viewsize = geo_info.width-40;

   WEIGHT_squaresize = 
              (maxunits > WEIGHT_viewsize/5)?5:(int)(WEIGHT_viewsize/maxunits);
   WEIGHT_windowsize = WEIGHT_squaresize*maxunits+40;
   XtResizeWindow(WEIGHT_GRAPH_displaywidget);


   /* clear the window */
   
   XSetForeground(ui_display, WEIGHT_GRAPH_gc[20], 
		  BlackPixel(ui_display,WEIGHT_GRAPH_screen));
   XSetBackground(ui_display, WEIGHT_GRAPH_gc[20], 
		  WhitePixel(ui_display,WEIGHT_GRAPH_screen));
   XClearWindow(ui_display,WEIGHT_GRAPH_win);



   /* draw the weight diagramm */

   n = 1;
   next = krui_getFirstUnit();
   krui_setCurrentUnit(next);
   do{
      if((pred=krui_getFirstPredUnit(&strength)) !=0){
         do{
	    dx = dy = WEIGHT_squaresize;
	    xx = (int)(pred-1)*WEIGHT_squaresize+20;
	    yy = (int)(next-1)*WEIGHT_squaresize+20;
	    strength = (strength > 9)?  9: strength;
	    strength = (strength <-9)? -9: strength;
	    XFillRectangle(ui_display,WEIGHT_GRAPH_win,
			   WEIGHT_GRAPH_gc[(int)(10.0 + (float)strength)],
			   xx,yy,dx,dy);
	 }while( (pred=krui_getNextPredUnit(&strength)) != 0);
      }
    }while((next=krui_getNextUnit()) != 0);


   /* set output grid of units */

   dash[0] = dash[1] = (int)(WEIGHT_squaresize*0.5 +0.5);
   XSetDashes(ui_display,WEIGHT_GRAPH_gc[20],
	      (int)(WEIGHT_squaresize*0.5 +0.5)+20,dash,2);
   XSetLineAttributes(ui_display,WEIGHT_GRAPH_gc[20],0,LineOnOffDash,
		      CapButt,JoinMiter);
   step = WEIGHT_gridWidth;
/*   for(n=1; n<=maxunits; n+=step){*/
   n = step;
   while(n <= maxunits){
      sprintf(outString,"%d",n);
      p1.x = (int)(WEIGHT_squaresize*n-WEIGHT_squaresize*0.5)+20;
      p1.y = 10;
      p2.x = (maxunits)*WEIGHT_squaresize+30;
      p2.y = (int)(WEIGHT_squaresize*n-WEIGHT_squaresize*0.5)+20;
      p3.x = 0; 
      p3.y = p2.y;
      p4.x = p1.x;
      p4.y = (maxunits)*WEIGHT_squaresize+20;
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[20],p1.x,p1.y,
		  outString,strlen(outString));
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[20],p2.x,p2.y,
		  outString,strlen(outString));
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[20],p3.x,p3.y,
		  outString,strlen(outString));
      XDrawString(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[20],
		  p4.x,p4.y+20,outString,strlen(outString));
      XDrawLine(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[20],
		p1.x,p1.y,p4.x,p4.y);
      XDrawLine(ui_display,WEIGHT_GRAPH_win,WEIGHT_GRAPH_gc[20],
		p2.x,p2.y,p3.x+10,p3.y);
      n += step;
   }
   XSetLineAttributes(ui_display,WEIGHT_GRAPH_gc[20],0,
		      LineSolid,CapButt,JoinMiter);


   /* get everything on the screen */

   XFlush(ui_display);
}




/*****************************************************************************
  FUNCTION : ui_displWeightsFromUpdate

  PURPOSE  : Interface program, used if net was updated and has to be redrawn.
  NOTES    : 
  UPDATE   : 29.01.92
******************************************************************************/
void ui_displWeightsFromUpdate()
{
   int maxunits;

   if( !WEIGHT_WINDOW_created )return;


   if(((maxunits=krui_getNoOfUnits()) != old_maxunits)  && old_maxunits){
      /* New network was loaded with different # of units *
       * ==> new viewport has to be created               */
      XtDestroyWidget(WEIGHT_GRAPH_mainwidget);
      old_maxunits = maxunits;
      WEIGHT_WINDOW_created = 0;
      ui_displWeights(WEIGHT_GRAPH_displaywidget,NULL);
   }else{

      /* old network was only updated, so redraw suffice */
      if(ui_col_monochromeMode)
/*         ui_displWeightsBW(WEIGHT_GRAPH_displaywidget,NULL);*/
         ui_displWeightsBW();
      else
/*         ui_displWeightsCOL(WEIGHT_GRAPH_displaywidget,NULL);*/
         ui_displWeightsCOL();
   }
}



/*****************************************************************************
  FUNCTION : ui_uncreate

  PURPOSE  : notify xgui about the destruction of the window
  NOTES    : 
  UPDATE   : 29.01.92
******************************************************************************/
void ui_uncreate(dummy,w,call_data)
Widget dummy,w;
caddr_t call_data;
{
   XtDestroyWidget(w);
   WEIGHT_WINDOW_created = 0;
   old_maxunits = 0;
}



/*****************************************************************************
  FUNCTION : ui_weightInfo

  PURPOSE  : Program to display information about the weights
  NOTES    : 
  UPDATE   : 05.05.92
******************************************************************************/

void ui_weightInfo(w, display, event)
Widget w;
Display *display;
XEvent *event;
{
    Widget ui_wei_popup;
    static Bool ui_wei_exit = TRUE;
    int xval,yval;
    FlintType con_weight;
    char buf[40];
    Widget label,form;
    int n;
    Arg args[5];
    

    switch(event->type){
        case ButtonPress: 

             /* get unit numbers of clicked at connection */
	     xval = (int)((float)(event->xbutton.x-20) /
			  (float)(WEIGHT_squaresize))+1;
	     yval = (int)((float)(event->xbutton.y-20) /
			  (float)WEIGHT_squaresize)+1;

	     /* do nothing for invalid numbers */
	     if(xval<0 || yval<0)return;
	     if(xval>krui_getNoOfUnits() || yval>krui_getNoOfUnits())return;

	     /* create  label and popup */
	     if(krui_areConnected(xval,yval,&con_weight))
	        sprintf(buf,"Link %d -> %d \nhas weight %f"
			,xval,yval,con_weight);
	     else
	        sprintf(buf,"No connection from unit %d to unit %d\n",
		       xval,yval);
	     ui_wei_popup = XtCreatePopupShell("info",overrideShellWidgetClass,
					       ui_toplevel,NULL, 0);
	     form = XtCreateManagedWidget("form",formWidgetClass,ui_wei_popup, 
					   NULL,0);
	     n = 0;
	     XtSetArg(args[n], XtNfromHoriz, NULL); n++;
	     XtSetArg(args[n], XtNborderWidth, 0); n++;
	     XtSetArg(args[n], XtNlabel, buf); n++;
	     label = XtCreateManagedWidget("weiMsg",labelWidgetClass,form,
					   args,n);
	     XawFormDoLayout(form, True);
	     XtRealizeWidget(ui_wei_popup);
    
	     /* now move the popup to the mouse position */
	     {
	       Window root;
	       int    x, y;
	       unsigned int borderWidth, depth, width, height;
	       XWindowChanges xChange;

	       (void) XGetGeometry(ui_display, XtWindow(ui_wei_popup),
				   &root, &x, &y, &width, &height, 
				   &borderWidth, &depth);

	       xChange.x = event->xbutton.x_root;
	       xChange.y = event->xbutton.y_root;
	       xChange.stack_mode = (int) Above;
	       XConfigureWindow(ui_display, XtWindow(ui_wei_popup), 
				CWX BIT_OR CWY BIT_OR CWStackMode, &xChange);
	     }

	     XtPopup(ui_wei_popup, XtGrabExclusive);
	     ui_wei_exit = FALSE;
	     while (NOT ui_wei_exit) {
	        XtAppNextEvent(ui_appContext, event);
		if(event->type == ButtonRelease){
		   XtDestroyWidget(ui_wei_popup);
		   ui_wei_exit = TRUE;
		}
	        (void) XtDispatchEvent(event);
	     }
             break;

        default: /* ignore all other events */
             break;
    }
}




/*****************************************************************************
  FUNCTION : ui_setGrid

  PURPOSE  : Allows dynamical setting of the grid width
  NOTES    : 
  UPDATE   : 05.05.92
******************************************************************************/

void ui_setGrid(button,dia_win,call_data)
Widget  button;
Widget  dia_win;
caddr_t call_data;
{
    int depth;
    int test;


    test = (int)ui_xIntFromAsciiWidget(dia_win);
    if(test > 0)
      WEIGHT_gridWidth = test;
    else
      ui_confirmOk("Negative grid size not allowed!!");

    if((depth=DisplayPlanes(ui_display,ui_screen)) == 1)
       ui_displWeightsBW(WEIGHT_GRAPH_displaywidget,NULL);
    else
       ui_displWeightsCOL(WEIGHT_GRAPH_displaywidget,NULL);
}



/*****************************************************************************
  FUNCTION : ui_displScaleCOL

  PURPOSE  : Program to display the scale of colors
  NOTES    : 
  UPDATE   : 07.05.92
******************************************************************************/

void ui_displScaleCOL(w,call_data)
Widget w;
caddr_t call_data;

{
    Window scale_win;
    int n;
    int size = 10;

    scale_win = XtWindow(w);
    XDrawString(ui_display,scale_win,WEIGHT_GRAPH_gc[20],10,8,"min",3);
    for(n=1; n<=19; n++){
       XFillRectangle(ui_display,scale_win,WEIGHT_GRAPH_gc[n],
                      10*n,10,size,size);
    }
    XDrawString(ui_display,scale_win,WEIGHT_GRAPH_gc[20],179,8,"max",3);

}
