/*****************************************************************************
  FILE     : ui_edit.c
  SHORTNAME: edit.c

  PURPOSE  : callbacks for editing sites and f-types
  FUNCTIONS: 
  NOTES    :

  AUTHOR   : Tilman Sommer
  DATE     : 25.9.1990

             (c) 1992 by Tilman Sommer and the SNNS-Group
             
******************************************************************************/


#include "ui.h"

#include <X11/Xaw/List.h>

#include "ui_infoP.h"  /* ui_info_displayList() */
#include "ui_lists.h"
#include "ui_xWidgets.h"
#include "ui_confirmer.h"


/* widgets in the edit panels */

extern Widget ui_edit_siteFuncNameWidget, ui_edit_siteNameWidget;
extern Widget ui_edit_actFuncNameWidget, ui_edit_outFuncNameWidget;
extern Widget ui_edit_ftypeNameWidget;

/* list return values */

extern char   ui_list_returnName[MAX_NAME_LENGTH];
extern int    ui_list_returnIndex;

extern int    ui_kernelError; /* last reported kernel error */

/* the following values are available after the call of ui_list_buildList */
extern int   ui_list_noOfFtypes;     /* no of ftypes reported by the kernel */
extern int   ui_list_noOfSites;      /* no of sites reported by the kernel */
extern int   ui_list_noOfFTypeSites; /* no of sites at this ftype */

/* names */

char    ui_edit_siteName[MAX_NAME_LENGTH];
char    ui_edit_siteFuncName[MAX_NAME_LENGTH];
char    ui_edit_actFuncName[MAX_NAME_LENGTH];
char    ui_edit_outFuncName[MAX_NAME_LENGTH];
char    ui_edit_ftypeName[MAX_NAME_LENGTH];



/*****************************************************************************
  FUNCTION : ui_edit_setCurrentSite

  PURPOSE  : makes this side current in the panel
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/

void ui_edit_setCurrentSite(name)
     char   *name;
{
    bool successful;
    char *siteName, *func;


    if (name != NULL) {
	successful = krui_getFirstSiteTableEntry(&siteName, &func);
	while (successful) {
	    if (strcmp(siteName,name) == 0) { /* site entry found */
		sprintf(ui_edit_siteName, "%s", name);
		ui_xSetString(ui_edit_siteNameWidget, ui_edit_siteName);
		sprintf(ui_edit_siteFuncName, "%s", func);
		ui_xSetLabel(ui_edit_siteFuncNameWidget, ui_edit_siteFuncName);
		return;
	    } else 
		successful = krui_getNextSiteTableEntry(&siteName, &func);
	}
    }
    /* no such site was found or (name == NULL) */
    ui_xSetString(ui_edit_siteNameWidget, "");
    ui_xSetLabel(ui_edit_siteFuncNameWidget, "");
}



/*****************************************************************************
  FUNCTION : ui_edit_selectFunction

  PURPOSE  : popup a list with all site functions
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/


void  ui_edit_selectFunction(w, type, call_data)
     Widget   w;
     int      type;
     caddr_t  call_data;
{   
    static struct SimpleListType listFunc;

    switch (type) {
      case UI_EDIT_SITE_FUNC:
	sprintf(listFunc.msg,"%s","Select site function:");
	listFunc.unitPtr  = NULL;
	listFunc.widgetPtr= w;
	listFunc.listType = UI_LIST_SITE_FUNC;
	ui_info_displayList(w, (struct SimpleListType *) &listFunc, NULL);
	sprintf(ui_edit_siteFuncName,"%s",ui_list_returnName);
	ui_xSetLabel(ui_edit_siteFuncNameWidget, ui_edit_siteFuncName);
	break;
      case UI_EDIT_ACT_FUNC:
	sprintf(listFunc.msg,"%s","Select f-type activation function:");
	listFunc.unitPtr  = NULL;
	listFunc.widgetPtr= w;
	listFunc.listType = UI_LIST_FTYPE_ACT_FUNC;
	ui_info_displayList(w, (struct SimpleListType *) &listFunc, NULL);
	sprintf(ui_edit_actFuncName,"%s",ui_list_returnName);
	ui_xSetLabel(ui_edit_actFuncNameWidget, ui_edit_actFuncName);
	break;
      case UI_EDIT_OUT_FUNC:
	sprintf(listFunc.msg,"%s","Select f-type output function:");
	listFunc.unitPtr  = NULL;
	listFunc.widgetPtr= w;
	listFunc.listType = UI_LIST_FTYPE_OUT_FUNC;
	ui_info_displayList(w, (struct SimpleListType *) &listFunc, NULL);
	sprintf(ui_edit_outFuncName,"%s",ui_list_returnName);
	ui_xSetLabel(ui_edit_outFuncNameWidget, ui_edit_outFuncName);
	break;
    }
}


/*****************************************************************************
  FUNCTION : ui_edit_setSite

  PURPOSE  : change name and function of the current selected site
  NOTES    : This site is highlighted.
             Callback from a SET-Button.

  UPDATE   : 20.9.90
******************************************************************************/


void  ui_edit_setSite(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    XawListReturnStruct  *listStructPtr;

    listStructPtr = XawListShowCurrent(listDescriptorPtr->listWidget);
    
    if (listStructPtr->list_index == XAW_LIST_NONE) {
	ui_confirmOk("No site in the site list selected!");
	return;
    }

    ui_xStringFromAsciiWidget(ui_edit_siteNameWidget, ui_edit_siteName, 
			      MAX_NAME_LENGTH);
    ui_correctName(ui_edit_siteName);
    ui_checkError(krui_changeSiteTableEntry(listStructPtr->string,
					    ui_edit_siteName,
					    ui_edit_siteFuncName));
    if (ui_kernelError < 0)
	ui_confirmOk("Name or function invalid!");
    else {
	free((listDescriptorPtr->listPtr)[listStructPtr->list_index]);
	(listDescriptorPtr->listPtr)[listStructPtr->list_index] =
	    XtNewString(ui_edit_siteName);
	XawListChange(listDescriptorPtr->listWidget, 
		      listDescriptorPtr->listPtr, 
		      listDescriptorPtr->noOfItems, 0, True);
    }
}


/*****************************************************************************
  FUNCTION : ui_edit_listSetSite

  PURPOSE  : callback. Called when a site in the site list was seleted.
  NOTES    :

  UPDATE   : 28.8.1990
******************************************************************************/

void
ui_edit_listSetSite(w, editType, listStructPtr)
     Widget w;
     int    editType;
     XawListReturnStruct   *listStructPtr; /* call_data of list widgets */
{
    switch (editType) {
      case UI_EDIT_FTYPE:
	break;
      case UI_EDIT_SITE:
	ui_edit_setCurrentSite(listStructPtr->string);
	break;
    }
}





/*****************************************************************************
  FUNCTION : ui_edit_newSite

  PURPOSE  : callback. Called when the creation of a new site was requested.
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/


void  ui_edit_newSite(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    ui_xStringFromAsciiWidget(ui_edit_siteNameWidget, ui_edit_siteName, 
			      MAX_NAME_LENGTH);
    ui_correctName(ui_edit_siteName);
    if ((strlen(ui_edit_siteName) == 0) OR (strlen(ui_edit_siteFuncName) == 0)) {
	ui_confirmOk("Invalid name or function!");
	return;
    }

    ui_checkError(krui_createSiteTableEntry(ui_edit_siteName, 
					    ui_edit_siteFuncName));
    if (ui_kernelError >= 0) {
	if (ui_list_noOfSites++ == 0) {
	    /* first entry is "*** no sites ***" */
	    free((listDescriptorPtr->listPtr)[0]);
	    (listDescriptorPtr->listPtr)[0] = 
		XtNewString(ui_edit_siteName);
	} else
	    ui_xAddListEntry(listDescriptorPtr, ui_edit_siteName);
	XawListChange(listDescriptorPtr->listWidget, listDescriptorPtr->listPtr, 
		      listDescriptorPtr->noOfItems, 0, True);
    }
}




/*****************************************************************************
  FUNCTION : ui_edit_deleteSite

  PURPOSE  : callback. Called when a site should be deleted 
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/

void  ui_edit_deleteSite(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    XawListReturnStruct   *listStructPtr;
    int                   i;

    listStructPtr = XawListShowCurrent(listDescriptorPtr->listWidget);

    if ((listStructPtr->list_index == XAW_LIST_NONE) OR
	(ui_list_noOfSites <= 0))
	return; /* no site selected */

    ui_checkError(krui_deleteSiteTableEntry(listStructPtr->string));
    
    /* remove this entry from the array */
    free((listDescriptorPtr->listPtr)[listStructPtr->list_index]);
    for (i = listStructPtr->list_index + 1; 
	 i <= listDescriptorPtr->noOfItems - 1; i++)
	(listDescriptorPtr->listPtr)[i-1] = (listDescriptorPtr->listPtr)[i];
    if (ui_list_noOfSites-- == 1) { /* last site deleted */
	(listDescriptorPtr->listPtr)[0] = XtNewString("*** no sites");
    XawListChange(listDescriptorPtr->listWidget, 
		  listDescriptorPtr->listPtr, 
		  1, 0, True);
    } else
	XawListChange(listDescriptorPtr->listWidget, 
		      listDescriptorPtr->listPtr, 
		      --listDescriptorPtr->noOfItems, 0, True);
}



/*****************************************************************************
  FUNCTION : ui_edit_displayFType

  PURPOSE  : display given ftype 
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/

void  ui_edit_displayFType(name, listDescriptorPtr)
     char                  *name;
     struct SimpleListType *listDescriptorPtr;
{
    if (krui_setFTypeEntry(name)) {
	sprintf(ui_edit_ftypeName,   "%s", krui_getFTypeName());
	sprintf(ui_edit_actFuncName, "%s", krui_getFTypeActFuncName());
	sprintf(ui_edit_outFuncName, "%s", krui_getFTypeOutFuncName());
	ui_list_buildList(listDescriptorPtr);

	ui_xSetString(ui_edit_ftypeNameWidget , ui_edit_ftypeName);
	ui_xSetLabel(ui_edit_actFuncNameWidget, ui_edit_actFuncName);
	ui_xSetLabel(ui_edit_outFuncNameWidget, ui_edit_outFuncName);

	XawListChange(listDescriptorPtr->listWidget, 
		      listDescriptorPtr->listPtr, 	
		      listDescriptorPtr->noOfItems, 0, True);
    } else {
	ui_xSetString(ui_edit_ftypeNameWidget , "");
	ui_xSetLabel(ui_edit_actFuncNameWidget, "");
	ui_xSetLabel(ui_edit_outFuncNameWidget, "");
	
	listDescriptorPtr->noOfItems = 0;
	ui_xAddListEntry(listDescriptorPtr, "*** no sites ***");
	ui_list_noOfSites = 0;
	XawListChange(listDescriptorPtr->listWidget, 
		      listDescriptorPtr->listPtr, 	
		      listDescriptorPtr->noOfItems, 0, True);
    }
}



/*****************************************************************************
  FUNCTION : ui_edit_displayFirstFType

  PURPOSE  : display first ftype 
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/

void  ui_edit_displayFirstFType(listDescriptorPtr)
     struct SimpleListType *listDescriptorPtr;
{
    if (krui_setFirstFTypeEntry()) {
	sprintf(ui_edit_ftypeName,   "%s", krui_getFTypeName());
	ui_edit_displayFType(ui_edit_ftypeName, listDescriptorPtr);
    }
}




/*****************************************************************************
  FUNCTION : ui_edit_ftypeAddSite

  PURPOSE  : popup site list
             if a site was choosen add it to the list
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/


void  ui_edit_ftypeAddSite(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    static struct SimpleListType listSites;

    sprintf(listSites.msg,"%s","Select site to add:");
    listSites.unitPtr  = NULL;
    listSites.widgetPtr= w;
    listSites.listType = UI_LIST_SITE;
    ui_info_displayList(w, (struct SimpleListType *) &listSites, NULL);
    if (ui_list_returnIndex != XAW_LIST_NONE) {
	if (ui_list_noOfFTypeSites++ == 0) {
	    /* first entry is "*** no sites ***" */
	    /* replace the first dummy entry     */
	    free((listDescriptorPtr->listPtr)[0]);
	    (listDescriptorPtr->listPtr)[0] = 
		XtNewString(ui_list_returnName);
	} else
	    ui_xAddListEntry(listDescriptorPtr, ui_list_returnName);
	XawListChange(listDescriptorPtr->listWidget, listDescriptorPtr->listPtr, 
		      listDescriptorPtr->noOfItems, 0, True);
    }
}



/*****************************************************************************
  FUNCTION : ui_edit_ftypeDeleteSite

  PURPOSE  : remove highlighted site from list
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/

void  ui_edit_ftypeDeleteSite(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    XawListReturnStruct   *listStructPtr;
    int                   i;

    listStructPtr = XawListShowCurrent(listDescriptorPtr->listWidget);

    if ((listStructPtr->list_index == XAW_LIST_NONE) OR
	(ui_list_noOfFTypeSites <= 0))
	return; /* no site selected */

    /* remove this entry from the array */
    free((listDescriptorPtr->listPtr)[listStructPtr->list_index]);
    for (i = listStructPtr->list_index + 1; 
	 i <= listDescriptorPtr->noOfItems - 1; i++)
	(listDescriptorPtr->listPtr)[i-1] = (listDescriptorPtr->listPtr)[i];
    if (ui_list_noOfFTypeSites-- == 1) { /* last site deleted */
	(listDescriptorPtr->listPtr)[0] = XtNewString("*** no sites");
    XawListChange(listDescriptorPtr->listWidget, 
		  listDescriptorPtr->listPtr, 
		  1, 0, True);
    } else
	XawListChange(listDescriptorPtr->listWidget, 
		      listDescriptorPtr->listPtr, 
		      --listDescriptorPtr->noOfItems, 0, True);
}



/*****************************************************************************
  FUNCTION : ui_edit_chooseFType

  PURPOSE  : display a list of all existingf f-types
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/

void  ui_edit_chooseFType(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    static struct SimpleListType listFTypes;

    if (krui_setFirstFTypeEntry()) {
	sprintf(listFTypes.msg,"%s","Choose f-type to edit:");
	listFTypes.unitPtr  = NULL;
	listFTypes.widgetPtr= w;
	listFTypes.listType = UI_LIST_FTYPE_NAME;
	ui_info_displayList(w, (struct SimpleListType *) &listFTypes, NULL);
	sprintf(ui_edit_ftypeName,"%s", ui_list_returnName);
	ui_edit_displayFType(ui_edit_ftypeName, listDescriptorPtr);
    }
}



/*****************************************************************************
  FUNCTION : ui_edit_setFType

  PURPOSE  : update ftype name and functions; sites remain unchanged
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/
void  ui_edit_setFType(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    char buf[MAX_NAME_LENGTH];

    if (NOT krui_setFTypeEntry(ui_edit_ftypeName)) {
	sprintf(buf,"Invalid f-type: %s", ui_edit_ftypeName);
	ui_confirmOk(buf);
	return;
    }

    ui_xStringFromAsciiWidget(ui_edit_ftypeNameWidget, buf, 
			      MAX_NAME_LENGTH);
    ui_correctName(buf);
    if (strcmp(buf,ui_edit_ftypeName) != 0) {
	strcpy(ui_edit_ftypeName, buf);
	ui_checkError(krui_setFTypeName(ui_edit_ftypeName));
    }
    ui_checkError(krui_setFTypeActFunc(ui_edit_actFuncName));
    ui_checkError(krui_setFTypeOutFunc(ui_edit_outFuncName));
}




/*****************************************************************************
  FUNCTION : ui_edit_newFType

  PURPOSE  : create a new f-type
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/

void  ui_edit_newFType(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    ui_xStringFromAsciiWidget(ui_edit_ftypeNameWidget, ui_edit_ftypeName,
			      MAX_NAME_LENGTH);
    ui_correctName(ui_edit_ftypeName);

    if (strlen(ui_edit_ftypeName) > 0) {
	ui_checkError(krui_createFTypeEntry(ui_edit_ftypeName,
					    ui_edit_actFuncName,
					    ui_edit_outFuncName,
					    ui_list_noOfFTypeSites,
					    listDescriptorPtr->listPtr));
	if (ui_kernelError < 0)
	    ui_confirmOk("Creation of this f-type failed!");
    } else
	ui_confirmOk("No f-type name given!");
}



/*****************************************************************************
  FUNCTION : ui_edit_deleteFType

  PURPOSE  : delete f-type with the given name 
  NOTES    :

  UPDATE   : 20.9.1990
******************************************************************************/

void  ui_edit_deleteFType(w, listDescriptorPtr, call)
     Widget  w;
     struct SimpleListType *listDescriptorPtr;
     caddr_t call;
{
    ui_xStringFromAsciiWidget(ui_edit_ftypeNameWidget, ui_edit_ftypeName,
			      MAX_NAME_LENGTH);
    ui_correctName(ui_edit_ftypeName);
    ui_checkError(krui_deleteFTypeEntry(ui_edit_ftypeName));
    if (ui_kernelError < 0) {
	char buf[MAX_NAME_LENGTH];
	sprintf(buf,"Can't delete f-type: %s",ui_edit_ftypeName);
	ui_confirmOk(buf);
    } else
	ui_edit_displayFirstFType(listDescriptorPtr);
}









/* end of file */
/* lines: 570 */
