/*****************************************************************************
  FILE     : ui_file.c
  SHORTNAME: file.c

  PURPOSE  : popups a window with all filenames. The user may alter the names
             and may request a save or load operation with the correspondent
	     kind of file (NET, GUI, PAT, CFG, TXT).
  FUNCTIONS: 
  NOTES    : is called only during initialisation

  AUTHOR   : Ralf Huebner
  DATE     : 06.04.1992

  COPYRIGHT: (c) 1992 by Ralf Huebner and the SNNS-Group
             
******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <malloc.h>

#include "ui.h"


#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Cardinals.h>

#include "ui_file.h"
#include "ui_fileP.h"
#include "ui_xWidgets.h"


#define SELECTED_NAME_LENGTH   1024
#define MAX_DIR_ENTRIES        1024
#define DIR_BUFFER_LENGTH      4096


extern    Display      *ui_display;
Widget    ui_filePanel; 

static    Widget fileBox, selectorBox;

/* widgets of all input fields */

Widget    ui_fileNET, ui_filePAT, ui_fileCFG, ui_fileTXT, ui_path;

static Widget netToggle, patToggle, resToggle, cfgToggle, txtToggle;

/* strings storing the data of all input fields */

extern char   ui_filenameNET[];
extern char   ui_filenamePAT[];
extern char   ui_filenameRES[];
extern char   ui_filenameCFG[];
extern char   ui_filenameTXT[];
extern char   ui_pathname[];

Bool      ui_fileIsCreated = FALSE; /* file panel created yes or no */

static int currentFileType = UI_FILE_NET;
static char extMask[10];
static char dirEntries[DIR_BUFFER_LENGTH];
static char sortedDirEntries[DIR_BUFFER_LENGTH];
static char *dirPtr[MAX_DIR_ENTRIES];
static int maxEntries;
static struct stat *stat_buf;




/*****************************************************************************
  FUNCTION :                 UI_LOAD SELECTED FILE

  PURPOSE  : loads the current file
  RETURNS  :
  NOTES    : 
*****************************************************************************/



void 
ui_loadSelectedFile(w, button, call_data)

    Widget w;
    Widget button;
    caddr_t call_data;

{
    char selectedName[SELECTED_NAME_LENGTH];

    ui_xStringFromAsciiWidget(fileBox, selectedName, SELECTED_NAME_LENGTH);
    ui_xStringFromAsciiWidget(ui_path, ui_pathname, MAX_NAME_LENGTH);
    switch (currentFileType) {
        case UI_FILE_NET: strcpy (ui_filenameNET, selectedName);
                          ui_file_loadNet();
                          break;
	case UI_FILE_PAT: strcpy (ui_filenamePAT, selectedName);
                          ui_file_loadPatterns();
                          break;
	case UI_FILE_RES: strcpy (ui_filenameRES, selectedName);
                          ui_file_loadResult();
                          break;
	case UI_FILE_CFG: strcpy (ui_filenameCFG, selectedName);
                          ui_file_loadConfiguration();
                          break;
	case UI_FILE_TXT: strcpy (ui_filenameTXT, selectedName);
                          ui_file_loadText();
                          break;

    }
}



/*****************************************************************************
  FUNCTION :                 UI_SAVE SELECTED FILE

  PURPOSE  : saves the current file
  RETURNS  :
  NOTES    : 
*****************************************************************************/



void 
ui_saveSelectedFile(w, button, call_data)

    Widget w;
    Widget button;
    caddr_t call_data;

{
    char selectedName[SELECTED_NAME_LENGTH];

    ui_xStringFromAsciiWidget(fileBox, selectedName, SELECTED_NAME_LENGTH);
    ui_xStringFromAsciiWidget(ui_path, ui_pathname, MAX_NAME_LENGTH);
    switch (currentFileType) {
        case UI_FILE_NET: strcpy (ui_filenameNET, selectedName);
                          ui_file_saveNet();
                          break;
	case UI_FILE_PAT: strcpy (ui_filenamePAT, selectedName);
                          ui_file_savePatterns();
                          break;
	case UI_FILE_RES: strcpy (ui_filenameRES, selectedName);
                          ui_file_saveResult(w, button, call_data);
                          break;
	case UI_FILE_CFG: strcpy (ui_filenameCFG, selectedName);
                          ui_file_saveConfiguration();
                          break;
	case UI_FILE_TXT: strcpy (ui_filenameTXT, selectedName);
                          ui_file_saveText();
                          break;
    }
}


/*****************************************************************************
  FUNCTION :                 UI_EXPAND PATH

  PURPOSE  : expands to full pathname if path = dir/../file
  RETURNS  :
  NOTES    : 
*****************************************************************************/

void
ui_expandPath(path)

    char *path;

{
    char cwd[SELECTED_NAME_LENGTH];

    if (*path != '/') {
        getcwd(cwd, SELECTED_NAME_LENGTH);
        strcat(cwd, "/");
        strcat(cwd, path);
        strcpy(path, cwd);   
    }     
}


/*****************************************************************************
  FUNCTION :                 UI_CHECK PATH

  PURPOSE  : checks the existance of a directory
  RETURNS  : current directory if failed
  NOTES    : 
*****************************************************************************/

void
ui_checkPath(path)

    char *path;

{
    char cwd[SELECTED_NAME_LENGTH];
    char errorMsg[2048];
    DIR *dirp;

    dirp = opendir (path); 
    if (dirp == NULL) {
        getcwd(cwd, SELECTED_NAME_LENGTH);
        sprintf (errorMsg, "Error! Can't read path:\n%s\nchanging to\n%s\n", 
                   ui_pathname, cwd);
	ui_confirmOk(errorMsg);
        strcpy(path, cwd);
        ui_xSetString(ui_path, path);
    } else 	
        closedir (dirp);
}

/*****************************************************************************
  FUNCTION :                 UI_STR SORT

  PURPOSE  : sorts the current directory
  RETURNS  :
  NOTES    : modified quicksort algorithm for strings
*****************************************************************************/




void
ui_strSort (left, right)

    int left, right;

{

    int i, j;
    char *px, *pw;

    do {
        px = dirPtr[(left+right) / 2];
        i = left;
        j = right;
        do {
            while (strcmp(dirPtr[i], px) < 0)
                i++;
            while (strcmp(dirPtr[j], px) > 0)
                j--;
            if (i > j)
                break;
            pw = dirPtr[i];
            dirPtr[i] = dirPtr[j];
            dirPtr[j] = pw;
	} 
        while (++i <= --j);
        if (j - left < right - i) {
            if (left < j)
                ui_strSort (left, j);
            left = i;
            j = right;
        } else {
            if (i < right)
                ui_strSort (i, right);
            right = j;
            i = left;
        }
    }
    while (left < right);   

}



/*****************************************************************************
  FUNCTION :                 UI_READ DIRECTORY

  PURPOSE  : reads the contents of the current directory
  RETURNS  :
  NOTES    : puts 0x01 and 0x02 around the name if entry is a subdirectory
*****************************************************************************/



void
ui_readDirectory ()

{
    struct dirent *dp;   
    DIR *dirp;
    int i;
    char *sptr, *pptr;
    char newpath[MAX_NAME_LENGTH];
    char tempName[MAX_NAME_LENGTH];
    char errorMsg[255];
    
    dirEntries[0] = '\0';
    dirp = opendir (ui_pathname); 
    stat_buf = (struct stat *) malloc (sizeof (struct stat)); 
    for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
        strcpy (newpath, ui_pathname);
        strcat (newpath, "/");
        strcat (newpath, dp->d_name);
        stat (newpath, stat_buf);
        if (S_ISDIR(stat_buf->st_mode)   AND
           (S_IXUSR & stat_buf->st_mode) AND
           (S_IXGRP & stat_buf->st_mode) AND
           (S_IXOTH & stat_buf->st_mode))  {
            strcat (dirEntries, "\1");
            strcat (dirEntries, dp->d_name);
            strcat (dirEntries, "\2");
	    strcat (dirEntries, "\n");
        } else {
	    sptr = strrchr (dp->d_name, '.'); 
            if (sptr != NULL) {
                if (strcmp(sptr, extMask) == 0) {
   	            strcpy (tempName, dp->d_name);
	            pptr = strrchr (tempName, '.');
	            *pptr = '\0';
                    strcat (dirEntries, tempName);
                    strcat (dirEntries, "\n");
		}
	    }
	}
    }    
    closedir (dirp);
    free (stat_buf);
    maxEntries = 1;
    dirPtr[0] = dirEntries;
    for (sptr = dirEntries; *sptr != '\0'; sptr++) {
        if (*sptr == '\n') {
            *sptr = '\0';
            dirPtr[maxEntries] = sptr + 1;
            maxEntries++;
            if (maxEntries == MAX_DIR_ENTRIES) {
        	ui_confirmOk("Error! Directory too large");
                return;
	    }
	}
    }

    ui_strSort (0, maxEntries-2);

    sortedDirEntries[0] = '\0';
    for (i=0; i<maxEntries; i++) {
        strcat (sortedDirEntries, dirPtr[i]);
        strcat (sortedDirEntries, "\n");
    }

    for (sptr = sortedDirEntries; *sptr != '\0'; sptr++) {
        switch (*sptr) {
	    case '\1': *sptr = '[';
                       break;
	    case '\2': *sptr = ']';
                       break;
        }
    }
}




/*****************************************************************************
  FUNCTION :                 UI_SET EXT STRING

  PURPOSE  : sets the file extention to .txt, .pat, cfg or .txt
  RETURNS  :
  NOTES    : 
*****************************************************************************/



void 
ui_setExtString(w, fileType, call_data)

    Widget  w;
    int     fileType;
    caddr_t call_data;


{
    Arg		args[2];
    Cardinal	n;

    ui_xSetToggleState (netToggle, FALSE);    
    ui_xSetToggleState (patToggle, FALSE);    
    ui_xSetToggleState (resToggle, FALSE);    
    ui_xSetToggleState (cfgToggle, FALSE);    
    ui_xSetToggleState (txtToggle, FALSE);    
    switch (fileType) {
        case UI_FILE_NET: ui_xSetToggleState (netToggle, TRUE);
                          ui_xSetString(fileBox, ui_filenameNET);
                          strcpy (extMask, ".net");
                          break;
        case UI_FILE_PAT: ui_xSetToggleState (patToggle, TRUE);
                          ui_xSetString(fileBox, ui_filenamePAT);
                          strcpy (extMask, ".pat");
                          break;
        case UI_FILE_RES: ui_xSetToggleState (resToggle, TRUE);
                          ui_xSetString(fileBox, ui_filenameRES);
                          strcpy (extMask, ".res");
                          break;
        case UI_FILE_CFG: ui_xSetToggleState (cfgToggle, TRUE);
                          ui_xSetString(fileBox, ui_filenameCFG);
                          strcpy (extMask, ".cfg");
                          break;
        case UI_FILE_TXT: ui_xSetToggleState (txtToggle, TRUE);
                          ui_xSetString(fileBox, ui_filenameTXT);
                          strcpy (extMask, ".txt");
                          break;
    }
    currentFileType = fileType;
    ui_xStringFromAsciiWidget(ui_path, ui_pathname, MAX_NAME_LENGTH);
    ui_checkPath(ui_pathname);
    ui_readDirectory ();
    n = 0;
    XtSetArg(args[n], XtNstring, sortedDirEntries); n++;
    XtSetValues (selectorBox, args, n);
}



/*****************************************************************************
  FUNCTION :                 UI_CHANGE DIRECTORY

  PURPOSE  : changes the current path for the input
  RETURNS  :
  NOTES    : doesn't work with an auto mounter
*****************************************************************************/



void
ui_changeDirectory(dirName)

    char dirName[MAX_NAME_LENGTH];
     
{
    char *sptr;
    Arg		args[2];
    XawTextPosition  start, end, found;
    Cardinal	n;

    if ((dirName[0] == '.') AND (dirName[1] == '\0'))
        return;
    if ((dirName[0] == '.') AND (dirName[1] == '.')) {
      if (strcmp (ui_pathname, "/") == 0)
        return;
      sptr = strrchr(ui_pathname, '/');
       if (sptr == ui_pathname)
         sptr++;
       *sptr = '\0';
    } else {
       if (strcmp (ui_pathname, "/") != 0)
           strcat(ui_pathname, "/");
       strcat(ui_pathname, dirName);
    }
    ui_checkPath(ui_pathname);
    ui_readDirectory();
    ui_xSetString(ui_path, ui_pathname);
    n = 0;
    XtSetArg(args[n], XtNstring, sortedDirEntries); n++;
    XtSetValues (selectorBox, args, n);
}


/*****************************************************************************
  FUNCTION :                 UI_SELECTION EVENT PROC

  PURPOSE  : puts the current filename in the dialog wisgwt after the second
             mouseklick
  RETURNS  :
  NOTES    : 
*****************************************************************************/


void
ui_selectionEventProc(w, display, event)

    Widget           w;
    Display          *display;
    XEvent           *event;
{
    Widget           src;
    XawTextBlock     textBlock, totalText;
    XawTextPosition  start, end, found;
    Cardinal         n;
    Arg              args[5];
    char             selectedName[SELECTED_NAME_LENGTH];
    char             realName[SELECTED_NAME_LENGTH];
    char             *sptr;
    Boolean          nlStarts, nlEnds;
    int              i;


    selectedName[0] = '\0';
    switch (event->type) {
        case ButtonRelease:
            if (event->xbutton.button == 1) {
                 XawTextGetSelectionPos(selectorBox, &start, &end);
                 if (start != end) {
                     n = 0;
	             XtSetArg(args[n], XtNtextSource, &src); n++;
	             XtGetValues (selectorBox, args, n);
	             found = XawTextSourceRead(src, (int) start, &textBlock, 
                         (int) (end-start));
	             if (found > 0) {
                         nlStarts = nlEnds = FALSE;
	                 strncpy(selectedName, textBlock.ptr-1, (int) (end-start+2));
                         selectedName[end-start+2] = '\0';
        	         if (selectedName[0] == '\n')
                             nlStarts = TRUE;
	                 if (selectedName[strlen(selectedName)-1] == '\n')
                             nlEnds = TRUE;
                         if (nlStarts AND nlEnds) {
                             for (i=0; i<end-start; i++)
                                 realName[i] = selectedName[i+1];
                             realName[end-start] = '\0';
                             if (strchr(realName, '\n' != NULL))
                                 return;
                             if (realName[0] != '[') {
                                 n = 0;
                                 XtSetArg (args[n], XtNstring, realName); n++;
                                 XtSetValues (fileBox, args, n);
			     } else {
			         for (i=0; i<strlen(realName)-2; i++)
                                      realName[i] = realName[i+1];
				 realName[strlen(realName)-2] = '\0';
				 ui_changeDirectory(realName);
			     }
		         }
		     }
		 }
	  }
    }
}



/*****************************************************************************
  FUNCTION :                 UI_XCREATE FILE PANEL

  PURPOSE  : create the file selector panel
  RETURNS  : void
  NOTES    : the wigdet will be created in relative position to eachother
*****************************************************************************/

void
ui_xCreateFilePanel(parent)
     Widget parent;      /* the parent widget of the new form widget */

{
    int  fontWidth = 8;
    int  titelWidth  = 14 * fontWidth;
    int  nameWidth   = 15 * fontWidth;  


    Arg		args[20];
    Cardinal	n;
    Widget      dummy, button, label;

    ui_filePanel =
	XtCreateManagedWidget("fPanel", formWidgetClass, parent, NULL, ZERO);
    ui_expandPath(ui_pathname);
    ui_path =
	ui_xCreateDialogItem("path", ui_filePanel, ui_pathname, 0 /* free */, NULL, NULL);

    strcpy (extMask, ".net");
    currentFileType = UI_FILE_NET;

    ui_checkPath(ui_pathname);
    ui_readDirectory ();

    n = 0;
    XtSetArg(args[n], XtNwidth,  150); n++;
    XtSetArg(args[n], XtNheight, 245); n++;
    XtSetArg(args[n], XtNtype, XawAsciiString); n++;
    XtSetArg(args[n], XtNstring, sortedDirEntries); n++;
    XtSetArg(args[n], XtNscrollVertical, XawtextScrollAlways); n++;
    XtSetArg(args[n], XtNdisplayCaret, FALSE); n++;
    XtSetArg(args[n], XtNfromVert , ui_path);  n++;
    XtSetArg(args[n], XtNfromHoriz, NULL);  n++;
    XtSetArg(args[n], XtNleft  , XtChainLeft);   n++;
    XtSetArg(args[n], XtNright , XtChainLeft);   n++;
    XtSetArg(args[n], XtNtop   , XtChainTop);    n++;
    XtSetArg(args[n], XtNbottom, XtChainTop);    n++;	
    selectorBox = XtCreateManagedWidget("textWin", asciiTextWidgetClass, 
       ui_filePanel, args, n);

    XtAddEventHandler (selectorBox, ButtonReleaseMask, 
                        FALSE, ui_selectionEventProc, ui_display);

    fileBox =
	ui_xCreateDialogItem("filebox", ui_filePanel, "", nameWidth,
      			     selectorBox, ui_path);

    ui_xSetResize(fileBox, TRUE);
    ui_xSetString(fileBox, ui_filenameNET);

    dummy =
	ui_xCreateLabelItem(" ", ui_filePanel, 8, selectorBox, fileBox);

    dummy =
	ui_xCreateLabelItem(" ", ui_filePanel, 8, selectorBox, dummy);

    netToggle =
	ui_xCreateToggleItem("net", ui_filePanel, NULL,
        selectorBox, dummy);
    label =
	ui_xCreateLabelItem(" Network", ui_filePanel, titelWidth,
        netToggle, dummy);
    patToggle =
	ui_xCreateToggleItem("pat", ui_filePanel, NULL, 
        selectorBox , netToggle);
    label =
	ui_xCreateLabelItem(" Patterns", ui_filePanel, titelWidth,  
        patToggle, netToggle);
    resToggle =
	ui_xCreateToggleItem("res", ui_filePanel, NULL, 
        selectorBox , patToggle);
    label =
	ui_xCreateLabelItem(" Result File", ui_filePanel, titelWidth,  
        resToggle, patToggle);
    cfgToggle =
	ui_xCreateToggleItem("cfg", ui_filePanel, NULL,
        selectorBox , resToggle);
    label =
	ui_xCreateLabelItem(" Config", ui_filePanel, titelWidth,  
        cfgToggle, resToggle);
    txtToggle =
	ui_xCreateToggleItem("txt", ui_filePanel, NULL, 
        selectorBox , cfgToggle);
    label =
	ui_xCreateLabelItem(" Log File", ui_filePanel, titelWidth,  
        txtToggle, cfgToggle);

    XtAddCallback(netToggle, XtNcallback, ui_setExtString, (int) UI_FILE_NET);
    XtAddCallback(patToggle, XtNcallback, ui_setExtString, (int) UI_FILE_PAT);
    XtAddCallback(resToggle, XtNcallback, ui_setExtString, (int) UI_FILE_RES);
    XtAddCallback(cfgToggle, XtNcallback, ui_setExtString, (int) UI_FILE_CFG);
    XtAddCallback(txtToggle, XtNcallback, ui_setExtString, (int) UI_FILE_TXT);

    dummy =
	ui_xCreateLabelItem(" ", ui_filePanel, 8, selectorBox, txtToggle);

    dummy =
	ui_xCreateLabelItem(" ", ui_filePanel, 8, selectorBox, dummy);

    dummy =
	ui_xCreateLabelItem(" ", ui_filePanel, 8, selectorBox, dummy);

    button =
	ui_xCreateButtonItem("load", ui_filePanel, selectorBox , dummy);

    XtAddCallback(button, XtNcallback,
        ui_loadSelectedFile, NULL);

    button =
	ui_xCreateButtonItem("save", ui_filePanel, button , dummy);

    XtAddCallback(button, XtNcallback,
        ui_saveSelectedFile, NULL);

    XawFormDoLayout(ui_filePanel, True);

    ui_xSetToggleState (netToggle, TRUE);    
 
}







/* end of file */
/* lines: 521 */
