/*****************************************************************************
  FILE     : ui_remoteP.c
  SHORTNAME: remoteP.c

  PURPOSE  : routines to control learning, updating the net
  FUNCTIONS: 
  NOTES    : Most of this routines are callbacks of buttons in the remote-
             panel.

  AUTHOR   : Tilman Sommer
  DATE     : 18.7.1990

             (c) 1992 by Tilman Sommer and the SNNS-Group
             
******************************************************************************/

#include <stdio.h>


#include "ui.h"
#include "ui_mainP.h"
#include "ui_utilP.h"
#include "ui_selection.h"
#include "ui_netUpdate.h"

#include "kr_ui.h"

#include "ui_fileP.h"
#include "ui_confirmer.h"
#include "ui_textP.h"
#include "ui_xWidgets.h"
#include "ui_display.h"
#include "ui_remoteP.h"

#include "o_graph.h"


/* ascii and label widgets of remote panel containing data */
extern Widget          ui_numberOfStepsWidget, ui_numberOfCyclesWidget;
extern Widget          ui_numberOfStepsDoneWidget, ui_numberOfPatternWidget;
extern Widget          ui_showModeLabel;
extern Widget          ui_learnParameterWidgets[];
extern Widget          ui_initParameterWidgets[];
extern Widget          ui_updateParameterWidgets[];
/* jog weights */
extern Widget          ui_lowLimitWidget, ui_highLimitWidget;
extern char            ui_filenamePAT[];   /* current pattern file name */
extern int             ui_patternLoadMode; /* how to show output units when a
					      pattern is recalled */
extern struct PosType ui_gridPosMouse; /* current mouse position */

/* functions and variables exported by o_graph.c */

extern void            o_InitCurve();
extern int             o_LearnStepCount;
extern short           o_CurveLengths[MAX_CURVE_NO];
extern int             o_CurveNo;
extern int             o_PressPossible;
extern int             o_open;

/* current configuration of remote panel */
extern int             ui_numberOfCycles, ui_numberOfStepsDone;
extern int             ui_noOfCurrentPattern;
extern float           ui_learnParameters[];
extern float           ui_updateParameters[];
extern float           ui_initParameters[];
extern Bool            ui_shuffleFlg;

extern Bool            ui_remoteIsCreated; /* panel created yes or no */
extern XtAppContext    ui_appContext;      /* application context */
extern struct Ui_DisplayType *ui_currentDisplay;  /* current xgui display */


/* use of work procs (update steps and learning cycles) */
static XtWorkProcId  ui_workProcId =  0;         /* id reported by X */
static int           ui_numberOfWorkCycles = 0;  /* loops left to do */
static int           ui_numberOfLearnCycles = 0; /* loops to do */
static FlagType      ui_workType = UI_STEPS;     /* UI_STEPS or UI_LEARN */

static float         *ui_errorArray;
static int           ui_noErrorArrayElements;





/*****************************************************************************
  FUNCTION : ui_rem_displayCounter

  PURPOSE  : displays the value in ui_numberOfStepsDone in the remote panel
  RETURNS  : ---
*****************************************************************************/


void
ui_rem_displayCounter()
{
    char  buf[50];
    if (NOT ui_remoteIsCreated) return;

    sprintf(buf,"%d", ui_numberOfStepsDone);
    ui_xSetLabel(ui_numberOfStepsDoneWidget, buf);
}


/*****************************************************************************
  FUNCTION : ui_rem_displayPatternNumber

  PURPOSE  : displays the value in ui_numberOfCurrentPattern in the 
             remote panel
  RETURNS  : ---
*****************************************************************************/


void
ui_rem_displayPatternNumber()
{
    char  buf[50];

    if (NOT ui_remoteIsCreated) return;

    sprintf(buf,"%d", ui_noOfCurrentPattern);
    ui_xSetString(ui_numberOfPatternWidget, buf);
}




/*****************************************************************************
 FUNCTION : ui_rem_resetCounter()

  PURPOSE  : set counter to 0
  NOTES    :

  UPDATE   : 30.7.1990
******************************************************************************/


void
ui_rem_resetCounter()
{
    ui_numberOfStepsDone = 0;
    ui_rem_displayCounter();
}




/*****************************************************************************
  FUNCTION : ui_rem_incCounter()

  PURPOSE  : increment counter by 1
  NOTES    :

  UPDATE   : 30.7.1990
******************************************************************************/


void
ui_rem_incCounter()
{
    ui_numberOfStepsDone += 1;
    ui_rem_displayCounter();
}



/*****************************************************************************
  FUNCTION : ui_rem_resetNet

  PURPOSE  : Resets the net to the initial activation and resets the counter
  NOTES    :

  UPDATE   : 24.8.1990
******************************************************************************/

void
ui_rem_resetNet(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    ui_sel_reshowItems(ui_currentDisplay, UI_GLOBAL);
    ui_net_updateWhole(NULL, UI_GLOBAL, UI_DRAW_UNITS, UI_ERASE);
    krui_resetNet();
    ui_rem_resetCounter();
    ui_net_updateWhole(NULL, UI_GLOBAL, UI_DRAW_UNITS, UI_DRAW);
    ui_sel_reshowItems(ui_currentDisplay, UI_GLOBAL);
}






/*****************************************************************************
  FUNCTION : ui_rem_doInitialization

  PURPOSE  : get the parameters and call the kernel
  NOTES    : 

  UPDATE   : 18.3.1991
******************************************************************************/

void
ui_rem_doInitialization(w, client_data, call_data)
     Widget	w;		
     caddr_t    client_data, call_data;
{
    int i, err;
 
    if(o_open){
      o_CurveNo += ((o_CurveLengths[o_CurveNo] != 0) && (o_CurveNo < MAX_CURVE_NO-1)) ? (1) : (0);
      o_InitCurve();
      o_LearnStepCount = 0;
    }
 
    for (i=0; i< UI_NO_INIT_PARAMS; i++) {
	ui_initParameters[i] =
	    ui_xFloatFromAsciiWidget(ui_initParameterWidgets[i]);
    }
    ui_checkError(krui_initializeNet(ui_initParameters, UI_NO_INIT_PARAMS));
    if (ui_displ_isSomeWhereToShowWeights())
	ui_net_completeRefresh(NULL, UI_GLOBAL);
    else
	ui_net_updateWhole(NULL, UI_GLOBAL, UI_DRAW_UNITS, UI_DRAW);    
    ui_sel_reshowItems(ui_currentDisplay, UI_GLOBAL);
}



/*****************************************************************************
  FUNCTION : ui_rem_deleteNet

  PURPOSE  : clear SNNS kernel
  NOTES    : 

  UPDATE   : 18.3.1991
******************************************************************************/

void
ui_rem_deleteNet(w, client_data, call_data)
     Widget	w;		
     caddr_t    client_data, call_data;
{
    if (ui_confirmYes("CLEAR: The network and all patterns will be lost. Clear?")) {
	/* reset kernel */
	krui_deleteNet();
	/* reset gui */
	ui_sel_resetList();       /* reset selections */
	ui_net_completeRefresh(NULL, UI_GLOBAL); /* show net */
	ui_printMessage("");
	ui_stat_displayStatus(ui_gridPosMouse);
	ui_file_updateShellLabels();
	ui_info_makeUnitInfoPanelConsistent();	
        d3_clear_xyTranslationTable ();
    }
}




/*****************************************************************************
  FUNCTION : ui_rem_finishSteps

  PURPOSE  : finish performing update steps
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/


void ui_rem_finishSteps()
{
    ui_printMessage("");
    ui_sel_reshowItems(ui_currentDisplay, UI_GLOBAL); /* reshow selection markers */
}




/*****************************************************************************
  FUNCTION :                 UI_REM_PERFORM STEP

  PURPOSE  : called by pressing the button STEPS
  RETURNS  : void
*****************************************************************************/

Boolean 
ui_rem_performStep(client_data)
     caddr_t   client_data;
{
    if (ui_numberOfWorkCycles-- < 1) {
	ui_rem_finishSteps();
	return(TRUE);
    }
    /* erase the units first, if they have to be drawn with their values
       in numerical form at the bottom (setup parameter) */
    if (ui_displ_isSomeWhereToShowValues())
	ui_net_updateWhole(NULL, UI_GLOBAL, 
			   UI_DRAW_UNITS, UI_ERASE_BOTTOM);
    ui_checkError(krui_updateNet(ui_updateParameters, UI_NO_UPDATE_PARAMS));
    ui_rem_incCounter();
    ui_net_updateWhole(NULL, UI_GLOBAL, UI_DRAW_UNITS, UI_DRAW);
    return(FALSE);
}






/*****************************************************************************
  FUNCTION :                        UI_REM_STEPS PROC

  PURPOSE  : perform one or more steps, depending on the PANEL_CLIENT_DATA
             if STEP is pressed one update step in the current mode
	     if STEPS is pressed: get the number of steps to perform
	     from the panel and go.
  RETURNS  : void
  NOTES    :
*****************************************************************************/


void  
ui_rem_stepsProc(widget, multiStepPressed, call_data)
     Widget  widget;
     Bool    multiStepPressed;
     caddr_t call_data;
{ 
    char   buf[80];
    int    i;

    if (multiStepPressed) 
	/* MultiSTEPS button */
	if ((ui_numberOfWorkCycles = 
	     ui_xIntFromAsciiWidget(ui_numberOfStepsWidget)) <= 0)
	    return;
	
    sprintf(buf,"%d steps. Calculating ...", ui_numberOfWorkCycles);
    ui_printMessage(buf);
   

    for (i=0; i<UI_NO_UPDATE_PARAMS; i++)
	ui_updateParameters[i] = (float) 
	    ui_xFloatFromAsciiWidget(ui_updateParameterWidgets[i]);  
    ui_sel_reshowItems(ui_currentDisplay, UI_GLOBAL); /* erase selection markers */

    if (ui_workProcId)
	XtRemoveWorkProc(ui_workProcId); /* kill old workProc */
    ui_workType = UI_STEPS;
    ui_workProcId = XtAppAddWorkProc(ui_appContext, ui_rem_performStep, NULL);

}



/*****************************************************************************
  FUNCTION : ui_rem_moveInPatternsProc             

  PURPOSE  : callback. Called when clicked on a cursor button or the GOTO 
             button.

  RETURNS  : void
*****************************************************************************/


void
ui_rem_moveInPatternsProc(w, moveType, call_data)
     Widget    w;		
     int       moveType;
     caddr_t   call_data;
{

    int   temp;
    Bool  doMove = FALSE;


    if (krui_getNoOfPatterns() == 0) {
	ui_confirmOk("No patterns in memory!");
	return;
    }

    /* there are patterns */

    switch (moveType) {
      case UI_JUMP: 
	temp = ui_xIntFromAsciiWidget(ui_numberOfPatternWidget);
	if ((temp > 0) AND (temp <= krui_getNoOfPatterns())) {
	    ui_noOfCurrentPattern = temp;
	    ui_rem_displayPatternNumber();
	    doMove = TRUE;
	} else
	    ui_confirmOk("pattern goto: invalid number!");
	break;
      case UI_FIRST:
	if (ui_noOfCurrentPattern != 1) {
	    ui_noOfCurrentPattern = 1;
	    ui_rem_displayPatternNumber();
	    doMove = TRUE;
	}
	break;
      case UI_LAST:
	if (ui_noOfCurrentPattern != krui_getNoOfPatterns()) {
	    ui_noOfCurrentPattern = krui_getNoOfPatterns();
	    ui_rem_displayPatternNumber();
	    doMove = TRUE;
	}
	break;
      case UI_NEXT:
	if (++ui_noOfCurrentPattern > krui_getNoOfPatterns())
	    ui_noOfCurrentPattern = 1;
	ui_rem_displayPatternNumber();
	doMove = TRUE;
	break;
      case UI_PREVIOUS:
	if (--ui_noOfCurrentPattern < 1)
	    ui_noOfCurrentPattern = krui_getNoOfPatterns();
	ui_rem_displayPatternNumber();
	doMove = TRUE;
	break;
    }
    
    if (doMove) {
	/* new pattern: reset */
	ui_rem_resetCounter();
	
	krui_setPatternNo(ui_noOfCurrentPattern);
	ui_checkError(krui_showPattern(ui_patternLoadMode));
	ui_net_updateWhole(NULL, UI_GLOBAL, UI_DRAW_UNITS, UI_DRAW);
    }
}



/*****************************************************************************
  FUNCTION :                UI_REM_TEST PROC

  PURPOSE  : load the next (or first) pattern and perform the number of steps
             specified in the remote panel in "no of steps"
  RETURNS  : void
*****************************************************************************/


void
ui_rem_testProc(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    if (krui_getNoOfPatterns() == 0) {
	char buf[80];
	sprintf(buf,"No patterns present.");
	ui_printMessage(buf);
	return;
    }

    /* there are patterns */

    if (ui_noOfCurrentPattern > krui_getNoOfPatterns() - 1)
	ui_noOfCurrentPattern = 0; /* set before first pattern */

    /* new pattern: reset */
    ui_rem_resetCounter();

    krui_setPatternNo(++ui_noOfCurrentPattern);
    ui_checkError(krui_showPattern(ui_patternLoadMode));
    ui_rem_displayPatternNumber();
    if (ui_xIntFromAsciiWidget(ui_numberOfStepsWidget) < 1)
	ui_net_updateWhole(NULL, UI_GLOBAL, 
			   UI_DRAW_UNITS, UI_DRAW);
    else
	ui_rem_stepsProc(NULL, TRUE, NULL); /* multistep */
}




/*****************************************************************************
  FUNCTION : ui_rem_deleteAllPatternsProc

  PURPOSE  : delete all patterns in main memory. The contents of the current
             pattern file (if any) is left unchanged 
  NOTES    :

  UPDATE   : 24.8.1990
******************************************************************************/

void
ui_rem_deleteAllPatternsProc(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    if (ui_confirmYes("Delete all patterns. Are you sure?")) {
	krui_deleteAllPatterns();
	ui_noOfCurrentPattern = 0;
	ui_rem_displayPatternNumber();
    }
}




/*****************************************************************************
  FUNCTION : ui_rem_modifyPatternProc

  PURPOSE  : calls the kernal in order to store the current visible pattern
             replacing the current pattern. 
  NOTES    :

  UPDATE   : 18.3.1991
******************************************************************************/

void
ui_rem_modifyPatternProc(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    int temp = ui_xIntFromAsciiWidget(ui_numberOfPatternWidget);
    int err;

    if ((temp > 0) AND (temp <= krui_getNoOfPatterns())) {
	ui_noOfCurrentPattern = temp;
	krui_setPatternNo(ui_noOfCurrentPattern);
	ui_checkError(krui_modifyPattern());
	if (ui_noOfCurrentPattern > krui_getNoOfPatterns()) {
	    ui_noOfCurrentPattern = krui_getNoOfPatterns();
	    ui_rem_displayPatternNumber();
	} 
    } else
	ui_confirmOk("modify pattern: invalid pattern number!");
}



/*****************************************************************************
  FUNCTION : ui_rem_deletePatternProc

  PURPOSE  : calls the kernal in order to delete the current visible pattern.
  NOTES    :

  UPDATE   : 18.3.1991
******************************************************************************/

void
ui_rem_deletePatternProc(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    int temp = ui_xIntFromAsciiWidget(ui_numberOfPatternWidget);

    if ((temp > 0) AND (temp <= krui_getNoOfPatterns())) {
	ui_noOfCurrentPattern = temp;
	krui_setPatternNo(ui_noOfCurrentPattern);
	ui_checkError(krui_deletePattern());
	if (ui_noOfCurrentPattern > krui_getNoOfPatterns()) {
	    ui_noOfCurrentPattern = krui_getNoOfPatterns();
	    ui_rem_displayPatternNumber();
	} 
    } else
	ui_confirmOk("delete pattern: invalid pattern number!");
}




/*****************************************************************************
  FUNCTION : ui_rem_newPatternProc

  PURPOSE  : calls the kernal in order to store the current visible pattern
             as a new one. This new pattern will be appended.
  NOTES    :

  UPDATE   : 24.8.1990
******************************************************************************/

void
ui_rem_newPatternProc(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    ui_checkError(krui_newPattern());
    ui_noOfCurrentPattern = krui_getNoOfPatterns();
    ui_rem_displayPatternNumber();
}


    


/*****************************************************************************
  FUNCTION : ui_rem_printLearnInfo

  PURPOSE  : print information about last learning cycle
  NOTES    :

  UPDATE   : 13.2.1991
******************************************************************************/

void
ui_rem_printLearnInfo()
{
    char    buf[MAX_NAME_LENGTH];
    int     i;

    sprintf(buf,"   cycle %5d:", ui_numberOfWorkCycles + 1);
    ui_tw_printMessage(buf);
    for (i = 0; i < ui_noErrorArrayElements; i++) {
	sprintf(buf,"   error %2d = %10.5f", i + 1, ui_errorArray[i]);
	ui_tw_printMessage(buf);
	if (krui_getNoOfOutputUnits() > 0) {
	    sprintf(buf,"    ave = %10.5f\n", 
		    (ui_errorArray[i]/krui_getNoOfOutputUnits()));
	    ui_tw_printMessage(buf);
	} else
	    ui_tw_printMessage("\n");
    }
}


/*****************************************************************************
  FUNCTION : ui_rem_finishLearning

  PURPOSE  : finish performing learning cycles
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/


void ui_rem_finishLearning()
{
    if (ui_shuffleFlg) 
	/* this brings the patterns back into their old order before shuffling
	   was done. */
	ui_checkError(krui_shufflePatterns(FALSE));
    
    if (ui_displ_isSomeWhereToShowWeights())
	ui_net_completeRefresh(NULL, UI_GLOBAL);
    else
	ui_net_updateWhole(NULL, UI_GLOBAL, 
			   UI_DRAW_UNITS, UI_DRAW);
}



/*****************************************************************************
  FUNCTION : ui_rem_performLearn

  PURPOSE  : perform learning cycles
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/


Boolean
ui_rem_performLearn(learnType)
     int  learnType;

{
    krui_err err;

    if (ui_numberOfWorkCycles-- < 1) {
	ui_rem_finishLearning();
	return(TRUE);
    }

    switch (learnType) {
      case UI_LEARN_ALL:
	if (ui_shuffleFlg) 
	    ui_checkError(krui_shufflePatterns(ui_shuffleFlg));
	ui_checkError(err = krui_learnAllPatterns(ui_learnParameters, 
					    UI_NO_LEARN_PARAMS,
					    &ui_errorArray,
					    &ui_noErrorArrayElements));
        if (err != KRERR_NO_ERROR)
            return (TRUE);
	break;
      case UI_LEARN_SINGLE:
	if ((ui_noOfCurrentPattern <= krui_getNoOfPatterns()) AND
	    (ui_noOfCurrentPattern > 0)) {
	    ui_checkError(err = krui_learnSinglePattern(ui_noOfCurrentPattern,
						  ui_learnParameters,
						  UI_NO_LEARN_PARAMS,
						  &ui_errorArray,
						  &ui_noErrorArrayElements));
            if (err != KRERR_NO_ERROR)
                return (TRUE);
	} else
	    ui_confirmOk("Can't learn current pattern.");
	break;
    }
    if ((ui_numberOfWorkCycles == ui_numberOfLearnCycles) or
	(ui_numberOfWorkCycles == 0) or
	(ui_numberOfLearnCycles <= 10) or
	(((ui_numberOfWorkCycles+1) MOD (ui_numberOfLearnCycles DIV 10)) == 0)) 
	ui_rem_printLearnInfo();
    
    if(o_open){
      o_draw(ui_errorArray[0]/krui_getNoOfOutputUnits());
      if(ui_numberOfWorkCycles == 0){
        o_PressPossible = 1;
      }
    }
    return(FALSE);
}


/*****************************************************************************
  FUNCTION :                       UI_REM_LEARN PROC

  PURPOSE  : call the learn procedure of the kernal. The number of learn cycles
             is determined by the user and also the learn parameter.
  RETURNS  : implicite alteration of the network
  NOTES    : 
*****************************************************************************/

void
ui_rem_learnProc(w, learnType, call_data)
     Widget    w;		
     int       learnType;
     caddr_t   call_data;
{
    char  buf[MAX_NAME_LENGTH];
    int   i;
    
    o_PressPossible = 0;
    /* check now for errors and react ! */
    if (krui_getNoOfPatterns == 0) {
	ui_confirmOk("No patterns loaded/created!");
	return;
    }

    ui_numberOfWorkCycles = /* this value will be decreased */
	ui_xIntFromAsciiWidget(ui_numberOfCyclesWidget);
    ui_numberOfLearnCycles = /* this value stays constant */
	ui_numberOfWorkCycles;
    for (i=0; i<UI_NO_LEARN_PARAMS; i++)
	ui_learnParameters[i] = (float) 
	    ui_xFloatFromAsciiWidget(ui_learnParameterWidgets[i]);
    
    if (learnType == UI_LEARN_ALL)
	ui_tw_printMessage("\nLearning all patterns:\n");
    else
	ui_tw_printMessage("\nLearning current pattern:\n");

    sprintf(buf,"  cycles   : %d\n  parameter: %8.5f\n  #o-units : %d\n", 
	    ui_numberOfWorkCycles, ui_learnParameters[0], krui_getNoOfOutputUnits());
    ui_tw_printMessage(buf);
    
    if (ui_workProcId)
	XtRemoveWorkProc(ui_workProcId); /* kill old workProc */
    ui_workType = UI_LEARN;
    ui_workProcId = XtAppAddWorkProc(ui_appContext, ui_rem_performLearn, learnType); 

}


/*****************************************************************************
  FUNCTION : UI_REM_STOP PROC

  PURPOSE  : callback. Called after click on STOP button
  RETURNS  : 
  NOTES    :
*****************************************************************************/


void  
ui_rem_stopProc(widget, client_data, call_data)
     Widget  widget;
     caddr_t client_data, call_data;
{ 
    if (ui_workProcId) { 
	switch (ui_workType) {
	  case UI_STEPS: 
	    ui_rem_finishSteps();
	    break;
	  case UI_LEARN:
	    ui_rem_finishLearning();
	    ui_rem_printLearnInfo();
	    break;
	}
	XtRemoveWorkProc(ui_workProcId); /* kill workProc */
	ui_workProcId = 0;
	ui_numberOfWorkCycles = 0;
    }
    o_PressPossible = 1;
}



/*****************************************************************************
  FUNCTION :                  UI_REM_SHUFFLE PROC

  PURPOSE  : notify procedure of the toggle item, to remember the status of
             it.
  RETURNS  : changes the flag ui_shuffleFlg
*****************************************************************************/

void 
ui_rem_shuffleProc(w, client, call)
     Widget   w;
     caddr_t  client, call;
{
    ui_shuffleFlg = ui_xGetToggleState(w);
}



/*****************************************************************************
  FUNCTION :                    UI_REM_SHOW MODE PROC

  PURPOSE  : notify procedure of the panel cycle.
  RETURNS  : sets the variable ui_patternLoadMode
*****************************************************************************/

void
ui_rem_showModeProc(w, value, call_data)
     Widget w;
     int      value;
     caddr_t  call_data;
{    
    ui_patternLoadMode = (int) value;
    switch (value) {
      case OUTPUT_NOTHING:
	ui_xSetLabel(ui_showModeLabel, "none");
	break;
      case OUTPUT_OUT:
	ui_xSetLabel(ui_showModeLabel, "output");
	break;
      case OUTPUT_ACT:
	ui_xSetLabel(ui_showModeLabel, "activation");
	break;
    }
}





/*****************************************************************************
  FUNCTION : ui_rem_setRandomWeights

  PURPOSE  : set weights of all links randomly

  RETURNS  : 
*****************************************************************************/

void
ui_rem_setRandomWeights(w, randomType, call_data)
     Widget   w;
     int      randomType;
     caddr_t  call_data;
{    
    FlintType  low, high, temp;

    low  = (FlintType) ui_xFloatFromAsciiWidget(ui_lowLimitWidget);
    high = (FlintType) ui_xFloatFromAsciiWidget(ui_highLimitWidget);
    if (low > high) {
	temp = low;
	low  = high;
	high = temp;
    }

    switch (randomType) {
      case UI_RANDOM:
	/* krui_randomizeWeights(low, high); */	   
	break;
      case UI_JOG:
	krui_jogWeights(low, high);
	break;
    }
    if (ui_displ_isSomeWhereToShowWeights())
	ui_net_completeRefresh(NULL, UI_GLOBAL);
}




/*****************************************************************************
  FUNCTION : ui_rem_getKernelInfo

  PURPOSE  : get some info from the kernal. Output to text window
  NOTES    :

  UPDATE   : 1.2.1990
******************************************************************************/

void
ui_rem_getKernelInfo(w, button, call_data)
     Widget    w;		
     XtPointer button;
     caddr_t   call_data;
{
    char   buf[127];
    int    no_of_sites, no_of_links, no_of_STable_entries, no_of_FTable_entries;
    int    unit_bytes, site_bytes, link_bytes, 
           NTable_bytes, STable_bytes, FTable_bytes;

    sprintf(buf,"\n%s :\n\n", krui_getVersion());
    ui_tw_printMessage(buf);

    krui_getNetInfo(&no_of_sites, &no_of_links,
		    &no_of_STable_entries, &no_of_FTable_entries);

    sprintf(buf,"#input  units: %6d\n#output units: %6d\n#patterns    : %6d\n",
	    krui_getNoOfInputUnits(), krui_getNoOfOutputUnits(), 
	    krui_getNoOfPatterns());
    ui_tw_printMessage(buf);
    sprintf(buf,"#sites       : %6d\n#links       : %6d\n",
	    no_of_sites, no_of_links);
    ui_tw_printMessage(buf);
    sprintf(buf,"#STable entr.: %6d\n#FTable-Entr.: %6d\n\n",
	    no_of_STable_entries, no_of_FTable_entries);
    ui_tw_printMessage(buf);

    ui_tw_printMessage("sizes in bytes:\n");
    krui_getMemoryManagerInfo(&unit_bytes, &site_bytes, &link_bytes,
			      &NTable_bytes, &STable_bytes, &FTable_bytes);
    sprintf(buf,"units        : %7d\nsites        : %7d\nlinks        : %7d\n",
	    unit_bytes, site_bytes, link_bytes);
    ui_tw_printMessage(buf);
    sprintf(buf,"NTable       : %7d\nSTable       : %7d\nFTable       : %7d\n\n",
	    NTable_bytes, STable_bytes, FTable_bytes);
    ui_tw_printMessage(buf);
    sprintf(buf,"learning function  : %s\n", krui_getLearnFunc());
    ui_tw_printMessage(buf);
    sprintf(buf,"update function    : %s\n", krui_getUpdateFunc());
    ui_tw_printMessage(buf);
    sprintf(buf,"initialize function: %s\n", krui_getInitialisationFunc());
    ui_tw_printMessage(buf);

    ui_tw_printMessage("\n\n");
}






/* end of file */
/* lines: 921 */
