;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: ADVISOR -*-
;;; -----------------------------------------------------------------------
;;; File:         clause.l
;;; Description:  Grammatical systems for clause
;;; Author:       Michael Elhadad
;;; Created:      19 Dec 1991
;;; Modified:     07 Jul 92 - added past-participle mood treatment
;;;               14 Jul 92 - added Karen's constraint on pronoun/dative-move
;;;               18 Aug 92 - added adverb
;;;               19 Aug 92 - added scoped
;;;               27 Jun 93 - removed constraint that passive-prep be "by"
;;;               19 Oct 93 - added dots between binder and start in mood bound
;;;               05 Dec 93 - added possessive-relative
;;;               21 Dec 93 - fixed possessive-relative to recurse on scoper.
;;;                         - added generic-mood to make conjunction works.
;;; Package:      ADVISOR
;;; -----------------------------------------------------------------------

;;; (in-package "ADVISOR")

(require "transitivity" "transitivity")
(require "voice" "voice")
(require "circumstance" "circumstance")

;;; BRANCH FOR SIMPLE-CLAUSE
(def-conj simple-clause
  (cat simple-clause)
  (generic-cat clause)
  (complex none)
  
  ;; Some shorthands that are used for convenience...
  (verb {^ process})
  (proc {^ process})
  (cset ((- start)))
  (partic {^ participants})
  (circum   {^ circumstances})
  (process ((type {^ process-type})))
  (process-type {^ process process-type})
  (lex-roles {^ lexical-roles})
  
  ;; General structure of a clause
  ;; lexical-roles: semantic roles specific to a lexical item
  ;; participants: the semantic roles
  ;; oblique:      the obliqueness hierarchy of semantic roles
  ;; synt-roles:   the syntactic complements
  ;; circumstances: optional semantic roles (circumstantial)
  ;; adjuncts:      optional syntactic complements
  ;; Processing of nuclear participants:
  ;; Map from partic -> oblique -> synt-roles.
  ;; OR Map from lexical-roles -> oblique from lexicon.
  ;; First stage is transitivity, second is voice.
  ;; Government pattern of verbs is specified under subcat which is
  ;; unified with synt-roles (ala Mel'cuk and vaguely subcat of HPSG).
  (participants ((fset (agent affected created range
			      processor phenomenon
			      sayer addressee verbalization
			      carrier attribute
			      identified identifier
			      located location time
			      possessor possessed))))
  ;; Choice is either lexical-roles or partic or none but not both
  (alt type-of-roles 
      (((lexical-roles none)
	(partic given))
       ((lexical-roles given)
	(partic none)
	(process-type lexical))
       ((lexical-roles none)
	(partic none)
	(alt (((process-type natural-phenom))
	      ((partic any)))))))
  (oblique ((fset (1 2 3 4 5 6 7 8))))
  (synt-roles ((fset (subject object iobject 
			      subj-comp obj-comp 
			      dative by-obj))))
  (innermost-role {^ oblique 1})
  (:! mood)
  (:! transitivity)
  (:! voice)
  ;; END OF TREATMENT OF INHERENT PARTICIPANTS
  
  ;; Process optional participants
  (:& circumstantial)

  (alt adverb (:index adverb)
    (((adverb none))
     ((adverb given)
      (adverb ((cat adv))))))

  ;; DISPLACED CONSTITUENT PROCESSING
  ;; for interrogative and relative mood, the constituent under scope
  ;; is displaced.  We handle it in this alternation.
  (alt scoped-determination (:index scope)
    (((scope none))
     ((scope given) (scope ((clause-level ((scoped yes))))))))
  (alt embedded-determination (:wait synt-funct)
    (((synt-funct main-clause) (embedded no))
     ((control (not (equal #@{^ synt-funct} 'main-clause)))
      (embedded yes))))
  (:! agentless)
  (:! displaced-constituent)
  
  ;; If dative-move is not bound yet, it should be yes
  ;; Do not use dative-move if object is a pronoun
  ;; * Mary gave the library it.
  ;; Mary gave it to the library.
  (alt dative-move-default (:bk-class dative-move)
    (((dative-move yes)
      (oblique ((3 given)
		(3 ((cat ((alt (common proper)))))))))
     ((process-type lexical)
      (dative-move yes))
     ((dative-move no))))
  
  ;; check cat of each syntactic role subcategorization system
  (:& subcat)

  ;; START OF GENERAL THINGS: ORDERING PLUS AGREEMENT
  ;; General things: arrange syntactic roles together and do the agreements.
  ;; The patterns are here of course.
  
  ;; Focus first (change when add modifiers)
  ;; This is an example of mergeable pattern - where an ordering
  ;; constraint can have an effect on voice choice.
  ;; Not delicate enough though... so instead use conflation below.
  ;; (pattern ((* focus) dots))  
  (alt focus 
      (((focus none))
       ((focus given)
	(alt focus-given 
	    (((focus ((synt-funct given)))
	      (focus {^ synt-roles subject}))
	     ((focus ((realization head)))
	      (focus {^ synt-roles subject}))
	     ((focus ((synt-funct none)
		      (realization {^2 synt-roles subject})))))))))
  
  ;; Number and person agreement (verb is process)
  (verb ((cat verb-group)
	 (modality {^ ^ modality})
	 (adverb {^ ^ adverb})
	 (epistemic-modality {^ ^ epistemic-modality})
	 (deontic-modality {^ ^ deontic-modality})
	 (tense  {^ ^ tense})
	 (polarity {^ ^ polarity})
	 (insistence {^ ^ insistence})
	 (person {^ ^ synt-roles subject person})
	 (number {^ ^ synt-roles subject number})))
  
  ;; particle is for verbs like "take off"
  ;; This is for MOVABLE particles only.
  ;; non-movable particles should be treated as in the example in ir8.l
  (cset ((- particle)))
  (alt particle (:demo "Does the verb have a particle?")
    (:wait {^ verb particle})
    (((verb ((particle none))))
     ((verb ((particle given)))
      (particle ((cat adv)
		 (lex {^ ^ verb particle}))))))
  
  ;; Arrange order of complements
  ;; start is a dummy constituent used only for the ordering
  ;; constraints
  ;; A single pattern will do
  (pattern
   (dots start {^ synt-roles subject} dots process
	 dots {^ synt-roles iobject} dots
	 {^ synt-roles object} dots particle dots
	 {^ synt-roles subj-comp}
	 {^ synt-roles obj-comp}
	 dots
	 {^ synt-roles by-obj}
	 {^ synt-roles dative}
	 end-partic
	 dots))
  
  ;; Case assignment
  ;; Subject of present-participle mood can be possessive mood.
  %case%
  (synt-roles 
   ((alt (:wait {^ subject syntax case})
	(((subject ((syntax ((case subjective))))))
	 ((subject ((syntax ((case given))))))))
    (opt ((object  ((syntax ((case objective)))))))
    (opt ((iobject ((syntax ((case objective)))))))))
  %case%

  ;; Place optional roles (movable adjuncts are positioned in circumstance.l)
  %pat%
  (pattern (dots end-partic
		 {^ adjuncts accompaniment} 
		 {^ adjuncts behalf}  
		 {^ adjuncts reason}
		 {^ adjuncts purpose-pp}
		 dots                   ;; possibly time here if not fronted
		 {^ adjuncts at-loc}
		 {^ adjuncts from-loc}
		 {^ adjuncts to-loc}
		 {^ adjuncts on-loc}
		 {^ adjuncts in-loc} 
		 {^ adjuncts instrument} 
		 {^ adjuncts effect}
		 end-circum             ;; all position end after that
		 dots))
  %pat%
  
  )


(def-alt mood 
  ;; MOOD SYSTEM
  ;; For relative and interrogative, the mapping scope to role is
  ;; delayed after the transitivity system (same time as the voice
  ;; system does the mapping semantic/syntactic roles).
  ;; In this first system, all pre-selections that can be derived from
  ;; the mood are set.
  (:index mood) 
  (:demo "Deciding between mood finite and non-finite")

  (((mood finite)
    (cat verbal-clause)
    (alt finite (:index mood)
      (:demo 
       "Is the clause simple declarative, interrogative, relative ~
                or subordinate?")
      (((mood declarative)
	(generic-mood declarative)
	(pattern (dots start dots)))
       
       ((mood interrogative)
	(generic-mood interrogative)
	;; for both yes-no and wh questions, front the tensed part
	;; of the verb group and the not particle.
	;; copy everything from tensed-feature except the gap
	;; (therefore cannot just conflate them).
	;; Note: these are all the features known to the morphology.
	(scope ((gap yes)))
	;; For wh questions don't use dative-move
	(alt (((mood wh)
	       (dative-move no))
	      ((mood yes-no))))
	(process ((interrogative {^ ^ mood})))
	(cset ((- fronted-aux)))
	(alt AUX (:wait {^ verb tensed-feature})
	  (((fronted-aux 
	     ((person {^ ^ verb tensed-feature person})
	      (number {^ ^ verb tensed-feature number})
	      (ending {^ ^ verb tensed-feature ending})
	      (tense  {^ ^ verb tensed-feature tense })
	      (cat    {^ ^ verb tensed-feature cat})
	      (lex    {^ ^ verb tensed-feature lex})))))))

       ((mood bound)
	;; For conjunctions like:
	;; I know who he is and whether he is coming.
	(generic-mood declarative)
	(pattern (binder dots start dots))
	(binder ((cat conj)))
	(opt binder
	     ((binder ((lex ((alt ("that" "whether" "if"))))))))
	(binder ((lex given)))
	;; If clause is scoped by a long distance dependency, don't use
	;; binder: the person who you think [that] won the prize.
	(alt bound-scoped (:index scoped) (:wait scoped)
	  (((scoped no))
	   ((scoped #(under yes))
	    (binder ((gap yes))))))
	(:! mood-bound))
       
       ;; relative -- mapping scope/role is done in voice system.
       ;; Just indicate here that scope has a gap.
       ((mood relative)
	(generic-mood relative)
	(dative-move no)
	(scope ((gap yes)))))))

   ((mood non-finite)
    (alt non-finite (:index mood)
      (:demo "Is the clause imperative, present-participle ~
                          or infinitive?")
      (((mood imperative)
	(generic-mood imperative)
	(cat verbal-clause)
	(modality none)
	(epistemic-modality none)
	(deontic-modality none)
	(process ((ending root)
		  (voice active)))
	(innermost-role ((gap yes))))
       
       ((mood present-participle)
	(generic-mood present-participle)
	(cat nominalized-ing)
	(process ((ending present-participle)))
	(alt (((binder given) 
	       (binder ((cat conj)))
	       (pattern (binder start dots)))
	      ((binder none))))
	(modality none)
	(epistemic-modality none)
	(deontic-modality none))

       ((mood past-participle)
	(generic-mood past-participle)
	(cat verbal-clause)
	(modality none)
	(epistemic-modality none)
	(deontic-modality none)
	(process ((ending past-participle)
		  (tensed-feature ((gap yes)))
		  (voice passive))))

       ;; binder in infinitive: how to do it, where to go
       ;; Temporary hack - should analyze it better.
       ((mood infinitive)
	(generic-mood infinitive)
	(cat verbal-clause)
	(modality none)
	(epistemic-modality none)
	(deontic-modality none)
	(alt (((binder given) 
	       (binder ((cat conj)))
	       (pattern (binder start dots)))
	      ((binder none))))
	(process ((ending infinitive)))))))))


;; Two types of bound clauses: 
;; bound-declarative: I know that he is coming.
;; bound-subjunctive: I demand that he come.
(def-alt mood-bound (:index mood)
  (((mood bound-declarative))
   ((mood bound-subjunctive)
    (cat verbal-clause)
    (modality none)
    (epistemic-modality none)
    (deontic-modality none)
    (process ((ending root))))))


;; ============================================================
;; DISPLACED CONSTITUENT
;; ============================================================

(def-alt displaced-constituent (:index mood)
  ;; This wait is necessary for long distance dependencies: you need to
  ;; wait until the scoped constituent is mapped in the embedded clauses
  ;; before processing it.
  (:wait {^ scope synt-funct})
  (((mood yes-no)
    ;; No displaced component except inversion of verb/subject
    ;; Adverb remains in regular position (unlike notf which is fronted).
    (pattern (dots fronted-aux fronted-not start dots)))
   ((mood wh)
    ;; Find pointer to scope realization - this is scoper.
    (alt (((scope ((synt-funct given)))
	   (scoper {^ scope}))
	  ((scope ((realization head)))
	   (scoper {^ scope}))
	  ((scoper {^ scope realization}))))
    (cset ((- scope scoper)))
    (scoper ((gap yes)
	     (clause-level ((scoped yes)))
	     (question-pronoun {^2 scope question-pronoun})
	     (question-embedded {^2 scope question-embedded})))
    (:! question-embedded)
    ;; Add the question element in front.
    ;; Adverb is fronted in cases like "who never owned this book"
    (pattern (dots question fronted-adverb fronted-aux fronted-not start dots)))
   
   ;; MOOD RELATIVE
   ((mood relative)
    ;; Find pointer to scope realization - this is scoper.
    (alt (((scope ((synt-funct given)))
	   (scoper {^ scope}))
	  ((scope ((realization head)))
	   (scoper {^ scope}))
	  ((scoper {^ scope realization}))))
    (scoper ((gap yes)
	     (clause-level ((scoped yes)))
	     (relative-marker {^2 scope relative-marker})
	     (relative-embedded {^2 scope relative-embedded})))
    (pattern (relative-marker start dots))
    (:! relative))
   
   ;; OTHER MOODS -- Nothing to do here
   ((mood declarative))
   ((mood bound))
   ((mood non-finite))))



(def-alt question-embedded (:index (scope question-embedded))
  ;; scope is a path to a semantic constituent on which the question is asked.
  ;; We expect to find in the semantic constituent the realization link to
  ;; the syntactic complement realizing it plus the following information:
  ;; question-embedded yes/no which determines whether the question is a
  ;; pronoun or a prep pronoun.
  ;; In the syntactic const. we find the prep to use for embedded questions
  ;; and the question-pronoun to use.  
  ;; Default for embedded is no.
  ;; ***** Should check for other cats (adverb, ap).
  ;; ***** Should check features of verb to decide whether
  ;; ***** to use a pp or an np as question element
  ;; ***** Should do questions like "to which house did you
  ;; ***** go?" (long distance - scope is sub-constituent) 
  ;; The features needed to determine a relative or question pronoun are:
  ;; case, restrictive, animate, possessive.
  ;; We pass them explicitly so that we do not depend on the cat affected
  ;; to the scoped constituent.
  (((scope ((question-embedded no)))
    (question {^ scoper question-pronoun})
    (question ((cat question-pronoun)
	       (restrictive {^2 scope restrictive})
	       (animate {^2 scope animate})
	       (possessive {^2 scope possessive})
	       (syntax {^2 scoper syntax})
	       (semantics ((index {^3 scoper semantics index})))))
    (alt (((question ((lex given)))
	   (cset ((- question))))
	  ((cset ((+ question)))))))
   ((scope ((question-embedded yes)))
    (cset ((+ question)))
    (question ((cat pp)
	       (alt (((prep {^2 scope question-prep})
		      (prep given))
		     ((prep {^2 scope prep})
		      (prep given))
		     ((prep {^2 scoper prep})
		      (prep given))
		     ((prep ((lex "of"))))))
	       (np {^2 scoper question-pronoun})
	       (np ((cat question-pronoun)
		    (restrictive {^3 restrictive})
		    (restrictive {^3 scope restrictive})
		    (animate {^3 scope animate})
		    (possessive {^3 scope possessive})
		    (syntax ((case objective)))
		    (semantics {^3 scoper np semantics})))
	       ;; If lex for pronoun is already given don't bother with it.
	       (alt (((np ((lex given)))
		      (cset ((- np prep))))
		     ((cset ((+ np) (- prep)))))))))))



(def-alt relative (:index (scoper relative-embedded))
  (:demo "Is the relative clause simple or embedded in a PP?")
  ;; (:wait (({^ scope cat} #(< lexical-cat))))
  (((mood simple-relative)
    ;; Example: the woman who lives there
    ;;          the man whom I know
    ;;          the reason why I came
    ;; Simple relative is the qualifier of an NP. The NP
    ;; is a constituent of the relative clause, as indicated
    ;; by the scope constituent:
    ;; if NP is medium, do (scope {^ partic medium}) in the relative
    ;; clause. Scope inherits the relevant features from the 
    ;; head of the enclosing NP.
    (pattern (relative-marker start dots))
    (cset ((- scope scoper)))
    (scoper ((gap yes)
	     (relative-embedded no)
	     (cat simple-np)
	     (lex {^3 head lex})
	     (semantics ((index {^4 semantics index})))))
    (relative-marker {^ scoper relative-marker})
    (relative-marker ((cat relative-pronoun)
		      (restrictive {^2 scope restrictive})
		      (animate {^2 scope animate})
		      (possessive {^2 scope possessive})
		      (restrictive {^2 restrictive})
		      (syntax {^2 scoper syntax})
		      (semantics {^2 scoper semantics})))
    (alt (((relative-marker ((lex given)))
	   (cset ((- relative-marker))))
	  ((relative-marker ((restrictive #(under yes))
			     (lex "that")))
	   (cset ((- relative-marker))))
	  ((cset ((+ relative-marker)))))))

   ;; possessive relative - scope is determiner of an NP
   ;; Ex: The man whose car I wash.
   ;; (mood possessive-relative)
   ;; (scope {^ partic affected})
   ((mood possessive-relative)
    ;; ***** Copying any type of NP in any form (annoying).
    ;; Done here only for common and partitive.
    (alt (((scoper ((cat #(under partitive))))
	   (cset ((- scope scoper)))
	   (relative-marker ((cat partitive)
			     (part {^2 scoper part})
			     (part-of {^2 scoper part-of})
			     (part-of ((possessor ((cat relative-pronoun)))))
			     (all {^2 scoper all})
			     (prep {^2 scoper prep}))))
	  ;; Note: need to process scoper here otw all the features at the
	  ;; top level of the NP won't be propagated as needed into head
	  ;; and others (syntax...).  This wouldn't work with 
	  ;; "The soil, whose layers are topsoil and subsoil."
	  ;; as number wouldn't be processed correctly - credit to Charles
	  ;; Brendan for finding this bug.
	  ((scoper ((cat np)))
	   (cset ((+ scoper relative-marker)))
	   (relative-marker ((cat np)
			     (lex {^2 scoper lex})
			     (head {^2 scoper head})
			     (possessor ((cat relative-pronoun)))
			     (cardinal {^2 scoper cardinal})
			     (ordinal {^2 scoper ordinal})
			     (reference {^2 scoper reference})
			     (semantics {^2 scoper semantics})
			     (syntax {^2 scoper syntax})
			     (describer {^2 scoper describer})
			     (classifier {^2 scoper classifier})
			     (qualifier {^2 scoper qualifier})))))))

   ;; embedded relative - scope is within a PP
   ((mood embedded-relative)
    (scoper ((relative-embedded yes)))
    ;; Example: the box in which we store the elixir
    ;;          an experience the likes of which you have never seen
    ;; Bind scope to embedding np head
    (scoper ((np ((semantics ((index {^5 semantics index})))))))
    (cset ((- scope scoper)))
    (relative-marker ((cat pp)
		      (pattern (prep np))
		      (prep {^2 scoper prep})
		      (np {^2 scoper relative-marker})
		      (np ((cat relative-pronoun)
			   (syntax ((case objective)))
			   (restrictive {^3 scope restrictive})
			   (animate {^3 scope animate})
			   (possessive {^3 scope possessive})
			   (semantics {^3 scoper np semantics})))))
    (alt (((relative-marker ((np ((lex given)))))
	   (cset ((- relative-marker))))
	  ((cset ((+ relative-marker)))))))

   
   ;; ***** be-deleted not implemented
   ;; Actually debatable whether useful: in the meantime, can
   ;; generate them as AP in qualifier (cf note below on
   ;; criteria to distinguish adjectives and participles).
   ;; Example: Goldwater /who was/ crushed by his defeat
   ;;          the team /that is/ expected to win
   ;;          an enchanting island /that is/ waiting to be seen
   #+ignore((mood be-deleted-relative))
   
   ;; ***** wh-nominal not implemented
   ;; Example: They were amazed at which one they chose
   ;;          I couldn't believe how many people understood
   ;;          What they did next was surprising
   ;; To rework when nominalization is implemented [cf Vendler]
   #+ignore((mood wh-nominal-relative))
   
   ;; ***** wh-ever-nominal not implemented
   ;; Example: Whoever did it will be sorry
   ;; Cf nominalization.
   #+ignore((mood wh-ever-nominal-relative))))



;; ============================================================
;; SUBCATEGORIZATION
;; ============================================================

(def-conj subcat
  ;; Get lexical information on government pattern
  (process ((subcat {^ ^ oblique})))
  ;; special treatment of subject clauses (this is the right time
  ;; because the subject is now bound). 
  (:! subject-mood)
  (:! subject-subcat)
  (:! object-subcat)
  (:! subj-comp-cat)
  (:! obj-comp-cat)
  (:! by-obj-cat)
  (:! dative-cat)
)  


(def-alt subject-mood	(:index mood) 
  (:demo "Is a subject required or does it need a special treatment?")
  (((mood finite)
    (synt-roles ((subject given))))
   ;; In a question if subject is not given, it may be because it is
   ;; the scope of the question.
   ((mood wh)
    (scope {^ synt-roles subject}))
   ((mood non-finite)
    (alt subject-mood-nf (:index mood)
      (:wait (({^ synt-roles subject cat} #(< lexical-cat))))
      (((mood infinitive)
	(syntax 
	 ((opt ((case given) 
		(case purposive) 
		(keep-in-order yes)
		({^ pattern} (in-order dots))))))
	;; When the clause is subject or purpose, and there is a subject,
	;; use a FOR-clause as in "for her to do it is a bold
	;; statement" or "in order for him to eat, we must cook"
	(alt keep-for
	    (:demo "Should we use a for in front of the subject?")
	  (((keep-for yes)
	    (syntax ((case ((alt (subjective purposive))))))
	    (synt-roles
	     ((subject given)
	      (subject ((gap none)
			(case objective)))))
	    (pattern (dots for start dots))
	    (for ((cat conj) (lex "for"))))
	   ((keep-for no)
	    (syntax ((case ((alt (subjective purposive))))))
	    (synt-roles ((subject ((gap yes))))))
	   ((syntax ((case objective)))))))
       ((mood present-participle)
	;; subject is optional or in possessive form
	(synt-roles
	 ((alt subject-subject (:wait {^ subject gap})
	    (((subject none))
	     ((subject ((gap yes))))
	     ((subject given)
	      (subject ((cat np)
			(gap none)
			(syntax ((case possessive)))))))))))
       ((mood past-participle)
	(synt-roles ((subject ((gap yes))))))
       ((mood imperative)
	;; subject is optional in realization
	(synt-roles ((alt (((subject none))
			   ((subject ((gap yes))))))))))))))



(def-alt subject-subcat 
  ;; Syntactic category of subject is compatible with verb?
  ;; This depends on particular verb, information we will get from the
  ;; lexicon. So far, we check whether subject-clause and object-clause
  ;; are ok. 
  ;; SUBJECT CAT = NP, CLAUSE.
  (((synt-roles ((subject none))))
   ((alt subject-subcat2 (:index (synt-roles subject cat))
      (:wait (({^ synt-roles subject cat} #(< lexical-cat))))
      (((synt-roles ((subject none))))
       ((synt-roles ((subject ((cat np))))))
       ((synt-roles ((subject ((cat #(under clause))))))
	(alt subject-clause	(:index (process subject-clause))
	  (:demo "For clausal subjects, what type of clause ~
                     must be used?")
	  (((process ((subject-clause infinitive)))
	    (synt-roles ((subject ((mood {^ ^ ^ process subject-clause}))))))
	   ((process ((subject-clause #(under present-participle))))
	    (synt-roles ((subject ((mood {^ ^ ^ process subject-clause}))))))
	   ((process ((subject-clause that)))
	    (synt-roles ((subject ((mood bound)
				   (binder ((lex ((alt (given "that"))))))))))))))
       ((synt-roles ((subject ((cat #(under list)))))))))
    (opt ((synt-roles ((subject ((synt-funct subject))))))))))


(def-alt object-subcat
  ;; OBJECT CAT = NP, CLAUSE, PP? (deal with?)
  (((synt-roles ((object none))))
   ((alt object-subcat1 (:index  (synt-roles object cat))
      (:wait (({^ synt-roles object cat} #(< lexical-cat))))
      (((synt-roles ((object none))))
       ((synt-roles ((object ((cat np))))))
       ((synt-roles ((object ((cat pp))))))
       ((synt-roles ((object ((cat #(under clause))))))
	(alt object-clause (:index (process object-clause))
	  (:demo "For clausal objects, what type of clause ~
                    must be used?")	  
	  (((process ((object-clause infinitive)))
	    (synt-roles ((object ((mood infinitive))))))
	   ((process ((object-clause #(under present-participle))))
	    (synt-roles ((object ((mood present-participle))))))
	   ((process ((object-clause that)))
	    (synt-roles ((object ((mood bound)
				  (binder ((lex ((alt (given "that"))))))))))))))
       ((synt-roles ((object ((cat #(under list)))))))))
    (opt ((synt-roles ((object ((synt-funct object))))))))))



(def-alt subj-comp-cat
  ;; SUBJ-COMP CAT = NP, AP, PP, ADV
  (((synt-roles ((subj-comp none))))
   ((synt-roles ((subj-comp ((alt subj-comp-cat1 (:index cat)
			       (:wait (({^ cat} #(< lexical-cat))))
			       (none
				((cat ap))
				((cat given))
				((cat #(under np)))
				((cat #(under pp)))
				((cat #(under list)))
				((cat #(under adv)))))))))
    (opt ((synt-roles ((subj-comp ((synt-funct subj-comp))))))))))


(def-alt obj-comp-cat
  ;; OBJ-COMP CAT = NP, AP, PP, ADV
  (((synt-roles ((obj-comp none))))
   ((synt-roles ((obj-comp ((alt obj-comp-cat1 (:index cat)
			      (:wait (({^ cat} #(< lexical-cat))))
			      (none
			       ((cat ap))
			       ((cat given))
			       ((cat #(under np)))
			       ((cat #(under pp)))
			       ((cat #(under list)))
			       ((cat #(under adv)))))))))
    (opt ((synt-roles ((obj-comp ((synt-funct obj-comp))))))))))


(def-alt by-obj-cat
  ;; BY-OBJ CAT = PP, set prep
  (((synt-roles ((by-obj none))))
   ((synt-roles ((alt by-obj-cat1 (:index (by-obj cat))
		   (((by-obj none))
		    ((by-obj given)
		     (by-obj ((cat pp)
			      (synt-funct by-obj)
			      (np ((cat np)
				   (syntax ((case objective)))))))))))))))

(def-alt dative-cat
  ;; DATIVE CAT = PP, set prep
  (((synt-roles ((dative none))))
   ((synt-roles 
     ((alt dative-cat1 
	(((dative none))
	 ((dative given)
	  (dative ((cat pp)
		   ({^ ^ process dative-prep} given)
		   (prep ((lex {^4 process dative-prep})))
		   (np ((cat np)
			(syntax ((case objective))))))))
	 ((dative ((cat pp)
		   (prep ((lex "to")))
		   (synt-funct dative)
		   (np ((cat np) 
			(syntax ((case objective)))))))))))))))


;; ============================================================
(provide "clause")
;; ============================================================
