/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <string.h>

#include "code_area.h"
#include "errors.h"
#include "instructions.h"
#include "pc.h"
#include "pred_table.h"
#include "string_table.h"
#include "cells.h"


global	offset	*predicate_table;
global	natural	predicate_table_size = DEFAULT_PREDICATE_TABLE_SIZE;


/*----------------------------------------------------------------------------
    allocate the memory for the predicate table, note the size should be a
    large prime number. Zeroise each element in the array.
----------------------------------------------------------------------------*/
global	void
initialise_predicate_table(void)
{
	int	i;

	if ((predicate_table = (offset *) malloc(
		predicate_table_size * sizeof(offset))) == NULL)
		fatal("not enough memory for predicate table %d",
		    predicate_table_size);
	for (i = 0; i < predicate_table_size; i++)
		predicate_table[i] = 0;
}

/*----------------------------------------------------------------------------
    find the code offset (into the code_area) for the predicate represented
    by string_offset/arity. The predicate name and arity are used to hash
    into the hash table.

    string_offset
		The offset of the predicate name into the string table.
    arity	arity of the predicate.
    code_offset	The offset into the code table that the current predicate 
		definition is.
    insert	The mode of insert, either TRUE or FALSE
		(with code_offset = NULL_OFFSET).

    get the hash index for the predicate (string_offset/arity)

    while location in the hash table is full do
	if the hash entry is for the same predicate (string_offset / arity)
	then
	    if in insert mode then
		clobber the old predicate defintion (ptr) with the new
		issue a warning that it has been redefined
	    return the ptr to the predicate definition's code
	if the hash index is greater than the size of the hash table then
	    if in insert mode then
	    	have run out of space in the hash table (fatal)
	    else
		return code_offset
	increment the hash index
    end-while

    add the entry to the mepty hash table position and return the ptr to the
    new definition of the predicate (code)
    
    
	+-------+	<- the entries in the hash table point to the start
	|string#|	   of the predicate entry
	+-------+
	| arity |
	+-------+
	|  size |
	+-------+	<- whereas we want to return a pointer to the start
	| code	|	   of the code.
	  ...
----------------------------------------------------------------------------*/
global	offset
lookup_predicate_table(offset string_offset, unsigned int arity, offset code_offset, int insert)
	      	              
	       	      
	      	            
	       	       		/* insert if not present */
{
reg	natural	i;
reg	natural	incr;
	offset	string_offset2;
	int	arity2;


	incr = 1;
	for (i = HashPredicate(string_offset, arity);
	     predicate_table[i];
	     i = (i+incr)%predicate_table_size, incr += 2)
	{
		pc = code_area + predicate_table[i];
		GetOffset(string_offset2);
		GetNumber(arity2);
		if (string_offset == string_offset2 && arity == arity2)
		{
			if (insert)
			{
				if (strcmp(string_table + string_offset,
					   "$query"))
					warning("%s/%d redefined",
						string_table+string_offset,
						arity);
				predicate_table[i] = code_offset;
				SkipConstant();
				return(code_offset + SizeOfOffset +
				    SizeOfNumber + SizeOfConstant);

			}
			SkipConstant();
			return(pc - code_area);
		}
		else if (incr%predicate_table_size == 0)
			if (insert)
				fatal("Out of space in predicate hash table %d",
					predicate_table_size);
			else
				return(code_offset);
	}

	if (insert)
		predicate_table[i] = code_offset;

	if (!predicate_table[i])
	     return(code_offset);
	pc = code_area + code_offset;
	SkipOffset();
	SkipNumber();
	SkipConstant();
	return(pc - code_area);
}
