/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "cells.h"
#include "errors.h"
#include "index.h"
#include "indexing.h"
#include "instructions.h"
#include "opcodes.h"
#include "pc.h"
#include "pred_table.h"
#include "resolve_object.h"

local	void	resolve_code(code *top, offset *string_map_tbl);

/*----------------------------------------------------------------------------
    resolve references to predicates and strings in the query code and code
    area
----------------------------------------------------------------------------*/
global	void
resolve_object(index *object_index)
{
	code	*top;
	natural	size;

	pc = object_index->obj_query_code_start;
	resolve_code(object_index->obj_query_code_top,
		object_index->string_map);

	pc = object_index->obj_code_area_start;
	while (pc < object_index->obj_code_area_top)
	{
		SkipOffset();
		SkipNumber();
		GetConstant(size);
		top = pc + size;
		resolve_code(top, object_index->string_map);
	}
}

/*----------------------------------------------------------------------------
	The following instructions have to be modified

	get_constant		offset	->	get_constant	newoffset
	put_constant		offset	->	put_constant	newoffset
	call_predicate		offset arity n	call_address	code_offset n
	execute_predicate	offset arity ->	execute_address	code_offset

	for all occurences of string offsets adjust the offset to the
	new offset value (for the combined string table)

	for the call and execute predicate, try to resolve the predicate 
	address (offset into the code table)

	if can find the predicate in the predicate table then 
		replace *_predicate with *_address and the code offset
	else
		update the new string offset for the functor name
----------------------------------------------------------------------------*/
local	void
resolve_code(code *top, offset *string_map_tbl)
{
	offset	atom;
	offset	code_offset;
	offset	string_offset;
	natural	arity;
	natural	n;
	natural	i;
	natural	j;
	boolean	insert;
	code	*pc_save;
	code	*table;
	offset	default_offset;
	index_entry	*index_table;

	insert = FALSE;

	while (pc < top)
	{
		switch (*pc++) 
		{
		when PUT_CONSTANT or
		     GET_CONSTANT:
			GetConstant(atom);
			if (IsAtom(atom))
			{
				UnGetConstant();
				PutConstant(Atom(string_map_tbl[
						RestOfConstant(atom)]));
			}
			SkipRegister();
		when PUT_NIL or
		     PUT_CONS or
		     PUT_X_OBJECT_VARIABLE or
		     PUT_Y_OBJECT_VARIABLE or
		     PUT_EMPTY_SUBSTITUTION or
		     SET_OBJECT_PROPERTY or
		     DETERMINE_PROPERTY or
		     GET_NIL or
		     GET_CONS:
			SkipRegister();
		when PUT_X_VARIABLE or
		     PUT_Y_VARIABLE or
		     PUT_X_VALUE or
		     PUT_Y_VALUE or
		     PUT_X_OBJECT_VALUE or
		     PUT_Y_OBJECT_VALUE or
		     PUT_UNSAFE_VALUE or
		     PUT_SUBSTITUTION_OPERATOR or
		     PUT_SUBSTITUTION or
		     PUT_PARALLEL_SUBSTITUTION_PAIR or
		     GET_X_VARIABLE or
		     GET_Y_VARIABLE or
		     GET_X_VALUE or
		     GET_Y_VALUE or
		     GET_X_OBJECT_VARIABLE or
		     GET_Y_OBJECT_VARIABLE or
		     GET_X_OBJECT_VALUE or
		     GET_Y_OBJECT_VALUE or
		     NOT_FREE_IN:
			SkipRegister();
			SkipRegister();
		when PUT_APPLY or
		     PUT_PAIR or
		     PUT_QUANTIFIER or
		     GET_APPLY or
		     GET_PAIR or
		     GET_QUANTIFIER:
			SkipRegister();
			SkipRegister();
			SkipRegister();
		when PUT_PARALLEL_SUBSTITUTION:
			SkipNumber();
			SkipRegister();
		when CALL_PREDICATE:
			GetPredAtom(atom);
			string_offset = string_map_tbl[atom];
			GetNumber(arity);
			UnGetNumber();
			UnGetPredAtom();
			pc_save = pc;
			code_offset = lookup_predicate_table(string_offset,
						arity, NULL_OFFSET, insert);
			pc = pc_save;
			if (code_offset != NULL_OFFSET)
			{
				UnGetInstruction();
				PutInstruction(CALL_ADDRESS);
				PutAddress(code_offset);
			}
			else
			{
				PutPredAtom(string_offset);
				SkipNumber();
			}
			SkipNumber();
		when CALL_ADDRESS:
			fatal("CALL_ADDRESS occurs in object file");
			SkipAddress();
			SkipNumber();
		when EXECUTE_PREDICATE:
			GetPredAtom(atom);
			string_offset = string_map_tbl[atom];
			GetNumber(arity);
			UnGetNumber();
			UnGetPredAtom();
			pc_save = pc;
			code_offset = lookup_predicate_table(string_offset,
						arity, NULL_OFFSET, insert);
			pc = pc_save;
			if (code_offset != NULL_OFFSET)
			{
				UnGetInstruction();
				PutInstruction(EXECUTE_ADDRESS);
				PutAddress(code_offset);
			}
			else
			{
				PutPredAtom(string_offset);
				SkipNumber();
			}
		when EXECUTE_ADDRESS:
			fatal("EXECUTE_ADDRESS occurs in object file");
			SkipAddress();
		when PROCEED:
		when GOTO:
		when ESCAPE:
			SkipNumber();
		when FAIL:
		when CUT:
		when DO_DELAYED_PROBLEMS:
		when ALLOCATE:
			SkipNumber();
		when DEALLOCATE:
		when TRY_ME_ELSE:
			SkipNumber();
			SkipOffset();
		when RETRY_ME_ELSE:
			SkipOffset();
		when TRUST_ME_ELSE_FAIL:
		when TRY:
			SkipNumber();
			SkipOffset();
		when RETRY:
			SkipOffset();
		when TRUST:
			SkipOffset();
		when SWITCH_ON_TERM:
			SkipOffset();
			SkipOffset();
			SkipOffset();
			SkipOffset();
			SkipOffset();
			SkipOffset();
		when SWITCH_ON_CONSTANT:
			GetNumber(n);
			table = pc;
			index_table = (index_entry *)malloc((unsigned)
					(TableSize(n) * sizeof(index_entry)));
			SkipConstant();
			GetOffset(default_offset);
			for (i = 0, GetConstant(index_table[i].atom),
			     GetOffset(index_table[i].address);
			     ! (index_table[i].atom == NULL &&
			        index_table[i].address == NULL) &&
					i < (TableSize(n) - 1);
			     i++, GetConstant(index_table[i].atom),
			     GetOffset(index_table[i].address))
			{
				UnGetOffset();
				UnGetConstant();
				PutConstant(NULL);
				PutOffset(default_offset);
			}
			if (i < (TableSize(n) - 1))
			{
				UnGetOffset();
				UnGetConstant();
				PutConstant(NULL);
				PutOffset(default_offset);
			}
			for (j = 0; j < i; j++)
				insert_constant_table(
					(IsAtom(index_table[j].atom) ?
						Atom(string_map_tbl[
							RestOfConstant(
							index_table[j].atom)]) :
						index_table[j].atom),
					index_table[j].address, table, n);
			free((char *)index_table);
			pc = table + TableSize(n) * SizeOfConstantTableEntry;
		when SWITCH_ON_STRUCTURE or
		     SWITCH_ON_QUANTIFIER:
			GetNumber(n);
			table = pc;
			index_table = (index_entry *)malloc((unsigned)
					(TableSize(n) * sizeof(index_entry)));
			SkipConstant();
			SkipNumber();
			GetOffset(default_offset);
			for (i = 0, GetConstant(index_table[i].atom),
			     GetNumber(index_table[i].arity),
			     GetOffset(index_table[i].address);
			     ! (index_table[i].atom == NULL &&
			        index_table[i].arity == NULL &&
			        index_table[i].address == NULL) &&
					i < (TableSize(n) - 1);
			     i++, GetConstant(index_table[i].atom),
			     GetNumber(index_table[i].arity),
			     GetOffset(index_table[i].address))
			{
				UnGetOffset();
				UnGetNumber();
				UnGetConstant();
				PutConstant(NULL);
				PutNumber(0);
				PutOffset(default_offset);
			}
			if (i < (TableSize(n) - 1))
			{
				UnGetOffset();
				UnGetNumber();
				UnGetConstant();
				PutConstant(NULL);
				PutNumber(0);
				PutOffset(default_offset);
			}
			for (j = 0; j < i; j++)
				insert_structure_table(
					Atom(string_map_tbl[RestOfConstant(
						index_table[j].atom)]),
					index_table[j].arity,
					index_table[j].address, table, n);
			free((char *)index_table);
			pc = table + TableSize(n) * SizeOfStructureTableEntry;
		when COMMIT:
		when BACK_TO:
		when COMMIT_STAGE:
		otherwise:
			fatal("unknown op-code %d\n", pc[-1]);
		}
	}
}
