/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <string.h>

#include "errors.h"
#include "instructions.h"
#include "name_table.h"
#include "string_table.h"

global	char	*string_table;
global	natural	string_table_size = DEFAULT_STRING_TABLE_SIZE;
global	char	*top_of_string_table;



/*----------------------------------------------------------------------------
    allocate memory for the string table, do not use the first character,
    so that string_offsets (into this table) for the name_table do not have
    zero values.
----------------------------------------------------------------------------*/

extern size_t fread (void *, size_t, size_t, FILE *);
extern int fgetc (FILE *);
extern offset lookup_name_table_offset (char *s, int name_table_tag);

global	void
initialise_string_table(void)
{

	if ((string_table = malloc(Kwords_to_chars(string_table_size)))
		== NULL)
		fatal("not enough memory for string_table %d K",
		    string_table_size);
	*string_table = '\0';
	top_of_string_table = string_table+1;

}

/*----------------------------------------------------------------------------
	put nil, cons and dollar in the string table.
----------------------------------------------------------------------------*/
global	void
insert_defaults_in_string_table(void)
{
	(void) add_name_string_offset("[]", ATOM_W);
	(void) add_name_string_offset(".", ATOM_W);
	(void) add_name_string_offset("$", ATOM_W);
}

/*----------------------------------------------------------------------------
add_name_string_hash(s, name_table_tag)   

    Find the hash value to name table for a string (string is to be
    a variable name) 
    If the variable name is not in name table
        add the string to the string table
        add the variable name to the name table
    Return hash value to name table
----------------------------------------------------------------------------*/
global natural
add_name_string_hash(char *s, int name_table_tag)
{
reg     natural i;
	
        i = lookup_name_table(s, name_table_tag);

	if ( NameT(name_table[i]) == NULL)
	     NameT(name_table[i]) = name_table_tag|add_new_string(s);
	  
        return(i);
}
/*----------------------------------------------------------------------------
add_name_string_offset(s, name_table_tag)   

    Find the hash value to name table for a string (string is to be
    a variable name) 
    If the variable name is not in name table
        add the string to the string table
        add the variable name to the name table
	return the offset of the new added string to the string table
    else
	return the offset of the string to the string table from name table
----------------------------------------------------------------------------*/
global offset
add_name_string_offset(char *s, int name_table_tag)
{
reg     natural i;
        offset  string_offset; 


        i = lookup_name_table(s, name_table_tag);

	if ( NameT(name_table[i]) == NULL)
	{ 
	     string_offset = add_new_string(s);
	     NameT(name_table[i]) = name_table_tag|string_offset;

        }
	else
	    string_offset = NameOffset(name_table[i]);
        return(string_offset);
}
/*------------------------------------------------------------------------------
add_new_string(s)  

    Check that string has not existed in the string table, must have been
    done before this procedure is called.

    Add the string to the string table. 
    Return the offset of the string from the start of the string table.
------------------------------------------------------------------------------*/
global	offset
add_new_string(char *s)
{
        offset string_offset;


	if ((top_of_string_table + strlen(top_of_string_table)+1) >
	    (string_table + Kwords_to_chars(string_table_size)))
		fatal("Out of space in string table %d K", string_table_size);

	string_offset = top_of_string_table - string_table;

	top_of_string_table = strcpy(top_of_string_table, s); 

        top_of_string_table += strlen(top_of_string_table)+1;

	return(string_offset);
}


/*----------------------------------------------------------------------------
    read the object file's string table  size
    skip the first char in the string table
    for every string in the object file's string table do until read all
	read in a string into the new string table
	if the string has not already been added to the table then
	    increment the top_of_string_table ptr past the string read
    return the number of strings read
----------------------------------------------------------------------------*/
global	int
merge_strings(offset *string_map, FILE *object_file, char *filename)
{
	natural	obj_string_table_size;
	int	count;
	int	i;
	char	*string_save;

	if (fread((char *)&obj_string_table_size, SizeOfConstant, 1,
		object_file) != 1 ||
	    fread(top_of_string_table, 1, 1, object_file) != 1)
		fatal("invalid file format for object file %s", filename);
	count = 1;
	for (i = 0; count < obj_string_table_size; i++)
	{
		string_save = top_of_string_table;
		while ((*string_save++ = (char) fgetc(object_file)) != '\0');
		if (string_save > (string_table +
				   Kwords_to_chars(string_table_size)))
			fatal("Out of space in string table %d K",
			      string_table_size);
		count += strlen(top_of_string_table) + 1;
		if (top_of_string_table == string_table +
		       (string_map[i] = lookup_name_table_offset(
					top_of_string_table, ATOM_W)))
			top_of_string_table += strlen(top_of_string_table)+1;
	}
	return(i);
}
