/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * compile_procs - Compile the procedures that were read by read_procs.
 */
:- dynamic nested/2.

/*----------------------------------------------------------------------------
compile_procedure(Predicate, Procedure) :-
    Compile the given Procedure (a list of clauses) with the name Predicate.

    Write out the indexing instructions for the procedure.
    Compile the individual clauses, writing out the code for each.
    Write out the "end of predicate" label.
    Compile any nested procedures for the predicate.
----------------------------------------------------------------------------*/
compile_procedure(Predicate, Procedure) :-
    do_and_clean_up(do_indexing(Predicate, Procedure)),
    compile_clauses(Procedure, Predicate),
    write_instruction(label(end(Predicate))),
    compile_nested(Predicate).

/*----------------------------------------------------------------------------
compile_nested(Predicate) :-
    Compile the nested clauses that were flattened out from the given
    predicate name.  The nested/2 fact is indexed on predicate name to
    allow multiple levels of nesting to be handled.  These facts are
    removed by a retractall in the read_procs module.
----------------------------------------------------------------------------*/
compile_nested(Predicate) :-
    nested(Predicate,NestedProcedure),
    compile_nested_clauses(NestedProcedure),
    fail.
compile_nested(_Predicate).

/*----------------------------------------------------------------------------
do_and_clean_up(X) :-
    Performs a call on X, and then performs a "fail", to clean up any used
    head space.  It then succeeds.
----------------------------------------------------------------------------*/
do_and_clean_up(X) :-
    call(X),
    fail.
do_and_clean_up(_X).

/*----------------------------------------------------------------------------
do_indexing(Predicate, Procedure) :-
    Derives the indexing instructions for the procedure and writes them
    to the ".qs" assembly code file.
----------------------------------------------------------------------------*/
do_indexing(Predicate, Procedure) :-
    write_instruction(label(Predicate)),
    index_procedure(Predicate, Procedure, IndexingInstructions),
    write_instructions(IndexingInstructions).

/*----------------------------------------------------------------------------
compile_nested_clauses(Ss) :-
    Compile the nested clauses Ss (produced by flattening ;'s and once's out
    of code).

    Take a fresh copy so instantiations in each clause do not affect one 
	another.
    Order the clauses into procedures for each predicate.
    Compile each procedure.
----------------------------------------------------------------------------*/
compile_nested_clauses(Ss) :-
    fresh_copy_clauses(Ss, Ss2),
    get_procedures(Ss2, Ps),
    compile_nested_procedures(Ps).

/*----------------------------------------------------------------------------
compile_nested_procedures(Procedures) :-
    compile the open list Procedures of nested procedures.

    To save space compile each procedure and then fail until there are no 
    more procedures to compile.
----------------------------------------------------------------------------*/
compile_nested_procedures(Procedures) :-
    member_ol(p(Predicate, Procedure), Procedures),
    tail_ol(Procedure, []),
    compile_procedure(Predicate, Procedure),
    fail.
compile_nested_procedures(_Procedures).

/*----------------------------------------------------------------------------
get_procedures(Ss, Ps) :-
    Sort the list of clauses Ss into procedures Ps for each predicate.
----------------------------------------------------------------------------*/
get_procedures([], _Ps).
get_procedures([S|Ss], Ps) :-
    insert_in_procedures(S, Ps),
    get_procedures(Ss, Ps).

/*----------------------------------------------------------------------------
compile_clauses(Clauses, Predicate) :-
    Compile each of the clauses in Clauses, writing the output to the
    ".qs" file.  Don't add labels if there is only one clause.
----------------------------------------------------------------------------*/
compile_clauses([Clause], Predicate) :-
    !,
    do_compile_clause(Predicate, 0, Clause).
compile_clauses(Clauses, Predicate) :-
    compile_clauses2(Clauses, 1, Predicate).

compile_clauses2([], _Num, _Predicate).
compile_clauses2([Clause|Clauses], Num, Predicate) :-
    do_compile_clause(Predicate, Num, Clause),
    Num2 is Num + 1,
    compile_clauses2(Clauses, Num2, Predicate).

do_compile_clause(Predicate, Label, Clause) :-
    do_and_clean_up(compile_and_write_clause(Predicate, Label, Clause)).

/*----------------------------------------------------------------------------
write_out_comment(Clause) :-
    Writes out a comment to the ".qs" file containing the Prolog code for
    the flattened clause.  It will only be written out in "debug" mode,
    as set in the "qc1" module.
----------------------------------------------------------------------------*/
write_out_comment(Clause) :-
    coption(debug),
    allocate_vars([Clause]),
    nl,
    write_clause_comment(Clause).

/*----------------------------------------------------------------------------
compile_and_write_clause(Predicate, Label, Clause) :-
    Compiles a clause for a predicate, and writes the QuAM code out to the
    ".qs" file.  If a label is required (Label \== 0), then it will be
    prepended to the written code.
----------------------------------------------------------------------------*/
compile_and_write_clause(Predicate, Label, Clause) :-
    (Label \== 0 -> write_instruction(label(Label)) ; true),
    compile_clause(Predicate, Clause,CC),
    write_instructions(CC).

/*----------------------------------------------------------------------------
compile_clause(Predicate, Clause, CC) :-
    To compile a Clause to CC. Predicate is the name to assert with nested
    clauses.

    pre-translate the cluase to remove Qu-Prolog shorthand notations.
    flatten all of the ors and onces into separate procedures, that is,
	flatten all of the choice points into straight line code.
	Trues can be removed at this point also since they are no longer 
	place holders.  These "nested" procedures are asserted in the
        fact "nested/2" for later compilation.
    Collect all of the variables that range over more than one clause
	(permanent variables).    
    Unravel the clause into QuAM instructions.
    Allocate all of the permanent variables (Y registers).
    Type the instructions for variables (variable, value, unsafe_value).
    Find out which temporary registers are alive at any point in time during 
	execution of the instructions of the clause.
    Allocate temporary registers based on which need to be alive at any point
	in time (X registers).
    Optimize the code to remove redundant instructions.

  +--  compile_nested_clauses <--+
  |				 |
  +>pretrans --> flatten_or_once*--> permvars --> permalloc -----+
				 |				 |
				 |				 |
				 +-> unravel --+		 |
					       |		 |
					       +--> valvar	 |
	   * ->flatten_ands->		       |		 |
					       +--> alive -----+ |
					       |	       | |
					       +--> tempalloc<-+ |
					       |		 |
					       +--> peephole <---+
----------------------------------------------------------------------------*/
compile_clause(Predicate, S, CC) :-
    pretrans(S, S2),
    flatten_or_once(S2, [S3|Ss3]),
    assert(nested(Predicate, Ss3)),
    % writeln(user_error, call:do_and_clean_up(write_out_comment(S3))),
    do_and_clean_up(write_out_comment(S3)),
    % writeln(user_error, exit:do_and_clean_up(write_out_comment(S3))),
    flatten_ands(S3, S4),
    '$vars_in_term'(S4, [], Variables),
    % writeln(user_error, call:'$select_sub_class2'(Variables, dynamic_object_var, OVs)),
    '$select_sub_class2'(Variables, dynamic_object_var, OVs),
    % writeln(user_error, exit:'$select_sub_class2'(Variables, dynamic_object_var, OVs)),
    '$map_vars_to_interpretation'(OVs, OVsRegs),
    % permvars(S4, PermanentVars),
    % writeln(user_error, call:permvars(S4, OVsRegs, PermanentVars)),
    permvars(S4, OVsRegs, PermanentVars),
    % writeln(user_error, exit:permvars(S4, OVsRegs, PermanentVars)),
    % unravel(S4, Instructions),
    % writeln(user_error, call:unravel(S4, OVsRegs, Instructions)),
    unravel(S4, OVsRegs, Instructions),
    % writeln(user_error, exit:unravel(S4, OVsRegs, Instructions)),
    permalloc(PermanentVars, EnvSize),
    valvar(Instructions),
    % writeln(user_error, call:alive(Instructions, Registers)),
    alive(Instructions, Registers),
    % writeln(user_error, exit:alive(Instructions, Registers)),
    % write_allocations(Instructions, Registers), % debugging
    % writeln(user_error, call:tempalloc(Instructions, Registers)),
    tempalloc(Instructions, Registers),
    % writeln(user_error, exit:tempalloc(Instructions, Registers)),
    % write_allocations(Instructions, Registers), % debugging
    % writeln(user_error, call:peephole(Instructions, EnvSize, CC)),
    peephole(Instructions, EnvSize, CC),
    % writeln(user_error, exit:peephole(Instructions, EnvSize, CC)),
    !.
compile_clause(_P, S, []) :-
    current_output(Current),
    set_output(user_error),
    writeln('Error: Could not compile clause'),
    portray_clause(S),
    set_output(Current).
