/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * flatten_or_once - Continue the predicate pre-processing begun by pretrans.
 */

/*----------------------------------------------------------------------------
flatten_or_once(Sentence, FlattenedSentences) :-
    FlattenedSentences is the list of sentences contained in Sentence, after
    flattening for choice points (;'s ) in the body.
    Remove any true predicates out of the new straight-line code.
    The head of FlattenedSentences is the direct flattened version of
    Sentence, and the tail contains the clauses that were flatttened out.

    e.g.

	f(A) :- 
		(g(A), !, once((h(A, B); h(B, A)))
		;i(A), !, j(B)
		;once(call(C))
		),
	        l(C, D).

    The input to this routine, and is translated to (the output)

	f(A) :- f$0(A, C), l(C, D).


	f$0(A, C) :- g(A), !, f$1(A).

	f$0(A, C) :- i(A), !, j(B).

	f$0(A, C) :- f$3(C).


	f$1(A) :- f$2(A), !.


	f$2(A) :- h(A, B).

	f$2(A) :- h(B, A).


	f$3(C) :- call(C), !.

----------------------------------------------------------------------------*/
flatten_or_once((Head :- Body), [NewClause|FlattenedSentences]) :-
    !,
    Head =.. [F|_Args],
    variables_in_formula(Head, Vars),
    name(F, Fs),
    flatten_or_once_body(Body, NewBody, Vars, Fs, FlattenedSentences),
    (remove_trues(NewBody, NewBody2) ->
	NewClause = (Head :- NewBody2)
    ;
	NewClause = Head
    ).
flatten_or_once(Sentence, [Sentence]).

/*----------------------------------------------------------------------------
flatten_or_once_body(Body, NewBody, Vars, Fs, NewClauses) :-
    
    Flatten out all ';'s in the body replacing them with calls to predicates
    (in NewClauses), that implement the ';'s.

    Body	The input body (rhs of the sentence).
    NewBody	The new body of the clause with all ';'s replaced with calls
		to routines that implement the ';'s.
    Vars	The variables that occur outside the body.
    Fs		The string representing the name of the predicate whose
		clause we are examining.
    NewClauses	The clauses created for the implementation of ';'s in the
		body.
----------------------------------------------------------------------------*/
flatten_or_once_body((F , AndF), (NewF , NewAndF), Vars, Fs, NewClauses) :-
    !,
    variables_in_formula(F, VarsF),
    variables_in_formula(AndF, VarsAndF),
    append(VarsF, Vars, VarsF2),
    append(VarsAndF, Vars, VarsAndF2),
    flatten_or_once_body(F, NewF, VarsAndF2, Fs, NewClausesF),
    flatten_or_once_body(AndF, NewAndF, VarsF2, Fs, NewClausesAndF),
    append(NewClausesF, NewClausesAndF, NewClauses).
flatten_or_once_body((F ; OrF), Formula, Vars, Fs, OrsClauses) :-
    !,
    new_predicate_name(Fs, G),
    ors_list((F ; OrF), Ors),
    variables_occuring_outside_list(Ors, Vars, Variables),
    tail_ol(Variables, []),
    Formula =.. [G|Variables],
    ors_clauses(Ors, Formula, Vars, Fs, OrsClauses).
flatten_or_once_body(once(Once), Formula, Vars, Fs, 
	[(Formula :- NewOnce, !)|NewClauses]) :-
    !,
    new_predicate_name(Fs, G),
    variables_occuring_outside(Once, Vars, Variables),
    tail_ol(Variables, []),
    Formula =.. [G|Variables],
    flatten_or_once_body(Once, NewOnce, Vars, Fs, NewClauses).
flatten_or_once_body(Body, Body, _Vars, _Fs, []).


/*----------------------------------------------------------------------------
ors_list(F, Ors) :-
    F is the ORing of the list of predicates Ors.

    e.g. F = (A ; (B ; C , D)), Ors = [A, B, (C, D)]
----------------------------------------------------------------------------*/
ors_list(F, [F]) :-
    var(F), !.
ors_list((F ; OrF), [F, Neck|Ors]) :-
    !,
    ors_list(OrF, [Neck|Ors]).
ors_list(F, [F]).

/*----------------------------------------------------------------------------
ors_clauses(Ors, Formula, Vars, F, OrsClauses) :-

    Flatten the list of options in the ';' (Ors) into clauses of the same
    predicate, flattening any or's in their body out into clauses of a 
    new predicate.

    Ors		The list of predicates contained in an ';', that is, the
		options for each clause, the unflattened bodies of the new
		clauses.
    Formula	The head of the new clauses.
    Vars	The list of variables that occur outside the ';'.
    F		The string representing the name of the predicate the ';'
		occured in.
    OrsClauses	The set of clauses made from flattening the ';'s out of the
		body.
----------------------------------------------------------------------------*/
ors_clauses([], _Formula, _Vars, _F, []).
ors_clauses([Or|Ors], Formula, Vars, F, OrsClauses2) :-
    flatten_or_once_body(Or, NewOr, Vars, F, NewClauses),
    ors_clauses(Ors, Formula, Vars, F, OrsClauses),
    append([(Formula :- NewOr)|NewClauses], OrsClauses, OrsClauses2).


/*----------------------------------------------------------------------------
variables_in_formula(Formula, Variables) :-
    Variables is the list of variables in the formula Formula, as
    determined by $isa_var/1.
----------------------------------------------------------------------------*/
variables_in_formula(Formula, Variables) :-
    get_variables(Formula, Variables),
    tail_ol(Variables, []).

/*----------------------------------------------------------------------------
get_variables_list(Formulas, Variables) :-
    Variables is the list of variables in the list of formula Formulas.
----------------------------------------------------------------------------*/
get_variables_list([], _Variables).
get_variables_list([Formula|Formulas], Variables) :-
    get_variables(Formula, Variables),
    get_variables_list(Formulas, Variables).

/*----------------------------------------------------------------------------
get_variables(Formula, Variables) :-
    Variables is the list of variables in the formula Formula.
----------------------------------------------------------------------------*/
get_variables(Formula, Variables) :-
    '$isa_var'(Formula), !,
    insert_ol2(Formula, Variables).
get_variables(Formula, Variables) :-
    Formula =.. [_F|Args],
    get_variables_list(Args, Variables).


/*----------------------------------------------------------------------------
variables_occuring_outside_list(Formulas, VarsOutSide, Variables) :-
    Similar to variables_in_formula/2, except variables are only added
    to the open list Variables if they are also in VarsOutSide.
----------------------------------------------------------------------------*/
variables_occuring_outside_list([], _VarsOutSide, _Variables).
variables_occuring_outside_list([Formula|Formulas], VarsOutSide, Variables) :-
    variables_occuring_outside(Formula, VarsOutSide, Variables),
    variables_occuring_outside_list(Formulas, VarsOutSide, Variables).

/*----------------------------------------------------------------------------
variables_occuring_outside(Formula, VarsOutSide, Variables) :-
    Similar to variables_in_formula/2, except variables are only added
    to the open list Variables if they are also in VarsOutSide.
----------------------------------------------------------------------------*/
variables_occuring_outside(Formula, VarsOutSide, Variables) :-
    '$isa_var'(Formula), !,
    (member2(Formula, VarsOutSide) -> insert_ol2(Formula, Variables) ; true).
variables_occuring_outside(Formula, VarsOutSide, Variables) :-
    Formula =.. [_F|Args],
    variables_occuring_outside_list(Args, VarsOutSide, Variables).

/*----------------------------------------------------------------------------
new_predicate_name(Fs, G) :-
    Return a new predicate name, G, from a base, Fs, and an extension N.

    F		The string representing the predicate name.
    G 		The new predicate name.

    e.g. F="foo", suffix(2), G = 'foo$2'
    if the new filename G is of the form "$query$<n>" then make it
    "$query$<n><working directory><filename>"
----------------------------------------------------------------------------*/
new_predicate_name(Fs, G) :-
    suffix(N),
    incr(suffix),
    name(N, Ns),
    append("$", Ns, Gs),
    append(Fs, Gs, Gs2),
    (append("$query", _Rest, Fs) ->
	filename_prefix(Prefix),
	absolute_file_name('.', WD),
	name(WD, WDs),
	append(WDs, [0'/|Prefix], NewPrefix),
	append(Gs2, NewPrefix, Gs3)
    ;
	Gs3 = Gs2
    ),
    name(G, Gs3).


/*----------------------------------------------------------------------------
remove_trues(Body, NewBody) :-
    Remove all of the useless calls to "true/0" in the straight line
    code in Body. 
----------------------------------------------------------------------------*/
remove_trues(Body, NewBody) :-
    rhs_list(Body, Bodies),
    remove_trues_in_list(Bodies, NewBodies),
    rhs_list(NewBody, NewBodies).


/*----------------------------------------------------------------------------
remove_trues_in_list(Bodies, NewBodies) :-
    NewBodies is Bodies with all occurences of 'true' removed.
----------------------------------------------------------------------------*/
remove_trues_in_list([], []).
remove_trues_in_list([true|Bodies], NewBodies) :-
    !,
    remove_trues_in_list(Bodies, NewBodies).
remove_trues_in_list([Formula|Bodies], [Formula|NewBodies]) :-
    remove_trues_in_list(Bodies, NewBodies).

/*----------------------------------------------------------------------------
flatten_ands(Sentence, FlattenedSentences) :-
    flatten terms anded together into separate anded elements in a RHS.
    e.g.
    (Head :- A, (B, C), D) --> (Head :- A, B, C, D)
----------------------------------------------------------------------------*/
flatten_ands((Head :- Body), (Head :- NewBody)) :-
    !,
    rhs_list(Body, BodyList),
    flatten_ands_body(BodyList, NewBodyList),
    rhs_list(NewBody, NewBodyList).
flatten_ands(Sentence, Sentence).


/*----------------------------------------------------------------------------
flatten_ands_body(Body, NewBody) :-
    flatten terms anded together into separate anded elements in a RHS.
    e.g.
    [A, (B, C), D] --> [A, B, C, D]
----------------------------------------------------------------------------*/
flatten_ands_body([], []).
flatten_ands_body([Term|Body], NewBodyOut) :-
    flatten_ands_term(Term, NewTerms),
    flatten_ands_body(Body, NewBody),
    append(NewTerms, NewBody, NewBodyOut).


/*----------------------------------------------------------------------------
flatten_ands_term(Term, NewTerms) :-
    (B, C)--> [B, C]
----------------------------------------------------------------------------*/
flatten_ands_term(Term, [Term]) :-
    var(Term), !.
flatten_ands_term((A, B), NewTerms) :-
    !,
    flatten_ands_term(A, As),
    flatten_ands_term(B, Bs),
    append(As, Bs, NewTerms).
flatten_ands_term(Term, [Term]).
