/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * fresh_copy - Create fresh copies of clauses and formulas.
 */

/*----------------------------------------------------------------------------
fresh_copy_clauses(Clauses, NewClauses) :-
    NewClauses is a fresh copy of all of the clauses in the list Clauses,
    that is, there are new variables for each clause.
----------------------------------------------------------------------------*/
fresh_copy_clauses([], []).
fresh_copy_clauses([Formula|Formulas], [NewFormula|NewFormulas]) :-
    fresh_copy_formula(Formula, _Mapping, NewFormula),
    fresh_copy_clauses(Formulas, NewFormulas).


/*----------------------------------------------------------------------------
fresh_copy_formula(Formula, Mapping, NewFormula) :-
    NewClauses is a fresh copy of the clause formula, that is, there are 
    new variables for each variable in the formula.
----------------------------------------------------------------------------*/
fresh_copy_formula(Formula, Mapping, NewFormula) :-
    var(Formula), !,
    map_variable(Formula, Mapping, NewFormula).
fresh_copy_formula(Formula, Mapping, NewFormula) :-
    Formula =.. [F|Args],
    fresh_copy_args(Args, Mapping, NewArgs),
    NewFormula =.. [F|NewArgs].


/*----------------------------------------------------------------------------
fresh_copy_args(Args, Mapping, NewArgs),
    NewArgs is Args with variables in Args mapping (Mapping) to new variables
    in NewArgs.
----------------------------------------------------------------------------*/
fresh_copy_args([], _Mapping, []).
fresh_copy_args([Arg|Args], Mapping, [NewArg|NewArgs]) :-
    fresh_copy_formula(Arg, Mapping, NewArg),
    fresh_copy_args(Args, Mapping, NewArgs).


/*----------------------------------------------------------------------------
map_variable(Var, Mapping, NewVar) :-
    NewVar is the Mapping of the variable Var as defined by the list Mapping.
    If no mapping exists then create a new variable.
----------------------------------------------------------------------------*/
map_variable(Var, Mapping, NewVar) :-
    var(Mapping), !,
    Mapping = [r(Var, NewVar)|_Tail].
map_variable(Var, [r(X, Y)|_Mapping], Y) :-
    Var == X, !.
map_variable(Var, [_VarMap|Mapping], NewVar) :-
    map_variable(Var, Mapping, NewVar).
