/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * index - Create the QuAM indexing instructions for a procedure.
 */

/*----------------------------------------------------------------------------
index_procedure(F/N, Clauses, IndexingInstructions) :-
    Generates the indexing instructions for a procedure with name F/N, and
    the given clauses.  IndexingInstructions is the resulting list of
    indexing instructions for the procedure.

    If there is only one clause no control instructions are needed just
	the labels
	    F/N:	<clauses instructions>
	    end(F/N):
    else
	    F/N:	switch_on_term(Var, Const, Apply, Pair, Quant, Obvar)
	    Var:	try(N, 1)
			retry(2)
			...
			retry(n-1)
			trust(n)
	    Const:	switch_on_constant(m, [default(Default),
					       cluster(Constant1, Constant1L), ...])
	    Default:	try(N, i)
			retry(j)
			...
			trust(k)

	    Constant1L:	try(N, o)
			retry(p)
			...
			trust(r)

	    Apply:	switch_on_structure(m, [default(Default),
						cluster(F/N, StruL), ...])
	    StruL:	....

	    1:		<clause 1's instructions>
	    2: 		<clause 2's instructions>

			....

	    n: 		<clause n's instructions>
	    end(F/N):

    where Var, Const, etc. are represented with n+1, n+2, ...

    For the switch_on_constant and switch_on_structure if they have only
    one table entry then these instructions are not used, instead the
    try, retry, ..., trust combination is moved up to the Const or Apply
    label respectively.

    If there is only one clause for a constant or apply then the cluster
    label is the clause label, e.g. Constant1L is "Lab" say if "Lab" is label 
    of the only clause.

    If there is only one clause with a variable in the first argument
    then the Default label is set directly to that clause.
----------------------------------------------------------------------------*/
index_procedure(_F/N, Clauses, IndexingInstructions) :-
    length(Clauses, Length),
    index_procedure2(N, Length, Clauses, IndexingInstructions).


index_procedure2(_N, 1, _Clauses, []) :-
    !.
index_procedure2(0, Length, _Clauses, IndexingInstructionsOut) :-
    !,
    enumerate_n_ints(Length, AllClauses),
    index_set_of_clauses(AllClauses, 0, IndexingInstructionsOut).
index_procedure2(N, Length, Clauses, IndexingInstructions) :-
    collect_first_arguments(Clauses, Args),
    build_decision_tree(Args, Length, DecisionTree),
    index_clauses(DecisionTree, N, IndexingInstructions).

/*----------------------------------------------------------------------------
collect_first_arguments(Clauses, Args) :-
    Args is the list of first arguments of each clause in the procedure.
    Since we are only indexing on the first level of the term, only need
	variable	X
	constant(a)	a
	apply(f/n)	f(t1, ..., tn)
	quantifier	x^t
	object_variable	x
    Then wrap argno(<term type>, i) where i is the clause number.
----------------------------------------------------------------------------*/
collect_first_arguments(Args, ArgNumbers) :-
    collect_first_arguments_numbered(Args, 1, ArgNumbers).

collect_first_arguments_numbered([], _N, []).
collect_first_arguments_numbered([Clause|Clauses], N, [argno(Arg, N)|Args]) :-
    M is N + 1,
    first_argument(Clause, Arg),
    collect_first_arguments_numbered(Clauses, M, Args).


/*----------------------------------------------------------------------------
first_argument(Clause,Arg) :-
    Sets Arg to the type of the first argument in the head of the
    given clause.
----------------------------------------------------------------------------*/
first_argument((Head :- _Body), Arg) :-
    !,
    first_argument_from_head(Head, Arg).
first_argument(Head, Arg) :-
    first_argument_from_head(Head, Arg).
    

first_argument_from_head(Head, Arg) :-
    Head =.. [_F, FirstArg|_Args],
    type_argument(FirstArg, Arg).


/*----------------------------------------------------------------------------
type_argument(Term, Arg) :-
    Determine the argument type (Arg) of a given Term.
----------------------------------------------------------------------------*/
type_argument(Term, variable) :-
    (var(Term)
    ; 
     Term =.. [F|Args],
     '$isa_substitution'(F, Args, _S, _T)
    ), !.
type_argument(Term, object_variable) :-
    '$isa_object_var'(Term), !.
type_argument(Term, constant(Term)) :-
    atomic(Term), !.
type_argument(Term, Type) :-
    Term =.. [F|Args],
    length(Args, N),
    ('$quantifier'(F) ->
	Type = quantifier(F/N)
    ;
	Type = apply(F/N)
    ).

/*----------------------------------------------------------------------------
build_decision_tree(Args, N, DecisionTree) :-
    Divide the list of clauses first args Args into a DecisionTree of the
    form of six classes:
	Variables
	Constants - further split up into clauses of the same constant
	Applys - further split up into clauses of the same functor and arity.
	Pairs
	Quant
	Obvars

    Note: "Pairs" is empty at present.

    for the instruction 
	switch_on_term(Var, Const, Apply, Pair, Quant, Obvar)
    N is the number of clauses in the procedure.
----------------------------------------------------------------------------*/
build_decision_tree(Args, N, dt(AllClauses, Vars, Consts, Applys, [],
				Quant, ObjVar)) :-
    enumerate_n_ints(N, AllClauses),
    partition_vars(Args, Vars),
    partition_types(Args, constant, Vars, Consts),
    partition_types(Args, apply, Vars, Applys),
    partition_types(Args, quantifier, Vars, Quant),
    partition_types(Args, object_variable, Vars, ObjVar).


/*----------------------------------------------------------------------------
enumerate_n_ints(N, NInts) :-
    NInts is the list of positive integers in the range 1 to N (inclusive).
----------------------------------------------------------------------------*/
enumerate_n_ints(N, NInts) :-
    enumerate_range_of_ints(1, N, NInts).

enumerate_range_of_ints(M, N, []) :-
    M > N, !.
enumerate_range_of_ints(M, N, [M|NInts]) :-
    MTmp is M + 1,
    enumerate_range_of_ints(MTmp, N, NInts).

/*----------------------------------------------------------------------------
partition_vars(Args, VarClauses) :-
    VarClauses is the list of clauses (numbers representing the clauses)
    where the first argument of the head is a variable.
    Args is the list of typed first arguments of each clause.
----------------------------------------------------------------------------*/
partition_vars([], []).
partition_vars([Arg|Args], VarClauses) :-
    partition_var(Arg, VarClauses, VarClausesTmp),
    partition_vars(Args, VarClausesTmp).

partition_var(argno(variable, N), [N|VarClauses], VarClauses) :-
    !.
partition_var(_Arg, VarClauses, VarClauses).


/*----------------------------------------------------------------------------
partition_types(Args, Type, Vars, Types) :-
    Take all of the arguments of the given Type in the list of first
    arguments of each clause Args, and sort them into lists of lists of the
    same subtype, with the variable cases added in sequence.
    The subtype is discarded for those cases where the value of the
    subtype is not important.  eg object_variable
----------------------------------------------------------------------------*/
partition_types(Args, Type, Vars, Types) :-
    partition_the_types(Args, Type, TypesOnly),
    tail_ol(TypesOnly, []),
    distribute_variables_clauses(TypesOnly, Vars, Types).

partition_the_types([], _Type, _TypesOnly).
partition_the_types([Arg|Args], Type, TypesOnly) :-
    partition_the_type(Arg, Type, TypesOnly),
    partition_the_types(Args, Type, TypesOnly).

partition_the_type(argno(Type, ClauseNo), Type, TypesOnly) :-
    !,
    insert_ol(cluster(_SubType, Clauses), TypesOnly),
    postpend_ol(ClauseNo, Clauses).
partition_the_type(argno(ArgType, ClauseNo), Type, TypesOnly) :-
    ArgType =.. [Type, SubType],
    !,
    insert_ol(cluster(SubType, Clauses), TypesOnly),
    postpend_ol(ClauseNo, Clauses).
partition_the_type(_Arg, _Type, _TypesOnly).

/*----------------------------------------------------------------------------
distribute_variables_clauses(TypesOnly, Vars, Types) :-
    Merge the clauses of a subtype (TypesOnly say for the atom a) with
    the clauses that have variables in their first argument positions.
    e.g. 
	f(a).
	f(X).
    the query :- f(a). will match both clauses, so they both have to be
    considered. The clauses are represented by numbers so sort them into
    their original sequence.
----------------------------------------------------------------------------*/
distribute_variables_clauses([], _Vars, []).
distribute_variables_clauses([cluster(SubType, Clauses)|TypesOnly], Vars,
	[cluster(SubType, NewClauses)|Types]) :-
    distribute_variables(Vars, Clauses, NewClauses),
    distribute_variables_clauses(TypesOnly, Vars, Types).
    
distribute_variables(Vars, Clauses, NewClauses) :-
    tail_ol(Clauses, []),
    merge(Vars, Clauses, NewClauses).

/*----------------------------------------------------------------------------
index_clauses(DecisionTree, N, IndexingInstructions) :-
    Converts an indexing DecisionTree for N clauses into the actual QuAM
    assembly instructions in IndexingInstructions.  This procedure will
    only be called from "index_procedure2/4" if the procedure has arity
    1 or more, and at least 2 clauses.

    dt([1, ..., n],
       [i, ..., j],
       [ac(<atom>, [o, .., p, i, ..., j]), ...],
       [ac(<f/n>, [q, .., r, i, ..., j]), ...],
       [],
       [ac(^, [s, ..., t, i, ..., j]),
       [ac(<x>, [u, ..., v, i, ..., j]), ...]).

    f\n:
		switch_on_term(variable_label, constant_label, apply_label,
		pair_label, quantifier_label, object_variable_label).

    variable_label:
		try(N, 1)
		retry(2)
		...
		trust(m)

    constant_label:
		switch_on_constant(n, list_of_constant_entries)

		where each entry is of the form atom : atom_label
    atom_label:
		try(N, i)
		retry(j)
		...
		trust(k)	where each clause can be either have the
				same atom or a variable as the first arg.
    apply_label:
		switch_on_structure(n, list_of_structure_entries)

		similarly for f/n : structure_label

    quantifier_label:
		try(N, l)
		retry(m)
		...
		trust(n)

    object_variable_label:
		try(N, o)
		retry(p)
		...
		trust(q)

----------------------------------------------------------------------------*/
index_clauses(dt(AllClauses, _Vars, [], [], [], [], []),
	N, IndexingInstructions) :-
    !,
    index_set_of_clauses(AllClauses, N, IndexingInstructions). 
index_clauses(dt(AllClauses, Vars, Consts, Applys, [], Quants, ObjVars), N,
	[switch_on_term(AllClausesLabel, ConstsLabel, ApplysLabel, fail,
	QuantsLabel, ObjVarsLabel)|IndexingInstructions]) :-
    insert_ol2(AllClausesLabel, Labels),
    index_set_of_clauses(AllClauses, N, AllIndexing), 
    index_set_of_clauses(Vars, N, VarIndexing),
    (VarIndexing \== [] ->
	insert_ol2(DefaultLabel, Labels),
    	NewVarIndexing = [label(DefaultLabel)|VarIndexing]
    ; Vars = [DefaultLabel] ->
	NewVarIndexing = []
    ;
	DefaultLabel = fail,
	NewVarIndexing = []
    ),
    index_subtypes(Consts, ConstsLabel, Labels, N, DefaultLabel,
	switch_on_constant, ConstIndexing),
    index_subtypes(Applys, ApplysLabel, Labels, N, DefaultLabel,
	switch_on_structure, ApplyIndexing),
    index_subtypes(Quants, QuantsLabel, Labels, N, DefaultLabel,
	switch_on_quantifier, QuantIndexing),
    index_no_value_subtypes(ObjVars, ObjVarsLabel, Labels, N, DefaultLabel,
	ObjVarsIndexing),
    append(ObjVarsIndexing, NewVarIndexing, OV),
    append(QuantIndexing, OV, QOV),
    append(ApplyIndexing, QOV, ApplyVar),
    append(ConstIndexing, ApplyVar, ConstApplyVar),
    append([label(AllClausesLabel)|AllIndexing], ConstApplyVar,
	IndexingInstructions),
    length(AllClauses, Length),
    set_labels_from_n(Labels, Length).


/*----------------------------------------------------------------------------
index_set_of_clauses(ClauseLabels, N, SetIndexingInstructions) :-
    Create the indexing instructions (a try/retry/trust block) for
    N clauses.

    e.g. ClauseLabels = [2, 5, 6, 8]
	 IndexingInstructions = [try(N, 2), retry(5), retry(6), trust(8)]
    if there is only zero or one clauses then no indexing is required.
----------------------------------------------------------------------------*/
index_set_of_clauses([], _N, []).
index_set_of_clauses([_Label], _N, []) :- !.
index_set_of_clauses([FirstLabel|Labels], N,
	[try(N, FirstLabel)|RetrysAndTrust]) :-
    generate_retries(Labels, RetrysAndTrust).

generate_retries([LastLabel], [trust(LastLabel)]) :- !.
generate_retries([Label|Labels], [retry(Label)|RetrysAndTrust]) :-
    generate_retries(Labels, RetrysAndTrust).


/*----------------------------------------------------------------------------
index_subtypes(Table, TableLabel, Labels, N, DefaultLabel, TypeOfSwitch,
	Indexing) :-
    Generates indexing instructions for a switching subtype (constant's and
    apply's).

    The following is an example of the instructions generated for the
    constant subtype :

	    Const:	switch_on_constant(m, [default(Default),
					       cluster(Constant1, Constant1L), ...])
	    Default:	try(N, i)
			retry(j)
			...
			trust(k)

	    Constant1L:	try(N, o)
			retry(p)
			...
			trust(r)

    Table	The switching table for a particular subtype, i.e. constant
		apply, etc.
    TableLabel	The label for the switch instruction for this table.
    Labels	The open list of all indexing instructions labels for the
		procedure.
    DefaultLabel
		The label for the set of default clauses for this subtype
		i.e. those that have variables in the first arg.
    TypeOfSwitch
		The type of switching instruction for the subtype, i.e.
		switch_on_constant, switch_on_structure, etc.
    Indexing	The indexing instructions for the subtype.

    If there are no table entries then
	the TableLabel (in the switch_on_term instruction) is fail
    else if there is one element then jump to it directly
    else jump to switch instruction

----------------------------------------------------------------------------*/
index_subtypes([], DefaultLabel, _Labels, _N, DefaultLabel, _TypeOfSwitch,
	       []) :- !.
index_subtypes([cluster(_SubType, [Label])], Label, _Labels, _N,
	_DefaultLabel, _TypeOfSwitch, []):- !.
index_subtypes(Table, TableLabel, Labels, N, DefaultLabel, TypeOfSwitch,
	[label(TableLabel), SwitchInstruction|ClusterInstructions]) :-
    insert_ol2(TableLabel, Labels),
    index_subtype_clusters(Table, Labels, N, NewTable, ClusterInstructions),
    size_table(Table, SizeTable),
    SwitchInstruction =.. [TypeOfSwitch, SizeTable,
		[default(DefaultLabel)|NewTable]].

    
/*----------------------------------------------------------------------------
index_subtype_clusters(Table, Labels, N, NewTable, ClustersInstructions) :-
    Create the index instructions for a cluster of a subtype e.g. the
    subtype Constant1 of the constant type.

    [cluster(Constant1, [o, p, ..., r]), ....]

    from this generate instructions

    [cluster(Constant1, Constant1L), ...]		% the table in the
							% switch instruction
    Constant1L:	try(N, o)
		retry(p)
		...
		trust(r)
    ....

----------------------------------------------------------------------------*/
index_subtype_clusters([], _Labels, _N, [], []).
index_subtype_clusters([Cluster|Clusters], Labels, N, [TableEntry|Table],
    ClustersInstructionsOut) :-
    index_subtype_cluster(Cluster, Labels, N, TableEntry, ClusterInstructions),
    index_subtype_clusters(Clusters, Labels, N, Table, ClustersInstructions),
    append(ClusterInstructions, ClustersInstructions,
	ClustersInstructionsOut).
    
index_subtype_cluster(cluster(SubType, [ClusterLabel]), _Labels, _N, 
	(SubType : ClusterLabel), []) :- !.
index_subtype_cluster(cluster(SubType, ClusterLabels), Labels, N, 
	(SubType : ClusterLabel), [label(ClusterLabel)|ClusterInstructions]):-
    index_set_of_clauses(ClusterLabels, N, ClusterInstructions),
    insert_ol2(ClusterLabel, Labels).
	 
/*----------------------------------------------------------------------------
index_no_value_subtypes(Table,TableLabel,Labels,N,DefaultLabel,Indexing) :-
    Similar to index_subtype.  The value of the subtype has been discarded
    as they do not provide further indexing.  eg object_variable.
----------------------------------------------------------------------------*/
index_no_value_subtypes([], DefaultLabel, _Labels, _N, DefaultLabel, []) :- !.
index_no_value_subtypes([cluster(_SubType, [Label])], Label, _Labels, _N,
	_DefaultLabel, []):- !.
index_no_value_subtypes([cluster(_SubType, ClusterLabels)], ClusterLabel,
	Labels, N, _DefaultLabel, [label(ClusterLabel)|ClusterInstructions]):-
    index_set_of_clauses(ClusterLabels, N, ClusterInstructions),
    insert_ol2(ClusterLabel, Labels).

    
/*----------------------------------------------------------------------------
size_table(Table, N) :-
    N is the smallest power of 2 that will contain all of the entries in 
    Table.
    e.g. say the length of Table is 3 then N = 2, since 3 < 2^2.
----------------------------------------------------------------------------*/
size_table(Table, N) :-
    length(Table, Length),
    min_power_of_two(Length, 0, M),
    N is M + 1.

min_power_of_two(Length, M, N) :-
    X is 1 << M,
    Length > X,
    !,
    M2 is M + 1,
    min_power_of_two(Length, M2, N).
min_power_of_two(_Length, N, N).

/*----------------------------------------------------------------------------
set_labels_from_n(Labels, N) :-
    Set the open list of Labels [X,Y,...|V] to Integers from N+1, N+2, ....
----------------------------------------------------------------------------*/
set_labels_from_n(Labels, _N) :-
    var(Labels), !.
set_labels_from_n([M|Labels], N) :-
    M is N + 1,
    set_labels_from_n(Labels, M).
