/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * interpret_term - Interpret Qu-Prolog terms into their Prolog equivalents.
 */

%	Interpreting Qu-Prolog terms
%	----------------------------
%	interpreter - uses read1Term/3
%	compiler - uses pread/1
%
%	interpreter :
%		. generate distinctness information from the parallel 
%		  substitutions occuring in the term
%		. interpret term to new structure on the heap
%		. record the largest extension used of persistent vars
%		  (meta and object).
%
%	compiler :
%		. record the largest extension used of persistent vars
%		  (meta and object).
%		. collect distinctness information and add to the body of
%		  the term
%		. compile the new (herbrand) with distinctness information
%		  added to the body
%
% {persistent_var, object_var, meta_var} < T. 
% {persistent_object_var, persistent_meta_var} < persistent_var. 
% {persistent_object_var, dynamic_object_var, local} < object_var. 
% {persistent_meta_var, dynamic_meta_var} < meta_var. 
%
%
%	Pers elements are X or x which were before #X and #x respectively
%	DOVs elements are x.
%	PSubs elements are [x1, ..., xn] the domain of parallel subs
%		where xi are x or #x
%
%	We need to form the distinctness information from PSubs, such that
%	x1 distinct_from x2, ..., x1 distinct_from xn,
%	..., xn-1 distinct_from xn.
%
% Note: This code cannot contain Qu-Prolog terms x, #X, #x, @x, q x t,
%	[t1/x1, ..., tn/xn] * t because they would have a different
%	interpretation in WAM and QuAM and because they need to be treated
%	at the herbrand level so we can parse them.
%
% ?????????????????????
% Note: var(Var) catches all meta variables, persistent or dynamic. This
%	may have an effect on the code for $isa_.....
% ?????????????????????

/*
'$interpret_qu_prolog'(Term, MetaNames, MetaVariables,
		     NewTerm, QuNames, QuVars) :-
    '$qu_prolog_term'(Term),
    '$make_quantifiers_anonymous'(Term, TermTmp),
    '$vars_in_term'(TermTmp, [], Variables),
    '$map_vars_to_interpretation'(Variables, MapVars),
    '$select_sub_class'(MapVars, object_var, DynamicOVs),
    '$create_distincts'(DynamicOVs, [], Distincts),
    '$select_sub_class'(MapVars, persistent_var, _Persistents),
    % '$set_persistent_bases'(Persistents), % to be used by new_persistent
    '$interpret_term'(TermTmp, MapVars, MetaNames, MetaVariables, NewTerm),
    '$execute_distincts'(Distincts),
    '$collect_qu_names_and_vars'(MapVars, MetaNames, MetaVariables,
				QuNames, QuVars),
    !.
'$interpret_qu_prolog'(Term, _, _, _, _, _) :-
    write(user_error, Term),
    writeln(user_error, ' cannot convert to Qu-Prolog term'),
    fail.
*/

'$filter_term_to_be_compiled'(Term, NewTerm) :-
    '$qu_prolog_term'(Term),
    '$make_quantifiers_anonymous'(Term, TermTmp),
    '$vars_in_term'(TermTmp, [], Variables),
    '$map_vars_to_vars'(Variables, MapVars),
    ('$select_sub_class2'(MapVars, persistent_var, []) % no persistents allowed
    ;
     writeln(user_error, 'Error: persistent variables not allowed'),
     writeln(user_error, Term),
     fail
    ),
    '$select_sub_class'(MapVars, dynamic_object_var, DynamicOVs),
    '$create_distincts'(DynamicOVs, [], Distincts),
    '$add_to_body'(Distincts, TermTmp, NewTerm), % using rhs_list/2
    !.
'$filter_term_to_be_compiled'(Term, _) :-
    write(user_error, Term),
    writeln(user_error, ' cannot convert to Qu-Prolog term'),
    fail.

'$add_to_body'([], Term, Term).
'$add_to_body'(Distincts, TermTmp, NewTerm):- 
    '$extract_distinct_goals'(Distincts, DistinctsList),
    rhs_list(DistinctGoals, DistinctsList),
    '$add_to_body2'(TermTmp, DistinctGoals, NewTerm).

'$extract_distinct_goals'([], []).
'$extract_distinct_goals'([_Identifier/Call|Distincts], [Call|DistinctsList]):-
    '$extract_distinct_goals'(Distincts, DistinctsList).

'$add_to_body2'((Head :- Body), DistinctGoals, (Head :- DistinctGoals, Body)).
'$add_to_body2'(Head, DistinctGoals, (Head :- DistinctGoals)).

%
% '$qu_prolog_term'(Term) :-
%	True, iff Term is a Qu-Prolog term.
%	Side-affect : gives error messages for badly formed terms.
%	

'$qu_prolog_term'(X) :-
    '$qu_prolog_term2'(X, []).

'$qu_prolog_term2'(X, LocalVars) :-
    '$isa_local'(X), !,
    (\+ member(X, LocalVars) ->
	write(user_error, 'Error : free occurrence of local object variable '),
	writeln(user_error, X),
	fail
    ;
	true
    ).
'$qu_prolog_term2'(X, _LocalVars) :-
    '$isa_var'(X), !.
'$qu_prolog_term2'(Term, LocalVars) :-
    Term =.. [F|Args],
    '$qu_prolog_term'(F, Args, LocalVars).

'$qu_prolog_term'(F, Args, LocalVars):-
    '$isa_substitution'(F, Args, Ss, T),
    !,
    '$qu_prolog_subs'(Ss, LocalVars, LocalVars2),
    '$qu_prolog_term2'(T, LocalVars2).
'$qu_prolog_term'(Q, [X, T], LocalVars) :-
    '$quantifier'(Q), !,
    (\+ '$isa_object_var'(X) ->
	write(user_error, 'Error : quantified position, '),
	write(user_error, X),
	writeln(user_error, ', must be an object-var'),
	fail
    ;
	true
    ),
    ('$isa_local'(X) ->
	LocalVars2 = [X|LocalVars]
    ;
	LocalVars2 = LocalVars
    ),
    '$qu_prolog_term2'(T, LocalVars2).
'$qu_prolog_term'(quant, [Q, X, T], LocalVars) :-
    '$quantifier'(Q), !,
    (\+ '$isa_object_var'(X) ->
	write(user_error, 'Error : quantified position, '),
	write(user_error, X),
	writeln(user_error, ', must be an object-var'),
	fail
    ;
	true
    ),
    ('$isa_local'(X) ->
	LocalVars2 = [X|LocalVars]
    ;
	LocalVars2 = LocalVars
    ),
    '$qu_prolog_term2'(T, LocalVars2).
'$qu_prolog_term'(_F, Args, LocalVars) :-
    '$qu_prolog_args'(Args, LocalVars).

'$qu_prolog_args'([], _LocalVars).
'$qu_prolog_args'([Term|Terms], LocalVars) :-
    '$qu_prolog_term2'(Term, LocalVars),
    '$qu_prolog_args'(Terms, LocalVars).

'$qu_prolog_subs'(Ss, LocalVarsIn, LocalVarsOut) :-
    nonvar(Ss),
    Ss = S1 * S2,
    !,
    '$qu_prolog_subs'(S1, LocalVarsIn, LocalVars2),
    '$qu_prolog_subs'(S2, LocalVars2, LocalVarsOut).
'$qu_prolog_subs'(Ss, LocalVarsIn, LocalVarsOut) :-
    '$isa_substitution'(Ss),
    '$qu_prolog_sub'(Ss, [], LocalVarsIn, LocalVarsOut).

'$qu_prolog_sub'([], _DomainIn, LocalVars, LocalVars).
'$qu_prolog_sub'([T/X|Ss], DomainIn, LocalVarsIn, LocalVarsOut) :-
    !,
    (\+ '$isa_object_var'(X) ->
	write(user_error, 'Error : the domain of '),
	write(user_error, T/X),
	writeln(user_error, ' must be an object var'),
	fail
    ;
	true
    ),
    (member(X, DomainIn) -> 
	write(user_error, 'Error : object var '), 
	write(user_error, X), 
	writeln(user_error, ' in substitution must be distinct'),
	fail
    ;
	true
    ),
    '$qu_prolog_term2'(T, LocalVarsIn),
    ('$isa_local'(X) ->
	LocalVars2 = [X|LocalVarsIn]
    ;
	LocalVars2 = LocalVarsIn
    ),
    '$qu_prolog_sub'(Ss, [X|DomainIn], LocalVars2, LocalVarsOut).
'$qu_prolog_sub'([X|_Ss], _DomainIn) :-
    write(user_error, 'Error : substitution '),
    write(user_error, X),
    writeln(user_error, ' needs a "/"'),
    fail.



% '$make_quantifiers_anonymous'(Term, NewTerm) :-
%	Replaces any occurences of q x t as a subterm in Term with q(x ^ t).
%	

'$make_quantifiers_anonymous'(X, X) :-
    '$isa_var'(X), !.
'$make_quantifiers_anonymous'(Term, NewTerm) :-
    Term =.. [F|Args],
    ('$quantifier'(F), Args = [X, T] ->
	NewF = F,
	'$make_quantifiers_anonymous'(T, NewT),
	NewArgs = [X ^ NewT]
    ;
     F = quant, Args = [Q, X, T], '$quantifier'(Q) ->
	NewF = Q,
	'$make_quantifiers_anonymous'(T, NewT),
	NewArgs = [X ^ NewT]
    ;
	NewF = F,
	'$make_quantifiers_anonymous_args'(Args, NewArgs)
    ),
    NewTerm =.. [NewF|NewArgs].

'$make_quantifiers_anonymous_args'([], []).
'$make_quantifiers_anonymous_args'([Term|Terms], [NewTerm|NewTerms]) :-
    '$make_quantifiers_anonymous'(Term, NewTerm),
    '$make_quantifiers_anonymous_args'(Terms, NewTerms).

/*
%
% '$parallel_subs_in_term'(Var, PSubsIn, PSubsOut) :-
%	collect all of the object vars that are distinct (contained 
%	in the same parallel substitution
%	

'$parallel_subs_in_term'(X, PSubsIn, PSubsIn) :-
    '$isa_var'(X), !.
'$parallel_subs_in_term'(Term, PSubsIn, PSubsOut) :-
    Term =.. [F|Args],
    ('$isa_substitution'(F, Args, Ss, T) ->
	'$parallel_subs_in_subs'(Ss, PSubsIn, PSubsTmp, [], DomainOut),
	'$parallel_subs_in_term'(T, [DomainOut|PSubsTmp], PSubsOut)
    ;
	'$parallel_subs_in_args'(Args, PSubsIn, PSubsOut)
    ).

'$parallel_subs_in_args'([], PSubsIn, PSubsIn).
'$parallel_subs_in_args'([Term|Terms], PSubsIn, PSubsOut) :-
    '$parallel_subs_in_term'(Term, PSubsIn, PSubsTmp),
    '$parallel_subs_in_args'(Terms, PSubsTmp, PSubsOut).

'$parallel_subs_in_subs'([], PSubsIn, PSubsIn, DomainIn, DomainIn).
'$parallel_subs_in_subs'([T/X|Ss], PSubsIn, PSubsOut, DomainIn, DomainOut) :-
    !,
    '$parallel_subs_in_term'(T, PSubsIn, PSubsTmp),
    '$parallel_subs_in_subs'(Ss, PSubsTmp, PSubsOut, [X|DomainIn], DomainOut).
*/

'$isa_substitution'(Ss) :-
     nonvar(Ss),
     isList(Ss),
     !.
'$isa_substitution'(Ss) :-
     nonvar(Ss),
     Ss = S1 * S2,
     '$isa_substitution'(S1),
     '$isa_substitution'(S2).

'$isa_substitution'((*), [Ss, T], Ss, T) :-
     '$isa_substitution'(Ss).



%
% '$vars_in_term'(Term, VariablesIn, VariablesOut) :-
%
%	Collect VariablesOut the set of local object vars in term union
%	the set VariablesIn.
%	
'$vars_in_term'(Term, VariablesIn, VariablesOut) :-
    '$isa_var'(Term), !,
    '$add_to_vars'(Term, VariablesIn, VariablesOut).
'$vars_in_term'(Term, VariablesIn, VariablesOut) :-
    Term =.. [_F|Args],
    '$vars_in_args'(Args, VariablesIn, VariablesOut).

'$vars_in_args'([], VariablesIn, VariablesIn).
'$vars_in_args'([Term|Terms], VariablesIn, VariablesOut) :-
    '$vars_in_term'(Term, VariablesIn, VariablesTmp),
    '$vars_in_args'(Terms, VariablesTmp, VariablesOut).


% '$interpret_term'(Term, Vars, InterpretedTerm) :-
%	True, iff Term is a Qu-Prolog term, where a Qu-Prolog term t is
%	described as:
%
%	t is a constant a
%	t is a meta var X
%	t is an object var x
%	t is a persistent meta var #X
%	t is a persistent object var #x
%	t is a local object var @x
%		where x represents x, #x or @x from now on.
%	t is a quantified term q x t,
%		where q is declared by :- op(999, quant, q),
%		and t is a Qu-Prolog term. 
%	t is a term with a parallel substitution [t1/x1, ..., tn/xn] * t
%		where each xi distinct_from xj, i<>j, and each ti and t are 
%		Qu-Prolog terms.
%	t is a constructor f(t1, ..., tn) and ti are Qu-Prolog terms.
%	
%	The herbrand forms: a, X, f(t1, ..., tn) stay the same, however
%	the interpretation of the herbrand input Term is different
%	inside the Qu-Prolog Abstract Machine.
%
%	This routine is to be used by read/1 for reading terms in the
%	interpreter.
%	Vars is the list of vars that occur in the term:
%	X, x, #X, #x, @x and their interpreted value in the machine.
%


/*
'$interpret_term'(Var, Vars, MetaNames, MetaVariables, NewVar) :-
    '$isa_var'(Var), !,
    '$dereference_var'(Var, Vars, MetaNames, MetaVariables, NewVar).
'$interpret_term'(Term, Vars, MetaNames, MetaVariables, NewTerm) :-
    Term =.. [F|Args],
    '$interpret_term'(F, Args, Vars, MetaNames, MetaVariables, NewTerm).

'$interpret_term'(Q, [X^T], Vars, MetaNames, MetaVariables, Quant) :-
    '$quantifier'(Q),
    !,	
    '$interpret_term'(X, Vars, MetaNames, MetaVariables, NewX),
    '$interpret_term'(T, Vars, MetaNames, MetaVariables, NewT),
    quantify(Quant, Q, NewX, NewT).
'$interpret_term'(F, Args, Vars, MetaNames, MetaVariables, SubsT):-
    '$isa_substitution'(F, Args, Ss, T), !,
    '$interpret_subs'(Ss, Vars, MetaNames, MetaVariables, NewSs),
    '$interpret_term'(T, Vars, MetaNames, MetaVariables, NewT),
    substitute(SubsT, NewSs, NewT).
'$interpret_term'(F, Args, Vars, MetaNames, MetaVariables, NewTerm) :-
    '$interpret_args'(Args, Vars, MetaNames, MetaVariables, NewArgs),
    NewTerm =.. [F|NewArgs].

'$interpret_args'([], _Vars, _MetaNames, _MetaVariables, []).
'$interpret_args'([Term|Terms], Vars, MetaNames, MetaVariables,
		[NewTerm|NewTerms]) :-
    '$interpret_term'(Term, Vars, MetaNames, MetaVariables, NewTerm),
    '$interpret_args'(Terms, Vars, MetaNames, MetaVariables, NewTerms).

'$interpret_subs'(Subs1 * Subs2, Vars, MetaNames, MetaVariables, NewSubs) :- 
    !,
    '$interpret_subs'(Subs1, Vars, MetaNames, MetaVariables, NewSubs1),
    '$interpret_subs'(Subs2, Vars, MetaNames, MetaVariables, NewSubs2),
    append(NewSubs1, NewSubs2, NewSubs).
'$interpret_subs'(Sub, Vars, MetaNames, MetaVariables, [NewSub]) :-
    '$isa_substitution'(Sub),
    '$interpret_sub'(Sub, Vars, MetaNames, MetaVariables, NewSub).

'$interpret_sub'([], _Vars, _MetaNames, _MetaVariables, []).
'$interpret_sub'([T/X|Ss], Vars, MetaNames, MetaVariables, [NewT/NewX|NewSs]) :-
    !,
    '$interpret_term'(X, Vars, MetaNames, MetaVariables, NewX),
    '$interpret_term'(T, Vars, MetaNames, MetaVariables, NewT),
    '$interpret_sub'(Ss, Vars, MetaNames, MetaVariables, NewSs).

%'$dereference_var'(Var, Mapping, MetaNames, MetaVariables, NewVar) :-
%	Dereference the var Var (herbrand term) to the Qu-Prolog
%	interpretation of the var, that has been made.

'$dereference_var'(Var, [p(A, NewVar, Interpreted)|_Mapping],
		 MetaNames, MetaVariables, NewVar) :-
    Var == A, !,
    (var(Interpreted) -> '$new_var'(Var, MetaNames, MetaVariables, NewVar),
			 Interpreted = true
    ;
			 true
    ).
'$dereference_var'(Var, [_Map|Mapping], MetaNames, MetaVariables, NewVar) :-
    '$dereference_var'(Var, Mapping, MetaNames, MetaVariables, NewVar).

*/

%'$dereference_var2'(Var, Mapping, NewVar) :-
%	Dereference the var Var (herbrand term) to the 
%	interpretation of the var: do not interpret into Qu-Prolog.

'$dereference_var2'(Var, [p(A, NewVar, _Interpreted)|_Mapping], NewVar) :-
    Var == A, !.
'$dereference_var2'(Var, [_Map|Mapping], NewVar) :-
    '$dereference_var2'(Var, Mapping, NewVar).
/*

% '$new_var'(Var, MetaNames, MetaVariables, Var) :-
% 	Map new vars as structures to their interpretation in the
%	Qu-Prolog abstract machine: X, x, #x, #X, @x.
%	The routines called by this routine have to be implemented in 'C'.

'$new_var'(Var, MetaNames, MetaVariables, NewVar) :-
    '$type_var'(Type, Var),
    (Type == persistent_meta_var ->
	arg(1, Var, V),
	get_persistent_name(V, MetaVariables, MetaNames, Name)
    ;
    Type == persistent_object_var ->
	arg(1, Var, Name)
    ;
	Name = Var
    ),
    '$new_var_type'(Type, Name, NewVar).

'$new_var_type'(dynamic_meta_var, Var, Var).
'$new_var_type'(dynamic_object_var, _Var, NewVar) :-
    '$new_dynamic_object_var'(NewVar).
'$new_var_type'(persistent_meta_var, Var, NewVar) :-
    '$new_persistent_meta_var'(Var, NewVar).
'$new_var_type'(persistent_object_var, Var, NewVar) :-
    '$new_persistent_object_var'(Var, NewVar).
*/
/*
 *  Local object variable is not allowed at inputs.
'$new_var_type'(local, _Var, NewVar) :-
    '$new_local'(NewVar).
*/

get_persistent_name(_, [], [], _) :-
    writeln(user_error, 'system error: unable to get persistent name');
    fail.
get_persistent_name(Var, [X|_], [NameL|_], Name) :-
    Var == X, !,
    name(Name, NameL).
get_persistent_name(Var, [_|MetaVariables], [_|MetaNames], Name) :-
    get_persistent_name(Var, MetaVariables, MetaNames, Name).

% '$map_vars_to_interpretation'(Vars, MapVars) :-
% 	Given a list of vars Vars, create a mapping to their
%	interpretation in the abstract machine (to be determined later).
'$map_vars_to_interpretation'([], []).
'$map_vars_to_interpretation'([Var|Vars],
			  [p(Var, _NewVar, _Interpreted)|MapVars]) :-
    '$map_vars_to_interpretation'(Vars, MapVars).

% '$map_vars_to_vars'(Vars, MapVars) :-
% 	Given a list of vars Vars, create a mapping to themselves
%	for use in the compiler.
'$map_vars_to_vars'([], []).
'$map_vars_to_vars'([Var|Vars], [p(Var, Var, true)|MapVars]) :-
    '$map_vars_to_vars'(Vars, MapVars).

/*
% '$map_vars_to_registers'(Mapping, X, NewMapping) :-
%  	CANNOT FIND WHERE IS USED
'$map_vars_to_registers'([], X, [MapX]) :-
    '$map_var_to_register'(X, MapX).
'$map_vars_to_registers'([p(Y, Register, true)|Mapping], X,
		[p(Y, Register, true)|Mapping]) :-
    Y == X, !.
'$map_vars_to_registers'([Map|Mapping], X, [Map|NewMapping]) :-
    '$map_vars_to_registers'(Mapping, X, NewMapping).

'$map_var_to_register'(Var, p(Var, Var, true)) :-
    var(Var), !.
'$map_var_to_register'(Var, p(Var, _Register, true)).
*/
    

% '$execute_distincts'(Distincts) :-
%	Call each of the goals in Distincts.
%	Elements of Distincts are of the form (x/y)/(x' distinct_from y')
%	where x' and y' are the interpretations of x and y respectively.

'$execute_distincts'([]).
'$execute_distincts'([_Identifier/Distinct|Distincts]) :-
    call(Distinct),
    '$execute_distincts'(Distincts).

'$add_to_vars'(X, Vars, Vars):-
    '$member2'(X, Vars), !.
'$add_to_vars'(X, Vars, [X|Vars]).

'$member2'(X, [Y|_Ys]) :-
    X == Y, !.
'$member2'(X, [_Y|Ys]) :-
    '$member2'(X, Ys).

'$select_sub_class'([], _ClassType, []).
'$select_sub_class'([p(Var, NewVar, InterpretedVar)|Variables], ClassType,
	          [p(Var, NewVar, InterpretedVar)|SubClass]) :-
    '$type_var'(ClassType, Var), !,
    '$select_sub_class'(Variables, ClassType, SubClass).
'$select_sub_class'([_VariableMap|Variables], ClassType, SubClass) :-
    '$select_sub_class'(Variables, ClassType, SubClass).

'$select_sub_class2'([], _ClassType, []).
'$select_sub_class2'([Var|Variables], ClassType, [Var|SubClass]) :-
    '$type_var'(ClassType, Var), !,
    '$select_sub_class2'(Variables, ClassType, SubClass).
'$select_sub_class2'([_Var|Variables], ClassType, SubClass) :-
    '$select_sub_class2'(Variables, ClassType, SubClass).

% '$type_var'(Type, Var) :-
%	Note: when Type is output, Var is input, this arrangement will
%	give the most specialised type, e.g. for X will give
%	dynamic_meta_var rather than meta_var.
%	This routine is usually use in two modes ++, -+
'$type_var'(Type, Var) :-
    var(Var), !, once((Type = dynamic_meta_var ; Type = meta_var)).
'$type_var'(Type, Var) :-
    Var =.. [F|Args],
    once('$type_var'(Type, F, Args)).

'$type_var'(persistent_meta_var, F, Args) :-
    '$isa_persistent_meta_var'(F, Args).
'$type_var'(dynamic_object_var, F, Args) :-
    '$isa_dynamic_object_var'(F, Args).
'$type_var'(persistent_object_var, F, Args) :-
    '$isa_persistent_object_var'(F, Args).
'$type_var'(local, F, Args) :-
    '$isa_local'(F, Args).
'$type_var'(meta_var, F, Args) :-
    '$isa_meta_var'(F, Args).
'$type_var'(object_var, F, Args) :-
    '$isa_object_var'(F, Args).
'$type_var'(persistent_var, F, Args) :-
    '$isa_persistent_var'(F, Args).

% '$isa_var'(Var) :-
%	True, iff Var is one of the classes of var: X, x, #X, #x, @x 
%
%			var
%		      /     |     \
%		     /      |      \
%		    /	    |       \
%		   /        |        \
%		  /         |         \
%	    persistent    object      meta
%           var 	  var         var
%	     /       \     /  \  \     ||
%	    +         +   /    \  \    ||
%	    |         |  /      \  +-- || ----------+
%	    |         | +	 +---- || -+	    |
%	    | +------ |-|--------------+|  |        |
%	    | |	      | |         +-----+  |	    |
%	    | |       | |         |        |	    |
%	persistent  persistent  dynamic  dynamic  local
%	meta_var    object_var meta_var object_var
%	 (#X)        (#x)        (X)      (x)      (@x)
%
'$isa_var'(X) :-
    var(X), !.
'$isa_var'(X) :-
    X =.. [F|Args],
    once('$isa_var'(F, Args)).

'$isa_var'(F, Args) :-
    '$isa_meta_var'(F, Args), !.
'$isa_var'(F, Args) :-
    '$isa_object_var'(F, Args), !.
'$isa_var'(F, Args) :-
    '$isa_persistent_var'(F, Args), !.



'$isa_meta_var'(X) :-
    var(X), !.
'$isa_meta_var'(Term) :-
    Term =.. [F|Args],
    '$isa_meta_var'(F, Args).

'$isa_object_var'(X) :-
    var(X), !, fail.
'$isa_object_var'(Term) :-
    Term =.. [F|Args],
    '$isa_object_var'(F, Args).

'$isa_persistent_var'(X) :-
    var(X), !, fail.
'$isa_persistent_var'(Term) :-
    Term =.. [F|Args],
    '$isa_persistent_var'(F, Args).

'$isa_unpersistent_var'(X) :-
    var(X), !.
'$isa_unpersistent_var'(X) :-
    '$isa_dynamic_object_var'(X), !.
'$isa_unpersistent_var'(X) :-
    '$isa_local'(X), !.

'$isa_persistent_meta_var'(X) :-
    var(X), !, fail.
'$isa_persistent_meta_var'(Term) :-
    Term =.. [F|Args],
    '$isa_persistent_meta_var'(F, Args).

'$isa_dynamic_object_var'(X) :-
    var(X), !, fail.
'$isa_dynamic_object_var'(Term) :-
    Term =.. [F|Args],
    '$isa_dynamic_object_var'(F, Args).

'$isa_persistent_object_var'(X) :-
    var(X), !, fail.
'$isa_persistent_object_var'(Term) :-
    Term =.. [F|Args],
    '$isa_persistent_object_var'(F, Args).

'$isa_local'(X) :-
    var(X), !, fail.
'$isa_local'(Term) :-
    Term =.. [F|Args],
    '$isa_local'(F, Args).

'$isa_meta_var'(F, Args) :-		% only handles case #X
    '$isa_persistent_meta_var'(F, Args).

'$isa_object_var'(F, Args) :-
    '$isa_dynamic_object_var'(F, Args), !.
'$isa_object_var'(F, Args) :-
    '$isa_persistent_object_var'(F, Args), !.
'$isa_object_var'(F, Args) :-
    '$isa_local'(F, Args), !.

'$isa_persistent_var'(F, Args) :-
    '$isa_persistent_meta_var'(F, Args), !.
'$isa_persistent_var'(F, Args) :-
    '$isa_persistent_object_var'(F, Args), !.

'$isa_persistent_meta_var'((#), [X]) :-
    var(X).

'$isa_persistent_object_var'((#), [X]) :-
    atom(X),
    '$object_var_prefix_declared'(X).

'$isa_dynamic_object_var'(X, []) :-
    atom(X),
    '$object_var_prefix_declared'(X).

'$isa_local'((@), [X]) :-
    atom(X),
    '$object_var_prefix_declared'(X).

/*
%'$create_all_distincts'(PSubs, DynamicOVs, DistinctsIn, DistinctsOut),
%
%	Note: this seems to make all of the object variables apart 
%	regardless of information about parallel subs. This was needed
%	historically because could have set a flag full_separation/1.
%	full_separation(on) had the affect that all object-variables in
%	a clause are deemed to be apart.
%	full_separtion(off) only object-variables in parallel subs are
%	deemed distinct.
%
% 		USED IN misc_ext_int.ql, which is not included anywhere

'$create_all_distincts'(PSubs, DynamicOVs, DistinctsIn, DistinctsOut) :-
    '$create_all_distincts2'([DynamicOVs|PSubs], DistinctsIn,
	DistinctsOut).

'$create_all_distincts2'([], DistinctsIn, DistinctsIn).
'$create_all_distincts2'([ParallelSub|ParallelSubs], DistinctsIn, DistinctsOut):-
    '$create_distincts'(ParallelSub, DistinctsIn, DistinctsTmp),
    '$create_all_distincts2'(ParallelSubs, DistinctsTmp, DistinctsOut).
*/

'$create_distincts'([], DistinctsIn, DistinctsIn).
'$create_distincts'([X|ParallelSub], DistinctsIn, DistinctsOut) :-
    '$create_xs_distincts'(ParallelSub, X, DistinctsIn, DistinctsTmp),
    '$create_distincts'(ParallelSub, DistinctsTmp, DistinctsOut).

'$create_xs_distincts'([], _X, DistinctsIn, DistinctsIn).
'$create_xs_distincts'([Y|ObjectVars], X, DistinctsIn, DistinctsOut) :-
    '$add_distinct'(X, Y, DistinctsIn, DistinctsTmp),
    '$create_xs_distincts'(ObjectVars, X, DistinctsTmp, DistinctsOut).

'$add_distinct'(p(X, NewX, _InterpretedX),
	      p(Y, NewY, _InterpretedY), DistinctsIn, DistinctsOut) :-
    (member((X/Y) / distinct_from(NewX, NewY), DistinctsIn) ->
	DistinctsOut = DistinctsIn
    ;
	DistinctsOut = [(X/Y) / distinct_from(NewX, NewY)|DistinctsIn]
    ).

'$quantifier'(F) :-
    '$qup_op'(_Prec, quant, F).

'$collect_qu_names_and_vars'([], _, _, [], []).
'$collect_qu_names_and_vars'([p(Name, Var, _)|MapVars],
			   MetaNames, MetaVariables,
			   [RealName|QuNames], [Var|QuVariables]) :-
    '$type_var'(Type, Name),
    '$get_real_name'(Type, Name, MetaVariables, MetaNames, RealName),
    '$collect_qu_names_and_vars'(MapVars, MetaNames, MetaVariables, QuNames,
				QuVariables), !.
'$collect_qu_names_and_vars'([_|MapVars], MetaNames, MetaVariables,
			   QuNames, QuVariables) :-
    '$collect_qu_names_and_vars'(MapVars, MetaNames, MetaVariables, QuNames,
				QuVariables).

'$get_real_name'(dynamic_meta_var, Name, MetaVars, MetaNames, RealVar) :-
    '$convert_name'(MetaVars, MetaNames, Name, RealVar).
'$get_real_name'(dynamic_object_var, Name, _, _, RealVar) :-
    name(Name, RealVar).
'$get_real_name'(persistent_meta_var, #(Name), MetaVars, MetaNames,
		[0'#|RealVar]) :-
    '$convert_name'(MetaVars, MetaNames, Name, RealVar).
'$get_real_name'(persistent_object_var, #(Name), _, _, [0'#|RealVar]) :-
    name(Name, RealVar).
'$get_real_name'(local, @(Name), _, _, [0'@|RealVar]) :-
    name(Name, RealVar).

'$convert_name'([X|_MetaVars], [RealVar|_MetaNames], Name, RealVar) :-
    X == Name, !.
'$convert_name'([_|MetaVars], [_|MetaNames], Name, RealVar) :-
    '$convert_name'(MetaVars, MetaNames, Name, RealVar).
/*
% '$set_persistent_bases'(_Persistents). % to be used by new_persistent
*/
