/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * peephole - Perform code optimisations on the QuAM assembly code.
 */

/*----------------------------------------------------------------------------
peephole(Is, EnvSize, Is2) :-
    Is2 is the optimised or cleaned up version of Is.

    Is		Instructions implementing the clause.
    EnvSize	The size of the environment (number of permanent registers in
		the clause.
    Is2		Optimized code.
----------------------------------------------------------------------------*/
peephole([_Head|Is], EnvSize, Is9) :-
    % write_allocations([_Head|Is], _),
	/* expand_substitution_instrs/2 and replace_empty_sub_instr/2
	 * must be together at the beginning of peephole */
    expand_substitution_instrs(Is, Is11),
    replace_empty_sub_instr(Is11, Is12),
    expand_instructions(Is12, Is1),
    optimize_get_puts(Is1, Is2),
    %write_allocations([_Head|Is2], _),
    optimize_put_puts(Is2, Is3),
    %write_allocations([_Head|Is3], _),
    optimize_get_y_puts(Is3, Is4),
    %write_allocations([_Head|Is4], _),
    optimize_instructions(Is4, Is13),
    %write_allocations([_Head|Is13], _),
    make_sub_before_build(Is13, Is5),
    trim_environment_instructions(Is5),
    insert_wakeup_call(Is5, Is10),
    last_call_optimisation(Is10, Is6),
    allocate_and_deallocate_env(EnvSize, Is6, Is7),
    nils_and_cons(Is7, Is8),
    clean_instructions(Is8, Is9).

/*----------------------------------------------------------------------------
expand_substitution_instrs(Is, Is1) :-
    Expands out the substitution instructions in the head matching and
    the setup for the first goal, removing any empty substitutions
    following "put_unsafe_value" instructions after the first goal.  For
    each "get_variable" instruction before the first call that is followed
    by a "put_value" instruction, and rX(I) has not been destroyed by any
    instructions between these two instructions, a "put_substitution"
    instruction is inserted after the "put_value", possible replacing
    an empty substitution.  Note that "eMTPY" indicates that the X and XS
    registers in an argument have not been "bound together", and so a
    "put_substitution_operator" instruction is generated to do so.  This
    avoids the XS register being overwritten.
----------------------------------------------------------------------------*/
expand_substitution_instrs(Is, Is1) :-
	append(Front, [call_predicate(F, N, EnvSize)|Back], Is),
	!,
	expand_substitution_instr(Front, Is2),
	remove_unsafe_substitution(Back, Back1),
	append(Is2, [call_predicate(F, N, EnvSize)|Back1], Is1).
expand_substitution_instrs(Is, Is1):-
	expand_substitution_instr(Is, Is1).

expand_substitution_instr(Is, Is1) :-
	append(Front, [get(variable, I, J)|Back], Is),
	x_register(I),
	!,
	(append(Front2, [put(value, I, K)|Back2], Back),
	 \+ (member(Instr, Front2),
	     destroys_register(Instr, I)) ->
		(Back2 = [put_substitution(eMPTY, K)|Back3] ->
		     append(Front2, [put_substitution_operator(I, K),
				put_substitution(empty, K)|Back3], Back4)
		;
		     once((Back2 = [put_substitution(empty, K)|Back3]
			   ;
		     	   Back2 = Back3
			  )),
		     append(Front2, [put(value, I, K),
				put_substitution(I, K)|Back3], Back4)
		)
	;
		Back4 = Back
	),
	expand_substitution_instr(Back4, Is2),
	append(Front, [get(variable, I, J)|Is2], Is1).
expand_substitution_instr(Is, Is).

remove_unsafe_substitution([], []).
remove_unsafe_substitution([put(unsafe_value, Y, X),
			    put_substitution(empty, X)|Back],
			    		[put(unsafe_value, Y, X)|Back1]) :-
	remove_unsafe_substitution(Back, Back1).
remove_unsafe_substitution([I|Back], [I|Back1]) :-
	remove_unsafe_substitution(Back, Back1).

/*----------------------------------------------------------------------------
replace_empty_sub_instr(Is, Is1) :-
    Replace any remaining "eMTPY" substitution types with "empty" after the
    processing performed by "expand_substitution_instrs" has finished.
----------------------------------------------------------------------------*/
replace_empty_sub_instr([], []).
replace_empty_sub_instr([put_substitution(eMPTY, K)|Is],
			[put_substitution(empty, K)|Is1]) :-
	replace_empty_sub_instr(Is, Is1).
replace_empty_sub_instr([I|Is], [I|Is1]) :-
	replace_empty_sub_instr(Is, Is1).

/*----------------------------------------------------------------------------
make_sub_before_build(Is12, Is13) :-
    Before a term is used in building a complex term (structure or
    quantifier), group the substitution and the term together.
    This is needed only in situation like:
	p(f(A, B), R) :- R = [A,B].
    where A and B cover a term with substitution.
    The situation applies up to the first goal.
----------------------------------------------------------------------------*/
make_sub_before_build(Is1, Is) :-
	append(Front1, [call_predicate(F, N, EnvSize)|Back], Is1),
	!,
	insert_sub_op_instr(Front1, [], Front2),
	append(Front2, [call_predicate(F, N, EnvSize)|Back], Is).
make_sub_before_build(Is1, Is):-
	insert_sub_op_instr(Is1, [], Is).

insert_sub_op_instr([], Is, Is).
insert_sub_op_instr(Is1, Back1, Back) :-
	break_up_list(Is1, Front, I),
	((	extract_problem_reg_from_put_apply_or_quant(I, Reg),
	  	\+ inst_destroy_reg_by_put(Front, Reg)
	 )
	->
		/* Insert a "put_substitution_operator" instruction. */
		Back2 = [put_substitution_operator(Reg, Reg), I|Back1]
	;
		Back2 = [I|Back1]
	),
	insert_sub_op_instr(Front, Back2, Back).

break_up_list([I], [], I).
break_up_list([I|Is1], [I|Is], Back) :-
	break_up_list(Is1, Is, Back).

extract_problem_reg_from_put_apply_or_quant(put_apply(_, R, _), R).
extract_problem_reg_from_put_apply_or_quant(put_quant(_, _, R), R).

inst_destroy_reg_by_put(Instr, Reg) :-
	member(I, Instr),
	destroys_register_by_put(I, Reg),
	!.

/*----------------------------------------------------------------------------
expand_instructions(Is, Is1) :-
    Expands out "put_object" and "get_quantifier" instructions to their
    more complete QuAM equivalents.
----------------------------------------------------------------------------*/
expand_instructions([], []).
expand_instructions([I|Is1], NI):-
    expand_instruction(I, NI-Is2),
    expand_instructions(Is1, Is2).

expand_instruction(put_object(value, I, J), [put_object(value, I, J)|Is]-Is):-
    !.
expand_instruction(put_object(unsafe_value, I, J), [put_object(value, I, J)|Is]-Is):-
    !.
expand_instruction(put_object(variable, rY(I), rX(J)),
	[put_object(variable, rY(I), rY(I)), put_object(value, rY(I), rX(J))|Is]-Is):-
    !.
expand_instruction(put_object(variable, rX(I), rX(J)),
	[put_object(variable, rX(J), rX(J)), put_object(value, rX(J), rX(I))|Is]-Is):-
    I \== J,
    !.
expand_instruction(get_quantifier(I, J, K),
	[not_free_in(I, K), get_quantifier(I, J, K)|Is]-Is):-
    !.
expand_instruction(I, [I|Is]-Is).

/*----------------------------------------------------------------------------
optimize_get_puts(Is, Is2) :-
    remove matching get_variable A B, and put_value A B (maybe multiple).

    if there are no puts at all then do not remove get_variable A B
	if instruction is get_variable A B
	remove all put_value A B instructions (up to the first call) that have
	nothing destroying B before it.
	if there are no put_value A DontCare instructions (up to the first
	    call) that do not have an instruction that modifies A preceding
	    it then remove get_variable A B
----------------------------------------------------------------------------*/
optimize_get_puts([], []).
optimize_get_puts([get(variable, A, X)|Is], Is4) :-
    x_register(A),
    !,
    once((append(Front, [call_predicate(_F, _N, _EnvSize)|_Back], Is);
	  Front = Is)),
    (member(put(value, A, _Destination), Front) ->
	remove_redundant_put_values(put, A, X, Is, Is2),
	(remove_redundant_get_variable(A, Is2) ->
	    Is4 = Is3
	;
	    Is4 = [get(variable, A, X)|Is3]
	)
    ;
	Is4 = [get(variable, A, X)|Is3],
	Is2 = Is
    ),
    optimize_get_puts(Is2, Is3).
optimize_get_puts([get_object(variable, A, X)|Is], Is4) :-
    x_register(A),
    !,
    once((append(Front, [call_predicate(_F, _N, _EnvSize)|_Back], Is);
	  Front = Is)),
    (member(put(value, A, _Destination), Front) ->
	remove_redundant_put_values(put, A, X, Is, Is2),
	Is4 = [get_object(variable, A, X)|Is3]
    ;
	Is4 = [get_object(variable, A, X)|Is3],
	Is2 = Is
    ),
    optimize_get_puts(Is2, Is3).
optimize_get_puts([I|Is], [I|Is2]) :-
    optimize_get_puts(Is, Is2).

/*----------------------------------------------------------------------------
remove_redundant_put_values(Opcode, A, X, Is, Is2) :-
    remove a put instruction that has no register movement i.e.
	get_variable	X2 X1
	put_value	X2 X1

    but only if there are no calls or instructions which clobber the
    destination  in the put_value instruction.
	get_variable	X2 X1
	call ...
	put_value	X2 X1	%can't remove this instruction

	get_variable	X2 X1
	put_value	X0 X1
	escape ...
	put_value	X2 X1	%can't remove this instruction

----------------------------------------------------------------------------*/
remove_redundant_put_values(Opcode, A, X, Is, Is2) :-
    (append(Front, [call_predicate(F, N, EnvSize)|Back], Is) ->
	remove_redundant_puts(Opcode, A, X, Front, List),
	append(List, [call_predicate(F, N, EnvSize)|Back], Is2)
    ;
	remove_redundant_puts(Opcode, A, X, Is, Is2)
    ), !.
remove_redundant_put_values(_Opcode, _A, _X, Is, Is).

/*----------------------------------------------------------------------------
remove_redundant_puts(Opcode, A, X, Is, Is2) :-
    The list of instructions Is contains a redundant put_value instruction
    which puts the value A into register X, which already contains A. Is2 is
    this list of instructions with this redundant instruction removed.

    Note: we want to remove multiple copies of this same instruction if
	    they are all redundant.
    Fails if there are no redundant put instructions.
----------------------------------------------------------------------------*/
remove_redundant_puts(Opcode, A, X, Is, Is3) :-
    remove_redundant_put(Opcode, A, X, Is, Is2),
    (
	remove_redundant_puts(Opcode, A, X, Is2, Is3)
    ;
	Is3 = Is2
    ), !.


/*----------------------------------------------------------------------------
remove_redundant_put(Opcode, A, X, Is, Is2) :-
----------------------------------------------------------------------------*/
remove_redundant_put(Opcode, A, X, Is, Is2) :-
    Inst =.. [Opcode, value, A, X],
    append(Front, [Inst|Back], Is),
    \+ (member(Instruction, Front), 
	(destroys_register(Instruction, X);
	 destroys_register(Instruction, A))),
    (Back = [put_substitution(_, X)|Back2]
    ;
     Back2 = Back
    ),
    append(Front, Back2, Is2).

/*----------------------------------------------------------------------------
destroys_register(Instruction, X) :-
    true if the Instruction puts anything into register X.

    for get		check everyone except last
	put_variable	both
	put		only last
----------------------------------------------------------------------------*/
destroys_register(Instruction, X) :- destroys_register_by_put(Instruction, X).
destroys_register(Instruction, X) :- destroys_register_by_get(Instruction, X).

destroys_register_by_put(put_constant(_Constant, X), X).
destroys_register_by_put(put_apply(_Functor, _Argument, X), X).
destroys_register_by_put(put_quantifier(_ObjVar, _Term, X), X).
destroys_register_by_put(put(variable, _Source, X), X) :- !.
destroys_register_by_put(put(variable, X, _Destination), X) :- !.
destroys_register_by_put(put(_Type, _Source, X), X).
destroys_register_by_put(put_object(variable, _Source, X), X) :- !.
destroys_register_by_put(put_object(variable, X, _Destination), X) :- !.
destroys_register_by_put(put_object(_Type, _Source, X), X).

destroys_register_by_get(get_apply(X, _Argument, _Destination), X) :- !.
destroys_register_by_get(get_apply(_Functor, X, _Destination), X).
destroys_register_by_get(get_quantifier(_ObjVar, X, _Destination), X).
destroys_register_by_get(get(_Type, X, _Destination), X).
destroys_register_by_get(get_object(_Type, X, _Destination), X).

/*----------------------------------------------------------------------------
remove_redundant_get_variable(A, Is) :-
    remove the get_variable A B instruction if there are no put_value A C
    or get_value A D instructions before the next call in Is.
----------------------------------------------------------------------------*/
remove_redundant_get_variable(A, Is) :-
    once((
	    append(Front, [call_predicate(_F, _N, _EnvSize)|_Back], Is)
	;
	    Front = Is
	)),
    \+ once((member(put(value, A, _DontCare), Front);
	    member(get(value, A, _DontCare), Front))).


/*----------------------------------------------------------------------------
optimize_put_puts(Is, Is2) :-
    remove the matching put_value to put_variable A B
    (put_value A B (maybe multiple)).

    if instruction is put_variable A B
    remove all put_value A B instructions (up to the first call) that have
    nothing destroying A or B before it.
----------------------------------------------------------------------------*/
optimize_put_puts([], []).
optimize_put_puts([put(variable, A, X)|Is], [put(variable, A, X)|Is3]) :-
    x_register(A),
    !,
    remove_redundant_put_values(put, A, X, Is, Is2),
    optimize_put_puts(Is2, Is3).
optimize_put_puts([put_object(variable, A, X)|Is],
		  [put_object(variable, A, X)|Is3]) :-
    x_register(A),
    !,
    remove_redundant_put_values(put_object, A, X, Is, Is2),
    optimize_put_puts(Is2, Is3).
optimize_put_puts([I|Is], [I|Is2]) :-
    optimize_put_puts(Is, Is2).

/*----------------------------------------------------------------------------
optimize_get_y_puts(Is, Is2) :-
    remove the matching put_y_value to get_y_variable A B
    (put_y_value A B (maybe multiple)).

    if instruction is get_y_variable A B
    remove all put_y_value A B instructions (up to the first call) that have
    nothing destroying B before it.
----------------------------------------------------------------------------*/
optimize_get_y_puts([], []).
optimize_get_y_puts([get(variable, A, X)|Is], [get(variable, A, X)|Is3]) :-
    y_register(A),
    !,
    remove_redundant_put_values(put, A, X, Is, Is2),
    optimize_get_y_puts(Is2, Is3).
optimize_get_y_puts([get_object(variable, A, X)|Is],
		[get_object(variable, A, X)|Is3]) :-
    y_register(A),
    !,
    remove_redundant_put_values(put, A, X, Is, Is2),
    optimize_get_y_puts(Is2, Is3).
optimize_get_y_puts([I|Is], [I|Is2]) :-
    optimize_get_y_puts(Is, Is2).

/*----------------------------------------------------------------------------
optimize_instructions(Is, Is2) :-
    Remove no-op instructions from the list Is to give Is2.  Remove any
    non-empty substitutions that follow the no-op instructions.
----------------------------------------------------------------------------*/
optimize_instructions([], []).
optimize_instructions([I|Is], Is2) :-
    optimize_instruction(I), !,
    (Is = [put_substitution(Type, _)|Is1],
     Type \== empty
    ;
     Is = Is1
    ),
    optimize_instructions(Is1, Is2).
optimize_instructions([I|Is], [I|Is2]) :-
    optimize_instructions(Is, Is2).

/*----------------------------------------------------------------------------
optimize_instruction(I) :-
    The instruction I can be removed if its source is the same as its
    destination for get_variable, get_value, get_object, put_value and
    put_object, since these are noops.
----------------------------------------------------------------------------*/
optimize_instruction(get(variable, X, X)).
optimize_instruction(get(value, X, X)).
optimize_instruction(get_object(value, X, X)).
optimize_instruction(put(value, X, X)).
optimize_instruction(put_object(value, X, X)).

/*----------------------------------------------------------------------------
trim_environment_instructions(Is) :-
    Trim the environment back as permanent registers are no longer used.
    The permanent registers are numbered N ... 0 where N is the next 
    permanent to become dead (unused) and 0 is the last.

    The trimming is done by adding the parameter, current environment size,
    to the call instruction, which will trim the environment back to this
    value.

    Is		Instructions
----------------------------------------------------------------------------*/
trim_environment_instructions([]).
trim_environment_instructions([I|Is]) :-
    trim_environment(I, Is),
    trim_environment_instructions(Is).


/*----------------------------------------------------------------------------
trim_environment(I, Is) :-
    The environment is trimmed by seeing which permanent registers are still
    in use after the current instruction, the largest permanent register is
    the size of the environment.

    Only calculate the environment size for call instructions, since they do
    the trimming.

    I		current instruction
    Is		instructions after the current
----------------------------------------------------------------------------*/
trim_environment(call_predicate(_F, _N, EnvSize), Is) :-
    !,
    largest_permanent(Is, 0, EnvSize).
trim_environment(_I, _Is).


/*----------------------------------------------------------------------------
largest_permanent(Instructions, Size, Size2) :-
----------------------------------------------------------------------------*/
largest_permanent([], Size, Size).
largest_permanent([Instruction|Instructions], Size, Size3) :-
    largest_permanent_instruction(Instruction, Size, Size2),
    largest_permanent(Instructions, Size2, Size3).


/*----------------------------------------------------------------------------
largest_permanent_instruction(Instruction, Size, Size2) :-
----------------------------------------------------------------------------*/
largest_permanent_instruction(fail, Size, Size) :-
    !.
largest_permanent_instruction(cut, Size, Size) :-
    !.
largest_permanent_instruction(escape(_F, _N), Size, Size) :-
    !.
largest_permanent_instruction(call_predicate(_F, _N, _EnvSize), Size, Size) :-
    !.
largest_permanent_instruction(put_constant(_A, _X), Size, Size) :-
    !.
largest_permanent_instruction(get_constant(_A, _X), Size, Size) :-
    !.
largest_permanent_instruction(Instruction, Size, Size2) :-
    Instruction =.. [_PutorGet, _Type|Regs],
    largest_permanent_in_regs(Regs, Size, Size2).


/*----------------------------------------------------------------------------
largest_permanent_in_regs(Regs, Size, Size2) :-
----------------------------------------------------------------------------*/
largest_permanent_in_regs([], Size, Size).
largest_permanent_in_regs([Reg|Regs], Size, Size3) :-
    largest_permanent_in_reg(Reg, Size, Size2),
    largest_permanent_in_regs(Regs, Size2, Size3).


/*----------------------------------------------------------------------------
largest_permanent_in_reg(Reg, Size, Size2) :-
----------------------------------------------------------------------------*/
largest_permanent_in_reg(Reg, Size, Size2) :-
    (y_register(N, Reg) ->
	M is N + 1,
	(M > Size ->
	    Size2 = M
	;
	    Size2 = Size
	)
    ;
	Size2 = Size
    ).


/*----------------------------------------------------------------------------
insert_wakeup_call(Is, Is2) :-
    Inserts "do_delayed_problems" instructions into the QuAM code at
    various points.
----------------------------------------------------------------------------*/
insert_wakeup_call(Is, Is2):-
    append(Front, [call_predicate(F, N, E)|Is1], Is),
    F/N \== (distinct_from)/2,
    !,
    (member(X, Front),	% variables are bound "get"
     functor(X, Opcode, _N),
     get_instruction(Opcode),
     append(Front, [do_delayed_problems, call_predicate(F, N, E)|Is1], Is2)
    ;
     Is2 = Is
    ).
insert_wakeup_call(Is, Is):-
    append(_Front, [fail], Is),
    !.
insert_wakeup_call(Is, Is2):-
    !,			% variables are bound 'escape' predicates
    append(Is, [do_delayed_problems], Is2).

get_instruction(get_constant).
get_instruction(get_apply).
get_instruction(get_quantifier).
get_instruction(get).
get_instruction(get_object).

/*----------------------------------------------------------------------------
last_call_optimisation(Is, Is2) :-
    Get the last instruction from the list of instructions for the clause
    Is, and optimize (call -> execute, fail -> fail, otherwize instruction
    -> instruction + proceed).

    Is		The list of instructions
    Is2		Is after optimising last instruction.
----------------------------------------------------------------------------*/
last_call_optimisation([], [proceed]) :-
    !.
last_call_optimisation(Is, Is2) :-
    append(FrontIs, [LastInstruction], Is),
    optimize_last_instruction(LastInstruction, LastInstructions),
    append(FrontIs, LastInstructions, Is2).


/*----------------------------------------------------------------------------
optimize_last_instruction(LastInstruction, LastInstructions) :-
    If the last instruction is a call replace it with an execute
    instruction (can deallocate the environment).
    else if a fail instruction leave as is
    else add a proceed instruction to the end.

    LastInstruction	The last instruction in the clause
    LastInstructions	The new list of last instructions in the clause.

    ???????????????????????????????????????????????????????????????????
    ???????????????????????????????????????????????????????????????????
	Needs more work, need to go backwards to find the last
	call_predicate and change to an execute -> affecting allocate and 
	deallocate.

	The real problem is  allocate and deallocate, cant change

	    call
	    cut 

	to

	    execute
	    cut
	since execute won't come back
    ???????????????????????????????????????????????????????????????????
    ???????????????????????????????????????????????????????????????????
    ???????????????????????????????????????????????????????????????????
----------------------------------------------------------------------------*/
optimize_last_instruction(call_predicate(F, N, _EnvSize),
	[execute_predicate(F, N)]):- !.
optimize_last_instruction(fail, [fail]) :- !.
optimize_last_instruction(LastInstruction, [LastInstruction, proceed]).


/*----------------------------------------------------------------------------
allocate_and_deallocate_env(EnvSize, Is, Is2) :-
    Allocate an environment if there is more than one goal (since instructions
    have had last call optimization, then if there is a call instruction means
    that there are at least two calls (one is an execute)).
----------------------------------------------------------------------------*/
allocate_and_deallocate_env(EnvSize, Is, Is3) :-
    ((member(call_predicate(_F, _N, _NumPermanent), Is)
     ;
      member(put(_Type, rY(_N), _Dest), Is)
     ;
      member(put_object(_Type, rY(_N), _Dest), Is)
     ; 
      /* This extra disj included for signal handling 27-11-91 */
      member(escape(_E, _N), Is)
     ) ->
	allocate_env(EnvSize, Is, Is2),
	append(Front, [LastInstruction], Is2),
	append(Front, [deallocate, LastInstruction], Is3)
    ;
	Is3 = Is
    ), !.


/*----------------------------------------------------------------------------
allocate_env(EnvSize, Is, Is2) :-
    Force the allocate down as far as possible, that is, do not allocate
    until needed. Either before the first use of a permanent variable or the
    first call.
----------------------------------------------------------------------------*/
allocate_env(EnvSize, [I|Is], [allocate(EnvSize), I|Is]) :-
    instruction_uses_env(I), !.
allocate_env(EnvSize, [I|Is], [I|Is2]) :-
    allocate_env(EnvSize, Is, Is2).


/*----------------------------------------------------------------------------
instruction_uses_env(I) :-
    True, if the instruction I uses a permanent variable or is a call (since
    we already know that the predicate contains more than one call).
----------------------------------------------------------------------------*/
instruction_uses_env(call_predicate(_F, _N, _Env)) :- !.
instruction_uses_env(escape(_F, _N)) :- !.
instruction_uses_env(get_constant(_A, Y)) :-
    !,
    y_register(Y).
instruction_uses_env(put_constant(_A, Y)) :-
    !,
    y_register(Y).
instruction_uses_env(Instruction) :-
    Instruction =.. [_Instr, _Type|Regs],
    member(Y, Regs),
    y_register(Y).


/*----------------------------------------------------------------------------
nils_and_cons(Is, Is2) :-
    Is2 is the list of instructions Is replacing all occurences of put
    and get '.' and '[]' with cons and nils respectively.
----------------------------------------------------------------------------*/
nils_and_cons([], []).
nils_and_cons([I|Is], [I2|Is2]) :-
    nil_and_con(I, I2),
    nils_and_cons(Is, Is2).

/*----------------------------------------------------------------------------
nil_and_con(Is, Is2) :-
    Is is the instruction I replacing all occurences of put and get
    '.' and '[]' with cons and nils respectively.
----------------------------------------------------------------------------*/
nil_and_con(put_constant(., X), put_cons(X)) :-
    !.
nil_and_con(put_constant([], X), put_nil(X)) :-
    !.
nil_and_con(get_constant(., X), get_cons(X)) :-
    !.
nil_and_con(get_constant([], X), get_nil(X)) :-
    !.
nil_and_con(Is, Is).

/*----------------------------------------------------------------------------
clean_instructions(Is, Is2) :-
    Is2 is Is with all registers converted from rX(n) to n, rY(n) to n, and
    put_value to put_x_value or put_y_value etc.  This generates the
    final QuAM code for the clause to be written to the ".qs" file.
----------------------------------------------------------------------------*/
clean_instructions([], []).
clean_instructions([I|Is], [I2|Is2]) :-
    clean_instruction(I, I2),
    clean_instructions(Is, Is2).


/*----------------------------------------------------------------------------
clean_instruction(I, I2) :-
    I2 is I with all registers converted from rX(n) to n, rY(n) to n, and
    put_value to put_x_value or put_y_value etc.
----------------------------------------------------------------------------*/
clean_instruction(put(variable, rX(X1), rX(X2)),
		  put_x_variable(X1, X2)) :-
    !.
clean_instruction(put(variable, rY(X1), rX(X2)),
		  put_y_variable(X1, X2)) :-
    !.
clean_instruction(put(value, rX(X1), rX(X2)),
		  put_x_value(X1, X2)) :-
    !.
clean_instruction(put(value, rY(X1), rX(X2)),
		  put_y_value(X1, X2)) :-
    !.
clean_instruction(put(unsafe_value, rY(X1), rX(X2)),
		  put_unsafe_value(X1, X2)) :-
    !.
clean_instruction(put_object(variable, rX(X1), rX(X1)),
		  put_x_object_variable(X1)) :-
    !.
clean_instruction(put_object(variable, rY(X1), rY(X1)),
		  put_y_object_variable(X1)) :-
    !.
clean_instruction(put_object(value, rX(X1), rX(X2)),
		  put_x_object_value(X1, X2)) :-
    !.
clean_instruction(put_object(value, rY(X1), rX(X2)),
		  put_y_object_value(X1, X2)) :-
    !.
clean_instruction(put_substitution(empty, rX(X1)),
		  put_empty_substitution(X1)) :-
    !.
clean_instruction(get(variable, rX(X1), rX(X2)),
		  get_x_variable(X1, X2)) :-
    !.
clean_instruction(get(variable, rY(X1), rX(X2)),
		  get_y_variable(X1, X2)) :-
    !.
clean_instruction(get(value, rX(X1), rX(X2)),
		  get_x_value(X1, X2)) :-
    !.
clean_instruction(get(value, rY(X1), rX(X2)),
		  get_y_value(X1, X2)) :-
    !.
clean_instruction(get_object(variable, rX(X1), rX(X2)),
		  get_x_object_variable(X1, X2)) :-
    !.
clean_instruction(get_object(variable, rY(X1), rX(X2)),
		  get_y_object_variable(X1, X2)) :-
    !.
clean_instruction(get_object(value, rX(X1), rX(X2)),
		  get_x_object_value(X1, X2)) :-
    !.
clean_instruction(get_object(value, rY(X1), rX(X2)),
		  get_y_object_value(X1, X2)) :-
    !.
clean_instruction(I, I2) :-
    I =.. [Type|Args],
    remove_register_functors(Args, NewArgs),
    I2 =.. [Type|NewArgs].


/*----------------------------------------------------------------------------
remove_register_functors(Args, NewArgs) :-
    NewArgs is Args with all registers converted from rX(n) to n, rY(n).
----------------------------------------------------------------------------*/
remove_register_functors([], []).
remove_register_functors([Arg|Args], [N|NewArgs]) :-
    once((x_register(N, Arg) ; y_register(N, Arg); N = Arg)),
    remove_register_functors(Args, NewArgs).
