/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * read_procs - Read procedures from the source file and compile them.
 */

:- dynamic last_pred/2, redef/1, no_default_query/0.
?- dynamic '$qup_op'/3.

/*----------------------------------------------------------------------------
read_procedures :-
    Read the procedures from the input file and compile them into the
    output file.
----------------------------------------------------------------------------*/
read_procedures :-
    seeing(CurrentInput),
    input_filename(FileName),
    telling(CurrentOutput),
    output_filename(FileName2),
    tell(FileName2),
    (see(FileName) ->
	true
    ;
	write(user_error, 'Error : file '),
	write(user_error, FileName),
	writeln(user_error, ' does not exist'),
	fail
    ),
    read_procedures2,
    told,
    tell(CurrentOutput),
    seen,
    see(CurrentInput).


/*----------------------------------------------------------------------------
read_procedures2 :-
    Do the read/compile of the input file.  As the input is read, a fail is
    done after compiling each procedure to clean up the heap space.  However,
    a procedure change is only detected when the next procedure's first
    clause is read, so this is asserted in the fact "last_pred/2", to be
    picked up after the fail.  The clause "no_default_query" is asserted 
    if queries were encountered in the file, otherwise a default query must 
    be compiled.  The 'redef' predicate is used to keep track of predicates 
    already seen, to issue warnings.
----------------------------------------------------------------------------*/
read_procedures2 :-
    % assert(no_default_query),		% Stop "undefined" messages
    % retract(no_default_query),
    % assert(redef(a)),
    % abolish(redef, 1),		% Remove redefinition tracking
    % assert(nested(a,b)),
    % abolish(nested, 2),
    assert(last_pred(_Name,_List)),
    read_all_procedures,
    (\+ no_default_query ->		% Add default query if necessary
	do_read_compile('$query'/0,[('$query' :- fail) | _Tail])
     ;
    	retract(no_default_query)
    ),
    abolish(last_pred, 2).


/*----------------------------------------------------------------------------
read_all_procedures :-
    Reads all procedures in the input file in a repeat-fail loop.
----------------------------------------------------------------------------*/
read_all_procedures :-
    repeat,
    read_next_procedure,		% Read the next procedure
    !.					% At end of file when here


/*----------------------------------------------------------------------------
read_next_procedure :-
    Reads the next procedure from the input file, picking up a previously
    read clause asserted by the last call to 'read_next_procedure'.  This
    procedure will succeed at end of file only.
----------------------------------------------------------------------------*/
read_next_procedure :-
    last_pred(Name,List),		% get the previously read line
    read_proc_clauses(Name,List,NewName,NewList),
    abolish(last_pred, 2),		% remove current last_pred defn
    assert(last_pred(NewName,NewList)),	% assert a new one
    NewName == end_file(marker).	% fail if not at end of file


/*----------------------------------------------------------------------------
read_proc_clauses :-
    Read the clauses of the procedure, and compile the procedure once
    all clauses have been read in.
----------------------------------------------------------------------------*/
read_proc_clauses(Name,List,NewName,NewList) :-
    read(Sentence),
    (\+ Sentence == end_of_file ->
	'$filter_term_to_be_compiled'(Sentence, NewSentence),
	procedure_name(NewSentence,ProcName),
	(ProcName = Name ->	   % must be '=' to bind Name on first call
	    add_sentence(NewSentence,List),
	    read_proc_clauses(Name,List,NewName,NewList)
	 ;
	    do_read_compile(Name,List),
	    NewName = ProcName,
	    add_sentence(NewSentence,NewList)
	)
     ;
	do_read_compile(Name,List),
	NewName = end_file(marker)
    ).


/*----------------------------------------------------------------------------
add_sentence(Sentence, List) :-
    Add a sentence to a list doing query tranformations as necessary.
----------------------------------------------------------------------------*/
add_sentence((:- Goal),List) :-
    % once((rhs_list(Goal, GoalList), call_query(GoalList) ; true)),
    postpend_ol(('$query' :- once(Goal), fail), List),
    !.
add_sentence((?- Goal),List) :-
    once((rhs_list(Goal, GoalList), call_query(GoalList) ; true)),
    postpend_ol(('$query' :- once(Goal), fail), List),
    !.
add_sentence(Sentence,List) :-
    postpend_ol(Sentence, List).

/*----------------------------------------------------------------------------
call_query(GoalList) :-
    Execute a list of goals in a (?- Goal) query.  Quantifier operator
    declarations are modified to use 'fxy' declarations instead for the
    Nu-Prolog compiler version.  Note that 'fxy' is not defined in
    Qu-Prolog, so the second clause will fail, and the correct quantifier
    declaration will be made.
----------------------------------------------------------------------------*/
call_query([]).
call_query([op(Prec, quant, Q)|Goals]):-
    assert('$qup_op'(Prec, quant, Q)),
    call_query(Goals).
call_query([G|Goals]):-
    call(G),
    call_query(Goals).

/*----------------------------------------------------------------------------
procedure_name(Sentence,Name) :-
    Get the name of a procedure from the given sentence.  If it is a query,
    then assert 'no_default_query' if it is not already asserted.
----------------------------------------------------------------------------*/
procedure_name((:- _Goal),'$query'/0) :-
    !,
    (\+ no_default_query -> assert(no_default_query)
    ;
			    true
    ).
procedure_name((?- _Goal),'$query'/0) :-
    !,
    (\+ no_default_query -> assert(no_default_query)
    ;
			    true
    ).
procedure_name(Sentence,Name) :-
    predicate_name(Sentence,Name).


/*----------------------------------------------------------------------------
do_read_compile(Predicate,Procedure) :-
    Compile a procedure definition.  If the predicate name is not bound, 
    then ignore the procedure (file is empty - nothing to be compiled).
    The "; true" guards against the compilation failing.  Redefinition of
    the predicate is checked for amongst the 'redef' clauses, giving a
    warning.
----------------------------------------------------------------------------*/
do_read_compile(Predicate,Procedure) :-
    (\+ var(Predicate) ->
	  (redef(Predicate) ->
		write(user_error,'Warning : redefined predicate '),
		write(user_error,Predicate),
		writeln(user_error,' ignored')
	   ;
		assert(redef(Predicate)),
    	  	tail_ol(Procedure,[]),
    	  	(compile_procedure(Predicate,Procedure) ; true),
		abolish(nested, 2)	% Remove nested clause asserts
	  )
    ;
	true
    ).


/*----------------------------------------------------------------------------
insert_in_procedures(Sentence, Procedures) :-
    Insert the Sentence into its predicates procedure definition in 
    Procedures.
----------------------------------------------------------------------------*/
insert_in_procedures((:- Goal), Procedures) :-
    !,
    % once((call(Goal) ; true)),
    insert_ol(p('$query'/0, Procedure), Procedures),
    postpend_ol(('$query' :- once(Goal), fail), Procedure).
insert_in_procedures((?- Goal), Procedures) :-
    !,
    once((call(Goal) ; true)),
    insert_ol(p('$query'/0, Procedure), Procedures),
    postpend_ol(('$query' :- once(Goal), fail), Procedure).
insert_in_procedures(Sentence, Procedures) :-
    predicate_name(Sentence, Predicate),
    insert_ol(p(Predicate, Procedure), Procedures),
    postpend_ol(Sentence, Procedure).
    
/*----------------------------------------------------------------------------
predicate_name(Sentence, PredicateName) :-
    PredicateName is the name (F/N) of the predicate of which Sentence is a
    member of its definition.
----------------------------------------------------------------------------*/
predicate_name((Head :- _Body), F/N) :-
    !,
    Head =.. [F|Args],
    length(Args, N).
predicate_name(Head, F/N) :-
    Head =.. [F|Args],
    length(Args, N).
