/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * valvar - Determine the "get" and "put" instruction types of QuAM code.
 */

/*----------------------------------------------------------------------------
valvar(Instructions) :-
    Set the type of the put and get instructions to either: variable, value,
    or unsafe value, otherwise leave.
----------------------------------------------------------------------------*/
valvar(Instructions) :-
    type_instructions(Instructions, []).


/*----------------------------------------------------------------------------
type_instructions(Instructions, VarsSeen) :-
    Type each put and get instruction.
	
    VarsSeen 	The list of Prolog variables and Y registers seen sofar
		in the clause.
----------------------------------------------------------------------------*/
type_instructions([], _VarsSeen).
type_instructions([Instruction|Instructions], VarsSeen) :-
    type_instruction(Instruction, Instructions, VarsSeen, VarsSeen2),
    type_instructions(Instructions, VarsSeen2).

/*----------------------------------------------------------------------------
type_instruction(Instruction, Instructions, VarsSeen, VarsSeen2) :- 

    set the type of the put or get instructions to one of (variable, value,
    or unsafe value).

    VarsSeen	The variables and Y registers seen sofar in the clause.

    put_unsafe_value Y X 
	if Y is a permanent_register, and there are no occurrences after 
	the current call.

    otherwise
	if Instruction = [put, put_object, get, get_object] and the Type of
	    the instruction has not been determined (i.e. the cases where the
	    source is a variable) then
	    if the variable has occured previously then
		Instruction value Source Destination
	    else
		Instruction variable Source Destination
	else
	    leave instruction as is.
    
	For permanent (Y) variables
	    variable
	    value
	    ...
	    value
	    unsafe_value
----------------------------------------------------------------------------*/
type_instruction(put(variable, Source, Destination), _Instructions,
	VarsSeen, VarsSeen2) :-
    nonvar(Source),
    y_register(Source),
    \+ member2(Source, VarsSeen), !,
    (member2(Destination, VarsSeen) ->
	VarsSeen2 = [Source|VarsSeen]
    ;
	VarsSeen2 = [Destination, Source|VarsSeen]
    ).
type_instruction(put(unsafe_value, Source, _Destination), Instructions,
	VarsSeen, VarsSeen) :-
    nonvar(Source),
    y_register(Source),
    \+ permament_occurs_after_call(Source, Instructions), !.
    
type_instruction(Instruction, _Instructions, VarsSeen, VarsSeen2) :-
    (Instruction =.. [F, Type, Source, _Destination],
     member(F, [get, get_object, put, put_object]) ->
	set_instruction_type(Type, Source, VarsSeen, VarsSeen2)
    ;
	VarsSeen2 = VarsSeen
    ).

/*----------------------------------------------------------------------------
set_instruction_type(Type, Source, VarsSeen, VarsSeen2) :-
    Type is the type of put or get instruction (variable, value) to be 
    determined by the source parameter to the instruction.
    
    Type 	The type of the instruction {variable, value, unsafe_value}
    Source 	The input parameter to the instruction.
    VarsSeen	The variables seen previously
    VarsSeen2	Variables seen in previously and this instruction.

    if the type of the instruction has no been decided already then
	if the source is a variable that has been seen previously then
	    the type is value
	else 
	    the type of the get or put instruction is variable.
    
----------------------------------------------------------------------------*/
set_instruction_type(Type, Source, VarsSeen, VarsSeen2) :-
    var(Type), !,
    (member2(Source, VarsSeen) ->
	Type = value,
	VarsSeen2 = VarsSeen
    ;
	Type = variable,
	VarsSeen2 = [Source|VarsSeen]
    ).
set_instruction_type(_Type, _Source, VarsSeen, VarsSeen).


/*----------------------------------------------------------------------------
permament_occurs_after_call(Y, Instructions) :-
    True, if the permanent register, Y, occurs after the next call in the set
    of instructions, Instructions.
----------------------------------------------------------------------------*/
permament_occurs_after_call(Y, Instructions) :-
    append(_FrontInstructions,
	[call_predicate(_F, _N, _EnvSize)|BackInstructions], Instructions),
    member(Instruction, BackInstructions),
    permanent_occurs_in_instruction(Y, Instruction).


/*----------------------------------------------------------------------------
permanent_occurs_in_instruction(Y, Instruction) :-
    The permanent register occurs in Instructions.
    Permanent registers only occur as the source to the instructions.
----------------------------------------------------------------------------*/
permanent_occurs_in_instruction(Y, Instruction) :-
    Instruction =.. [_InstructionClass, _Type, Source, _Destination],
    Y == Source.
