/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * var_names - Declarations to produce more meaningful output variable names.
 */

:- dynamic next_meta_variable/1, next_object_variable/1,
		next_anonymous_variable/1.

/*----------------------------------------------------------------------------
get_next_meta_var(Y) :-
    Y is the next meta variable name.
----------------------------------------------------------------------------*/
get_next_meta_var(Y) :-
    next_meta_variable(Y),
    new_meta_var(Y, Z),
    replace(next_meta_variable(Y), next_meta_variable(Z)).

/*----------------------------------------------------------------------------
get_next_ov_var(Y) :-
    Y is the next object variable name.
----------------------------------------------------------------------------*/
get_next_ov_var(Y) :-
    next_object_variable(Ext),
    incr(next_object_variable),
    name(Ext, ExtStr),
    append("x", ExtStr, New),
    name(Y, New).

/*----------------------------------------------------------------------------
new_meta_var(Y, Z) :-
    Y is a meta variable name, Z is the next meta-variable name in 
    sequence.
----------------------------------------------------------------------------*/
new_meta_var(Y, Z) :-
    name(Y, Cs),
    next_meta_var_string(Cs, Ds),
    name(Z, Ds).


/*----------------------------------------------------------------------------
new_anonymous_variable(Y) :-
    Y is a new meta variable name.
----------------------------------------------------------------------------*/
new_anonymous_variable(Y) :-
    next_anonymous_variable(Y),
    name(Y, Cs),
    next_meta_var_string(Cs, Ds),
    name(Z, Ds),
    replace(next_anonymous_variable(Y), next_anonymous_variable(Z)).


/*----------------------------------------------------------------------------
next_meta_var_string(Cs, Ds) :-
    Ds is the next variable name string after Cs.

    	A	AA	AAA	AAAA etc
    	.. 	.. 	..
    	Z	AZ	AAZ
		BA	ABA
		..	..
		BZ	ABZ
		..	..
		ZA	ZYA
		..	..
		ZZ	ZZZ

----------------------------------------------------------------------------*/
next_meta_var_string(Cs, Ds) :-
    append(Rest, [C], Cs),
    (\+([C] = "Z") ->
	D is C + 1,
	append(Rest, [D], Ds)
    ;
	(\+(Rest = []) ->
	    next_meta_var_string(Rest, NewRest),
	    append(NewRest, "A", Ds)
	;
	    Ds = "AA"
	)
    ).

/*----------------------------------------------------------------------------
initialise_for_renaming :-
    initialize the meta-variable name to be used next in renaming, also 
    initialize the extension of the object-variable for renaming 
    object-variables in clauses asserted.

----------------------------------------------------------------------------*/
initialise_for_renaming :-
    assert(next_meta_variable('A')),
    assert(next_object_variable(1)).

/*----------------------------------------------------------------------------
clean_up_after_renaming :-
    remove meta-variable name to be used next and the extension of the 
    object-variable used next.

----------------------------------------------------------------------------*/
clean_up_after_renaming :-
    retract(next_meta_variable(_MetaVar)),
    retract(next_object_variable(_ObjectVar)).

/*----------------------------------------------------------------------------
allocate_vars(Ss) :-
    Allocates variable names to all variables in the structure Ss.
----------------------------------------------------------------------------*/
allocate_vars(Ss) :-
    initialise_for_renaming,
    instantiate_vars(Ss),
    clean_up_after_renaming.
 

/*----------------------------------------------------------------------------
instantiate_vars(Args) :-
    Traverses the structure of Args, allocating the variable names.
----------------------------------------------------------------------------*/
instantiate_vars([]).
instantiate_vars([Arg|Args]) :-
    var(Arg),
    !,
    get_next_meta_var(Arg),
    instantiate_vars(Args).
instantiate_vars([Arg|Args]) :-
    Arg =.. [_F|FsArgs],
    instantiate_vars(FsArgs),
    instantiate_vars(Args).
