 /* 
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
  */ 


#include "name_table.h"
#include "bind.h"
#include "string_table.h"
#include "cells.h"
#include "copy_term.h"
#include "data_area.h"
#include "database.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "examine_term.h"
#include "main.h"
#include "simplify.h"
#include "substitution.h"
#include "unify.h"
#include "x_registers.h"
#include "system.h"

global  natural delayed_stack_size = DEFAULT_DELAYED_STACK_SIZE;
global  delayed *delayed_stack;
global  delayed *top_delayed_stack;
global  delayed *end_delayed_stack;
global  cell    *delayed_problems_pointer = NULL;

local	boolean	not_free_make_progress;
local	boolean	is_free_in;

/*
 *
 * if t1 and t2, where delay is called, can be used as suspended variable,
 * the following procedure is used:
 * 1. meta variable & object variable -> meta variable
 * 2. both t1 and t2 have the sacell data type -> t1
 *
 * Note:  'left' must contain the term which the 'variable' is suspended on.
 */


/*-----------------------------------------------------------------------------
delay(type, variable, left, right)
	Add the problem to the associated variable if none of similar form
	is found.  While scanning through the list, it also removes any 
	duplicate problems similar to the new problem.  The duplicate problems
	arise, for example
		Existing problems:
			x not_free_in B
			y not_free_in B
		Later processing causes:
			x = y
		Adding similar problem again:
			x not_free_in B
		So:
			y not_free_in B can be removed.
-----------------------------------------------------------------------------*/

int same_term (VALUE *t1, VALUE *t2);
int reduction (cell objvar, VALUE *term);
int red_rule_1 (cell objvar, cell domain, cell range, VALUE *term);
int red_rule_2 (cell objvar, cell domain, cell range, VALUE *term);
int red_rule_3 (cell objvar, cell domain, cell range, VALUE *term);
extern void write_termv (VALUE *term);
extern int fprintf (FILE *, const char *, ...);
extern void writeln_termv (VALUE *term);

global	void
delay(int type, cell *variable, cell left, cell right)
{
	cell	*d, *d1;
	cell	var;
	VALUE	new;
	VALUE	old;
	boolean found = FALSE;

	if (top_delayed_stack+1 > end_delayed_stack)
		fatal("out of delayed stack space %d K", delayed_stack_size);
	else
	{
		if (type == NOT_FREE)
		{
				/* avoid duplicate NOT_FREE delayed problem */
			new.sub = EMPTY_SUB;
			new.term = left;
			object_dereference(&right);
			for (d = (cell *)RestOfVariable(*variable);
			     d != NULL;
			     d = Next(d))
                        {
				if (DelayedType(d) == NOT_FREE &&
				    DelayedSolved(d) == UNSOLVED)
				{
				        var = DelayedTerm(d);
				        object_dereference(&var);
				        old.sub = EMPTY_SUB;
				        old.term = DelayedVar(d);
				        if (right == var &&
                                            same_term(&new, &old))
				        {
						if (found)
						{
							Solved(d1);
						}
						else
						{
							found = TRUE;
						}
						d1 = d;
                                        } 
				}
                        }
			if (found)
			{
				return;
			}
		}

		if (type == NOT_FREE && Frozen(right) &&
		    flag_value(Atom(add_name_string_offset("frozen_nfi_test",
							   ATOM_W)))
					 == Atom(add_name_string_offset("on",
					 ATOM_W)))
		{
			/*
			 * FUDGE!!!!!   pjr
			 */
			is_free_in = TRUE;
		}
		else
		{ 
			top_delayed_stack->tag = type;
			top_delayed_stack->var = left;
			top_delayed_stack->term = right;
			top_delayed_stack++;

			set(variable, (cell)(Tag(*variable)|
						Temperature(*variable)|
				     DelayLink((cell)(top_delayed_stack-1),
						RestOfVariable(*variable))));

		}
	}

}
/*------------------------------------------------------------------------------
same_term(t1, t2)
    Boolean function - simple comparision between two terms.
    Both terms are dereferenced, so we have substitutions on
    one side, and terms on the other.
    Conditions for two terms to be the same are: 
	  1. terms are equal
	  2. substitutions are of the same size 
	  3. corresponding dereferenced domains are equal and 
	     corresponding dereferenced ranges are equal 

------------------------------------------------------------------------------*/
global	boolean
same_term(VALUE *t1, VALUE *t2)
{

        cell i, j, ri, rj, di, dj; 
	int k, k1; 

        dereference(t1);
        dereference(t2);

        if (t1->term != t2->term)
        {
	        return(FALSE);
        }
        else
        {
	        for (i = t1->sub, j = t2->sub; i != EMPTY_SUB && j != EMPTY_SUB;
		     i = NextSub(i), j = NextSub(j))
                { 
		        k1 = Size(i);

		        if (k1 != Size(j))
                        {
		                return(FALSE);
                        }
		        for (k = 1; k <= k1; k++) 
                        {
		                VariableDereference(ri, Range(i, k)); 

		                VariableDereference(rj, Range(j, k)); 

		                ObjectDereference(di, Domain(i, k)); 
		      
		                ObjectDereference(dj, Domain(j, k)); 

		                if (ri != rj || di != dj)
                                {
			                return(FALSE);
                                }
		        }
                } 
	        if (i == EMPTY_SUB && j == EMPTY_SUB)
                {
		        return(TRUE); 
                }
                else
                {
		        return(FALSE); 
                }
	}
}
/*------------------------------------------------------------------------------
do_delayed_problem(d)
    For the given delayed problem 

------------------------------------------------------------------------------*/
do_delayed_problem(cell *d)
{
	VALUE	variable;
	VALUE	term;
        boolean b;
                 
	variable.sub = EMPTY_SUB;
	variable.term = DelayedVar(d);
	term.sub = EMPTY_SUB;
	term.term = DelayedTerm(d);
	switch (DelayedType(d))
	{
	when UNIFY:
		return(unify(&variable, &term));
	when NOT_FREE:
		dereference(&term);
		return(not_free_in(term.term, &variable));
	}
}

/*
 * if (there is problems to add)
 *      move to the last problem in dest
 *      add problems to tail
 */
/*------------------------------------------------------------------------------
append_delayed_problems(dest, problems)

------------------------------------------------------------------------------*/
global  void
append_delayed_problems(cell *dest, cell *problems)
{
	cell	*d;

        if (problems != NULL)
        {
		if (RestOfVariable(*dest) == NULL)
		{
                        set(dest, VARIABLE|Temperature(*dest)|(cell)problems);
                }
		else
		{
		        for (d = (cell *)RestOfVariable(*dest);
			     Next(d) != NULL; d = Next(d))
			;
                        set((cell *)(&NextField(d)), (cell)problems);
                }
        }
}
/*------------------------------------------------------------------------------
add_wake_up_problems(problems)

------------------------------------------------------------------------------*/
global  void
add_wake_up_problems(cell *problems)
{
	cell	*d;
	cell	*d1;
       

        for (d = problems; d != NULL; d = d1)
	{
	        d1 = Next(d);
                if (DelayedSolved(d) == UNSOLVED)
	        {
/*
		        Solved(d);    
*/
                        set((cell *)(&NextField(d)), (cell)delayed_problems_pointer);
                        delayed_problems_pointer = d;
                }
        }
}
/*------------------------------------------------------------------------------
not_free_in(objvar, term)
    Boolean function - checking whether object variable is not free in the term.
    It returns:
       TRUE  - if object variable is not free in the term or in the  
	       case when we don't know whether object variable is free 
	       or not and when we delay NOT_FREE problem.
       FALSE - if we know that object variable is free in the term.

------------------------------------------------------------------------------*/
global  boolean
not_free_in(cell objvar, VALUE *term)
{
        return(!freeness(objvar, term));
}
/*-----------------------------------------------------------------------------
freeness(objvar, term)
    Boolean function - checking whether object variable is free in the term.
    It returns:
       TRUE  - if we know that object variable is free in the term.
       FALSE - if object variable is not free in the term or in the  
	       case when we don't know whether object variable is free 
	       or not and when we delay NOT_FREE problem.
    Note: some reduction rules are applied on the term.
------------------------------------------------------------------------------*/
global  boolean
freeness(cell objectvar, VALUE *term)
{
	VALUE	val1;
	VALUE	val2;
local	cell	generate_objvar(cell objvar, cell sub);
	cell	s, objvar;
static	cell	unique = Atom(DOLLAR);

	ObjectDereference(objvar, objectvar);
	val1.sub = term->sub;
	val1.term = term->term;
	dereference(&val1);
	switch (Tag(val1.term))
	{
	when CONSTANT:
		not_free_make_progress = TRUE;
		return(FALSE);
	when APPLY:
		val2.sub = val1.sub;
		val2.term = Reference(&Argument(val1.term));
		val1.term = Reference(&Functor(val1.term));
		return(freeness(objvar, &val1) || freeness(objvar, &val2));
	when PAIR:
		val2.sub = val1.sub;
		val2.term = Reference(&Right(val1.term));
		val1.term = Reference(&Left(val1.term));
		return(freeness(objvar, &val1) || freeness(objvar, &val2));
	when QUANTIFIER:
		val1.sub = RenameSub(BoundVar(val1.term), unique, val1.sub);
		val1.term = Reference(&Body(val1.term));
		return(freeness(objvar, &val1));
	when REFERENCE:
		is_free_in = FALSE;
		reduction(objvar, &val1);
		return(is_free_in);
		/*
		delay(NOT_FREE, Location(val1.term),
		      make_substitution(&val1), objvar);
		*/      
	when OBJECT_REFERENCE:
		if (val1.sub == EMPTY_SUB)
		{
			not_free_make_progress = TRUE;
			if (objvar == val1.term)
			{
				return(TRUE);
                        }
			else if (!distinct_from(objvar, val1.term))
			{
				SetDistinct(objvar, val1.term);
			}
		}
		else
		{
			is_free_in = FALSE;
		        reduction(objvar, &val1);
			return(is_free_in);
		/*
		delay(NOT_FREE, Location(val1.term),
		      make_substitution(&val1), objvar);
                */
		}
	}
	return(FALSE);
}
/*-----------------------------------------------------------------------------
reduction(objvar, term)
   
    The term is dereferenced in the function where 'reduction' is colled
    from, so we have substitution and term separated.
    Copy term with swapped pointers onto the heap. 
    s1 * s2 * ... * sn * t   ->   sn * sn-1 * ... * s1 * t  
    Explaination why:
       Rules are always applied on term passing a chain of composition
       of substitutions from right to left.Copy of it with swapped    
       pointers is made, so the most right substitution is the first 
       in a chain and etc. It is much easier to deal with such ordered 
       chain, because we can start with the first substitution and  
       continue with next in recursive call. 
    Apply reduction rules on term.
   
-----------------------------------------------------------------------------*/
reduction(cell objvar, VALUE *term)
{

local   cell copy_swap_sub(cell sub);
local   void apply_rule(cell objvar, VALUE *term);
	cell	sub;


	sub = term->sub;
        term->sub = copy_swap_sub(term->sub);

        apply_rule(objvar, term);
	term->sub = sub;

}
/*-----------------------------------------------------------------------------
apply_rule(objvar, term)
    term =  sn * sn-1 * ... * s1 * t (substitution is with swapped pointers)
    Apply reduction rules on term recursively (recursive call is from one   
    of rules).
    If no rule can be applied 
        if substitution is empty and term is an object variable
            set not_free_make_progress to TRUE 
	    if object variable and term are equal
	        free in problem succeeds
            else if object variable and term arn't different
		     set them to be different
            else
	        continue
        else
	    copie substitution (or sub-substitution) with swapped back   
	    pointers
	    delay NOT_FREE problem : objvar not_free_in s1 * s2 *...*si * t,
	    where i =< n

-----------------------------------------------------------------------------*/
local void
apply_rule(cell objvar, VALUE *term)
{
	VALUE   domain;
	VALUE   term1;
	boolean can_app_rule = FALSE;
	cell	range;

local   cell    copy_swap_sub(cell sub);

	if (term->sub != EMPTY_SUB && Size(term->sub) == 1)
	{
	        range = Range(term->sub, 1);  
	        variable_dereference(&range);  

	        DereferenceTerm(domain, Domain(term->sub, 1)); 
                
                if (red_rule_1(objvar, domain.term, range, term) || 
                    red_rule_2(objvar, domain.term, range, term) || 
                    red_rule_3(objvar, domain.term, range, term))
	        {
		        can_app_rule = TRUE;
                }
        }
	 
	if (can_app_rule) 
	{
		not_free_make_progress = TRUE;
	}
	else
	{
	        if (term->sub == EMPTY_SUB && IsObjectReference(term->term))
	        {
	                not_free_make_progress = TRUE;
	                if (objvar == term->term)
	                {
		                is_free_in = TRUE;
                        }
		        else if (!distinct_from(objvar, term->term))
                        {
		                SetDistinct(objvar, term->term);
                        } 
	        }
	        else
	        {
		        if (!IsLocalObjectVariable(objvar))
		        {
		                if (term->sub == EMPTY_SUB)
                                {
			                term1.sub = EMPTY_SUB;
                                }
		                else
                                {
			                term1.sub = copy_swap_sub(term->sub);
                                }
	                        term1.term = term->term;

		                delay(NOT_FREE, Location(term1.term),
			              make_substitution(&term1), objvar); 
		        }
                }
        } 
}
/*----------------------------------------------------------------------------- 
red_rule_1()
    Boolean function, if conditions are satisfied, it just returns TRUE, 
    otherwise it returns FALSE.
    Rule:
	z is an object variable or a constant
        x not_free_in [z/x]*t => (x != z) -> success
-----------------------------------------------------------------------------*/
local boolean 
red_rule_1(cell objvar, cell domain, cell range, VALUE *term)
{

        if ((IsConstant(range) ||
	    ((IsObjectReference(range) &&
	    IsObjectVariable(ObjectValue(range))) &&
	    distinct_from(range, objvar))) && 
	    domain == objvar)  
        {
	        return(TRUE);
        } 
	else
        {
	        return(FALSE);
        } 
}
/*-----------------------------------------------------------------------------
red_rule_2() 
    Boolean function, if conditions are satisfied, it applies specified 
    rule and returns TRUE, otherwise it returns FALSE.
    Rule:
       x not_free_in [x/z]*t => x != z -> x not_free_in t and z not_free_in t 
-----------------------------------------------------------------------------*/
local boolean
red_rule_2(cell objvar, cell domain, cell range, VALUE *term)
{
	cell 	sub;

        if (objvar == range /* && distinct_from(objvar, domain) */) 
	{
	        sub = term->sub = NextSub(term->sub);
	        apply_rule(domain, term);
	        term->sub = sub;
	        apply_rule(objvar, term);
	        return(TRUE);
        }
	else
	{
	        return(FALSE);
	}
}
/*----------------------------------------------------------------------------
red_rule_3() 
    Boolean function, if conditions are satisfied, it applies specified 
    rule and returns TRUE, otherwise it returns FALSE.
    Rule:
	 y is an object variable or a constant
         x not_free_in [y/z]*t => x != y, x != z  ->  z not_free_in t 
-----------------------------------------------------------------------------*/
local boolean
red_rule_3(cell objvar, cell domain, cell range, VALUE *term)
{
                
        if ((IsConstant(range) ||  
            (IsObjectReference(range) &&
	    IsObjectVariable(ObjectValue(range)) &&
            distinct_from(range, objvar))) &&
	    distinct_from(domain, objvar))
	{
		term->sub = NextSub(term->sub);
		apply_rule(objvar, term);
		return(TRUE);
        } 
	else
	{
	        return(FALSE);
	}
}
/*-----------------------------------------------------------------------------
copy_swap_sub()

    [r1/x1]* ...*[rn/xn] -> [rn/xn]*...*[r1/x1]   

    Copy substitution onto the heap and swap pointers at the same time.
    New proper properties are put. 
    It returns pointer to the new substitution. 
-----------------------------------------------------------------------------*/
local cell 
copy_swap_sub(cell sub)
{
        cell    previous;
	cell    copied;
        cell    s;
	cell    property;
	cell    prop;
        cell    prev_prop;
        int     i;

        previous = EMPTY_SUB; 
	prev_prop = INVERTIBLE;

	for (s = sub; s != EMPTY_SUB; s = NextSub(s))
        {
                i = determine_property(s);

                prop =  ContainLocal(s, 1) ? INVERTIBLE : 
	             		             i;
                property = prop > prev_prop ? prop : prev_prop; 

                copied = NewSubstitution(Table(s), previous, property);

                previous = (cell)copied;

	        prev_prop = property;
		     
        }
        return(previous);
}
/*------------------------------------------------------------------------------
generate_objvar(objvar, sub)

    If substitution is empty or no substitution contains
       the object variable as a domain
	return a pointer to an empty substitution
    else
        return a pointer to the most right substitution which 
	contains the object variable as a domain
------------------------------------------------------------------------------*/
local	cell
generate_objvar(cell objvar, cell sub)
{
	cell	s;

	if (sub == EMPTY_SUB)
        {
		return(sub);
        }
	else
	{
		s = generate_objvar(objvar, NextSub(sub));
		if (s != EMPTY_SUB || !in_domain(objvar, sub))
                {
			return(s);
                }
		else 
                {
			return(sub);
                }
	}
}
/*----------------------------------------------------------------------------- 
$first_delayed_problem()
    If there is any  unsolved delayed problem in the delayed stack
        returns the first one the begining of the delayed stack 
    else
        returns FALSE
------------------------------------------------------------------------------*/
global  boolean
esc_first_delayed_problem(void)
{
        VALUE   val;
        delayed *d;
 
        for (d = delayed_stack; d < top_delayed_stack; d++)
        {
                if ((d->tag & SOLVEDMASK) == UNSOLVED)
                {
                        val.sub = EMPTY_SUB;
                        val.term = PtrToInt(d);
                        return(unify(XV(0), &val));
                }
        }
        return(FALSE);
}
/*------------------------------------------------------------------------------
$get_delayed_problem()
     If dereferenced value in X(0) register is not integer 
	 return FALSE
     else
	 return type, term and variable of the given delayed problem
	 from register X(0) in registers X(1), X(2) and X(3)  
 
------------------------------------------------------------------------------*/
global  boolean
esc_get_delayed_problem(void)
{
        delayed *d;
        VALUE   type;
	VALUE	term;
	VALUE	var;
 
        if (! IsInteger(Xdref(0)))
        {
               return(FALSE);
        }
        else
        {
               d = (delayed *)IntToPtr(X(0));
               type.sub = EMPTY_SUB;
               type.term = (d->tag & TYPEMASK) == UNIFY ?
                            Atom(add_name_string_offset("=", ATOM_W)) :
                            Atom(add_name_string_offset("not_free_in", ATOM_W));
	       term.sub = EMPTY_SUB;
	       term.term = d->term;
	       var.sub = EMPTY_SUB;
	       var.term = d->var;
               return(unify(XV(1), &type) &&
                      unify(XV(2), &term) &&
                      unify(XV(3), &var));
        }
}
/*------------------------------------------------------------------------------
$next_delayed_problem()
     If dereferenced value in X(0) register is not integer 
	 return FALSE
     else
         if there is any  unsolved delayed problem in the delayed 
	    stack after the given one in X(0) register
	     return the first next after that one 
         else
	     return FALSE
 
------------------------------------------------------------------------------*/
global  boolean
esc_next_delayed_problem(void)
{
        VALUE   val;
        delayed *d;

        if (! IsInteger(Xdref(0)))
        {
                return(FALSE);
        }
        else
        {   
                for (d = (delayed *)IntToPtr(X(0)) + 1;
                     d < top_delayed_stack;
                     d++)
                {
                        if ((d->tag & SOLVEDMASK) == UNSOLVED)
                        {
                                val.sub = EMPTY_SUB;
                                val.term = PtrToInt(d);
                                return(unify(XV(1), &val));
                        }
                }
                return(FALSE);
        }
}
/*------------------------------------------------------------------------------
$check_delayed_problem()
    If the value in register X(0) is not an atom  
	return FALSE
    else
	if this atom is "="  
	    add "=" to the name table and to the string   
	    table if it hasn't already been added 
	    if there is at least one unsolved 
	       delayed problem of UNIFY type  
		return TRUE
            else
		return FALSE
	else if this atom is "not_free_in"  
	         add "not_free_in" to the name table and to the   
	         string table if it hasn't already been added 
	         if there is at least one unsolved 
	            delayed problem of NOT_FREE type  
		     return TRUE
                 else
		     return FALSE
        else
	    return FALSE

------------------------------------------------------------------------------*/
global	boolean
esc_check_delayed_problem(void)
{
	dereference(XV(0));
	if (! IsAtom(X(0)))
        { 
		return(FALSE);
        }
	else if (RestOfConstant(X(0)) == lookup_name_table_offset("=", ATOM_W))
        {
		return(test_delayed_problem(UNIFY));
        }
	else if (RestOfConstant(X(0)) == lookup_name_table_offset("not_free_in",
	                                                          ATOM_W))
        {
		return(test_delayed_problem(NOT_FREE));
        }
	return(FALSE);
}
/*------------------------------------------------------------------------------
test_delayed_problem(type)
    If there is at least one unsolved delayed problem of 
       the specified type in the delayed stack
	return TRUE
    else
	return FALSE

------------------------------------------------------------------------------*/
global	boolean
test_delayed_problem(int type)
{
	delayed	*d;

	for (d = delayed_stack; d < top_delayed_stack; d++)
        {
		if ((d->tag & SOLVEDMASK) == UNSOLVED && d->tag == type)
                {
			return(TRUE);
                }
        }
	return(FALSE);
}

/*
 * It solves all the problems in the wake up queue. 
 */

global	boolean
try_delay_problems(void)
{
	for (;;)
	{
                while(delayed_problems_pointer != NULL && 
                      DelayedSolved(delayed_problems_pointer) == SOLVED)
                {
	                delayed_problems_pointer = 
                                Next(delayed_problems_pointer); 
                }
                if (delayed_problems_pointer == NULL)
                {
                        break;
                }
                Solved(delayed_problems_pointer); 
	        if (!do_delayed_problem(delayed_problems_pointer))
	        {
		        return(FALSE);
	        }
	}
	return(TRUE);
}

/*
 * retry_delay_problems
 *	This is a Prolog predicate.  It solves all the problems in the
 *	wake up queue first and then retries all the problems in the delayed
 *	stack.
 */

global	boolean
retry_delay_problems(void)
{
	return(retry_delay());
}
/*------------------------------------------------------------------------------
retry_delay()
	Retry all the problems in the delayed stack.
------------------------------------------------------------------------------*/
global	boolean
retry_delay(void)
{
local	boolean	make_progress(delayed *new, delayed *old);
	boolean	progress;
	VALUE	variable;
	VALUE	term;
	delayed	*d;
	delayed *previous;
	delayed *current_top = delayed_stack;
extern	boolean debug_retry;
	

        delayed_problems_pointer = NULL;

	do
	{
		progress = FALSE;
		for (d = delayed_stack, current_top = top_delayed_stack;
		     d < current_top;
		     d++)
		{
			if ((d->tag & SOLVEDMASK) == UNSOLVED)
			{
				previous = top_delayed_stack;
				variable.sub = EMPTY_SUB;
				variable.term = d->var;
				term.sub = EMPTY_SUB;
				term.term = d->term;
				if (debug_retry)
				{
					write_termv(&term);
					switch (d->tag & TYPEMASK)
					{
					when UNIFY:
						fprintf(stderr, " = ");
						break;
					when NOT_FREE:
						fprintf(stderr, " nfi ");
						break;
					}
					writeln_termv(&variable);
				}
				switch (d->tag & TYPEMASK)
				{
				when UNIFY:
					if (! unify(&variable, &term))
                                        {
					        return(FALSE); 
                                        }
				when NOT_FREE:
					not_free_make_progress = FALSE;
					dereference(&term);
					if (freeness(term.term, &variable))
                                        {
					        return(FALSE);
                                        }
				}
				if (make_progress(previous, d))
				{
					set((cell *)&(d->tag),
					    (d->tag&~SOLVEDMASK)|SOLVED);
					progress = TRUE;
				}
			}
	        }
	} while (progress);
	return(TRUE);
}
/*------------------------------------------------------------------------------
retry_nfi_delay()
	Retry only the NFI problems in the delayed stack.
------------------------------------------------------------------------------*/
global	boolean
retry_nfi_delay(void)
{
local	boolean	make_progress(delayed *new, delayed *old);
	boolean	progress;
	VALUE	variable;
	VALUE	term;
	delayed	*d;
	delayed *previous;
	delayed *current_top = delayed_stack;
extern	boolean debug_retry;
	

        delayed_problems_pointer = NULL;

	do
	{
		progress = FALSE;
		for (d = delayed_stack, current_top = top_delayed_stack;
		     d < current_top;
		     d++)
		{
			if ((d->tag & SOLVEDMASK) == UNSOLVED &&
			    (d->tag & TYPEMASK) == NOT_FREE)
			{
				previous = top_delayed_stack;
				variable.sub = EMPTY_SUB;
				variable.term = d->var;
				term.sub = EMPTY_SUB;
				term.term = d->term;
				if (debug_retry)
				{
					write_termv(&term);
					switch (d->tag & TYPEMASK)
					{
					when UNIFY:
						fprintf(stderr, " = ");
						break;
					when NOT_FREE:
						fprintf(stderr, " nfi ");
						break;
					}
					writeln_termv(&variable);
				}
				not_free_make_progress = FALSE;
				dereference(&term);
				if (freeness(term.term, &variable))
				{
					return(FALSE);
				}
				if (make_progress(previous, d))
				{
					set((cell *)&(d->tag),
					    (d->tag&~SOLVEDMASK)|SOLVED);
					progress = TRUE;
				}
			}
	        }
	} while (progress);
	return(TRUE);
}
/*-----------------------------------------------------------------------------
make_progress(new, old)

-----------------------------------------------------------------------------*/
local	boolean
make_progress(delayed *new, delayed *old)
{
	cell	*var;
local	boolean	equiv(cell new, cell old);

	switch (old->tag & TYPEMASK)
	{
	when UNIFY:
		if (top_delayed_stack != new &&
		    equiv(new->var, old->var) &&
		    equiv(new->term, old->term))
		{
				/* dequeue from delayed variable */
			var = Location(IsSubstitution(new->var) ?
					Term(new->var) :
					new->var);
			*var = Tag(*var)|Temperature(*var)|
				((int)Next((cell *)RestOfVariable(*var)));
				/* reclaim space if make no progress */
			top_delayed_stack--;
			top_of_heap -= 2;
			if (IsSubstitution(new->var))
                        {
				top_of_heap -= 2;
                        }
			if (IsSubstitution(new->term))
                        {
				top_of_heap -= 2;
                        }
			if ((top_of_trail-1)->address == var)
                        {
				top_of_trail--;
                        }
			if (delayed_problems_pointer != NULL)
			{
				if ((top_of_trail-1)->address ==
					(cell *)&delayed_problems_pointer)
                                {
					top_of_trail--;
                                }
				delayed_problems_pointer = NULL;
			}
			return(FALSE);
		}
		return(TRUE);
	when NOT_FREE:
		   /* new delayed problem is created so make progress */ 
		return(top_delayed_stack != new || not_free_make_progress);
	}
}
/*------------------------------------------------------------------------------
equiv(new, old)
    Dereference both arguments so we have substitutions 
    and terms separated. 
    If pointers to the substitutions and pointers  
       to the terms are equal  	
	return TRUE
    else
	return FALSE
------------------------------------------------------------------------------*/
local	boolean
equiv(cell new, cell old)
{
	VALUE	newval;
	VALUE	oldval;

	DereferenceTerm(newval, new);
	DereferenceTerm(oldval, old);
	return(newval.sub == oldval.sub && newval.term == oldval.term);
}
