/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "cells.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "substitution.h"
#include "x_registers.h"

/*----------------------------------------------------------------------------
dereference(t)
    Removes any unnecessery REFERENCE and OBJECT_REFERENCE cell. 
    Does substitution evaluation.
    Parametar t is a pointer to VALUE type, it means that it consists of
    a substitution and a term, and dereference leaves it so substitution 
    and term are separated (term is with no substitution, what is not the
    case at the entrance of the procedure).

    Depending of the data object type in the term dereference performs:
	1. CONSTANT: remove all substitution and exit the procedure.
		     [t1/x1] * ... * [tn/xn] * c -> c
    
	2. REFERENCE: followe the REFERENCE chain. If the final data object 
		      is a variable (A), substitutions like:  
                          a. [x/x]
                          b. [x/v] (* where v is a local object variable)
			  c. [t/x] and x not_free_in A
                      will be removed.

	3. OBJECT_REFERENCE: followes the REFERENCE chain.After that one  
			     of following occurs: 

             .No substitution: exit the procedure.
    
	     .Local object variable: search through the most left domain of  
			             each substitution to find the maching 
				     local variable, and replace by the
				     range. If unable to find the matching 
				     local object variable, the procedure 
				     terminates.

	     .Object variable: if one of the domain is the same as the term 
			       substitution and term will be replaced by the
			       range -  [t/x]*x => t.  
			       If one of the domain is different from the
			       term substitution and term will be replaced
			       by the term - [t/x]*y => y.
			       If substitution is of the size one and range
			       and domain are the same, such substitution 
			       will be removed -  [x/x]*y => y
			       In other cases dereference terminetes. 
                               Like in REFERENCE case substitutions like:
                                  a. [x/x]
                               will be removed.

	4. SUBSTITUTION_OPERATOR: The substitution will be placed in tempo- 
				  rary substitution registers (XS) while the
				  is stored in the X register.
	5. Other data object: exit the procedure.  

Explanation - why substitutions like  [t1/x] * ... * [tn/x] * [x/v] 
              (where v is a local object variable) is removed:

[t1/x] * ... * [tn/x] * [x/v] * y =   ( [x/v]  comes from right hand side,  
                        |       /       where it was [v/x].When it was moved
                        |    /          to the left constrain x not_free_in y
		        |/ 		was built)
[t1/x] * ... * [tn/x] * y =           ( as v is different from y) 
               |        /
               |    /
               |/ 
[t1/x] * ... * y =                    ( as x is different from y, we know  
|             /                         that from  x not_free_in y ) 
|         /
|    /
|/ 
y 
----------------------------------------------------------------------------*/

extern int distinct_from (cell objvar1, cell objvar2);

global void
dereference(VALUE *t)
{
	cell    v, u;
reg	natural i, j;
local	void	drop_substitution(VALUE *t);

	for (;;)
	{
		switch (Tag(t->term))
		{
		when CONSTANT:
			t->sub = EMPTY_SUB;
			return;
		when REFERENCE:
			variable_dereference(&(t->term));
			if (IsReference(t->term))
			{
				drop_substitution(t);
				return;
			}
		when SUBSTITUTION_OPERATOR:
			t->sub = add_substitution(Substitution(t->term),
						  t->sub);
			t->term = Reference(&Term(t->term));
		when OBJECT_REFERENCE:
			object_dereference(&(t->term));
			if (t->sub == EMPTY_SUB)
				return;
			else if (IsLocalObjectVariable(t->term))
			{
				while (t->sub != EMPTY_SUB &&
				       t->term != Domain(t->sub, 1))
					t->sub = NextSub(t->sub);
				if (t->sub == EMPTY_SUB)
					return;
				else
				{
					t->term = Reference(&Range(t->sub,
								   1));
					t->sub = NextSub(t->sub);
				}
			}
			else
			{
				for (i = Size(t->sub), j = 1; j <= i; j++)  
				{    
					v = Domain(t->sub, j);
					object_dereference(&v);
					if (t->term == v)
					{   
						  /* [t/x]*x => t */
						t->term = Reference(&
							Range(t->sub, j));
						break;
					}
					else if (distinct_from(t->term, v))
						  /* [t/x]*y => y */
						continue;
					else if (i == 1 &&
					 	 (u = Reference(&Range(t->sub,
									1)),
						  object_dereference(&u),
					  	  v == u))
						  /* [x/x]*y => y */
						continue;
					else
						return;
				}
				t->sub = NextSub(t->sub);
			}
		otherwise:
			return;
		}
	}
}
/*-----------------------------------------------------------------------------
variable_dereference(t)
    Removes any unnecessery REFERENCE and OBJECT_REFERENCE cells. 

                   t    --------------    --------------
      It stops at: ---> |REFERENCE   |--->|VARIABLE    |
                        --------------    --------------
      or at any cell different from REFERENCE and OBJECT_REFERENCE.

Note: it doesn't do substitution evaluaton. 
-----------------------------------------------------------------------------*/
global	void
variable_dereference(cell *t)
{
	while (IsReference(*t) && ! IsVariable(Value(*t)))
		*t = Value(*t);
	object_dereference(t);
}
/*-----------------------------------------------------------------------------
object_dereference(t)
    Removes any unnecessery OBJECT_REFERENCE cell.

                        t    ---------------------    ---------------------
      It stops at:	---->|OBJECT_REFERENCE   |--->|OBJECT_VARIABLE    |
		             ---------------------    ---------------------
      or at any cell different from OBJECT_REFERENCE.

Note: it doesn't do substitution evaluaton. 
-----------------------------------------------------------------------------*/
global	void
object_dereference(cell *t)
{
	while (IsObjectReference(*t) && ! IsObjectVariable(ObjectValue(*t)))
		*t = ObjectValue(*t);
}
/*-----------------------------------------------------------------------------
drop_substitution(t)
    Remove substitution like:
	a. [x/x]
        b. [x/v]  (where v is a local object variable)
        c. [t/x] and x not_free_in A
-----------------------------------------------------------------------------*/
local	void
drop_substitution(VALUE *t)
{
	cell	range, domain, bound, term;
	cell	*delays;
	boolean	found;

	while (t->sub != EMPTY_SUB)
	{
		VariableDereference(range, Reference(&Range(t->sub, 1)));
		ObjectDereference(domain, Domain(t->sub, 1));
		if (Size(t->sub) == 1 && range == domain)
		{
				/* [x/x] * A => A */
			t->sub = NextSub(t->sub);
		}
		else if (IsLocalObjectVariable(domain))
		{
				/* [x/v] * A => A */
			t->sub = NextSub(t->sub);
		}
		else if (Size(t->sub) == 1)
		{
				/* [t/x] * A and x not_free_in A => A */
			found = FALSE;
			delays = (cell *)RestOfVariable(Value(t->term));
			while (delays != NULL)
			{
				VariableDereference(bound,
					    Reference(&DelayedTerm(delays)));
				VariableDereference(term,
					    Reference(&DelayedVar(delays)));
				if (DelayedType(delays) == NOT_FREE &&
				    domain == bound &&
				    IsReference(term))
				{
					found = TRUE;
					t->sub = NextSub(t->sub);
					break;
				}
				else
					delays = Next(delays);
			}
			if (!found)
			{
				return;
			}
		}
		else
			return;
	}
}

/*----------------------------------------------------------------------------
yield_constant(sub, constant)
    Boolean function, whish returns:
       TRUE  - if substitution yields the constant (we can   
	       get the constant from the substitution).
       FALSE - if substitution doesn't yield the constant.
----------------------------------------------------------------------------*/
global	boolean
yield_constant(cell sub, cell constant)
{
	natural	i, j;
	VALUE	t;

	if (Invertible(sub) || ObjectVariablesOnly(sub))
	    /*
	       substitution is either empty or all ranges   
	       in it are only object variables 
	    */
		return(FALSE);
	else
		for (i = Size(sub), j = 1; j <= i; j++)
		{
			DereferenceTerm(t, Range(sub, j));
			if ((IsReference(t.term) && !Frozen(t.term)) ||
			    t.term == constant ||
			    (IsObjectReference(t.term) &&
			     yield_constant(t.sub, constant)))
                            /*  
                               - frozen variable is like a constant, 
			         but always different from any other
			       - object variable can't be bound to  
			         other then object variable, so we  
			         can expect only substitution to 
                                 yield the constant 
                            */
				return(TRUE);
		}
	return(yield_constant(NextSub(sub), constant));
}

/*----------------------------------------------------------------------------
yield_tag(sub, tag)
    Boolean function, whish returns:
       TRUE  - if substitution yields a data object with asked tag.  
       FALSE - if substitution doesn't yield a data object with asked tag. 
----------------------------------------------------------------------------*/
global	boolean
yield_tag(cell sub, int tag)
{
	natural	i, j;
	VALUE	t;

	if (Invertible(sub) || ObjectVariablesOnly(sub))
	    /*
	       substitution is either empty or all ranges   
	       in it are only object variables 
	    */
		return(FALSE);
	else
		for (i = Size(sub), j = 1; j <= i; j++)
		{
			DereferenceTerm(t, Range(sub, j));
			if ((IsReference(t.term) && !Frozen(t.term)) ||
			    Tag(t.term) == tag ||
			    (IsObjectReference(t.term) &&
			     yield_tag(t.sub, tag)))
				return(TRUE);
		}
	return(yield_tag(NextSub(sub), tag));
}

/*----------------------------------------------------------------------------
yield_object_variable(sub, objvar)
    Boolean function, whish returns:
       TRUE  - if substitution yields the object variable.  
       FALSE - if substitution doesn't yield the object variable.
----------------------------------------------------------------------------*/
global	boolean
yield_object_variable(cell sub, cell objvar)
{
	natural	i, j;
	VALUE	t;

	if (sub == EMPTY_SUB)
		return(FALSE);
	else
		for (i = Size(sub), j = 1; j <= i; j++)
		{
			DereferenceTerm(t, Range(sub, j));
			if ((IsReference(t.term) && !Frozen(t.term)) ||
			    (IsObjectReference(t.term) &&
			     (!distinct_from(t.term, objvar) ||
			      yield_object_variable(t.sub, objvar))))
				return(TRUE);
		}
	return(yield_object_variable(NextSub(sub), objvar));
}

/*----------------------------------------------------------------------------
yield_any_object_variable(sub)
    Boolean function, whish returns:
       TRUE  - if substitution yields any object variable.  
       FALSE - if substitution doesn't yield any object variable.
----------------------------------------------------------------------------*/
global	boolean
yield_any_object_variable(cell sub)
{
	natural	i, j;
	VALUE	t;

	if (sub == EMPTY_SUB)
		return(FALSE);
	else
		for (i = Size(sub), j = 1; j <= i; j++)
		{
			DereferenceTerm(t, Range(sub, j));
			if ((IsReference(t.term) && !Frozen(t.term)) ||
			    (IsObjectReference(t.term) &&
			     !IsLocalObjectVariable(t.term)))
				return(TRUE);
		}
	return(yield_any_object_variable(NextSub(sub)));
}
