/* the hash table for object vars */

/* This module stolen in part from 
    termin.c - Edinburgh term input procedure for IC-Prolog ][
    Written by Frank McCabe, Philip Schwarz and Damian Chu
    Imperial College, Winter 1989

    Modifications:
    5/3/90      dac
        added garbage collection
    1/11/90     Gerard Ellis
        interfaced to QuAM data structures
        removed garbage collection
*/

/*
 * The head of the chain is returned by hash_object_var.
 * if (names are equal), object_var returns true
 * if ((names are not equal) && (next is NO_NEXT)), object_var returns false
 * otherwise repeat for next
 */

#include <string.h>
#include "cells.h"
#include "data_area.h"
#include "dereference.h"
#include "defs.h"
#include "errors.h"
#include "name_table.h"
#include "primitives.h"
#include "string_table.h"
#include "read.h"
#include "x_registers.h"

#define OBVAR_TABLE_BLOCKSIZE 100
#define OBVAR_STRINGS_BLOCKSIZE 500

typedef struct OBJECT_VAR_SHELL {
	unsigned		noff; 	/* offset into obvar_strings */
	struct OBJECT_VAR_SHELL	*next;	/* link to next hash table entry */
} object_var_shell, *obvarp;

static obvarp	object_var_table, 	/* the bottom of the hash table */
	table_top,		/* the top of the table */
	prev_entry,		/* the previous entry position */
	next_free;		/* the next free position */

char *obvar_strings;		/* array of strings */
unsigned current_string_pos;	/* offset into this array of next position */

unsigned	num_entries=0;
unsigned	table_size=0;
unsigned 	obvar_strings_size=0;

#define	Noff(p)		((p)->noff)
#define	Nexto(p)		((p)->next)
#define	NO_NEXT		((obvarp)1)
#define	Name(p)		(&(obvar_strings[Noff(p)]))
#define	NoNext(p)	((p)->next == NO_NEXT)	/* dummy address */
#define	HasNext(p)	((p)->next > NO_NEXT)	/* address if there is a link */
#define	Filled(p)	(Nexto(p))		/* if empty next is null */

/* forward reference */
void hash_obvar(char *s);


extern int hash_variable_name (char *variable_name);
extern int unify (VALUE *t1, VALUE *t2);

void
init_object_vars(void)
{
	if(!(object_var_table = 
			(obvarp)calloc(	table_size=OBVAR_TABLE_BLOCKSIZE,
					sizeof(object_var_shell)
				       )
	    ))
		fatal("unable to allocate space for object variables");
	table_top = object_var_table + table_size;
	next_free = table_top - 1;
	prev_entry = table_top - 1;
	
	if(!(obvar_strings =
			malloc( obvar_strings_size=(OBVAR_STRINGS_BLOCKSIZE) )
	    ))
		fatal("unable to allocate space for object variables strings");
	current_string_pos = 0;

	/*
	hash_obvar("x");
	hash_obvar("y");
	hash_obvar("z");
	*/
}

void 
expand_obvar_strings(void)
{
	if(!(obvar_strings =
		    realloc( obvar_strings,
			     obvar_strings_size+=(OBVAR_STRINGS_BLOCKSIZE) )
           ))
                fatal("unable to reallocate space for object variables strings");

}

unsigned
add_obvar_string(char *s)
{
    int	l = strlen(s)+1;
    unsigned return_pos = current_string_pos;
	if((current_string_pos += l) >= obvar_strings_size)
		expand_obvar_strings();
	strcpy(obvar_strings + return_pos, s);
	return(return_pos);
}

void 
rehash_object_vars(void)
{
	obvarp old = object_var_table;
	if(!(object_var_table =
			(obvarp)calloc(	table_size+=OBVAR_TABLE_BLOCKSIZE,
					sizeof(object_var_shell)
				       )
	    ))
		fatal("unable to allocate space for object variables");
	next_free = object_var_table + table_size - 1;

	{ obvarp tmp;
	  for(tmp = old ;tmp < table_top; tmp++)
		if(Filled(tmp))
			hash_obvar(Name(tmp));
	}
	table_top = object_var_table + table_size;
	free(old);
}

/* assigns to next field of o the next_free */
/* assigns to next_free the next available position */
/* returns previous next_free */
static obvarp
get_next(obvarp o)
{
	obvarp ret_next = next_free;
	Nexto(o) = next_free;
	/* find new next_free */
	next_free--;
	while(Filled(next_free) && (next_free >= object_var_table))
		next_free--;
	return(ret_next);
}


/* returns whether s has been declared as an object var 
 * sets pos to pointer into object_var_table of s if it exists
 * sets pos to tail of chain if position of hash is filled
 * otherwise sets pos to position of hash 
 */
boolean
is_obvar(char *s, obvarp *ppos)
{
	*ppos=object_var_table + hash_variable_name(s) % table_size;

        if(Filled(*ppos)) {
            boolean found;
            while (!(found = (!strcmp(s,Name(*ppos))))
                   &&
                   HasNext(*ppos)
                  )
                *ppos = Nexto(*ppos);
            if (found) return(TRUE);
	}
	return(FALSE);
}
	
/* enter a string in the obvar hash table */
void
hash_obvar(char *s)
{
	obvarp pos;
	if(!is_obvar(s, &pos)){
	    /* no entry exists yet */
		/* if have hashed onto next_free */
		if(pos == next_free)
			(void)get_next(pos);
		/* if position filled */
		if(Filled(pos)) pos=get_next(pos);

		/* put new entry at position pos */
		Noff(pos)=add_obvar_string(s);
		Nexto(pos)=NO_NEXT;
		num_entries++;
		prev_entry = pos;

		/* if table is full, rehash now */
		if(next_free < object_var_table)
			rehash_object_vars();
	}
}

/* returns the portion of the string not including an underscore */
char *
get_prefix(char *s)
{
	char *tmp, *tmpbase;

	tmp = (char *)strdup(s);
	tmpbase = tmp;
	while(*tmp && (*tmp != '_'))
		tmp++;
	*tmp = (char)0;
	return(tmpbase);
}

/* $object_var ***************************************************************
declare x as an object variable prefix by asserting it in the table 
---------------------------------------------------------------------------*/
global boolean
esc_object_var(void)
{
	if( IsAtom(Xdref(0)) )
		{
		(void)hash_obvar(get_prefix(String(X(0))));
		return(TRUE);
		}
	else
		return(FALSE);
}
/*****************************************************************************/

/* boolean $object_var_prefix_declared *****************************************
	return TRUE iff the prefix of the string argument has been declared as
	an object variable.
	else return guess wot.
-----------------------------------------------------------------------------*/
boolean
esc_object_var_prefix_declared(void)
{
	if( IsAtom(Xdref(0)) )
		{
		obvarp	dummy;
		return(is_obvar(get_prefix(String(X(0))), &dummy));
		}
	else
		return(FALSE);
}
/*****************************************************************************/

/* boolean $object_var_prefix(Objvar) ****************************************
	return TRUE and unify variable argument with some object variable if
	any have been declared
	else return FALSE
-----------------------------------------------------------------------------*/
boolean
esc_object_var_prefix(void)
	/* VARIABLE X; */
{
	if(num_entries)
		{
		VALUE val;
		obvarp tmp = table_top - 1;

		/* find some object var */
		val.sub = EMPTY_SUB;
		val.term = Atom(add_name_string_offset(Name(prev_entry),
		                                       ATOM_W));
		return(unify(XV(0), &val));
		}
	else
		return(FALSE);
}
/*****************************************************************************/

global boolean
object_var_prefix_declared(char *s)
{
	obvarp	dummy;
	return(is_obvar(get_prefix(s), &dummy));
}
/*****************************************************************************/

global char *
object_var_prefix(void)
{
	return(Name(prev_entry));
}

/*----------------------------------------------------------------------------
object_var_prefix_first() 
    gives name of the first declared object variable for a prefix 

----------------------------------------------------------------------------*/
global char *
object_var_prefix_first(void)
{
         return(Name(table_top - 1)); 
}

