/* the hash table for operators */

/* This module stolen in part from 
    termin.c - Edinburgh term input procedure for IC-Prolog ][
    Written by Frank McCabe, Philip Schwarz and Damian Chu
    Imperial College, Winter 1989

    Modifications:
    5/3/90      dac
        added garbage collection
    1/11/90     Gerard Ellis
        interfaced to QuAM data structures
        removed garbage collection
*/

/*
 * The head of the chain is returned by hash_variable_name.
 * if (names are equal), op() returns precedence
 * if ((names are not equal) && (lnk is NO_NEXT)), op() returns 0
 * otherwise repeat for lnk
 */

#include <string.h>
#include "cells.h"
#include "data_area.h"
#include "dereference.h"
#include "defs.h"
#include "errors.h"
#include "name_table.h"
#include "ops.h"
#include "primitives.h"
#include "string_table.h"
#include "termio.h"
#include "x_registers.h"

#define OP_TABLE_BLOCKSIZE 100
#define OP_STRINGS_BLOCKSIZE 500

/* global table pointers */
static operatorpo	op_table, 		/* the bottom of the hash table */
	table_top,		/* the top of the table */
	next_free;		/* the next free position */

static unsigned	table_size=0;

#define	NO_NEXT		((operatorpo)1)
#define	NoLnk(p)	((p)->lnk == NO_NEXT)	/* dummy address */
#define	HasLnk(p)	((p)->lnk > NO_NEXT)	/* address if there is a link */
#define	Filled(p)	(Lnk(p))		/* if empty next is null */


extern int unify (VALUE *t1, VALUE *t2);

void
init_ops(void)
{
	if(!(op_table = 
			(operatorpo)calloc(	table_size=OP_TABLE_BLOCKSIZE,
					sizeof(operator)
				       )
	    ))
		fatal("unable to allocate space for ops");
	table_top = op_table + table_size;
	next_free = table_top - 1;
}

void
add_op_string(operatorpo p, char *s)
{
    strncpy(Op(p),s,(MAXOPLEN - 1));
}

void 
rehash_ops(void)
{
	operatorpo old = op_table;
	if(!(op_table =
			(operatorpo)calloc(	table_size+=OP_TABLE_BLOCKSIZE,
					sizeof(operator)
				       )
	    ))
		fatal("unable to allocate space for ops");
	next_free = op_table + table_size - 1;

	{ operatorpo tmp;
	  for(tmp = old ;tmp < table_top; tmp++)
		if(Filled(tmp))
			hash_op(Op(tmp),
				/*
				Hsh(tmp),
				*/
				Prefixp(tmp),
				Preform(tmp),
				Infleft(tmp),
				Infixp(tmp),
				Infright(tmp),
				Postleft(tmp),
				Postfixp(tmp),
				Quantp(tmp)
				/*
				,
				Lnk(tmp)
				*/
				);
	}

	table_top = op_table + table_size;
	free(old);
}

/* assigns to next field of o the next_free */
/* assigns to next_free the next available position */
/* returns previous next_free */

static operatorpo
get_next(operatorpo o)
{
	operatorpo ret_next = next_free;
	/* twoBytes  exper; */
	Lnk(o) = next_free;
	/* exper = (twoBytes)next_free; */
	/* find new next_free */
	next_free--;
	while(Filled(next_free) && (next_free >= op_table))
		next_free--;
	return(ret_next);
}

operatorpo next_filled_pos;

static operatorpo
next_filled(operatorpo o)
{
	if(next_filled_pos < o)
		o = next_filled_pos;
	else
		{
		o--;
		while(!Filled(o) && (o >= op_table))
			o--;
		next_filled_pos = o;
		}
	/* make new next filled */
	next_filled_pos--;
	while(!Filled(next_filled_pos) && (next_filled_pos >= op_table))
	    next_filled_pos--;
	
	return(o);
}

/* same as in read.c */
global int
hash_variable_name(char *variable_name)
{
    int i;
    int len = strlen(variable_name);
    int hash = 0;

    while (*variable_name) {
        hash += *variable_name++;
    }
    return((hash & 0x0FFFF) | len<< 16);
}

/* returns whether s has been declared as a op 
 * sets pos to pointer into op_table of s if it exists
 * sets pos to tail of chain if position of hash is filled
 * otherwise sets pos to position of hash 
 */

boolean
is_operator(char *s, operatorpo *ppos)
{
	*ppos=op_table + hash_variable_name(s) % table_size;

        if(Filled(*ppos)) {
            boolean found;
            while (!(found = (!strncmp(s,Op(*ppos),MAXOPLEN - 1)))
                   &&
                   HasLnk(*ppos)
                  )
                *ppos = Lnk(*ppos);
            if (found) return(TRUE);
	}
	return(FALSE);
}

#define Hash_op_error(o) { warning("illegal op redefinition: %s",o);	\
			   return(FALSE);				\
			   }
	
/* enter an operator in the op hash table */
/* if an entry exists mask it with new entry */
/* returns success value */

#define Defd(p) ((p) != undefprior)

boolean
hash_op(char *o, int prep, int prefm, int infl, int infp, int infr, int postl, int postp, int quantp)
{
	operatorpo pos;
	if(is_operator(o, &pos))
	    {
	    /* operator redefinition */
	    if(quantp)
		{
	    /*
		if(Defd(Postfixp(pos)) 
		   || Defd(Infixp(pos)) 
		   || Defd(Prefixp(pos))
		   )  
			Hash_op_error(o);
		*/
		Quantp(pos) = quantp;
		return(TRUE);
		}
	    if(prep) 
		{
		/*
		if(Defd(Quantp(pos))) Hash_op_error(o);
		if(Defd(Postfixp(pos))) Hash_op_error(o);
		if(Defd(Infixp(pos)) && Infixp(pos) != prep) Hash_op_error(o);
		*/
		Prefixp(pos) = prep;
		Preform(pos) = prefm;
		return(TRUE);
		}
	    if(infp) 
		{
		/*
		if(Defd(Quantp(pos))) Hash_op_error(o);
		if(Defd(Prefixp(pos)) && Prefixp(pos) != infp) Hash_op_error(o);
		if(Defd(Postfixp(pos)) 
		   && Postfixp(pos) != infp
		   ) Hash_op_error(o);
		   */
		Infixp(pos) = infp;
		Infleft(pos) = infl;
		Infright(pos) = infr;
		return(TRUE);
		}
	    if(postp) 
		{
		/*
		if(Defd(Quantp(pos))) Hash_op_error(o);
		if(Defd(Prefixp(pos))) Hash_op_error(o);
		if(Defd(Infixp(pos)) && Infixp(pos) != postp) Hash_op_error(o);
		*/
		Postfixp(pos) = postp;
		Postleft(pos) = postl;
		return(TRUE);
		}
	    }
	else
	    {
	    /* no current entry */
	    if(pos == next_free)
		(void)get_next(pos);
	    /* if position filled */
	    if(Filled(pos))
	       pos=get_next(pos);

	    /* put new entry at position pos */
	    add_op_string(pos,o);
	    Prefixp(pos) = (prep?prep:undefprior);
	    Preform(pos) = (prefm?prefm:undefprior);
	    Infleft(pos) = (infl?infl:undefprior);
	    Infixp(pos) = (infp?infp:undefprior);
	    Infright(pos) = (infr?infr:undefprior);
	    Postleft(pos) = (postl?postl:undefprior);
	    Postfixp(pos) = (postp?postp:undefprior);
	    Quantp(pos) = (quantp?quantp:undefprior);
	    Lnk(pos)=NO_NEXT;

	    /* if table is full, rehash now */
	    if(next_free < op_table)
		rehash_ops();
	    return(TRUE);
	    }
}


operatorpo
oplook(char *name)
{
	operatorpo	tmp;
	if(is_operator(name, &tmp))
	    return(tmp);
	else
	    return((operatorpo)0);
}

/*
    Tests to see if a string is an operator.
    Returns the corresponding entry in the table.
*/
operatorpo
is_op(token *tok)
{
    switch(tok->tt) {
        case solo:
        case graph:
        case lower:
        case quoted:
        case comma:
        case semicolon: {
            register
            char        *s      = tok->buff;
            operatorpo  dummy;

            return(oplook(s));
        }

        default:                /* by default a token is not an operator */
            return(NULL);
    }
}


unsigned
quantifier(char *q)
{
	operatorpo tmp;
	if(!(tmp = oplook(q))) 
	    return 0;
	else
	    return(Defd(Quantp(tmp)) ? Quantp(tmp) : 0);
}

global boolean
esc_op_(void)
    /*
    int Prec
    int Assoc
    Atom Op
    */
	{
	if(IsInteger(Xdref(1))
	   &&
	   IsAtom(Xdref(2))
	   &&
	   IsInteger(Xdref(0))
	   )
		{
		char *op=String(X(2));
		int prec = IntOf(X(0));
		int assoc = IntOf(X(1));
		switch(assoc)	
		    {
		    when FX:
			return(hash_op(op, prec, prec - 1, 0, 0, 0, 0, 0, 0));
		    when FY:
			return(hash_op(op, prec, prec, 0, 0, 0, 0, 0, 0));
		    when XF:
			return(hash_op(op, 0, 0, 0, 0, 0, prec - 1, prec, 0));
		    when YF:
			return(hash_op(op, 0, 0, 0, 0, 0, prec, prec, 0));
		    when XFX:
			return(hash_op(op, 0, 0, prec-1, prec, prec-1, 0, 0, 0));
		    when XFY:
			return(hash_op(op, 0, 0, prec - 1, prec, prec, 0, 0, 0));
		    when YFX:
			return(hash_op(op, 0, 0, prec, prec, prec - 1, 0, 0, 0));
		    when YFY:
			return(hash_op(op, 0, 0, prec, prec, prec, 0, 0, 0));
		    when QUANT:
			return(hash_op(op, 0, 0, 0, 0, 0, 0, 0, prec));
		    }
		}
	else
		return(FALSE);
	}

int 
get_prec(operatorpo op, int arity)
{
	if(arity == 2) return(Infixp(op));
	if(Defd(Quantp(op))) return(Quantp(op));
	if(Defd(Prefixp(op))) return(Prefixp(op));
	return(Postfixp(op));
}

int 
get_assoc(operatorpo op, int arity)
{
	if(arity == 1)
		{
		if(Defd(Quantp(op)))
			return(QUANT);
		if(Defd(Prefixp(op)))
			return((Preform(op) < Prefixp(op))? FX : FY);
		else
			return((Postleft(op) < Postfixp(op))? XF : YF);
		}
	else
		{
		if(Infleft(op) == Infright(op))
			{
			if(Infixp(op) > Infleft(op)) return(XFX);
			else return(YFY);
			}
		else
			{
			if(Infixp(op) > Infleft(op)) return(XFY);
			else return(YFX);
			}
		}
}

char * 
get_op(operatorpo op)
{
	return(Op(op));
}

static operatorpo last_filled_pos;

global boolean
esc_first_op(int Ref)
{
	operatorpo reference = table_top;
	VALUE ref;

	next_filled_pos = reference;
	reference = next_filled(reference);

	/* set last_filled_pos position */
	last_filled_pos = op_table;
	while(!Filled(last_filled_pos) && (last_filled_pos < table_top))
		last_filled_pos++;

	ref.term = Integer(((int)reference - (int)op_table)
			    /
			    sizeof(operator));
	ref.sub = EMPTY_SUB;
	return(unify(&ref, XV(0)));
}


global boolean
esc_some_op(void)
{
	int arity;
	/* arity must not be a variable */


	if(!IsInteger(Xdref(4))) return(FALSE);
	arity = IntOf(X(4));

	/* if operator name is constant, hash to find it */
	if(IsAtom(Xdref(3)))
		{
		/* op name given so we can hash */
		operatorpo op;
		if(!(op = oplook(String(X(3))))) return(FALSE);
		{
		VALUE prec, assoc, opname;
		int p;
		if(!Defd(p = get_prec(op, arity)))
		    return(FALSE);
		prec.term = Integer(p);
		assoc.term = Integer(get_assoc(op, arity));
		opname.term = Atom(add_name_string_offset(get_op(op), ATOM_W));

		prec.sub = assoc.sub = opname.sub = EMPTY_SUB;

		return(unify(&prec, XV(1))
		       &&
		       unify(&assoc, XV(2))
		       &&
		       unify(&opname, XV(3))
		       );
		}
		}
	else
		{
		/* no op given, just look at ref position */
		int ref;
		/* ref must not be a variable */
		if(!IsInteger(Xdref(0))) return(FALSE);
		ref = IntOf(X(0));

		{
		VALUE prec, assoc, opname;
		int p;
		if(!Defd(p = get_prec(op_table + ref, arity)))
			return(FALSE);
		prec.term = Integer(p);
		assoc.term = Integer(get_assoc(op_table + ref, arity));
		opname.term = Atom(add_name_string_offset(
				   get_op(op_table + ref), ATOM_W));

		prec.sub = assoc.sub = opname.sub = EMPTY_SUB;

		return(unify(&prec, XV(1))
		       &&
		       unify(&assoc, XV(2))
		       &&
		       unify(&opname, XV(3))
		       );
		}
		}
}

global boolean
esc_next_op(void)
{
	operatorpo reference;
	VALUE nextref; 
	int refoffset;

	/* ref must not be a variable */
	if(!IsInteger(Xdref(0))) return(FALSE);
	refoffset = IntOf(X(0));

	reference = next_filled(op_table + refoffset);

	nextref.term = Integer(((int)(reference) - (int)op_table) 
			       /
			       sizeof(operator));
	nextref.sub = EMPTY_SUB;
	return(unify(&nextref, XV(1)));
}

/* check if any more filled positions */
global boolean
esc_last_op(void)
{
	int refoffset;

	/* ref must not be a variable */
	if(!IsInteger(Xdref(0))) return(FALSE);
	refoffset = IntOf(X(0));

	if( ((op_table + refoffset) <= last_filled_pos)
	    ||
	    last_filled_pos >= table_top)
		return(TRUE);
	else
		return(FALSE);
	}

