/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include "data_area.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "substitution.h"
#include "unify.h"
#include "x_registers.h"

/*------------------------------------------------------------------------------
   if (one of the range in the substitution is non object variable)
  	return property OTHERS;
   else
  	return property OBJECT VARIABLE ONLY;
 
------------------------------------------------------------------------------*/
global	int
determine_property(cell sub)
{
	cell	r;
	natural i, j;

	for (i = Size(sub), j = 1; j <= i; j++)
	{
		r = Reference(&Range(sub, j));
		variable_dereference(&r);
		if (!IsObjectReference(r))
			return(OTHERS);
	}
	return(OBJECT_VARIABLES_ONLY);
}

/*------------------------------------------------------------------------------
invert(sub, t)
    invert substitution with local object variables.
    Given [yn/vn] * ... * [y1/v1] * [v1/x1] * ... * [vn/xn]
    inverts to [xn/vn] * ... * [x1/v1] * [v1/y1] * ... * [vn/yn]
    note: The order of the inverted substitution is IMPORTANT.  The inverted
  	  substitutions behaves a bit like bracketing.  If the domain is
  	  original object variable, a free_in test must be done.
    Potential bug:  [v/z] * z = [v/y] * [y/v'] * [v'/x] * A, A can be x or y.
   
------------------------------------------------------------------------------*/
global	boolean
invert(cell sub, VALUE *t)
{
	cell	objvar;

	if (sub != EMPTY_SUB)
	{
		if (!invert(NextSub(sub), t))
			return(FALSE);
		else
		{
			objvar = Domain(sub, 1);
			object_dereference(&objvar);
			if (!IsLocalObjectVariable(objvar) &&
			    freeness(objvar, t))
				return(FALSE);
			else
				t->sub = add_substitution(t->sub,
					RenameSub(Range(sub, 1), objvar,
						  EMPTY_SUB));
		}
	}
	return(TRUE);
}

/*------------------------------------------------------------------------------
add_substitution(front, back)
   Optimisation can be performed when the substitution is added to the front.
                  s1 <- [x/v] |<- [v/x] <-
                              |
                           s1 |<-
------------------------------------------------------------------------------*/
global	cell
add_substitution(cell front, cell back)
{
	cell	next;
	int	property;

	if (front == EMPTY_SUB)
		return(back);
	else if (back == EMPTY_SUB)
		return(front);
	else
	{
		next = add_substitution(NextSub(front), back);
		if (next != EMPTY_SUB &&
		    IsLocalObjectVariable(Range(front, 1)) &&
		    Range(front, 1) == Domain(next, 1) &&
		    Domain(front, 1) == Range(next, 1))
			return(NextSub(next));
		else
		{
			property = Property(next) > Property(front) ?
					Property(next) : Property(front);
			return(NewSubstitution(Table(front), next, property));
		}
	}
}
/*------------------------------------------------------------------------------
copy_substitution(sub, stop_point)
    Copy susubstitutions from the most left one to the stop point 
    onto the heap. 
    New proper properties are put.
    It returns pointer to the new built substitution.
------------------------------------------------------------------------------*/
global	cell
copy_substitution(cell sub, cell stop_point)
{
	cell	next, copied;
	int	property, prop;

	if (sub == stop_point)
		return(EMPTY_SUB);
	else
	{
		next = copy_substitution(NextSub(sub), stop_point);
		if (Others(next))
			property = OTHERS;
		else
		{
			prop = ContainLocal(sub, 1) ? INVERTIBLE :
						      determine_property(sub);
			property = Property(next) > prop ? Property(next) :
							   prop;
		}
		copied = NewSubstitution(Table(sub), next, property);
		return(copied);
	}
}
/*------------------------------------------------------------------------------
in_range(objvar, sub)
    If the object variable occurs as the most right range in at least 
       one of substitutions in a composition of substitutions 
	returns TRUE
    else
	returns FALSE
note: Variable dereference has been done for the most right range.
------------------------------------------------------------------------------*/
global	boolean
in_range(cell objvar, cell sub)
{
	VALUE	val;

	for (;sub != EMPTY_SUB; sub = NextSub(sub))
	{
		DereferenceTerm(val, Reference(&Range(sub, 1)));
		if (objvar == val.term)
			return(TRUE);
        }
	return(FALSE);
}

/*------------------------------------------------------------------------------
in_domain(objvar, sub)
    If the object variable occurs as a domain in the substitution 
	returns TRUE
    else
	returns FALSE
note: Only one substitution has been passed as a parametar (not 
      composition of substitution). 
      Full dereference for domain has been done. 
------------------------------------------------------------------------------*/
global	boolean
in_domain(cell objvar, cell sub)
{
	VALUE	val;
	natural i, j;

	for (i = Size(sub), j = 1; j <= i; j++)
	{
		DereferenceTerm(val, Domain(sub, j));
		if (objvar == val.term)
			return(TRUE);
	}
	return(FALSE);
}

/*------------------------------------------------------------------------------
in_all_domain(objvar, sub)
------------------------------------------------------------------------------*/
global	boolean
in_all_domain(cell objvar, cell sub)
{
	return(sub == EMPTY_SUB ? FALSE :
				(in_domain(objvar, sub) ||
				 in_all_domain(objvar, NextSub(sub))));
}
/*------------------------------------------------------------------------------
same_domain(sub1, sub2)

------------------------------------------------------------------------------*/
global	boolean
same_domain(cell sub1, cell sub2)
{
local	boolean same_domain_sub(cell sub1, cell sub2);

	return(Size(sub1) == Size(sub2) &&
	       same_domain_sub(sub1, sub2) &&
	       same_domain_sub(sub2, sub1));
}

/*------------------------------------------------------------------------------
same_domain_sub(sub1, sub2)
------------------------------------------------------------------------------*/
local	boolean
same_domain_sub(cell sub1, cell sub2)
{
local	boolean same_domain_element(cell objvar, cell sub);
	VALUE	val;
	natural i, j;

	for (i = Size(sub1), j = 1; j <= i; j++)
	{
		DereferenceTerm(val, Domain(sub1, j));
		if (! same_domain_element(val.term, sub2))
			return(FALSE);
	}
	return(TRUE);
}
/*------------------------------------------------------------------------------
same_domain_element(objvar, sub)
------------------------------------------------------------------------------*/
local	boolean
same_domain_element(cell objvar, cell sub)
{
	VALUE	val;
	natural i, j;

	for (i = Size(sub), j = 1; j <= i; j++)
	{
		DereferenceTerm(val, Domain(sub, j));
		if (objvar == val.term)
			return(TRUE);
	}
	return(FALSE);
}
