/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <sys/wait.h>
#include <sys/types.h>
#include <sys/times.h>

#include "cells.h"
#include "code_area.h"
#include "data_area.h"
#include "database.h"
#include "debug.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "errors.h"
#include "io.h"
#include "main.h"
#include "name_table.h"
#include "string_table.h"
#include "system.h"
#include "unify.h"
#include "x_registers.h"

local	QUFLAG	flag_table[FLAG_NUMBER];
local	natural	flag_num = 0;

/*----------------------------------------------------------------------------
get_args(argv)
    argv is the list of atoms on the mainline call.
    Put the list of atoms onto the heap in WAM format
    e.g. [abc, def, ghi] is @(@(., abc), @(@(., def), @(@(., ghi), [])))

	atoms([], []).
	atoms([H|T], @(@(., H), Rest)) :-
	    atoms(T, Rest).
----------------------------------------------------------------------------*/

extern int fprintf (FILE *, const char *, ...);
int exv (char *filename, char **argv);
extern int execvp (const char *, char *const *);
extern pid_t fork (void);
extern int pipe (int *);
extern int close (int);
extern int dup2 (int, int);
extern int esc_fdopen (int fd, char *mode, cell *fd_term);
extern pid_t vfork (void);

global	boolean
get_args(void)
{
	cell	head_of_list;
reg	cell	*end_of_list;
reg	cell	value;
	VALUE	val;

	natural	i;

	end_of_list = &head_of_list;
	i = 0;
	while (argv_save[i] != (char *)NULL)
	{
		*end_of_list = value = Apply();
		Functor(value) = Apply();
		Functor(Functor(value)) = Atom(CONS);
		Argument(Functor(value)) = 
			Atom(add_name_string_offset(argv_save[i++], ATOM_W));
		end_of_list = &Argument(value);
	}
	*end_of_list = Atom(NIL);
	val.term = head_of_list;
	val.sub = EMPTY_SUB;
	return(unify(XV(0), &val));
}


/*----------------------------------------------------------------------------
$getwd(WorkingDirectory) :-
	Exit Qu-Prolog with exit code Code.
----------------------------------------------------------------------------*/
global	boolean
esc_getwd(void)
{
	char	*cwd;
	char	*getcwd(char *, size_t);
	VALUE	val;

	if ((cwd = getcwd((char *)NULL, CWD_SIZE)) == NULL)
		fatal("could not get working directory");
	else
	{
		val.term = Atom(add_name_string_offset(cwd, ATOM_W));
		val.sub = EMPTY_SUB;
		free(cwd);
		return(unify(XV(0), &val));
	}
	return(FALSE);
}

/*----------------------------------------------------------------------------
$exit(Code) :-
	Exit Qu-Prolog with exit code Code.
----------------------------------------------------------------------------*/
global	boolean
esc_exit(void)
{
	int	i;
	if (IsInteger(Xdref(0)))
	{
		i = IntOf(X(0));
		exit(i);
	}
	else
		exit(1);
	return(TRUE);
}

global	long	last_system_time = 0;
global	long	last_user_time = 0;

/*----------------------------------------------------------------------------
statistics(Statistics) :-
    Compatible with NU-Prolog statistics (here is the blurb for NU-Prolog's
    version)

    Provides statistics about NU-Prolog's use of resources. Statistics is a
    list of elements of the form Name = Value where Name is an atom and
    Value is a list of numbers. Values of Name include

	memory		[total memory, 0]
	program		[total program space allocated, 0]
	delayed		[delayed stack in use, delayed stack free]
	global		[global stack in use, global stack free]
	local		[local stack in use, local stack free]
	trail		[trail stack in use, trail stack free]
	user_time		[cpu time used, cpu time used since last
			call to statistics]
	system_time		[system cpu time used, system cpu time used
			since last call to statistics]
	time		sum of user_time and system_time

    Times are given in milliseconds, and sizes in bytes. The exact
    definitions of some of these quantities may vary from machine to
    machine.
----------------------------------------------------------------------------*/
global	boolean
statistics(void)
{
	struct	tms	buffer;
	int	last_user_microsecs;
	int	last_system_microsecs;
	int	user_microsecs;
	int	system_microsecs;
	VALUE	list;
extern	caddr_t	sbrk(int);

	(void)times(&buffer);

	user_microsecs = buffer.tms_utime * (1000 / (double) Hz);
	last_user_microsecs = (buffer.tms_utime - last_user_time) *
		(1000 / (double) Hz);
	last_user_time = buffer.tms_utime;

	system_microsecs = buffer.tms_stime * (1000 / (double) Hz);
	last_system_microsecs = (buffer.tms_stime - last_system_time) *
		(1000 / (double) Hz);
	last_system_time = buffer.tms_stime;


    list.term = 
		Cons(Group(time, user_microsecs + system_microsecs,
			 last_user_microsecs + last_system_microsecs),
		Cons(Group(stime, system_microsecs, last_system_microsecs),
		Cons(Group(utime, user_microsecs, last_user_microsecs),
		Cons(Group(global, (((char *)top_of_heap) - ((char *)heap)),
			 (((char *)stack) - ((char *)top_of_heap))),
		Cons(Group(local, (((char *)top_of_stack) - ((char *)stack)),
			 (((char *)end_of_stack) - ((char *)top_of_stack))),
		Cons(Group(trail, (((char *)top_of_trail) - ((char *)trail)),
			 (((char *)end_of_trail) - ((char *)top_of_trail))),
		Cons(Group(delayed, (((char *)top_delayed_stack) -
						((char *)delayed_stack)),
			 	    (((char *)end_delayed_stack) -
						((char *)top_delayed_stack))),
		Cons(Group(program, Kwords_to_chars(code_area_size +
			 string_table_size), 0),
		Cons(Group(memory, (int)sbrk(0), 0),
		     Atom(NIL))))))))));
	list.sub = EMPTY_SUB;
	return(unify(XV(0), &list));
}

global boolean
esc_write_time(void)
{
	struct	tms	buffer;
        int     last_user_microsecs;
        int     last_system_microsecs;
        int     user_microsecs;
        int     system_microsecs;

        (void)times(&buffer);

        user_microsecs = buffer.tms_utime * (1000 / (double) Hz);
        last_user_microsecs = (buffer.tms_utime - last_user_time) *
                (1000 / (double) Hz);
        last_user_time = buffer.tms_utime;

        system_microsecs = buffer.tms_stime * (1000 / (double) Hz);
        last_system_microsecs = (buffer.tms_stime - last_system_time) *
                (1000 / (double) Hz);
        last_system_time = buffer.tms_stime;

	fprintf(stderr, " [time = [%d, %d], ",
                user_microsecs + system_microsecs,
                last_user_microsecs + last_system_microsecs);

	return(TRUE);
}

global	boolean
esc_write_statistics(void)
{

#ifdef	EBUG

	struct	tms	buffer;
	int	last_user_microsecs;
	int	last_system_microsecs;
	int	user_microsecs;
	int	system_microsecs;
extern	caddr_t	sbrk(int);

	(void)times(&buffer);

	user_microsecs = buffer.tms_utime * (1000 / (double) Hz);
	last_user_microsecs = (buffer.tms_utime - last_user_time) *
		(1000 / (double) Hz);
	last_user_time = buffer.tms_utime;

	system_microsecs = buffer.tms_stime * (1000 / (double) Hz);
	last_system_microsecs = (buffer.tms_stime - last_system_time) *
		(1000 / (double) Hz);
	last_system_time = buffer.tms_stime;

	write_termv(XV(0));

	fprintf(stderr, " [time = [%d, %d], ",
		user_microsecs + system_microsecs,
		last_user_microsecs + last_system_microsecs);
	fprintf(stderr, "stime = [%d, %d], ",
		system_microsecs, last_system_microsecs);
	fprintf(stderr, "utime = [%d, %d], ",
		user_microsecs, last_user_microsecs);
	fprintf(stderr, "global = [%d, %d], ",
		(((char *)top_of_heap) - ((char *)heap)),
		(((char *)stack) - ((char *)top_of_heap)));
	fprintf(stderr, "local = [%d, %d], ",
		(((char *)top_of_stack) - ((char *)stack)),
		(((char *)end_of_stack) - ((char *)top_of_stack)));
	fprintf(stderr, "trail = [%d, %d], ",
		(((char *)top_of_trail) - ((char *)trail)),
		(((char *)end_of_trail) - ((char *)top_of_trail)));
	fprintf(stderr, "pstack = [%d, %d], ",
		(((char *)top_of_pstack) - ((char *)pstack)),
		(((char *)end_of_pstack) - ((char *)top_of_pstack)));
	fprintf(stderr, "delayed = [%d, %d], ",
		(((char *)top_delayed_stack) - ((char *)delayed_stack)),
		(((char *)end_delayed_stack) - ((char *)top_delayed_stack)));
	fprintf(stderr, "program = [%d, %d], ",
		Kwords_to_chars(code_area_size + string_table_size), 0);
	fprintf(stderr, "memory = [%d, %d]]\n",
		(int)sbrk(0), 0);

#endif /* EBUG */

	return(TRUE);
}

/*----------------------------------------------------------------------------
$label(X) :-
	true, iff X is points to the last choice point.
----------------------------------------------------------------------------*/
global	boolean
esc_label(void)
{
	VALUE	val;

	val.term = PtrToInt(last_choice_point);
	val.sub = EMPTY_SUB;
	return(unify(XV(0), &val));
}

/*----------------------------------------------------------------------------
$cut(X) :-
	true, iff X is an integer.
----------------------------------------------------------------------------*/
global	boolean
esc_cut(void)
{
	cell	*p;

	if (IsInteger(Xdref(0)) &&
	    (p = (cell *) IntToPtr(X(0))) >= stack &&
	    p <= end_of_stack)
	{
	    last_choice_point = (CHOICE *) p;
	    return(TRUE);
	}
	return(FALSE);
}

/*----------------------------------------------------------------------------
$system(ProgramName, Arguments, Status) :-
    True iff Status unifies with the status returned from unix after calling
    the program of ProgramName with the list of arguments Arguments.
----------------------------------------------------------------------------*/
global	boolean
esc_system(void)
{
	char	**argv;
	VALUE	val;
	extern	char	**get_args_from_list(VALUE *term);
    
	if (IsAtom(Xdref(0)) && (argv = get_args_from_list(XV(1))) != NULL)
	{
		val.term = Integer(exv(String(X(0)), argv));
		val.sub = EMPTY_SUB;
		return(unify(XV(2), &val));
	}
	else
		return(FALSE);
}

/*----------------------------------------------------------------------------
exec(ProgramName, Arguments) :-
    The exec system call overlays the process that is running with a new
    program and begins execution of the program at its entry point. 
    This predicate will not return a value and will either terminate
    normally (program successful) or a fatal condition aborts Qu-Prolog.
----------------------------------------------------------------------------*/
global	boolean
exec(void)
{
	char	**argv;
	extern	char	**get_args_from_list(VALUE *term);
    
	if (IsAtom(Xdref(0)) && (argv = get_args_from_list(XV(1))) != NULL)
		return(execvp(String(RestOfConstant(X(0))), argv));
	else
		return(FALSE);
}

global	boolean
esc_fork(void) /* fork() */
{
	int	status;
reg	int	child_pid;
reg	int	wait_pid;

	switch (child_pid = fork())
	{
	when -1:
		warning("cannot fork");
		return(FALSE);
	when 0:
		return(FALSE);
	otherwise:
		return(TRUE);
	}
}


global	boolean
esc_fork3(void) /* fork(Pid, Read, Write) */
{
	VALUE	in;
	VALUE	out;
	VALUE	pid;
	int	fd1[2];
	int	fd2[2];
reg	int	child_pid;


	if (pipe(fd1) == -1)
	{
		warning("Unable to create pipe in $fork3()");
		return(FALSE);
	}
	else if (pipe(fd2) == -1)
	{
		close(fd1[0]);
		close(fd1[1]);
		warning("Unable to create pipe in $fork3()");
		return(FALSE);
	}

	switch (child_pid = fork())
	{
	when -1:
		close(fd1[0]);
		close(fd1[1]);
		close(fd2[0]);
		close(fd2[1]);
		warning("cannot fork in $fork3()");
		return(FALSE);
	when 0:
		close(fd1[0]);
		dup2(fd1[1], 1);
		close(fd1[1]);

		close(fd2[1]);
		dup2(fd2[0], 0);
		close(fd2[0]);
		return(FALSE);
	otherwise:
		return((close(fd1[1]),
			in.sub = EMPTY_SUB,
			esc_fdopen(fd1[0], "r", &(in.term))) &&
		       (close(fd2[0]),
		        out.sub = EMPTY_SUB,
		        esc_fdopen(fd2[1], "w", &(out.term))) &&
		       (pid.term = Integer(child_pid),
			pid.sub = EMPTY_SUB,
			unify(XV(0), &pid)) && 
		       unify(XV(1), &in) &&
		       unify(XV(2), &out));
	}
}

/*----------------------------------------------------------------------------
    return an array of string extracted from the list of atoms.

    the last element is set to NULL. (n + 1 elements in the array
    for list of length n arguments). There must be at least one atom in the
    list.
----------------------------------------------------------------------------*/
global	char**
get_args_from_list(VALUE *term)
{
	VALUE	val;
	VALUE	val1;
	VALUE	val_h;
	char	**argv;
	natural	argc;
	natural	i;

	if (length(term, &argc) && argc > 0)
		argv = (char **) malloc((argc + 1) * sizeof(char *));
	else
		return(NULL);
    
	/* if here then know term is a list  (thanks to length/2) */

	val.term = term->term;
	val.sub = term->sub;


	for (i = 0; i < argc; i++)
	{
		DereferenceTerm(val, val.term);
		if (IsAtom(DerefTerm(val1, Head(val_h, val.term))))
			argv[i] = String(val1.term);
		else
			return(NULL);
		
		val.term = Tail(val.term);
        }
	argv[argc] = (char *) NULL;
	return(argv);
}

/*----------------------------------------------------------------------------
length([], 0).
length([H|T], N) :-
    length(T, M),
    N is M + 1.
----------------------------------------------------------------------------*/
global	boolean
length(VALUE *list, unsigned int *n)
{
	VALUE	val;
	natural	m;

	if ((DereferenceTerm(val, list->term), IsList(val.term)) &&
	    (val.term = Tail(val.term),val.sub = EMPTY_SUB, length(&val, &m)))
	{
		*n = m + 1;
		return(TRUE);
		
	}
	else if (IsNIL(val.term))
	{
		*n = 0;
		return(TRUE);
	}
	else
		return(FALSE);
	
}


/*----------------------------------------------------------------------------
	Compare with NU-Prologs exv.
	Gerard Ellis steals the code from NU-Prolog.
----------------------------------------------------------------------------*/
local int
exv(char *filename, char **argv)
{
	int	status;
reg	int	child_pid;
reg	int	wait_pid;

	switch (child_pid = vfork())
	{
	when -1:
		warning("cannot fork %s", filename);
		return(-1);
	when 0:
		execvp(filename, argv);
		warning("cannot execute %s", filename);
		exit(1);
	otherwise:
		while ((wait_pid = wait(&status)) != child_pid)
			if(wait_pid == -1)
				fatal("system error in exv");
		return((status >> 8) & 255);
	}
}

global cell *read_count_ptr;
global cell *write_count_ptr;
global cell *underline_ptr;

global void 
init_flags(void)
{
	flag_table[0].name = Atom(add_name_string_offset("read_count", ATOM_W));
	flag_table[0].state = Integer(0);
	read_count_ptr = &(flag_table[0].state);
	flag_table[1].name = Atom(add_name_string_offset("write_count",
                                                         ATOM_W));
	flag_table[1].state = Integer(0);
	write_count_ptr = &(flag_table[1].state);
	flag_table[2].name = Atom(add_name_string_offset("underline", ATOM_W));
	flag_table[2].state = Integer(0);
	underline_ptr = &(flag_table[2].state);
	flag_num = 3;
}

global	boolean
esc_setFlag(void)
{
	natural	i;

	if (! IsAtom(Xdref(0)) || ! (IsAtom(Xdref(1)) || IsInteger(X(1))))
		return(FALSE);
	else
	{
		for (i = 0; i < flag_num; i++)
			if (flag_table[i].name == X(0))
			{
				flag_table[i].state = X(1);
				return(TRUE);
			}
		if ((flag_num + 1) >= FLAG_NUMBER)
			fatal("too many internal flags - %d maximum",
				FLAG_NUMBER);
		else
		{
			flag_table[flag_num].name = X(0);
			flag_table[flag_num].state = X(1);
			flag_num++;
		}
	}
	return(TRUE);
}

global	boolean
esc_getFlag(void)
{
	cell	f;

	if (IsAtom(Xdref(0)) && (f = flag_value(X(0))) != NULL) {
		return(unify_constant(f, XV(1)));
	}
	return(FALSE);
}

global	cell
flag_value(int name)
{
	natural	i;

	for (i = 0; i < flag_num; i++)
		if (flag_table[i].name == name)
			return(flag_table[i].state);
	return(NULL);
}
