/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

#include <fcntl.h>
#include <string.h>
#include "cells.h"
#include "code_area.h"
#include "data_area.h"
#include "database.h"
#include "debug.h"
#include "defs.h"
#include "delayed_problems.h"
#include "dereference.h"
#include "examine_term.h"
#include "errors.h"
#include "io.h"
#include "name_table.h"
#include "object_vars.h"
#include "ops.h"
#include "persistent.h"
#include "pred_table.h"
#include "primitives.h"
#include "spy.h"
#include "string_table.h"
#include "system.h"
#include "termin.h"
#include "termio.h"
#include "write.h"
#include "write_read.h"
#include "var_ren_table.h"
#include "x_registers.h"
#ifdef X11
#include "xwidgets.h"
#endif	/* X11 */

#define	OpenPrecedenceParen(pi, po)	( (pi >= po)			\
				  	  ?	(esc_put('('), 	\
						 remember_to_close = 1)	\
					  :	0			\
					)

#define	FinishPrecedenceParen	{ if(remember_to_close) esc_put(')'); }

/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/

static	unsigned inlist=0, outlist=0; /* some global variables for portray */
static	natural  write_type=0;
static  cell     *next_var;  /* for a Prolog list of variables - VarNames */

/*---------------------------------------------------------------------------

      There are three kinds of write: ordinary,
				      portraying  and
				      writeR  
    
  
       ---------------                       --------------
       |   $writec   |        $writeq        |  $writeRc  | 
       |             |           |           |            |
       |             |           |           |            |   
       ---------------           |           --------------
              |                  |                 |
        initialise input         |            initialise variable   
        and output list          |            name table            
	for ordinary write       |            for writeR 
	and portraying           |                 |  
              |                  |                 |  
              |                  |                 |  
              \                  |                 /  
		 \               |              / 
	            \            |          /
	               \         |      /
	                   \     | /
                        --------------- 
			|  $write     |
			|             |
			|             |
                        ---------------

/*---------------------------------------------------------------------------
$writec(Term, Precedence, Portray, InList, OutList )
      
      Term       = the term being read in (output parametar)
      Precedence = an input integer usually 1200  
      Portray    = an input parametar which is true if the function  
		   $writec is called from Qu-Prolog predicate portraycl 
		   otherwise it is false
      InList     = input list of meta and object variables with thair   
		   corresponding strings which will be printed out
      OutList    = extension of InList with new meta and object vari-  
		   ables which occure in Term, but which havn't been 
		   in InList

      Note: InList and OutList are used for portraying only
---------------------------------------------------------------------------*/

extern unsigned int start_rename_list (void);
extern int unify (VALUE *t1, VALUE *t2);
extern void finish_rename_list (unsigned int list_offset);
extern char *is_renamed (cell *variable, unsigned int list_offset);
extern int fprintf (FILE *, const char *, ...);
extern int fflush (FILE *);
extern unsigned int is_ob_renamed (cell *obvariable, unsigned int list_offset);
extern void set (cell *var, cell term);

global boolean
esc_writec(void)
{    
static  void    esc_write(cell *term, int prec, int complete);
	cell	subop;
	int     i;

	
	
	if (!IsInteger(Xdref(1))) 
	{
	    error("Internal error - non-integer precedence to $writec");
	    return(FALSE);
	}
	    /* initialise portray information */
	if (IsAtom(Xdref(2)) && !strcmp(String(X(2)), "true")) 
	{
	    write_type = PORTRAYING;
	    if(IsReference(Xdref(4)))
		outlist = 1;
	    else 
		outlist = 0;
	    if(IsNIL(Xdref(3)))
		inlist = start_rename_list();
	    else
	        if (IsInteger(X(3)))
		    inlist = IntOf(X(3));
	        else 
		{
		    error("Illegal format for rename list in portraycl");
		    return(FALSE);
	        }
	}
	else 
	{
	    write_type = WRITE_ORD;
	       /* for ordinary write, InList and OutList  
                  are initialised to empty lists        */     
	    inlist = 0;
	    outlist = 0;
	}
	    /* do writing */
	subop = (cell)(SUBSTITUTION_OPERATOR|(int)XV(0));
	esc_write(&subop, IntOf(X(1)), TRUE);
	    /* return portray information */
        if (!(write_type == PORTRAYING))
	    return(TRUE);


	if (outlist) 
	{
	    VALUE out;
	    out.term = Integer(inlist);
	    out.sub = EMPTY_SUB;
	    return(unify(&out, XV(4)));
	}
	else 
	{
	    finish_rename_list(inlist);
	    return(TRUE);
	}
}

/*---------------------------------------------------------------------------
$writeq -
	Same as $writec except any unsafe atom is quoted.
---------------------------------------------------------------------------*/
global	boolean
esc_writeq(void)
{
static  void    esc_write(cell *term, int prec, int complete);
	cell	subop;

	if (!IsInteger(Xdref(1))) 
	{
	    error("Internal error - non-integer precedence to $writeq");
	    return(FALSE);
	}
	write_type = WRITE_Q;
	subop = (cell)(SUBSTITUTION_OPERATOR|(int)XV(0));
	esc_write(&subop, IntOf(X(1)), TRUE);
	return(TRUE);
}

/*----------------------------------------------------------------------------
$writeRc(Term, Precedence, VarNames)

     Term     - a term to be written
     VarNames - a list of all variables written out during  this write

     Initialise variable name table(a table used to 
     store pointers to the name table)
     Remember names already given to variables
     If a variable has not been given a name
        a new name is generated for this variable

----------------------------------------------------------------------------*/
global boolean
esc_writeRc(void)
{
static	void	esc_write(cell *term, int prec, int complete);
	cell	subop;
	natural i;
	VALUE   varnames;

	
	varnames.term = Atom(NIL);
	varnames.sub = EMPTY_SUB;
	next_var = &(varnames.term);
	if (!IsInteger(Xdref(1))) 
	{
	    error("Internal error - non-integer precedence to $writeRc");
	    return(FALSE);
	}

             /* initialise variable_name_table */ 
	
        for (i = 0; i < VAR_NAME_TABLE_SIZE; i++)
	    variable_name_table[i] = NULL;

	     /* do writing */ 
	write_type = WRITE_R;
	subop = (cell)(SUBSTITUTION_OPERATOR|(int)XV(0));
	esc_write(&subop, IntOf(X(1)), TRUE);

        return(unify(&varnames, XV(2))); 
	
}

/*----------------------------------------------------------------------------
$write(term, prec, complete)
----------------------------------------------------------------------------*/
static void
esc_write(cell *term, int prec, int complete)
{
static	void	esc_write_tail(cell *term, int prec, int complete);
static	void	esc_write_functor(cell *term, int prec, int complete);
static	void	esc_write_sub(cell sub, int complete);
static  void    esc_write_op(cell *term, operatorpo op, int prec, int p, int arity, int complete);
static  void    esc_write_quant(cell term1, cell term2, int prec, int complete);
static  void    esc_writeRc_var(cell variable, unsigned int *counter, char *root_name);
static	boolean	safe_string(char *s);
	boolean	remember_to_close = 0;
	VALUE	val;
        VALUE *term_p;
        VALUE val_h;
	FILE	*fp;
	char  *ob_root;


	 
#ifdef	X11
	if (complete) 
	    esc_new_term_write();
#endif	/* X11 */
	DereferenceTerm(val, *term);
	if (val.sub != EMPTY_SUB)
	{
		int subprec = get_prec(oplook("*"), 2);
		if(OpenPrecedenceParen(subprec, prec))
			prec = subprec;
		esc_write_sub(val.sub, FALSE);
		esc_put('*');
	}
	switch (Tag(val.term)) 
	{ 
	when CONSTANT:
		
		if (IsAtom(val.term)) 
                {
		        char *s=String(val.term);

			if(write_type != WRITE_Q || safe_string(s))
			    esc_write_string(s);
			else {
			    boolean operator = FALSE;

			    if (oplook(s))
			    {
				operator = TRUE;
			    	esc_put('(');
			    }
			    esc_put('\'');
			    while(*s) {
				esc_put(*s);
				if(*s == '\'') esc_put(*s);
				if(*s == '\\') esc_put(*s);
				s++;
			    }
			    esc_put('\'');
			    if (operator)
			    {
			    	esc_put(')');
			    }
			}
		}
		else if (IntOf(val.term) < 0 && 
                         get_prec(oplook("-"), 2) >= prec)
                {
                        esc_put('(');
			esc_write_int(IntOf(val.term));
                        esc_put(')');
		}
                else
                {
			esc_write_int(IntOf(val.term));
		}
	when CONS:
		esc_put('.');
	when APPLY:
		if (IsList(val.term))
		{
			int commaprec = get_prec(oplook(","), 2);

			esc_put('[');
			esc_write(&Head(val_h, val.term), commaprec, complete);
			esc_write_tail(&Tail(val.term), commaprec, complete);
			esc_put(']');
		}
		else
                {
			int     arity;
			int     p;
			operatorpo op;
			VALUE	val2;
			if(IsQuantifier(
				 DerefTerm(val2, 
					   Reference(&Argument(val.term)))
					))
			    esc_write_quant(val.term, val2.term, prec, complete);
			else
			    /* check whether functor is an operator of 
			       arity less then 3 with defined priority */
	        	if ((op = oplook(String(fetch_functor(val.term)))) &&
			    ((arity = (int)fetch_arity(val.term)) < 3 ) && 
			    ((p = get_prec(op, arity)) != undefprior)
			   )
			    
		            esc_write_op(&(val.term), op,prec,p,arity, complete); 
                        
		        else
		        {
				/* if the functor isn't an operator of
				   arity less then 3 with defined priority
				   then we write apply structure as a functor  
				   followed by arguments in parenthases */ 
			    
			    int commaprec = get_prec(oplook(","), 2);

		            esc_write_functor(&Functor(val.term), commaprec,
					   complete);
		            esc_write(&Argument(val.term), commaprec, complete);
			    esc_put(')');
		        }
		}
	when PAIR:
		esc_write_string("PAIR");
	when QUANTIFIER:
		esc_write(&BoundVar(val.term), prec, complete);
		esc_write_string(" ^ ");
		esc_write(&Body(val.term), prec, complete);
	when VARIABLE:		/* OBJECT_VARIABLE case as well */
		
		if (InPersistentStack(Reference(term))) {
		    esc_put('#');
		    esc_write_string(
			    String(get_persistent_name2(Reference(term))));
		}
		else {
		    switch (write_type)
		    {
		    when PORTRAYING:
			    /* Write corresponding string for the
			       meta variable.For doing that list of 
                               pairs - (meta variable, renamed name) 
			       is used.If meta variable is not in this 
			       list, next unused string from a list 
			       (A, B, ... Z,
				AA, AB, ... AZ, 
				BA, BB, ... BZ, ...)
				will be printed out for it, and a pair
				(meta variable, the string) will be added
				to the list of pairs
                             */
			esc_write_string(is_renamed(term, inlist));
                    when WRITE_ORD
		    or	 WRITE_Q:
			    if (PtrToName(Reference(&(val.term))) != NULL)
				     /* variable has a name so use it */
				esc_write_string(String3(Reference(&(val.term))));
			    else
				     /* automatically generated meta  
					variable name will be written */  
			    	WriteVariableName((int)(term - heap));
                    when WRITE_R:
			     /* explanation is in function 
				$writeRc_var in this file  */
			esc_writeRc_var(Reference(&(val.term)),
				     &meta_var_extention, "X");
                    otherwise:
		        fprintf(stderr, "Internal Error: OTHER write VARIABLE"); 	
                    }
		}
	when REFERENCE:
		if (InPersistentStack(val.term)) {
		    esc_put('#');
		    esc_write_string(
			    String(get_persistent_name(val.term)));
		}
		else {
		    switch (write_type)
		    {
		    when PORTRAYING:
			    /* the same as in VARIABLE case, 
			       but Location of term is taken */
			esc_write_string(is_renamed(Location(val.term), inlist));
                    when WRITE_ORD
		    or	 WRITE_Q:
			if (PtrToName(val.term) != NULL)
					/* variable has a name so use it */
				esc_write_string(String3(val.term));
			else
				    /* the same as in VARIABLE case, 
				       but Location of term is taken */
				WriteVariableName((int)(Location(val.term) -
							heap));
                    when WRITE_R:
			     /* explanation is in function 
				$writeRc_var in this file  */
			esc_writeRc_var(val.term, &meta_var_extention, "X");
                    otherwise:
		        fprintf(stderr, "Internal Error:OTHER write REFERENCE");
                    }
		}
	when OBJECT_REFERENCE:
		if (IsLocalObjectVariable(val.term)) {
			esc_put('@');
			WriteObjectVariableName((int)(Location(val.term) -
						heap));
		}
		else if (InPersistentStack(val.term)) {
			esc_put('#');
			esc_write_string(
				String(get_persistent_name(val.term)));
		}
		else {
		    switch (write_type)
		    {
		    when PORTRAYING:
			    /* Write corresponding string for the
			       object variable.The string consists  
			       of the prefix of first defined object
			       variable, '_' and a number.The number  
			       is an ordinal number of object variable  
                               in the list (if the object variable hasn't 
			       been in the list, it will be added)
                             */
		        esc_write_string(object_var_prefix_first());
		        esc_put('_');
			esc_write_int(is_ob_renamed(&(val.term), inlist));
                    when WRITE_ORD
		    or	 WRITE_Q:
			if (PtrToName(val.term) != NULL)
			{
					/* variable has a name so use it */
				esc_write_string(String3(val.term));
			}
			else
			{
				     /* automatically generated object  
					variable name will be written */  
				WriteObjectVariableName((int)(Location(val.term) -
							heap));
			}
                    when WRITE_R:
			ob_root = object_var_prefix_first();
			if (*ob_root == '\0')
			{
				WriteObjectVariableName((int)(Location(val.term) -
				  			heap));
			}
			else
			{
			     /* explanation is in function 
				$writeRc_var in this file  */
				esc_writeRc_var(val.term, &object_var_extention,
						ob_root);
			}
                    otherwise:
		        fprintf(stderr, "Internal Error:OTHER write OBJECT_REFERENCE");
                    }
		}
	otherwise:
		fprintf(stderr, "Internal Error: OTHER");
	} 
	FinishPrecedenceParen;
#ifdef	X11
	if(complete) esc_write_done();
#endif	/* X11 */
}

/*----------------------------------------------------------------------------
$write_sub(sub, complete)
----------------------------------------------------------------------------*/
static	void
esc_write_sub(cell sub, int complete)
{
        natural j;

	if (sub == EMPTY_SUB)
	{
		return;
	}
        if (NextSub(sub) != EMPTY_SUB)
	{/* write all subs */
		esc_write_sub(NextSub(sub), complete);
		esc_put('*');
	}
	{/* write a single substitution list */
	int prec = get_prec(oplook("/"), 2);
	esc_put('[');
	for (j = Size(sub); j > 1; j--)
	{
		esc_write(&Range(sub, j), prec, FALSE);
		esc_put('/');
		esc_write(&Domain(sub, j), prec, FALSE);
		esc_put(','); esc_put(' ');
	}
	esc_write(&Range(sub, j), prec, FALSE);
	esc_put('/');
	esc_write(&Domain(sub, j), prec, FALSE);
	esc_put(']');
	}
}

/*----------------------------------------------------------------------------
$write_functor(term, prec, complete)
     write functor and arguments in parenthases separated with commas  

----------------------------------------------------------------------------*/
static	void
esc_write_functor(cell *term, int prec, int complete)
{
	VALUE	val;

	DereferenceTerm(val, *term);

	if (IsApply(val.term))
	{
		esc_write_functor(&Functor(val.term), prec, complete);
		esc_write(&Argument(val.term), prec, complete);
		esc_put(','); esc_put(' ');
	}
	else
	{
		esc_write(term, prec, FALSE);
		esc_put('(');
	}

}

/*----------------------------------------------------------------------------
$write_tail(term, prec, complete)
----------------------------------------------------------------------------*/
static	void
esc_write_tail(cell *term, int prec, int complete)
{
	VALUE	val;
	VALUE	val_h;

	
	DereferenceTerm(val, *term);
	if (val.sub == EMPTY_SUB && IsList(val.term))
	{
		esc_put(',');  esc_put(' ');
		esc_write(&Head(val_h, val.term), prec, complete);
		esc_write_tail(&Tail(val.term), prec, complete);
	}
	else if (!IsNIL(val.term))
	{
		esc_put('|');
		esc_write(term, prec, complete);
	}
}

/*----------------------------------------------------------------------------
$write_delay(var, prec, complete)
    write delayed problems associated with the variable
----------------------------------------------------------------------------*/
static	void
esc_write_delay(cell var, int prec, int complete)
{
	cell	*d;

	for (d = (cell *)RestOfVariable(var); d != NULL; d = Next(d))
	{
		esc_write(&DelayedTerm(d), prec, complete);
		switch (DelayedType(d))
		{
		when UNIFY:
			esc_write_string(" = ");
		when NOT_FREE:
			esc_write_string(" not_free_in ");
		}
		esc_write(&DelayedVar(d), prec, complete);
		esc_put('\n');
	}
}

/*----------------------------------------------------------------------------
$write_op(term, op, prec, p, arity, complete)
     for an operator functor and arguments will be written 
     in a proper order 
----------------------------------------------------------------------------*/
static void 
esc_write_op(cell *term, operatorpo op, int prec, int p, int arity, int complete)
{
static void        esc_write_op_f(cell *term, int prec, int complete);
static void	   esc_write_operator(cell *term, int prec, int complete);
           VALUE    val, fn, arg;
	   boolean  bracket;

           DereferenceTerm(val, *term);
           if (p >= prec)
              esc_put('(');
               
           switch (get_assoc(op, arity))
	   {
	   when XFX or XFY or YFX or YFY:
	           esc_write_op_f(&Functor(val.term), p, complete); 
                   esc_put(' ');
                   esc_write(&Argument(val.term), p, complete);
           when FX or FY:
                esc_write_operator(&Functor(val.term), p, FALSE);
		DereferenceTerm(fn, Functor(val.term));
		DereferenceTerm(arg, Argument(val.term));
		if ((strcmp(String(fn.term), "-") == 0 ||
		     strcmp(String(fn.term), "+") == 0
		    ) &&
		    IsInteger(arg.term) && IntOf(arg.term) >= 0
		   )
		{
			bracket = TRUE;
			esc_put('(');
		}
		else
		{
			bracket = FALSE;
                	esc_put(' ');
		}
                esc_write(&Argument(val.term), p, complete);
		if (bracket)
		{
			esc_put(')');
		}
           when XF or YF:
                esc_write(&Argument(val.term), p, complete);
                esc_put(' ');
                esc_write_operator(&Functor(val.term), p, FALSE);
           }            
           if (p >= prec)
              esc_put(')');
}
/*------------------------------------------------------------------------------
$write_op_f(term, prec, complete)
------------------------------------------------------------------------------*/
static void 
esc_write_op_f(cell *term, int prec, int complete)
{
static void        esc_write_operator(cell *term, int prec, int complete);
           VALUE    val;

           DereferenceTerm(val, *term);
	           esc_write(&Argument(val.term), prec, complete); 
                   esc_put(' ');
	           esc_write_operator(&Functor(val.term), prec, FALSE); 
}

static void
esc_write_operator(cell *term, int prec, int complete)
{
static  boolean safe_string(char *s);
	VALUE	val;

#ifdef	X11
	if (complete) 
	    esc_new_term_write();
#endif	/* X11 */
	DereferenceTerm(val, *term);
	if (IsAtom(val.term)) 
	{
		char *s=String(val.term);

		if(write_type != WRITE_Q || safe_string(s))
		    esc_write_string(s);
		else {
		    esc_put('\'');
		    while(*s) {
			esc_put(*s);
			if(*s == '\'') esc_put(*s);
			if(*s == '\\') esc_put(*s);
			s++;
		    }
		    esc_put('\'');
		}
	}
}
/*------------------------------------------------------------------------------
$write_quant(term1, term2, prec, complete)
     write a quantified term
-----------------------------------------------------------------------------*/
static void
esc_write_quant(cell term1, cell term2, int prec, int complete)
{
	int 	p;
	VALUE 	funcval;

	DereferenceTerm(funcval, Functor(term1));
	if((p = get_prec(oplook(String(funcval.term)), 1)) >= prec)
		{
		 esc_put('(');
		}
	esc_write(&Functor(term1), p, FALSE);
	esc_put(' ');
	esc_write(&BoundVar(term2), p, complete);
	esc_put(' ');
	esc_write(&Body(term2), p, complete);
	if(p >= prec)
		esc_put(')');
}

/*-----------------------------------------------------------------------------
$writeRc_var(variable, counter, root_name)  

    variable =  pointer to the variable with  REFERENCE tag 

    counter will be updated on higher level

description:
    if root_name == NULL (ie. no object variable declaration)
	write the variable as anonymous object variable
    else if a variable has been given a name (ptr_to_name != NULL) 
        print the variable name
    else 
	generate a new variable name
	print the new variable name
	add the string for the name to the string table
	set value of the variable name to the variable
	set pointer to name to name table entry
    if the variable name is not in variable name table
	add the variable name to variable name table
        add the variable to a Prolog list of variables 

-----------------------------------------------------------------------------*/
static void
esc_writeRc_var(cell variable, unsigned int *counter, char *root_name)
{
        VALUE   val_h;
        char    *name;
        cell    *h;
        natural i;

        char    *generate_new_name(char *root_name, unsigned int *next_free);
        void    add_var_in_vnt(unsigned int i, NAME *ptr_to_name_table);
        int     var_in_vnt(unsigned int i, char *name);
local   natural add_name_string_hash(char *, int);




      if (root_name == NULL)
      {
	  WriteObjectVariableName((int)(Location(variable) - heap));
      }
      else if (PtrToName(variable) != NULL)
      {
	  /* variable has been given a name */

	  name = String3(variable);
	  esc_write_string(name);
      }
      else
      {
	  /* variable has not been given a name */
	  
          name = generate_new_name(root_name, counter); 
	  esc_write_string(name);
              /* add name to name_table and to string_table 
		 i = hash value of name to name_table   */
	  i = add_name_string_hash(name, VARIABLE_W); 
              /* set value of name to variable  */ 
          set(&ValueOfName(name_table[i]), variable);
              /*set ptr_to_name to name_table */
          set(&PtrToName(variable), (cell)&(name_table[i])); 
      }

      i = hash_var_name_table(name, VAR_NAME_TABLE_SIZE);
      if (var_in_vnt(i, name) < 0)
      {
	      /* add name to variable_name_table  */
	  add_var_in_vnt(i, Named(variable)); 
	      /* add variable to h */ 
	  InstantiateTail(next_var, h, val_h);
	  *next_var = Atom(NIL);
	  *h = variable;
      }
      
}
/*----------------------------------------------------------------------------
write_safe_string(name);
             fprintf(Qstdout, "%s", name);      
         else
	     fprintf(Qstdout, "\'%s\'", name);
----------------------------------------------------------------------------*/
static void
write_safe_string(char *name)
{
static	boolean	safe_string(char *s);

	if (safe_string(name)) 
		esc_write_string(name);      
	else
	{
		esc_put('\'');
		esc_write_string(name);
		esc_put('\'');
	}
}
/*----------------------------------------------------------------------------
safe_string(s)
----------------------------------------------------------------------------*/
static	boolean
safe_string(char *s)
{
static	boolean	safe_char(char c);

	if (!(*s))
		return(FALSE);
	while (*s)
		if (!safe_char(*s++))
			return(FALSE);
	
	return(TRUE);
}

/*----------------------------------------------------------------------------
safe_char(c)
    True, if the character c is an alphanumeric char or an underscore or
    dollar sign.
----------------------------------------------------------------------------*/
static boolean
safe_char(char c)
{
static	boolean	member(char c, char *s);

	return(member(c, "$abcdefghijklmnopqrstuvwxyz_0123456789"));
}

/*----------------------------------------------------------------------------
member(c, s)
    True, iff the character c is a member of the string s.
----------------------------------------------------------------------------*/
static	boolean
member(char c, char *s)
{
	while (*s)
		if (c == *s++)
			return(TRUE);

	return(FALSE);

}
