/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

Term1 \= Term2 :-
    \+ (Term1 = Term2).

Term1 \== Term2 :-
    \+ (Term1 == Term2).

Term1 =\= Term2 :-
    \+ (Term1 =:= Term2).

Term1 @< Term2 :-
    compare((<), Term1, Term2).
Term1 @> Term2 :-
    compare((>), Term1, Term2).

Term1 @=< Term2 :-
    compare(Comp, Term1, Term2),
    (Comp == (=); Comp == (<)).

Term1 @>= Term2 :-
    compare(Comp, Term1, Term2),
    (Comp == (=); Comp == (>)).

/*----------------------------------------------------------------------------
From page 50 Nu-Prolog Manual

Logical term comparison observes the following standard ordering:

(1)	Variables are equal if they are identical.
(2)	Numbers are ordered numerically, with integers less than their
	floating point equivalent. Atoms are ordered lexiographically.
(3)	Numbers are less than atoms.
(4)	If term X has arity smaller than Y, it is smaller.
(5)	Term X is less than term Y if the functor of X is lexiographically
	less than the functor of Y, and X and Y are equal under conditions
	(1) to (4).
(6)	Term X is less than term Y if an argument of X is less than the
	corresponding argument of Y, and X, Y and their preceeding arguments
	are equal under conditons (1) to (5).

	Non-logical term comparison is defined by the extending the logical
	ordering with the following ordering for variables:

(1a)	Two distinct variables have an ordering  based on the local
	implementation.
(1b)	A variable is less than a non-variable.
(1c)	Two distinct object variables have an ordering  based on the local
	implementation.
(1d)	An object variable is less than a variable.
----------------------------------------------------------------------------*/
compare(Comp, Term1, Term2) :-
    is_object_var(Term1), !,
    '$compare_object_var'(Comp, Term1, Term2).
compare(Comp, Term1, Term2) :-
    var(Term1), !,
    '$compare_var'(Comp, Term1, Term2).
compare(Comp, Term1, Term2) :-
    atom(Term1), !,
    '$compare_atom'(Comp, Term1, Term2).
compare(Comp, Term1, Term2) :-
    integer(Term1), !,
    '$compare_integer'(Comp, Term1, Term2).
compare(Comp, Term1, Term2) :-
    '$compare_functor'(Comp, Term1, Term2).

'$compare_object_var'((=), Term1, Term2) :-
    Term1 == Term2, !.
'$compare_object_var'((<), Term1, Term2) :-
    '$var_less_than'(Term1, Term2), !.
'$compare_object_var'((>), Term1, Term2) :-
    '$var_less_than'(Term2, Term1), !.
'$compare_object_var'((<), _Term1, _Term2).

'$compare_var'((>), _Term1, Term2) :-
    is_object_var(Term2), !.
'$compare_var'((=), Term1, Term2) :-
    Term1 == Term2, !.
'$compare_var'((<), Term1, Term2) :-
    '$var_less_than'(Term1, Term2), !.
'$compare_var'((>), Term1, Term2) :-
    '$var_less_than'(Term2, Term1), !.
'$compare_var'((<), _Term1, _Term2).

'$compare_atom'((>), _Term1, Term2) :-
    is_object_var(Term2), !.
'$compare_atom'((>), _Term1, Term2) :-
    var(Term2), !.
'$compare_atom'((=), Term1, Term2) :-
    Term1 == Term2, !.
'$compare_atom'((>), _Term1, Term2) :-
    integer(Term2), !.
'$compare_atom'(Comp, Term1, Term2) :-
    name(Term1, Name1),
    name(Term2, Name2),
    '$compare_name'(Comp, Name1, Name2), !.
'$compare_atom'((<), _Term1, _Term2).

'$compare_name'((=), [], []).
'$compare_name'(Comp, [Char1|Name1], [Char2|Name2]) :-
    '$compare_integer'(IntComp, Char1, Char2),
    (IntComp == (=) ->
	'$compare_name'(Comp, Name1, Name2)
    ;
	Comp = IntComp
    ), !.

'$compare_integer'((>), _Term1, Term2) :-
    is_object_var(Term2), !.
'$compare_integer'((>), _Term1, Term2) :-
    var(Term2), !.
'$compare_integer'((=), Term1, Term2) :-
    Term1 == Term2, !.
'$compare_integer'((<), Term1, Term2) :-
    integer(Term2),
    Term1 < Term2, !.
'$compare_integer'((>), Term1, Term2) :-
    integer(Term2),
    Term1 > Term2, !.
'$compare_integer'((<), _Term1, Term2) :-
    \+ integer(Term2).


'$compare_functor'((>), _Term1, Term2) :-
    is_object_var(Term2), !.
'$compare_functor'((>), _Term1, Term2) :-
    var(Term2), !.
'$compare_functor'((>), _Term1, Term2) :-
    integer(Term2), !.
'$compare_functor'((>), _Term1, Term2) :-
    atom(Term2), !.
'$compare_functor'(Comp, Term1, Term2) :-
    functor(Term1, F1, N1),
    functor(Term2, F2, N2),
    '$compare_integer'(CompN, N1, N2),
    (CompN == (=) ->
	compare(CompAtom, F1, F2),
	(CompAtom == (=) ->
	    Term1 =.. [F1|Args1],
	    Term2 =.. [F2|Args2],
	    '$compare_arg_list'(Comp, Args1, Args2)
	;
	    Comp = CompAtom
	)
    ;
	Comp = CompN
    ), !.

'$compare_arg_list'((=), [], []).
'$compare_arg_list'(Comp, [Arg1|Args1], [Arg2|Args2]) :-
    compare(CompArg, Arg1, Arg2),
    (CompArg == (=) ->
	'$compare_arg_list'(Comp, Args1, Args2)
    ;
	Comp = CompArg
    ), !.
