/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */

:- dynamic '$current_input'/1.
:- dynamic '$current_output'/1.

:- assert('$current_input'(user_input)).
:- assert('$current_output'(user_output)).

/*----------------------------------------------------------------------------
see(Stream) :-
----------------------------------------------------------------------------*/

see(Stream) :-
    (Stream = '$stream'(_M, _N); Stream = user_input), !,
    setInput(Stream).
see(File) :-
    \+ atom(File),
    error( 'Error in '), error( see(File)),
    errorln( ' atom or string or stream expected'), !, fail.
see(File) :-
    open(File, read, Stream), !,
    setInput(Stream).

current_input(Stream):-
    '$current_input'(Stream).

set_input(Stream) :-
    abolish('$current_input', 1),
    assert('$current_input'(Stream)),
    '$set_input'(Stream).

seeing(Stream) :-
    '$current_input'(Stream).

seen :-
    seeing(Stream),
    setInput(user_input),
    %once(retract('$current_input'(Stream))),
    %assert('$current_input'(user_input)),
    close(Stream).


    
/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
tell(Stream) :-
    (Stream = '$stream'(_M, _N); Stream = user_output; Stream = user_error), !,
    setOutput(Stream).
tell(File) :-
    open(File, write, Stream), !,
    setOutput(Stream).
tell(File) :-
    error( 'Error in '), error( tell(File)),
    errorln( ' atom or string or stream expected'), fail.

current_output(Stream):-
    '$current_output'(Stream).

set_output(Stream) :-
    abolish('$current_output', 1),
    assert('$current_output'(Stream)),
    '$set_output'(Stream).

telling(Stream) :-
    '$current_output'(Stream).

told :-
    telling(Stream),
    setOutput(user_output),
    %once(retract('$current_output'(Stream))),
    %assert('$current_output'(user_output)),
    close(Stream).


/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
close(Stream) :-
    '$close'(Stream), !.
close(Stream) :-
    error( 'Error in '), error( close(Stream)),
    errorln( ' atom or string or stream expected'), fail.

open(FileName, Mode, Stream) :-
    '$open'(FileName, Mode, Stream), !.
% open(FileName, Mode, Stream) :-
%     error( 'Error in '), error( open(FileName, Mode, Stream)),
%     errorln( ' atom or string or stream expected'), fail.

/*----------------------------------------------------------------------------
eof(Term) :-
----------------------------------------------------------------------------*/
eof(Term) :-
    var(Term), !.
eof(end_of_file).
    
/*----------------------------------------------------------------------------
----------------------------------------------------------------------------*/
putl(Stream, String) :-
    stream_o(Stream, putl(String)).

putl([]).
putl([Char|String]) :-
    put(Char),
    putl(String).

getl(Stream, String) :-
    stream_i(Stream, getl(String)).

getl(String) :-
    get0(Char),
    !,
    ( Char = -1 ->
	String = []
    ; Char = 10 ->   % newline
	String = [Char]
    ; String = [Char|TempString],
      getl(TempString)
    ).


stream_i(Stream, Function) :-
    '$current_input'(CurrentInput),
    '$set_input'(Stream),
    once((Function; ('$set_input'(CurrentInput), fail))),
    '$set_input'(CurrentInput).

stream_o(Stream, Function) :-
    '$current_output'(CurrentOutput),
    '$set_output'(Stream),
    once((Function; ('$set_output'(CurrentOutput), fail))),
    '$set_output'(CurrentOutput).

read(Stream, Term) :-
    stream_i(Stream, read(Term)).

readR(Stream, Term) :-
    stream_i(Stream, readR(Term)).

sread(Stream, String, Term) :-
    stream_i(Stream, sread(String, Term)).

get(Stream, Char) :-
    stream_i(Stream, get(Char)).

get0(Stream, Char) :-
    stream_i(Stream, get0(Char)).

writeln(Term) :-
    write(Term), nl.

term_widget_writeln(Widget, Term) :-
    '$write_to_term_widget'(Widget),
    widget_writeln(Widget, Term).

widget_writeln(Widget, Term) :-
    '$write_to_text_widget'(Widget),
    write(Term), nl,
    '$close_write_to_text_widget'.

nl :-
    put(10). 	% newline

widget_nl(Widget) :-
    widget_put(Widget, 10).	% newline

widget_put(Widget, C) :-
	'$write_to_text_widget'(Widget),
	put(C),
	'$close_write_to_text_widget'.

writeln(Stream, Term) :-
    stream_o(Stream, writeln(Term)).

write(Stream, Term) :-
    stream_o(Stream, write(Term)).

writeq(Stream, Term) :-
    stream_o(Stream, writeq(Term)).

writep(Stream, Term) :-
    stream_o(Stream, writep(Term)).

nl(Stream) :-
    stream_o(Stream, nl).

portray_clause(Stream, Term) :-
    stream_o(Stream, portray_clause(Term)).

flushOutput(Stream) :-
    stream_o(Stream, flush).

flush(Stream) :-
    stream_o(Stream, flush).

put(Stream, Char) :-
    stream_o(Stream, put(Char)).

put0(Stream, Char) :-
    stream_o(Stream, put0(Char)).

tab(Stream, N) :-
    stream_o(Stream, tab(N)).
    
tab(0) :- !.
tab(N) :-
    put(0' ),
    M is N - 1,
    tab(M).

skip(Stream, Char) :-
    stream_i(Stream, skip(Char)).

skip(Char) :-
    repeat,
    get(C),
    '$skipped'(C, Char).

'$skipped'(-1, _Char) :- !.
'$skipped'(C, C) :- !.
'$skipped'(C, Chars) :-
    member(C, Chars).

'$write_tokens'([]).
'$write_tokens'([Token|Tokens]) :-
    '$write_token'(Token),
    '$write_tokens'(Tokens).

'$write_token'([[]|end_of_file]) :- !.
'$write_token'([VariableName|var]) :-
    !,
    write(VariableName).
'$write_token'([Atom|atom]) :-
    !,
    write(Atom).
'$write_token'([QuotedAtom|quoted]) :-
    !,
    put(0''),
    write(QuotedAtom),
    put(0'').
'$write_token'([Number|number]) :-
    !,
    write(Number).
'$write_token'([String|string]) :-
    !,
    write(String).

/* New read */
ged(X) :-
    '$readc'(X, 1200, _, 'Qu-Prolog').

ged(Stream, Term) :-
    stream_i(Stream, ged(Term)).


/* counts for reading and writing */
io_count(read, C) :-
    var(C), !, 
    '$getFlag'(read_count, C).

io_count(read, C) :-
    integer(C), !, 
    '$setFlag'(read_count, C).

io_count(write, C) :-
    var(C), !, 
    '$getFlag'(write_count, C).

io_count(write, C) :-
    integer(C), !, 
    '$setFlag'(write_count, C).

/* predicates for indenting text */
set_indent(I) :- ipSet('$indent', I).

indent :- 
    ipLookup('$indent', I), 
    I1 is 4 * I,
    tab(I1).

inc_indent :-
    ipLookup('$indent', I),
    I1 is I + 1,
    ipSet('$indent', I1).

