/*
 * QU-PROLOG COPYRIGHT NOTICE, LICENCE AND DISCLAIMER.
 * 
 * Copyright 1993 by The University of Queensland, Queensland 4072 Australia
 * 
 * Permission to use, copy and distribute this software 
 * for any non-commercial purpose and without fee is hereby
 * granted, provided that the above copyright notice
 * and this permission notice and warranty
 * disclaimer appear in all copies and in supporting documentation, 
 * and that the name of The University of Queensland not be used in 
 * advertising or publicity pertaining to distribution of the software 
 * without specific, written prior permission.
 * 
 * Source code modifications are prohibited except where written agreement 
 * has been given in advance by The University of Queensland.
 * 
 * The University of Queensland disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and fitness.
 * In no event shall The University of Queensland be liable for any special,
 * indirect or consequential damages or any damages whatsoever resulting from
 * loss of use, data or profits, whether in an action of contract, negligence
 * or other tortious action, arising out of or in connection with the use or
 * performance of this software.
 */
/* predicates defined in this file 
   * a trailing `p' means a Herbrand term is expected
   * a `1Term' in the predicate name maens a name-var list is created
   * an `s' preceding the name means the read is done from a string
   * a `widget_' preceding the name means the read is done from a widget
   * an extra argument N in widget_reads gives the position to start reading
   * an extra two arguments NEV_in NEV_out maintains consistency across reads
     of variable renames
*/
/*
read(Term)
readp(Term)
read1Term(Term, NamesEqualsVars)
read1Termp(Term, NamesEqualsVars)
read(Term, NEV_in, NEV_out)
readp(Term, NEV_in, NEV_out)
sread(String,Term)
sreadp(String,Term)
sread(String,Term, NEV_in, NEV_out)
sreadp(String,Term, NEV_in, NEV_out)
sread1Term(String, Term, NamesEqualsVars)
sread1Termp(String, Term, NamesEqualsVars)
widget_read(Widget,Term)
widget_readp(Widget,Term)
widget_read(Widget,Term, NEV_in, NEV_out)
widget_readp(Widget,Term, NEV_in, NEV_out)
widget_read1Term(Widget, Term, NamesEqualsVars)
widget_read1Termp(Widget, Term, NamesEqualsVars)
widget_read(Widget,Term, N)
widget_readp(Widget,Term, N)
widget_read(Widget,Term, N, NEV_in, NEV_out)
widget_readp(Widget,Term, N, NEV_in, NEV_out)
widget_read1Term(Widget, Term, NamesEqualsVars, N)
widget_read1Termp(Widget, Term, NamesEqualsVars, N)
*/


read(Term) :-
    '$readc'(Term, 1200, _).


read1Term(Term, NamesEqualsVars) :-
    '$readc'(Term, 1200, NamesEqualsVars).


/*----------------------------------------------------------------------------
readR(Term, OldVars, NewNamePairs) 

    Term         = the term being read in  
    OldVars      = a Prolog list of variables that is input to read  
    NewNamePairs = a Prolog list of name pairs that is output from read
   
    Variables read in with the same name as variables in OldVars refer to 
    these variables.Otherwise variables read in refer to new variables. 
    If the name of a variable read in clashes with another variable 
    currently in use, then a variant name is generated for the new variable.

----------------------------------------------------------------------------*/

readR(Term, OldVars, NewNamePairs) :-
    '$readRc'(Term, 1200, OldVars, NewNamePairs).

/*----------------------------------------------------------------------------
readR(Term) - Prolog predicate 

    Term         = the term being read in

    Same as readR/3 assuming (implicitly) all currently active variables are
    in OldVars. I. E. each variable read in with the same name as a currently
    active variable refer to that variable.

-----------------------------------------------------------------------------*/

readR(Term) :-
    '$readR1c'(Term, 1200, _).

readR1Term(Term, VarsList) :-
	'$readR1c'(Term, 1200, VarsList).

/*---------------------------------------------------------------------------*/
/* read/3 */
/* A NamesEqualsVars list NEV_in is passed in.
   Consistency with this list is maintained during the read.
   An updated list is unified with NEV_out 
 */
read(Term, NEV_in, NEV_out) :-
    read1Term(Term, NamesEqualsVars),
    '$consistent_lists'(NEV_in, NamesEqualsVars),
    append(NEV_in, NamesEqualsVars, NEV_out).

sread(String,Term) :-
    '$string_read'(String),
    read(Term),
    !,
    '$no_string_read'.
sread(_S,_T) :-
    '$no_string_read'.

sreadp(String,Term) :-
    '$string_read'(String),
    readp(Term),
    !,
    '$no_string_read'.
sreadp(_S,_T) :-
    '$no_string_read'.

sread(String,Term, NEV_in, NEV_out) :-
    '$string_read'(String),
    read(Term, NEV_in, NEV_out),
    !,
    '$no_string_read'.
sread(_S,_T, NEV_in, NEV_out) :-
    '$no_string_read'.

sreadp(String,Term, NEV_in, NEV_out) :-
    '$string_read'(String),
    readp(Term, NEV_in, NEV_out),
    !,
    '$no_string_read'.
sreadp(_S,_T, NEV_in, NEV_out) :-
    '$no_string_read'.

sread1Term(String, Term, NamesEqualsVars) :-
    '$string_read'(String),
    '$readc'(Term, 1200, NamesEqualsVars),
    !,
    '$no_string_read'.
sread1Term(_S,_T) :-
    '$no_string_read'.

sread1Termp(String, Term, NamesEqualsVars) :-
    '$string_read'(String),
    read1Termp(Term, NamesEqualsVars),
    !,
    '$no_string_read'.
sread1Termp(_S,_T) :-
    '$no_string_read'.

/* predicates for reading from asciiText widgets */
widget_read(Widget,Term) :-
    '$widget_read'(Widget),
    read(Term),
    !,
    '$no_string_read'.
widget_read(_S,_T) :-
    '$no_string_read'.

widget_readR(Widget,Term, Old, New) :-
    '$widget_read'(Widget),
    readR(Term, Old, New),
    !,
    '$no_string_read'.
widget_readR(_S,_T, _O, _N) :-
    '$no_string_read'.

widget_readR(Widget,Term) :-
    '$widget_read'(Widget),
    readR(Term),
    !,
    '$no_string_read'.
widget_readR(_S,_T) :-
    '$no_string_read'.

widget_readp(Widget,Term) :-
    '$widget_read'(Widget),
    readp(Term),
    !,
    '$no_string_read'.
widget_readp(_S,_T) :-
    '$no_string_read'.

widget_read(Widget,Term, NEV_in, NEV_out) :-
    '$widget_read'(Widget),
    read(Term, NEV_in, NEV_out),
    !,
    '$no_string_read'.
widget_read(_S,_T, NEV_in, NEV_out) :-
    '$no_string_read'.

widget_readp(Widget,Term, NEV_in, NEV_out) :-
    '$widget_read'(Widget),
    readp(Term, NEV_in, NEV_out),
    !,
    '$no_string_read'.
widget_readp(_S,_T, NEV_in, NEV_out) :-
    '$no_string_read'.

widget_read1Term(Widget, Term, NamesEqualsVars) :-
    '$widget_read'(Widget),
    read1Term(Term),
    !,
    '$no_string_read'.
widget_read1Term(_S,_T, _N) :-
    '$no_string_read'.

widget_read1Termp(Widget, Term, NamesEqualsVars) :-
    '$widget_read'(Widget),
    read1Termp(Term),
    !,
    '$no_string_read'.
widget_read1Termp(_S,_T, _N) :-
    '$no_string_read'.

/* predicates for reading from asciiText widgets at position N */
widget_read(Widget,Term, N) :-
    '$widget_read'(Widget),
    '$string_read_position'(N),
    read(Term),
    !,
    '$no_string_read'.
widget_read(_S,_T, _N) :-
    '$no_string_read'.

widget_readp(Widget,Term, N) :-
    '$widget_read'(Widget),
    '$string_read_position'(N),
    readp(Term),
    !,
    '$no_string_read'.
widget_readp(_S,_T, _N) :-
    '$no_string_read'.

widget_read(Widget,Term, N, NEV_in, NEV_out) :-
    '$widget_read'(Widget),
    '$string_read_position'(N),
    read(Term, NEV_in, NEV_out),
    !,
    '$no_string_read'.
widget_read(_S,_T, _N, NEV_in, NEV_out) :-
    '$no_string_read'.

widget_readp(Widget,Term, N, NEV_in, NEV_out) :-
    '$widget_read'(Widget),
    '$string_read_position'(N),
    readp(Term, NEV_in, NEV_out),
    !,
    '$no_string_read'.
widget_readp(_S,_T, _N, NEV_in, NEV_out) :-
    '$no_string_read'.

widget_read1Term(Widget, Term, NamesEqualsVars, N) :-
    '$widget_read'(Widget),
    '$string_read_position'(N),
    read1Term(Term, NamesEqualsVars),
    !,
    '$no_string_read'.
widget_read1Term(_S,_T, _NEV, _N) :-
    '$no_string_read'.

widget_read1Termp(Widget, Term, NamesEqualsVars, N) :-
    '$widget_read'(Widget),
    '$string_read_position'(N),
    read1Termp(Term, NamesEqualsVars),
    !,
    '$no_string_read'.
widget_read1Termp(_S,_T, _NEV, _N) :-
    '$no_string_read'.

/* to ensure consistency of read variable mappings */
'$consistent_lists'([], _L) :- !.
'$consistent_lists'(_L, []) :- !.
'$consistent_lists'([H1|T1], [H2|T2]) :- 
	'$consistent_entries'(H1, H2),
	'$consistent_lists'([H1], T2),
	'$consistent_lists'(T1, [H2|T2]).
'$consistent_entries'(A = Alpha, B = Beta) :- 
	A == B -> Alpha = Beta ; true.
