;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                       ;;;
;;;       Paris-Vencovska Model of Belief Revision        ;;;
;;;                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;; Explanations and running instructions.
;;;---------------------------------------


;;;/********************************************************/
;;;/*                                                      */
;;;/*                 General logic macros                 */
;;;/*                                                      */
;;;/********************************************************/


(defmacro exists (var_list_pair s_expr)
   "Calls to this macro have the form:
       (exists (var list_of_vals) s_expr)
    It returns a non-nil value just in case setq-ing <var>
    to one or more values in <list_of_vals> will cause 
    <s_expr> to evaluate to a non-nil value"
    
   (let ((domain (gensym)) (answer (gensym)))
      `(do ((,domain ,(cadr var_list_pair) (cdr ,domain))
            (,answer nil (or ,answer
                             (progn nil 
                                (setq ,(car var_list_pair) (car ,domain))
                                ,s_expr))))
           ((null ,domain) ,answer))))
  

(defmacro forall (var_list_pair s_expr)
   "Calls to this macro have the form:
       (forall (var list_of_vals) s_expr)
    It returns a non-nil value just in case setq-ing <var>
    to every value in <list_of_vals> will cause 
    <s_expr> to evaluate to a non-nil value"
    
   (let ((domain (gensym)) (answer (gensym)))
      `(do ((,domain ,(cadr var_list_pair) (cdr ,domain))
            (,answer t (and ,answer
                            (progn nil 
                               (setq ,(car var_list_pair) (car ,domain))
                               ,s_expr))))
           ((null ,domain) ,answer))))
  



;;;/********************************************************/
;;;/*                                                      */
;;;/*             Formula evaluation functions             */
;;;/*                                                      */
;;;/********************************************************/

  
 
(defstruct formula 
   (functor nil)            ;;; <nil> indicates atomic
   (sub_formula1 nil)       ;;; first argument
   (sub_formula2 nil)       ;;; second argument (not used for <->)
   (truth_value nil))       ;;; used for destructive modification in evaluation
  

(defun mutilate_eval (formula scenario)
   "Evaluates formula <formula> with respect to scanario <scenario>.
    As the evaluation proceeds (as the recursion passes back up), the
    data-structure <formula> is desctructively modified, so that, if
    the evaluation cannot proceed at some point, a partially evaluated
    version remains, with the interfering proposition being returned"
   
   (cond ((eq (formula-truth_value formula) 'true) 
          'true)                                   ;;; Result has already been decided
         ((eq (formula-truth_value formula) 'false)
          'false)                                  ;;; Result has already been decided
         ((null (formula-functor formula))         ;;; If <formula> is atomic 
          (setf (formula-truth_value formula)      ;;; then use special function
                (mutilate_eval_atomic formula scenario)))
         ((eq (formula-functor formula) '&)        ;;; else, if <formula> is a conjunction
          (mutilate_eval_conjunction formula scenario))
         ((eq (formula-functor formula) '-)        ;;; else, if <formula> is a negation
          (mutilate_eval_negation formula scenario))))
 
(defun mutilate_eval_atomic (atomic_formula scenario)
   "Evaluates atomic formula <atomic_formula> with respect to scanario <scenario>.
    Returns truth value <true> or <false>, or <atomic_formula> itself if the 
    truth value cannot be decided. Note that the proposition letter will be
    passed up to the top of the original proposition from which this call
    was made."
    
   (cond ((true_in (formula-sub_formula1 atomic_formula) scenario)
          (setf (formula-truth_value atomic_formula) 'true))
         ((false_in (formula-sub_formula1 atomic_formula) scenario)
          (setf (formula-truth_value atomic_formula) 'false))
         (t 
          (setf (formula-truth_value atomic_formula) 
                (formula-sub_formula1 atomic_formula)))))
;;; Note that <atomic_formula> is an instance of the <formula> structure, 
;;; with the slot <formula-sub_formula1> filled by a proposition_lettr (which
;;; is a LISP atom.
                                                                         

(defun mutilate_eval_conjunction (formula scenario)
   "Evaluates formula <formula> with respect to scanario <scenario>.
    A sub-function of mutilate_eval.
    Deals with the case where the formula is a conjunction."
   
    (let ((result (mutilate_eval (formula-sub_formula1 formula) scenario)))
       (cond ((not (eq result 'true))
              (setf (formula-truth_value formula) result))
             (t
              (setf (formula-truth_value formula)
                    (mutilate_eval (formula-sub_formula2 formula) scenario))))))
     
;;; Note that <mutilate_eval> will never be called on the second conjunct
;;; if the first conjunct evaluates as false, or fails to evaluate.


(defun mutilate_eval_negation (formula scenario)
   "Evaluates formula <formula> with respect to scanario <scenario>.
    A sub-function of mutilate_eval.
    Deals with the case where the formula is a negation."

    (let ((result (mutilate_eval (formula-sub_formula1 formula) scenario)))
       (cond ((eq result 'true) 
              (setf (formula-truth_value formula) 'false))
             ((eq result 'false) 
              (setf (formula-truth_value formula) 'true))
             (t
             (setf (formula-truth_value formula)
                   (formula-truth_value (formula-sub_formula1 formula)))))))
;;; Note that this function reverses the truth-value stored at the embedded 
;;; sub-formula, unless the latter is a proposition letter (indicating that the 
;;; truth value could not be decided. In this case, the proposition letter is 
;;; simply copied into the truth-value of formula.


(defun formula_copy (formula)
      "Make a copy of a data-structure representing a formula"
      
   (cond ((symbolp formula) formula)
         (formula
          (make-formula :functor (formula-functor formula)
                        :sub_formula1 (formula_copy (formula-sub_formula1 formula))
                        :sub_formula2 (formula_copy (formula-sub_formula2 formula))
                        :truth_value (formula-truth_value formula)
           ))))


;;;/********************************************************/
;;;/*                                                      */
;;;/*           Scenario manipulation functions            */
;;;/*                                                      */
;;;/********************************************************/


(defstruct scenario (positives nil)    ;;; list of atoms which are true
                    (negatives nil)    ;;; list of atoms which are false
                    (potential 1))     ;;; can be any positive number

(setq *scenarios* nil)                 ;;; global variable storing scenarios

          
(defun start_again nil
   "Initialize <*scenarios*> to <nil>"
   
   (setq *scenarios* nil))


(defun true_in (proposition_letter scenario)
   "Decides whether atomic <proposition_letter> is true in <scenario>"

   (member proposition_letter (scenario-positives scenario)))
;;; Note that <proposition_letter> is a single proposition letter (an atom)
;;; rather than an atomic formula (which is an instance of the <formula> 
;;; structure.


(defun false_in (proposition_letter scenario)
   "Decides whether <proposition_letter> is false in <scenario>"

   (member proposition_letter (scenario-negatives scenario)))
;;; Note that <proposition_letter> is a single proposition letter (an atom)
;;; rather than an atomic formula (which is an instance of the <formula> 
;;; structure.

(defun decided_in (proposition_letter scenario)
   "Decides whether <proposition_letter> is either true or false in <scenario>"

   (member proposition_letter (append (scenario-positives scenario)
                                      (scenario-negatives scenario))))
;;; Note that <proposition_letter> is a single proposition letter (an atom)
;;; rather than an atomic formula (which is an instance of the <formula> 
;;; structure.

   
(defun test_consistent (scenario1 scenario2)
   "Returns non-nil if <scenario1> is consistent with <scenario2>"
 (and 
   (not (intersect (scenario-positives scenario1)
                   (scenario-negatives scenario2)))
   (not (intersect (scenario-positives scenario2)
                   (scenario-negatives scenario1)))))
       
(defun intersect (set1 set2)
   "Returns the intersection of two sets"

   (cond ((null set1) nil)
         ((member (car set1) set2) (cons (car set1) 
                                         (intersect (cdr set1) set2)))
         ((intersect (cdr set1) set2))))
   

;;; Strategy is to check that no proposition that is true in
;;; <scenario1> is false in <scenario2> and that no proposition
;;; that is false in <scenario1> is true in <scenario2>
           
(defun find_all_positive (proposition)
   "Find all those scenarios which decide a given proposition
    as being true"
   
   (do ((scenarios *scenarios* (cdr scenarios))
        (positive_scenarios nil 
         (cond ((true_in proposition (car scenarios)) 
                (cons (car scenarios) positive_scenarios))
               (t positive_scenarios))))
          ((null scenarios) positive_scenarios)))
 
(defun find_all_negative (proposition)
   "Find all those scenarios which decide a given proposition
    as being false"
   
   (do ((scenarios *scenarios* (cdr scenarios))
       (negative_scenarios nil 
         (cond ((false_in proposition (car scenarios)) 
                (cons (car scenarios) negative_scenarios))
               (t negative_scenarios))))
          ((null scenarios) negative_scenarios)))


(defun find_consistent_positive (proposition scenario)
   "Returns the list of all and only those scenarios which
    are consistent with <scenario> and which decide <proposition>
    as true"
    
   (mapcan #'(lambda (scenario_x)
               (cond ((test_consistent scenario scenario_x)
                      (list scenario_x))))
            (find_all_positive proposition)))
;;; Strategy  is to get the list of scenarios that decide <proposition>
;;; as true and simply to pick out those elements which are consistent  
;;; with <scenario>


(defun find_consistent_negative (proposition scenario)
   "Returns the list of all and only those scenarios which
    are consistent with <scenario> and which decide <proposition>
    as false"
    
   (mapcan #'(lambda (scenario_x)
               (cond ((test_consistent scenario scenario_x)
                      (list scenario_x))))
            (find_all_negative proposition)))
;;; Strategy  is to get the list of scenarios that decide <proposition>
;;; as false and simply to pick out those elements which are consistent  
;;; with <scenario>


(defun scenario_merge (scenario1 scenario2)
   "Returns the result of merging <scenario1> and <scenario2>,
    which are assumed to be consistent. The potential is taken
    from <scenario2>"
    
   (make-scenario
      :positives 
        (union (scenario-positives scenario1)
               (scenario-positives scenario2))
      :negatives 
        (union (scenario-negatives scenario1)
               (scenario-negatives scenario2))
      :potential 
            (scenario-potential scenario2)))
      

(defun find_extensions (proposition scenario)
   "Returns the list of all those scenarios formed by exending <scenario>
    consistently by some other consistent scenario that decides <proposition>
    positively or negatively"
    
   (append
      (mapcar #'(lambda (scenario_x) (scenario_merge scenario scenario_x))
              (find_consistent_positive proposition scenario))
      (mapcar #'(lambda (scenario_x) (scenario_merge scenario scenario_x))
              (find_consistent_negative proposition scenario))))
        


;;;/********************************************************/
;;;/*                                                      */
;;;/*       Degree of belief calculation functions         */
;;;/*                                                      */
;;;/********************************************************/


(defun compute_bel (f current_scenario)
      "compute the degree of belief in formula <f> in
       <current_scenario>, looking at extensions of <current_scenario>
       if necessary"

   (let ((formula (formula_copy f)))
    (mutilate_eval formula current_scenario)
;;; True to evaluate the formula in current scenario. The answer will either 
;;; be <true>, <false>, or some proposition letter. The last case arises when 
;;; the formula could not be fully evaluated, and the proposition letter 
;;; indicates where the evaluation process got stuck. This is then the 
;;; proposition letter which is used to extend to a richer scenario. It is 
;;; important to remember that, in this case, some of the subformula of 
;;; <formula> may have been succesfully evluated. These truth-values are 
;;; recorded in the data-structure <formula> in the obvious places. This 
;;; obviates the need for recalculation when the current scenario is extended.
   
   (cond ((eq (formula-truth_value formula) 'true) 1)
;;; If formula is definitely true in current scenario, then 1
         
         ((eq (formula-truth_value formula) 'false) 0)
;;; If formula is definitely false in current scenario, then 0
      
         (t
;;; Else, truth-value of formula contains the next proposition
;;; letter that needs to be settled.

          (do ((extensions (find_extensions (formula-truth_value formula)
                                            current_scenario)
                           (cdr extensions))
;;; So we get all ways of extending current scenario so as to decide the missing
;;; proposition letter (which is stored in the truth value of <formula>)
;;; and we sum the belief in <formula> in each of them, weighted by potential

               (so_far_belief 0.0 (+ so_far_belief
                                   (* (compute_bel 
                                         (formula_copy formula)
                                         (car extensions))
                                      (scenario-potential (car extensions)))))
               (so_far_potential 0.0 (+ so_far_potential
                                      (scenario-potential (car extensions)))))
              ((null extensions) (/ so_far_belief so_far_potential)))))))
;;; Notice that, in calling <compute_bel> recursively on the new, extended 
;;; scenarios, the parameter <formula> cannot simply be passed; rather a 
;;; copy of it must be made. This is because <compute_bel> calls 
;;; <mutilate_eval>, which, as the name suggests, does surgery on the 
;;; formula. When <mutlate_eval> is called on this copy, subformulae which 
;;; have truth-values <true> and <flase> already recorded there are not 
;;; re-processed. Otherwise, processing proceeds normally.


(defun monty_compute_bel (formula current_scenario number_of_runs)
      "compute the degree of belief in <formula> in
       <current_scenario>, using Monte Carlo method"

   (do ((counter number_of_runs (1- counter))
        (total_true 0 (+ total_true 
                         (single_run (formula_copy formula)  
                                     current_scenario))))
       ((zerop counter) (float (/ total_true number_of_runs)))))


(defun single_run (formula current_scenario)
   "Do a single run of the monte Carlo simulation. Note that 
    this mutillates the data-structure <formula>"
   
   (let ((current_val (mutilate_eval formula current_scenario)))
;;; Note that <mutilate_eval> mutilates <formula>. Thus, if the
;;; result is neither <true> nor <false>, when <formula> is recomputed
;;; in extensions of <current_scenario>, already-performed compuations
;;; are not repeated.

      (cond ((eq current_val 'true) 1)
            ((eq current_val 'false) 0)
            (t
             (single_run formula 
                         (find_random_extension current_val
                                                current_scenario))))))
                              

(defun potential_accumulate (set_of_scenarios)
   "Takes a set of scenarios and adds the potential of the first to the 
    second, then the (new) potential of the second to the third, and so
    on. The scenarios are desctructively modified (in regard of their
    potential, and the resulting new potential of the last scenario is 
    returned."
    
   (do ((set_of_scenarios1 set_of_scenarios (cdr set_of_scenarios1))
        (current_accumulation 0))
       ((null set_of_scenarios1) (float current_accumulation))
;;; Coerces output to be float so that random number generator in 
;;; <random_choose> works correctly.
       (setq current_accumulation 
             (+ current_accumulation 
                (scenario-potential (car set_of_scenarios1))))
       (setf (scenario-potential (car set_of_scenarios1)) 
             current_accumulation)))
    
(defun random_choose (set_of_scenarios)
   "Chooses a scenario at random from <set_of_scenarios> according to  
   their potential."
   
   (let ((cumulative_potential (potential_accumulate set_of_scenarios)))
;;; Calling this function mutilates the potential entries. It also returns 
;;; the last (and hence greatest) potential, coerced to be a float.

      (let ((random_number (random cumulative_potential)))
;;; get a random number between 0 and this total potential
         (do ((set_of_scenarios1 set_of_scenarios (cdr set_of_scenarios1)))
             ((< random_number (scenario-potential
                                 (car set_of_scenarios1))) 
              (car set_of_scenarios1))))))
 
         
(defun find_random_extension (proposition scenario)
   "Returns a randomly chosen scenario formed by exending <scenario>
    consistently by some other consistent scenario that decides <proposition>
    positively or negatively"
    
   (random_choose (find_extensions proposition scenario)))
;;; Note that <find_extensions> creates a set of *new* scenarios. Thus, the
;;; fact that random choose mutilates these scenarios does not matter.


   

;;;/********************************************************/
;;;/*                                                      */
;;;/*             Scenario definition functions            */
;;;/*                                                      */
;;;/********************************************************/


(defun input_scenario nil
   "This function asks for input to create a scenario"
   
   (prog (temp_input scenario) 
      (setq scenario (make-scenario))    
label1                   
     (terpri)
     (princ "Please input list of true proposition letters:  ")
     (setq temp_input (read))
     (cond ((not (atomic_list_check temp_input)) 
            (terpri) 
            (princ "There is a problem. You have not input a list of")
            (terpri)
            (princ "proposition letters. Please enter true propositions again.")
            (terpri)
            (go label1)))
     (setf (scenario-positives scenario) temp_input)
label2
     (terpri)
     (princ "Please input list of false proposition letters:  ")
     (setq temp_input (read))
     (cond ((not (atomic_list_check temp_input)) 
            (terpri) 
            (princ "There is a problem. You have not input a list of")
            (terpri) 
            (princ "proposition letters. Please enter false propositions again.")
            (terpri)
            (go label2)))
     (setf (scenario-negatives scenario) temp_input)
label3
     (terpri)
     (princ "Please input potential:  ")
     (setq temp_input (read))
     (cond ((not (potential_check temp_input)) 
            (terpri) 
            (princ "There is a problem. The potential you have input is")
            (terpri) 
            (princ "not a positive number. Please enter it again.")
            (terpri)
            (go label3)))
     (setf (scenario-potential scenario) temp_input)
     
     (terpri)
     (cond ((not (test_consistent scenario scenario)) 
            (terpri) 
            (princ "There is a problem. The scenario you have input is")
            (terpri) 
            (princ "inconsistent. Please start again.")
            (terpri)
            (go label1)))
     (push scenario *scenarios*)
     (return scenario)))

(defun atomic_list_check (sexpr)
   "Check that an input s-expression is a list of atoms"

   (cond ((null sexpr))
         ((and (listp sexpr)
               (symbolp (car sexpr))
               (car sexpr)
               (atomic_list_check (cdr sexpr))))))
      

(defun potential_check (sexpr)
   "Check that an input s-expression is a positive number"
   
   (and (numberp sexpr) (> sexpr 0)))


(defun input_formula nil
   "Read a formula from input and create a formula data-structure"
   
   (prog (temp_input formula)
label
      (terpri)
      (princ "Please input a formula:   ")
      (setq temp_input (read))
      (terpri)
      (setq formula (make-formula))
      (cond ((not (formula_create temp_input formula))
             (terpri)
             (princ "Formula improperly input. Please try again.")
             (terpri)
             (go label)))          
      (cond ((not (subsetp (get_prop_letters_from_formula formula)
                           (get_all_proposition_letters)))
             (princ "Warning: Formula contains proposition letters which are not mentioned in any scenario.")
             (terpri)))
      (return formula)))

(defun get_prop_letters_from_formula (formula)
    "Get all the proposition letters mentioned in the formula <formula>"
      (cond  ((null (formula-functor formula))  
              (list (formula-sub_formula1 formula)))     
             ((eq '- (formula-functor formula)) 
              (get_prop_letters_from_formula (formula-sub_formula1 formula)))
             (t 
              (union (get_prop_letters_from_formula (formula-sub_formula1 formula))
                     (get_prop_letters_from_formula (formula-sub_formula2 formula))))))

(defun formula_create (input_formula formula)
   "Takes a formula <input_formula> in a standard lisp input
    format and inserts the relevant bits in a the formula 
    data_structure <formula> (which it assumes already exists).
    The function returns <t> if the conversion succeeded (i.e. if 
    the input formula was legal), otherwise <nil>."
 
   (cond ((and (symbolp input_formula) (not (eq input_formula t)) input_formula)
          (setf (formula-functor formula) nil)
          (setf (formula-sub_formula1 formula) input_formula))
         ((not (listp input_formula)) nil)

         ((eq (cadr input_formula) '&)
          (setf (formula-functor formula) '&)
          (setf (formula-sub_formula1 formula)
                (make-formula))
          (setf (formula-sub_formula2 formula)
                (make-formula))
          (and  (formula_create (car input_formula) (formula-sub_formula1 
                   formula))
                (formula_create (caddr input_formula) (formula-sub_formula2 
                   formula))))

         ((eq (car input_formula) '-)
          (setf (formula-functor formula) '-)
          (setf (formula-sub_formula1 formula)
                (make-formula))
          (formula_create (cadr input_formula) (formula-sub_formula1 
                formula)))

	 ((eq (cadr input_formula) '->)
          (formula_create 
                (list '- (list (car input_formula) '&
                               (list '- (caddr input_formula))))
                formula))

         ((eq (cadr input_formula) '\/)
          (formula_create
                (list '- (list (list '- (car input_formula)) '&
                               (list '- (caddr input_formula))))
                formula))))
             
;;;/********************************************************/
;;;/*                                                      */
;;;/*           Scenario-list checking functions           */
;;;/*                                                      */
;;;/********************************************************/

                
(defun get_all_proposition_letters nil
   "Returns the list of all proposition letters which appear in 
    any of the scenarios in <*scenarios*>"
    
   (do ((scenarios *scenarios* (cdr scenarios))
        (propositions nil (union (scenario-positives (car scenarios))
                            (union (scenario-negatives (car scenarios))
                                   propositions))))
       ((null scenarios) propositions)))
 

(defun check_completeness_condition nil
   "Checks whether, for each possible scenario and for 
    each proposition in the propositions mentioned in <*scenarios*> 
    there exists a scenario in <*scenarios*> which is 
    consistent with the former scenario and which decides the proposition"
    
   (forall (scenario (get_generated_scenarios))
      (forall (proposition (get_all_proposition_letters))
         (exists (ent_scenario *scenarios*)
            (and (test_consistent scenario ent_scenario)
                 (or (decided_in proposition scenario)
                     (decided_in proposition ent_scenario)))))))
   


(defun get_generated_scenarios nil
   "Returns the list of all possible scenarios made up from the proposition
    letters in the scenarios declared for the ent"
    
   (mapcan #'positives_to_scenarios (power_set (get_all_proposition_letters))))                         

         
(defun positives_to_scenarios (set_of_propositions)
   "Takes a set of positive propositions and returns all the 
    possible scenarios with this set of propositions as positives" 
    
   (mapcar #'(lambda (negatives_x)
                 (make-scenario :positives set_of_propositions
                                :negatives negatives_x))
           (power_set (set-difference (get_all_proposition_letters) 
                                  set_of_propositions))))
                      

(defun power_set (set)
   "Returns the list of all sublists of <set>"
   
   (cond ((null set) (list nil))
         (t
          (let ((new_element (car set))
                (old_power_set (power_set (cdr set))))
             (append (mapcar #'(lambda (set_x) 
                                  (cons new_element set_x))
                             old_power_set)
                     old_power_set)))))
           
