/*
  title: ctable.c
  purpose: Two dimensional display mode colour-locking support (used by
    projectors.c).  Ensures colour entries are uniquely allocated.  Allows
    colour preferences to be specified during allocation.  Handles mouse
    events arising from the main display canvas and creates pop-up menus
    allowing manual color allocation etc. . .
  
  authors:  Gareth Lee.
  date:     10-02-94
  modified: 29-03-94

  Copyright (C) 1994 Gareth Lee.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  10-02-94: Routines extracted from projectors.c during a code rationalization.
  28-03-94: A mouse event driver was added for the display window.  This will
    allow selection of displayed trajectory labels to offer user specified
    colour changes.
  29-03-94: Color preferences added to the AllocateColorEntry routine.    
*/
#include <stdio.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/cursor.h>

/* default floating point type */
typedef REAL real;

#include "projectors.h"
#include "fview.h"
#include "interface.h"
#include "fviewio.h"

/* for use creating parentless wdigets */
#define NO_PARENT ((Xv_opaque) 0)

typedef struct
{
  int links;
  int key;
} ColorStruct;

ColorStruct ColorLock[MAX_COLORS];    /* Ensures unique allocation of colors */

/* Pop-up menu objects */
Menu_item LabelPointSetItem;
Menu LabelMenu, LabelColorsMenu;

/*****************************************************************************/
/* Colour table management functions                                         */
/*****************************************************************************/

/*
  SetupColorEntries: Initialize colour allocation table.
*/
void SetupColorEntries(void)
{
  int i;
  
  for (i = 0; i < MAX_COLORS; i++)
  {
    ColorLock[i].links = 0;
    ColorLock[i].key = FVIEW_COLR_NOPREF;
  }
}

/*****************************************************************************/

/*
  AllocateColorEntry: Allocate a unique color for drawing.
*/
int AllocateColorEntry(int key)
{
  int i;

  /* if a colour preference key is specified then try to find a match */
  if (key != FVIEW_COLR_NOPREF)
  {
    for (i = 0; i < MAX_COLORS; i++)
      if (ColorLock[i].key == key)
      {
        ColorLock[i].links++;
        return i;
      }
  }

  /* allocate a unique color */
  for (i = 0; i < MAX_COLORS; i++)
    if (ColorLock[i].links == 0)
      break;
  if (i == MAX_COLORS)
  {
    printf("Error: Failed to find a free color in which to draw\n");
    Quit(-1);
  }
  else
  {
    ColorLock[i].links = 1;           /* reserve color to ensure exclusivity */
    if (key != FVIEW_COLR_NOPREF)
      ColorLock[i].key = key;            /* save a key for the colour record */
  }

  return i;
}

/*****************************************************************************/

void AllocateSpecificColor(int color, int key)
{
  if (color < 0 || color >= MAX_COLORS)
  {
    printf("Error: specific color (%d) requested which is out of range\n",
           color);
    Quit(-1);
  }
  ColorLock[color].links++;    /* form an extra link to the specified colour */

  /*
    if the new colour did not have a previous key set then it's key field may
    be requesitioned.  Otherwise it may already be in use.
  */
  if(ColorLock[color].key == FVIEW_COLR_NOPREF && key != FVIEW_COLR_NOPREF)
    ColorLock[color].key = key;
}

/*****************************************************************************/

/*
  FreeColorEntry: reclaim a color previously allocated.
*/
void FreeColorEntry(int color)
{
  int links = ColorLock[color].links;

  if (links == 0)
  {
    printf("Error: Unused color should never be freed\n");
    Quit(-1);
  }
  if (links >= 1)
  {
    ColorLock[color].links--;
    if (links == 1)            /* this was the last link to the colour entry */
      ColorLock[color].key = FVIEW_COLR_NOPREF;                 /* reset key */
  }
}

/*****************************************************************************/
/* Mouse handling and pop-up menu support functions for the display canvas   */
/*****************************************************************************/

void LabelPointSetCallback(void)
{
  SpeechRecord *sd;
  
  sd = (SpeechRecord *) xv_get(LabelMenu, MENU_CLIENT_DATA);
  if (sd->ptset)
    sd->ptset = 0;                      /* set continuous trajectory drawing */
  else
    sd->ptset = 1;                            /* display data as a point set */
  RefreshGraphics();
}

/*****************************************************************************/

void LabelColorsCallback(Menu menu, Menu_item item)
{
  int color;
  SpeechRecord *sd;
  
  color = (int) xv_get(item, MENU_CLIENT_DATA);
  sd = (SpeechRecord *) xv_get(LabelMenu, MENU_CLIENT_DATA);
  FreeColorEntry(sd->color);              /* free previous colour allocation */
  AllocateSpecificColor(color, sd->color_key);  /* indicate new color choice */
  sd->color = color;      /* accept new color by registering in SpeechRecord */
  RefreshGraphics();
}

/*****************************************************************************/

void CreateTrajectoryMenu(void)
{
  const int ncols = 4;                               /* display in 4 columns */

  int i, nrows;
  Menu_item mi;

  /* create colours pull-right menu */
  nrows = MAX_COLORS / ncols;                 /* decide how many rows to use */
  LabelColorsMenu =
    xv_create(NO_PARENT, MENU,
              MENU_NROWS, nrows,
              MENU_NCOLS, ncols,
              MENU_NOTIFY_PROC, LabelColorsCallback,
              NULL);
  for (i = 0; i < MAX_COLORS; i++)
  {
    mi = (Menu_item)
      xv_create(NO_PARENT, MENUITEM,
                MENU_STRING, TrajectoryColors[i],
                MENU_CLIENT_DATA, i,
                NULL);
    xv_set(LabelColorsMenu, MENU_APPEND_ITEM, mi, NULL);
  }

  /* Create remaining menu options */
  LabelPointSetItem = (Menu_item)
    xv_create(NO_PARENT, MENUITEM,
              MENU_STRING, "Point set",
              MENU_NOTIFY_PROC, LabelPointSetCallback,
              NULL);
  LabelMenu =
    xv_create(NO_PARENT, MENU,
              MENU_APPEND_ITEM, LabelPointSetItem,
              MENU_ITEM,
                MENU_STRING, "Colours",
                MENU_PULLRIGHT, LabelColorsMenu,
                NULL,
              NULL);
}

/*****************************************************************************/

/*
  MouseSelection: Handler for selection of a label entry within the canvas
*/
void MouseSelection(Event *event, SpeechRecord *sd)
{
  const char delimiter = '/';                         /* file path delimiter */
  static char menu_title_buffer[100] = "(???)";     /* title for pop-up menu */
  
  static int menu_allocated = 0;
  char *cp;

  if (!menu_allocated)
  {
    CreateTrajectoryMenu();
    menu_allocated = 1;
  }
  cp = strrchr(sd->file_name, delimiter);   /* strip off path from file_name */
  if (cp == NULL)
    cp = sd->file_name;        /* use whole file name in delimiter not found */
  else
    cp++;
  sprintf(menu_title_buffer, "(%s)", cp);
  xv_set(LabelMenu,
         MENU_TITLE_ITEM, menu_title_buffer,
         MENU_CLIENT_DATA, sd,       /* save sd for use by callback routines */
         NULL);
  if (sd->ptset)
    xv_set(LabelPointSetItem, MENU_STRING, "Trajectory", NULL);
  else
    xv_set(LabelPointSetItem, MENU_STRING, "Point set", NULL);
  menu_show(LabelMenu, DisplayCanvas, event, NULL);
}

/*****************************************************************************/

/*
  MouseClick: Handler for all mouse down events within the window which do not
    correspond to selection of a label entry.
*/
void MouseClick(int x, int y, short action)
{
  switch (action)
  {
  case ACTION_SELECT:
    printf("Mouse left click at (%d,%d)\n", x, y);
    break;
  case ACTION_ADJUST:
    printf("Mouse middle click at (%d,%d)\n", x, y);
    break;
  case ACTION_MENU:
    printf("Mouse right click at (%d,%d)\n", x, y);
    break;
  }
}

/*****************************************************************************/

/*
  DisplayEventHandler: processes mouse events arising from the display canvas
    and decides whether they correspond to (i) the selection of a label entry
    within the canvas or (ii) any other mouse down event within the canvas.
*/
void DisplayEventHandler(Xv_window window, Event *event)
{
  static SpeechRecord *selected = NULL;

  int s, mx, my, rx, ry;
  short action;
  SpeechRecord *sd;

  action = event_action(event);
  if (action == ACTION_NULL_EVENT)                 /* ignore spurious events */
    return;

  mx = event_x(event);                    /* location at which event occured */
  my = event_y(event);

  /* ascertain whether the event was the selection of a trajectory record */
  /* but only allow selection when displaying labels */
  if (display_label_mode && action == ACTION_MENU)
  {
    switch (display_mode)
    {
    case two_dims:
    case spectrogram:
      /* check all displayed utterances in two dimensions */
      for (s = 0; s < samples; s++)
      {
        sd = spdata[s];
        rx = mx - sd->label_box.x;
        ry = sd->label_box.y - my;
        if (rx >= 0 && rx < sd->label_box.width  &&
            ry >= 0 && ry < sd->label_box.height)
          if (event_is_down(event))
          {
            MouseSelection(event, sd);
            selected = sd;
            return;
          }
      }
      break;
    case three_dims:
      /* check the single displayed utterance in three dimensions */
      sd = spdata[utterance_index];
      rx = mx - sd->label_box.x;
      ry = sd->label_box.y - my;
      if (rx >= 0 && rx < sd->label_box.width  &&
          ry >= 0 && ry < sd->label_box.height)
        if (event_is_down(event))
        {
          MouseSelection(event, sd);
          selected = sd;
          return;
        }
      break;
    }  /* switch (display_mode) */
  }

  /* process other mouse down events within the display window */
  if (event_is_down(event))
    MouseClick(mx, my, action);
}

/*****************************************************************************/

/* end of ctable.c */
