/*
  title: nist.c
  purpose: Format converting filter to interface NIST (*.wav) files with
    the fview package.

  authors:  Gareth Lee and Roberto Togneri.
  date:     28-10-93
  modified: 26-04-94

  Copyright (C) 1994 Gareth Lee and Roberto Togneri.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  28-10-93: Based on program NIST2DAT.C by Roberto Tognieri.
  26-04-94: Byte swapping code added to simplify porting to machines with a
    different byte ordering. Normally `noswap' should be used on Motorola/Sun
    platforms whilst `swap' should be enabled on Intel/DEC machines.
*/
#define VERSION "1.2"

/*======================== STANDARD INCLUDE FILES ==========================*/
#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include <string.h>
#include <memory.h> 
#include <assert.h>

#ifdef LINUX
#include <endian.h>
#endif
#ifdef SUNOS
#include "../endian.h"
#endif
#include "sfheader.h"
#include "../fviewio.h"
#include "swappers.h"

#define PRE_HDR_SIZE (16)
#define MAX_HDR_SIZE (10240)

/* Strings used in parsing NIST ascii header */
#define SAMPLE_COUNT_STR  "sample_count -i"
#define SAMPLE_FREQ_STR  "sample_rate -i"

char    pre_header[PRE_HDR_SIZE];
int     header_size;
char    header[MAX_HDR_SIZE];                     /* NIST header information */
FviewSampleHeader fsh;                                 /* output file header */

/* reorder bytes in all binary values read */
#if (BYTE_ORDER == LITTLE_ENDIAN)
int     byte_swap = 0;                            /* Intel/DEC byte ordering */
#elif (BYTE_ORDER == BIG_ENDIAN)
int     byte_swap = 1;                         /* Motorola/Sun byte ordering */
#else
#error byte order undefined/unknown
#endif

extern char *optarg;                   /* for use with the getopt() function */

typedef short sample_t;  /* 16 bit signed data type used for reading samples */

main(int argc, char *argv[])
{
  int   i, opt;
  FILE	*fdata;
  char	nistfile[1024];
  char	*sptr, *hptr;
  char magic[100];
  sample_t data;                                        /* 16 bit data value */

  assert(sizeof(sample_t) == 2);     /* check that data types are compatible */
  
  /* read command line options */
  while ((opt = getopt(argc, argv, "B:f:m:")) != -1)
    switch (opt) {
    case 'B':
      if (*optarg == 's' || *optarg == 'S')
        byte_swap = !byte_swap;
      break;
    case 'f':
      strcpy(nistfile, optarg);
      break;
    case 'm':
      strcpy(magic, optarg);
      break;
    case '?':
      fprintf(stderr, "usage: nist -f file_name -m magic number (char[3])\n");
      fprintf(stderr, "nist: option -%c not supported\n", opt);
      exit(-1);
    }

  if ((fdata = fopen(nistfile, "r")) == NULL)
  {
    fprintf(stderr,"nist: error opening %s for reading\n", nistfile);
    exit(-1);
  }

  /* read NIST ASCII pre_header */
  if (fread(pre_header, sizeof(char), PRE_HDR_SIZE, fdata) != PRE_HDR_SIZE)
  {
    fprintf(stderr, "Error: cannot read pre-header record\n");
    exit(-1);    
  }

  /* check NIST magic number */
  sptr = magic;
  hptr = pre_header;
  while (*sptr != '\0')
    if (*sptr++ != *hptr++)
      break;
  if (*sptr != '\0')
  {
    fprintf(stderr,"nist: magic number incorrect (%s != %s)\n", header, magic);
    exit(-1);
  }

  /* Find size of main header and read in */
  if (sscanf(pre_header + 8, "%d", &header_size, fdata) != 1)
  {
    fprintf(stderr, "Error: cannot decipher header size\n");
    exit(-1);
  }
  header_size -= PRE_HDR_SIZE;
  if (fread(header, sizeof(char), header_size, fdata) != header_size)
  {
    fprintf(stderr, "Error: cannot read header record\n");
    exit(-1);
  }
  assert(header_size == 1008);

  /* initialise data header file */
  strcpy(magic, FVIEW_SAMPLE_MAGIC);
  memcpy(fsh.magic, magic, 8);

  /* Parse header for number of samples */
  hptr = strstr(header, SAMPLE_COUNT_STR);
  if (hptr == NULL)
  {
    fprintf(stderr, "nist: error parsing NIST ASCII header for '%s'\n",
            SAMPLE_COUNT_STR);
    exit(-1);
  }
  else
    hptr += strlen(SAMPLE_COUNT_STR);
  if (sscanf(hptr, "%d", &(fsh.number_samples)) != 1)
  {
    fprintf(stderr, "nist: error reading sample_count from NIST header\n");
    exit(-1);
  }

  /* Parse header for samples frequency */
  hptr = strstr(header, SAMPLE_FREQ_STR);
  if (hptr == NULL)
  {
    fprintf(stderr, "nist: error parsing NIST ASCII header for '%s'\n",
            SAMPLE_FREQ_STR);
    exit(-1);
  }
  else
    hptr += strlen(SAMPLE_FREQ_STR);
  if (sscanf(hptr, "%d", &(fsh.sample_freq)) != 1)
  {
    fprintf(stderr, "nist: error reading sample_count from NIST header\n");
    exit(-1);
  }

  /* Find the extent of the header text */
  for (i = 0; i < header_size; i++)
    if (header[i-1] == '\n' && header[i] == '\n')
      break;
  if (i < header_size)
    fsh.info_size = i;
  else
  {
    fprintf(stderr, "nist: error finding size of NIST header\n");
    exit(-1);
  }
  
  /* write header file */
  fwrite(&fsh, sizeof(FviewSampleHeader), 1, stdout);
  fwrite(header, sizeof(char), fsh.info_size, stdout);
  
  /* transfer data */
  for (i = 0; i < fsh.number_samples; i++)
  {
    if (fread(&data, sizeof(sample_t), 1, fdata) != 1)
    {
      fprintf(stderr, "nist: sample data finished prematurely\n");
      exit(-1);
    }
    if (byte_swap)
      SWAP2(&data);
    fwrite(&data, sizeof(sample_t), 1, stdout);
  }
  
  fclose(fdata);

  fprintf(stderr, "nist (%s): %s (%d samps)\n",
                                        VERSION, nistfile, fsh.number_samples);
}
