;****************************************************************
;
;  Graphics package promises to respond to the following:
;
;    (make-window x y w h) => window
;    (make-position x y)   => position
;
;    (disp.init win display) => boolean
;    (disp.terminate) => void
;    (disp.reset) => void
;
;    (disp.text-at string position) => boolean
;    (disp.center-text string position) => boolean
;    (disp.center-text-in-window string window) => boolean
;    (disp.write-text-within-window string window font) => boolean
;    (disp.text-extent string) => position (offset)
;
;    (disp.draw-rectangle window) => boolean
;    (disp.draw-line pos1 pos2)  => boolean
;    (disp.draw-disconnected-lines list-of-lines) => boolean
;    (disp.draw-circle center-pos radius) => boolean
;
;    (disp.draw-connected-lines position-list) => boolean
;    (disp.draw-disconnected-lines position-list) => boolean
;
;    (disp.clear-rectangle window) => boolean
;    (disp.clear-rectangle-interior window) => boolean
;    (disp.clear-display) => boolean
;
;    (disp.fill-rectangle window) => boolean
;    (disp.fill-triangle pos1 pos2 pos3) => boolean
;    (disp.fill-circle center-pos radius) => boolean
;
;    (disp.tiny-font) => font-id
;    (disp.small-font) => font-id
;    (disp.medium-font) => font-id
;    (disp.large-font) => font-id
;    (disp.huge-font) => font-id
;    (disp.standard-font) => font-id
;    (disp.get-font-id font-name-string) => font-id
; 
;    (disp.with-font font-id  --body--)
;    ***** NLA  (disp.with-origin origin-pos --body--)  ******
;    (disp.with-clip-window  win --body--)
;    (disp.with-erasure  --body--)
;

;***************************************************************
;  Windows as vectors

(defun make-window (x y w h)
  (vector x y w h))

(defun make-position (x y)
  (vector x y))

(defun make-point (x y)
  (make-position x y))

(defun x-coord (obj)
  (elt obj 0))

(defun y-coord (obj)
  (elt obj 1))

(defun width (obj)
  (elt obj 2))

(defun height (obj)
  (elt obj 3))

(defun window? (obj) 
  (and (vectorp obj) (= 4 (length obj))))

(defun position? (obj) 
  (and (vectorp obj) (= 2 (length obj))))

(defun vertical-midpoint (window)
  (+ (y-coord window) (round (/ (height window) 2))))

(defun horizontal-midpoint (window)
  (+ (x-coord window) (round (/ (width window) 2))))

(defun add-window-offset (win1 win2)
  (if (and (window? win1) (or (window? win2) (position? win2)))
      (make-window (+ (x-coord win1) (x-coord win2)) 
                   (+ (y-coord win1) (y-coord win2))
                   (width win1)
                   (height win1))
      NIL))

(defun add-position-offset (pos1 pos2)
  (if (and (or (window? pos1) (position? pos1))
           (or (window? pos2) (position? pos2)))
      (make-position (fx+ (x-coord pos1) (x-coord pos2)) 
                     (fx+ (y-coord pos1) (y-coord pos2)))
      NIL))

(defun unpack-window (win)
  (values (x-coord win) (y-coord win) (width win) (height win)))

(defun unpack-position (pos)
  (values (x-coord pos) (y-coord pos)))

;******************************************************************

(defconstant   *tiny-font-path*     "5x8")
(defconstant   *small-font-path*    "6x10")
(defconstant   *medium-font-path*   "8x13")
(defconstant   *large-font-path*    "9x15")
(defconstant   *huge-font-path*     "10x20")

(defvar *tiny-font-index* nil)
(defvar *small-font-index* nil)
(defvar *medium-font-index* nil)
(defvar *large-font-index* nil)
(defvar *huge-font-index* nil)

(defun init-fonts ()
  (setf *tiny-font-index*   (grafp_load_font *tiny-font-path*))
  (setf *small-font-index*  (grafp_load_font *small-font-path*))
  (setf *medium-font-index* (grafp_load_font *medium-font-path*))
  (setf *large-font-index*  (grafp_load_font *large-font-path*))
  (setf *huge-font-index*   (grafp_load_font *huge-font-path*))
  (values))

(defun disp.tiny-font () *tiny-font-index*)
(defun disp.small-font () *small-font-index*)
(defun disp.medium-font () *medium-font-index*)
(defun disp.large-font () *large-font-index*)
(defun disp.huge-font () *huge-font-index*)
(defun disp.standard-font () (disp.medium-font))


(defun disp.init (win display-string)
  (multiple-value-bind (x y w h) (unpack-window win)
    (grafp_init x y w h display-string)
    (init-fonts)))

(defun disp.terminate ()
  (grafp_terminate))

(defun disp.text-at (str pos)
  (multiple-value-bind (x y) (unpack-position pos)
    (grafp_text_at str x y)))

(defun disp.center-text (str pos)
  (multiple-value-bind (x y) (unpack-position pos)
    (grafp_center_text str x y)))

(defun disp.center-text-in-window (str win)
  (grafp_center_text str 
					 (horizontal-midpoint win) 
					 (vertical-midpoint win)))

(defun disp.write-text-within-window (str win font)
  (disp.with-font font
    (grafp_center_text str (horizontal-midpoint win) (vertical-midpoint win))))

(defun disp.text-extent (str)
  (grafp_text_extent str))

(defun disp.draw-rectangle (win)
  (multiple-value-bind (x y w h) (unpack-window win)
    (grafp_draw_rectangle x y w h)))

(defun disp.draw-line (pos1 pos2)
  (grafp_draw_line (x-coord pos1) (y-coord pos1) (x-coord pos2) (y-coord pos2)))

(defun disp.draw-disconnected-lines (list-of-positions)
  (cond 
   ((null list-of-positions) (values))
   ((< (length list-of-positions) 2)
    (error "Disconnected lines got non-multiple of four"))
   (else 
    (disp.draw-line (car list-of-positions) (cadr list-of-positions))
    (disp.draw-disconnected-lines (cddr list-of-positions)))))

(defun disp.draw-connected-lines (list-of-positions)
  (cond
   ((<= (length list-of-positions) 1) (values))
   (else 
    (disp.draw-line (car list-of-positions) (cadr list-of-positions))
    (disp.draw-connected-lines (cdr list-of-positions)))))

(defun disp.draw-circle (center-pos r)
  (grafp_draw_circle (x-coord center-pos) (y-coord center-pos) r))

(defun disp.clear-rectangle (win)
  (multiple-value-bind (x y w h) (unpack-window win)
    (grafp_clear_rectangle x y w h)))

(defun disp.clear-rectangle-interior (win)
  (multiple-value-bind (x y w h) (unpack-window win)
    (grafp_clear_rectangle (+ x 1) (+ y 1) (- w 2) (- h 2))))

(defun disp.clear-display ()
 (grafp_clear_display))

(defun disp.fill-rectangle (win)
  (multiple-value-bind (x y w h) (unpack-window win)
    (grafp_fill_rectangle x y w h)))

(defun disp.fill-triangle (p1 p2 p3)
  (multiple-value-bind (x1 y1) (unpack-position p1)
	(multiple-value-bind (x2 y2) (unpack-position p2)
	  (multiple-value-bind (x3 y3) (unpack-position p3)
		(grafp_fill_triangle x1 y1 x2 y2 x3 y3)))))

(defun disp.fill-circle (center-pos r)
  (grafp_fill_circle (x-coord center-pos) (y-coord center-pos) r))

(defun disp.get-event ()
    (if (> (grafp_check_event) 0)
		(values T (grafp_last_key) (grafp_last_x) (grafp_last_y))
		(values NIL 0 0 0)))

(defvar *event-queue* '())

(defun check-for-events ()
  (multiple-value-bind (found? key x y) (disp.get-event)
	(cond
	  ((not found?) (values))
	  (T (setf *event-queue* (cons (list key x y) *event-queue*))
		 (check-for-events)))))

(defun unload-events ()
  (let ((old-event-queue (reverse *event-queue*)))
	(setf *event-queue* '())
	old-event-queue))

(defun event-and-input-loop ()
  (cond
    ((not (process-command (get-command)))
     (values))
    (T (check-for-events)
       (process-events)
       (event-and-input-loop))))

(defun process-events ()
  (do ((events (unload-events) (cdr events)))
      ((null events) (return))
    (format T "Event is ~a~%" (car events))))
	
(defun get-command ()
  (format T "Command: ")
  (read *terminal-io*))

(defun process-command (cmd)
  (cond
    ((eq cmd 'QUIT) NIL)
    (T (write *terminal-io* cmd)
	   T)))