(require 'clx)
(use-package 'xlib)

(defconstant *standard-font-name* "8x13")
(defconstant *max-fonts*     25)

;*****************************************************;
;*    X globals                                      *;
;*****************************************************;

(defvar *mydisplay* nil)     ; Display *
(defvar *mywindow* nil)      ; Window
(defvar *mygc* nil)          ; GC
(defvar *myscreen* nil)      ; int 
(defvar *default-foreground* nil)   ; unsigned long 
(defvar *default-background* nil)   ;

;*****************************************************;
;*   Other globals                                   *;
;*****************************************************;

(defvar *erasing* nil)
(defvar *xorigin* 0)
(defvar *yorigin* 0)
(defvar *next-font-index* 0)

(defvar *font-ids* (make-array (list *max-fonts*)))
(defvar *current-font-index* 0)

;**************************************************;
;  Former macros, now utility functions

(defun flush ()
 (display-force-output *mydisplay*))

(defun originate (x y)
  (values (+ x *xorigin*)
	  (+ y *yorigin*)))

(defun originate-point-list (point-list &key (destructive nil))
  (cond
    ((null point-list) nil)
    (t (multiple-value-bind (new-x new-y) 
	   (originate (car point-list) 
		      (cadr point-list))
	 (cond
	   (destructive
	    (setf (car point-list) new-x)
	    (setf (cadr point-list) new-y)
	    (originate-point-list (cddr point-list) :destructive T))
	   (t (cons new-x 
		    (cons new-y 
			  (originate-point-list (cddr point-list)
						:destructive NIL)))))))))

;**********************************************************************;
;**********************************************************************;
;***       Generic drawers for lines, circles, and polygons           *;
;**********************************************************************;
;**********************************************************************;

(defun really-draw-line (old-x1 old-y1 old-x2 old-y2)
  (multiple-value-bind (x1 y1) (originate old-x1 old-y1)
    (multiple-value-bind (x2 y2) (originate old-x2 old-y2)
      (draw-line *mywindow* *mygc* x1 y1 x2 y2)))
  (flush))

(defun really-draw-circle (old-x old-y radius filled?)
  (multiple-value-bind (x y) (originate old-x old-y)
    (draw-arc *mywindow* 
	      *mygc*
	      (- x radius) (- y radius)
	      (* radius 2) (* radius 2)
	      0 (* 2 pi)
	      filled?)
    (flush)))

(defun really-draw-lines (points filled?)
  (let ((new-points (originate-point-list points)))
    (draw-lines *mywindow* *mygc* new-points filled?)
    (flush)
    1))

(defun really-draw-polygon (points filled?)
  (let* ((end-points (list (car points) (cadr points)))
	 (new-points (nconc (copy-list points) end-points)))
    (originate-point-list new-points :destructive T)
    (draw-lines *mywindow* *mygc* new-points :fill-p filled?)
    (flush)
    1))

(defun really-draw-rectangle (old-x old-y width height filled?)
  (multiple-value-bind (x y) (originate old-x old-y)
    (draw-rectangle *mywindow*  *mygc* x y width height filled?)
    (flush)))

;*******************************************************************

(defun grafp_draw_rectangle (x y width height)
  (really-draw-rectangle x y width height NIL))

(defun grafp_draw_line (x1 x2 y1 y2)
  (really-draw-line x1 x2 y1 y2))

(defun grafp_draw_circle (x y radius)
  (really-draw-circle x y radius NIL))

(defun grafp_draw_triangle (&rest points)
  (really-draw-polygon points NIL))

;**************

(defun grafp_fill_rectangle (old-x old-y width height)
  (really-draw-rectangle old-x old-y width height t))

(defun grafp_fill_triangle (&rest points)
  (really-draw-polygon points T))

(defun grafp_fill_circle (old-x old-y radius)
  (really-draw-circle old-x old-y radius T))

;**********************************************************************;
;**********************************************************************;
;***       Clearing rectangles, display                               *;
;**********************************************************************;
;**********************************************************************;

(defun grafp_clear_rectangle (old-x old-y width height)
  (multiple-value-bind (x y) (originate old-x old-y)
    (clear-area *mywindow* :x x :y y :width width :height height
		:exposures-p nil)
    (flush)))

(defun grafp_clear_display ()
  (clear-area *mywindow*)
  (flush))


;**********************************************************************;
;**********************************************************************;
;***       Text                                                       *;
;**********************************************************************;
;**********************************************************************;

(defun grafp_text_at (str old-x old-y)
  (multiple-value-bind (x y) (originate old-x old-y)
	(draw-glyphs *mywindow* *mygc* (truncate x) (truncate y) str)
	(flush)
	1))

(defun grafp_center_text (str x y)
  (multiple-value-bind (width ascent)
      (grafp_text_extent str)
    (grafp_text_at str 
		   (- x (/ width 2))
		   (+ y (/ ascent 2)))))

(defun grafp_text_extent (str)
  (multiple-value-bind 
	(width ascent descent left right font-ascent direction fnd)
      (text-extents (gcontext-font *mygc*) str)
    (values width ascent)))


;**********************************************************************;
;**********************************************************************;
;***       Erasure                                                    *;
;**********************************************************************;

(defun grafp_get_erasure ()
  *erasing*)

(defun grafp_set_erasure (val)
  (cond
    ((= val 1)
     (setf (gcontext-foreground *mygc*) *default-background*)
     (setf *erasing* 1))
    (t (setf (gcontext-foreground *mygc*) *default-foreground*)
       (setf *erasing* 0))))

;**********************************************************************;
;**********************************************************************;
;***       Origin
;**********************************************************************;
;**********************************************************************;

(defun grafp_get_origin ()
  (values *xorigin* *yorigin*))

(defun grafp_set_origin (x y)
  (setf *xorigin* x)
  (setf *yorigin* y))

;**********************************************************************;
;**********************************************************************;
;***       Font stuff                                                 *;
;**********************************************************************;
;**********************************************************************;


(defun grafp_load_font (str)
  (cond
    ((> *next-font-index* *max-fonts*) -1)
    (T (setf (elt *font-ids* *next-font-index*) (open-font *mydisplay* str))
       (incf *next-font-index*)
       (- *next-font-index* 1))))

(defun grafp_get_font ()
  *current-font-index*)

(defun grafp_set_font (font_index)
  (setf (gcontext-font *mygc*) (elt *font-ids* font_index))
  (setf *current-font-index* font_index)
  (values))

(defun init_fonts ()
  (setf *next-font-index* 0)
  (grafp_set_font (grafp_load_font *standard-font-name*)))

;**********************************************************************;
;**********************************************************************;
;***       Initialize and terminate                                   *;
;**********************************************************************;
;**********************************************************************;

(defun grafp_init (x y w h &optional (disp ""))
  (setf *erasing* 0)
  (setf *xorigin* 0)
  (setf *yorigin* 0)
  (init_window x y w h disp)
  (init_fonts)
  1)

(defun init_window (x y width height disp)
  (setf *mydisplay* (open-display disp))
  (setf *myscreen*  (display-default-screen *mydisplay*))
  (setf *default-background* (screen-white-pixel *myscreen*))
  (setf *default-foreground* (screen-black-pixel *myscreen*))
  (setf *mywindow* 
	(create-window :parent (screen-root *myscreen*)
		       :x x 
		       :y y 
		       :width width 
		       :height height
		       :background *default-background*
                       :backing-store :always))
  (setf *mygc* (create-gcontext :drawable *mywindow* 
				:background *default-background*
				:foreground *default-foreground*))
  (setf (window-event-mask *mywindow*)
	(make-event-mask :key-press :exposure :owner-grab-button :button-press))
  (map-window *mywindow*)
;  (wait_for_exposure)
  1)

(defun wait_for_exposure ()
  (do ((finished nil))
      (finished 1)
    (event-case (*mydisplay* :timeout nil)
		(:exposure (count)
			   (when (zerop count) (setf finished t)))
		(t ()))))


;**********************************************************************;

(defun grafp_terminate ()
  (when (not (null *mygc*))
    (free-gcontext *mygc*)
    (setf *mygc* NIL))
  (when (not (null *mywindow*))
    (destroy-window *mywindow*)
    (setf *mywindow* NIL))
  (when (not (null *mydisplay*)) 
    (close-display *mydisplay*)
    (setf *mydisplay* NIL))
  1)


;**********************************************************************;
;*  Input (keyboard or mouse) ....
;*********************************************************************;

;* Return 1 iff there is an event, which may be keyboard or mouse      *;
;* input.  If there is, set variables key, x, and y, which can then    *;
;* be queried.                                                         *;

;(defvar *last-event-key* nil)
;(defvar *last-event-x*   nil) 
;(defvar *last-event-y*   nil)

;(defun generic-event-handler (&key :display :event-key :send-event-p 
;							  &rest event-slots)
;  (setf *last-event-key* (cadr (assq :code event-slots)))
;  (setf *last-event-x*   (cadr (assq :x event-slots)))
;  (setf *last-event-y*   (cadr (assq :y event-slots)))
;  t)
;
;(defun grafp_process_event (wait?)
;  (let ((gotone? (process-event *mywindow* 
;			:handler generic-event-handler 
;			:timeout 0)))
;	(cond
;	  ((and (not gotone?) wait?)
;	   (grafp_process_event wait?))
;	  ((not gotone?) (values nil nil nil))
;	  (t (values *last-event-key* *last-event-x* *last-event-y*)))))
;	
;(defun grafp_wait_for_input ()
;  (grafp_process_event T))
;

