;*******************************************************************
;  Main truck

;  Display-names is a list of symbols, e.g. FUEL-GAUGE, ARM1  that
;  identify the displayers in the list displays.
;  Note that each arm has two displayers, one for the arm and one 
;  for the associated bay.  The bay will have the name ARMNAME-BAY,
;  where ARMNAME always be either ARM1 or ARM2

(defun arm-bay-name (arm-name)
  (intern (string-append (symbol-name arm-name) "-BAY")))

(defun find-displayer (truck-disp disp-name)
  (let ((pos (position disp-name (display-names truck-disp) :test #'eq)))
	(if (null pos) 	
		nil	
		(nth pos (displays truck-disp)))))

;*******************************************************************

(defclass truck-disp  ()
  ((internal-window :accessor internal-window :initarg :internal-window)
   (bay-names     :accessor bay-names      :initarg :bay-names)
   (arm-names     :accessor arm-names      :initarg :arm-names)
   (display-names :accessor display-names  :initarg :display-names)
   (displays      :accessor displays       :initarg :displays)))

;*******************************************************************

(defmethod dm-add ((self truck-disp) appendage obj)
  (if (memq appendage (arm-names self))
	  (dm-add (find-displayer self (arm-bay-name appendage)) nil obj)
	  (dm-add (find-displayer self appendage) nil obj)))

(defmethod dm-del ((self truck-disp) appendage obj)
  (if (memq appendage (arm-names self))
	  (dm-del (find-displayer self (arm-bay-name appendage)) nil obj)
	  (dm-del (find-displayer self appendage) nil obj)))

(defmethod dm-redraw ((self truck-disp) appendage obj)
  (dm-redraw (find-displayer self appendage) '() obj))

(defmethod dm-empty ((self truck-disp) appendage)
  (if (memq appendage (arm-names self))
	  (dm-empty (find-displayer self (arm-bay-name appendage)) nil)
	  (dm-empty (find-displayer self appendage) nil)))

(defmethod dm-set ((self truck-disp) appendage val)
  (dm-set (find-displayer self appendage) '() val))

(defmethod dm-move ((self truck-disp) appendage loc)
  (dm-move (find-displayer self appendage) '() loc))

(defmethod display-window ((self truck-disp))
  (internal-window self))

(defun truck-display-subwindow (self appendage)
  (display-window (find-displayer self appendage)))

(defmethod dm-locate-obj ((self truck-disp) sym)
  (let ((the-position 
		 (position-if #'(lambda (displayer) 
                                  (dm-locate-obj displayer sym))
                              (displays self))))
	(if (null the-position)
            nil
            (list (nth the-position (display-names self))
                  (dm-locate-obj (nth the-position (displays self)) sym)))))

(defmethod dm-mnes ((self truck-disp) arm-name bay-name)
  (let ((arm-disp (find-displayer self arm-name))
		(space (next-empty-space self bay-name)))       
    (dm-move arm-disp '() (list bay-name space))))
                                           
(defmethod next-empty-space ((self truck-disp) bay-name)
  (next-empty-space (find-displayer self bay-name) '()))

;  This one should selectively refresh the arms, gauges, and cargo bays, 
;  but just for brevity we'll make it refresh everything.  Can change 
;  it back if the screen blinks too much too often when the arms move.

(defmethod refresh-arms ((self truck-disp))
  (dm-refresh self))

(defmethod dm-refresh ((self truck-disp))
  (truck-disp-draw-borders self)
  (mapc #'(lambda (disp-obj) (if disp-obj (dm-refresh disp-obj)))
		(displays self)))

;**********************************************************************

(defun make-truck-disp (arm-names bay-names fuel-capacity window-in)
  (let* ((the-displayer (make-instance 'truck-disp
									:arm-names arm-names
									:bay-names bay-names
									:internal-window window-in)))
	(truck-disp-set-displays the-displayer fuel-capacity)
	(truck-disp-draw-borders the-displayer)
	(mapc #'(lambda (arm-name) 
			  (set-truck (find-displayer the-displayer arm-name)
						 the-displayer))
		  arm-names)
	the-displayer))

(defun truck-disp-set-displays (td fuel-capacity)
  (let ((x (x-coord (internal-window td)))
		(y (y-coord (internal-window td)))
		(bay-names (bay-names td))
		(arm-names (arm-names td)))
	(setf (display-names td)
          (append '(tire-bay 
					weapon-bay 
					fuel-gauge 
					heading-gauge 
					speed-gauge 
					status-gauge)
				  bay-names
				  arm-names
				  (mapcar #'arm-bay-name arm-names)))
	(setf (displays td)
		  (list 
		   (make-single-bay-disp "TIRES"  (make-position (+ 5 x) (+ 40 y)))
           (make-single-bay-disp "WEAPONS" (make-position (+ 65 x) (+ 40 y)))
           (make-fuel-disp fuel-capacity (make-window (+ 125 x) (+ 40 y) 50 70))
           (make-heading-disp (make-window (+ 185 x) (+ 40 y)  50 50))
           (make-speed-disp (make-window (+ 245 x) (+ 40 y) 50 70))
           (make-status-disp (make-window (+ 305 x) (+ 40 y) 50 50))
           (truck-make-bay-disp (car bay-names) (+ x 5) (+ y 120) 360)
           (truck-make-bay-disp (cadr bay-names) (+ x 5) (+ y 200) 360)
           (make-arm-disp (+ x 332) (+ y 320) '(376 113 191 273 351 440) bay-names)
           (make-arm-disp (+ x 332) (+ y 410) '(381 108 196 268 346 435) bay-names)
		   (truck-make-bay-disp (car arm-names) (+ x 5) (+ y 280) 325)
		   (truck-make-bay-disp (cadr arm-names) (+ x 5) (+ y 370) 325)))
	(values)))


(defun truck-make-bay-disp (bay-name x y width)
  (make-bay-disp (copy-seq (symbol-name bay-name))
                 (make-window x y width (+ *small-icon-height* 35))
				 'right))

;************************************************************************

(defun truck-disp-draw-borders (self)
  (let* ((my-window (internal-window self))
		 (title-bar 30)
		 (text-width 100)
		 (x (x-coord my-window))
		 (y (y-coord my-window))
		 (w (width my-window))
		 (solid-width (truncate (/ (- w text-width) 2.0))))
	(disp.draw-rectangle my-window)
	(disp.with-clip-window my-window
	  (disp.fill-rectangle (make-window 0 0 solid-width title-bar))
	  (disp.fill-rectangle (make-window (+ solid-width text-width) 0 
										solid-width title-bar))
	  (disp.fill-rectangle (make-window 0 (- title-bar 2) w 2))
	  (disp.with-font (disp.large-font)
  	    (disp.center-text-in-window 
	      "TRUCK" 
		  (make-window solid-width y text-width (- title-bar 2)))))))


