;;; Note that when the documentation refers to operations,
;;; everything is now implemented in CLOS classes, instances
;;; and generic functions

;;; Everything that was a PACKAGE is now a PARCEL because of 
;;; conflicts in common lisp

;;; check printing etc

; --------------------------------------------------------------------
; ** Simulator support for  CONTAINERS 
; ----------------------------------------------------------
;
; Things defined in this file:
;
; Functions:
;
; Object builders:
;
; This file makes reference to things in OPERATIONS
;                                        LOWLEVEL
;                                        
; --------------------------------------------------------------------
; ** Simulator support for THINGOID CONTAINERS
; --------------------------------------------
; THINGOIDS can be put in CONTAINERS.  A CONTAINER is a thingoid that
;  can hold other thingoids and a generic container is a thingoid 
;  itself.  When a thingoid is inside a container it is invisible to
;  all other objects in the simulation unless it is of a class for
;  which the container is invisible.  Examples of such classes are
;  paper containers and radioactive thingoids.
; CONTAINERS can be placed inside other containers but not VESSELS.
; Both CONTAINERS and VESSELS are parcelS
; --------------------------------------------------------------------
; * Generic CONTAINER constructor
; -------------------------------
; This function creates a single object of type CONTAINER.  The object
;  created is also of type THINGOID, parcel and ENVIRONMENT (see the
;  file "OPERATIONS.T").  The properties of this object for the
;  simulation are given by the arguments.  Note that TRUCK cargo bays,
;  tire bays and weapons bays are generic containers.
;
; (SI:MAKE-CONTAINER-GENERIC id kind capacity bigness tool-needs visibles)
;  -    id: A globally unique symbol to differentiate this object from
;            all others.
;  -  kind: A symbol or list of symbols giving additional classes that
;            this object will belong to.
;  -  capacity: An integer giving the maximum sum of bignesses that this
;            object can hold.  Ie. gives a limit on the number of
;            other objects this object can contain.
;  - bigness: An integer giving the pseudo-size of this object
;            when empty.
;  - tool-needs: A list of object classes that a robot arm must already
;            hold before this object can be picked up.
;  - visibles: A list of object classes that this object is
;            transparent to.

(defclass container
  (parcel environment)
  ()
  )


;;; ** Operations handled by all objects

(defmethod update ((self container) number time)
  (mapc #'(lambda (x)
            (update x number time))
        (holdings self))
  (values))

(defmethod examine ((self container))
  (let ((things-held (holdings self))
        (id (unique-id self)))
    (cond ((null things-held)
           (post-sensor-datum 'container
                              id
                              '()
                              '()
                              '()))
          (t
           (mapcar #'(lambda (x)
                    (post-sensor-datum 'container           
                                       id                   
                                       (unique-id x)
                                       '()
                                       '()))
                things-held)))))


;;; ** Special ENVIRONMENT type operations


(Defmethod visible? ((self container) class) 
  (member class (visibles self) :test #'eq))

;;; ** Special THINGOID type operations ****


;;; ** special PARCEL type operations

(defmethod space-full ((self container)) 
  (- (capacity self) (space-empty self)))

(defmethod empty? ((self container)) 
  (null (holdings self)))

(defmethod gross-bigness ((self container))
  (cond ((holdings self)
         (+ (bigness self)
            (apply #'add
                   (mapcar #'(lambda (x)
                               (gross-bigness x))
                           (holdings self)))))
        (else (bigness self))))

;;; ** Special CONTAINER type operations

(defmethod will-hold? ((self container) thing)
  (<= (bigness thing) (space-empty self)))

(defmethod put-in ((self container) thing)
  (cond ((<= (bigness thing) (space-empty self))
         (setf (holdings self) (obj-list-put (holdings self) thing))
         (setf (space-empty self) (- (space-empty self) (bigness thing)))
         (setf (environment thing) self)  
         (cond ((displayer self)           
                (dm-add (displayer self) (id self) thing)))
         t)
        (t nil)))

(defmethod take-out ((self container) thing)
  (cond ((eq (id self)
              (unique-id (environment thing)))
         (setf (holdings self)
              (obj-list-del-id! (holdings self)
                                (unique-id thing)))
         (setf (space-empty self)
              (+ (space-empty self) (bigness thing)))
         (setf (environment thing) *undefined-object*)  
         (cond ((displayer self) (dm-del (displayer self) (id self) thing)))
         t)
        (t nil)))

(Defmethod get-if-holds-id ((self container) thing-id)
  (obj-list-get-id (holdings self) thing-id))



;;; display things

(defmethod show ((self container))
  (format t
          "Id:      ~a~&Bigness: ~a - Tools: ~a~&"
          (id self)
          (bigness self)
          (tool-needs self))
  (format t
          "Space:   ~a - Holding: ~a~&Let-See: ~a~&Held-By: ~a~&"
          (space-empty self)
          (holdings self)
          (visibles self)
          (environment self))
  t)

(defmethod show-contents ((self container)) 
  (format t "~a" (holdings self)))


(defmethod (setf displayer) :after (new-displayer (self container))
  (cond ((and (displayer self)
              (not (class? self 'arm))
              (not (class? self 'map-node)))
         (dm-empty (displayer self) (id self))
         (mapc #'(lambda (x)
                   (dm-add (displayer self) (id self) x))
               (holdings self)))))





