(comment (herald lowlevel))

; --------------------------------------------------------------------
; ** Basic things needed by other simulator files **
; --------------------------------------------------
; This file contains definitions for the following things:
;
; Functions:
;  (GET-IF-HOLDS-CLASS self class)    => objects or ()
;  (GET-IF-VISIBLE-CLASS self class)  => objects or ()
;  (HELD-BY-ENVIRONMENT? self env-id) => T or nil
;
;  (OBJ-LIST-PUT object list)             => list + object
;  (OBJ-LIST-GET-ID name list)            => object or ()
;  (OBJ-LIST-GET-CLASS class list)        => objects or ()
;  (OBJ-LIST-HOLDS-ID? name list)         => boolean
;  (OBJ-LIST-HOLDS-CLASS? class list)     => boolean
;  (OBJ-LIST-HOLDS-CLASSES? classes list) => boolean
;  (OBJ-LIST-DEL-ID! name list)           => list - object
;
;  (SI:MAP-DIRECTION-INDEX direction)    => unique integer
;  (SI:MAP-DIRECTION? direction)         => boolean
;  (SI:MAP-DIRECTION-OPPOSITE direction) => direction
;  (SI:MAP-DIRECTION-RIGHT direction)    => direction
;  (SI:MAP-DIRECTION-LEFT direction)     => direction
;
;  (CHECK-CLASS object class)  => error and returns valid object
;  (SI:BUMPINESS-VALUE? value) => boolean
;  (SI:SPEED-VALUE? value)     => boolean
;  (SI:STATE-VALUE? value)     => boolean
;
;  ??? More ???
; This file contains calls to NO external things.
; --------------------------------------------------------------------
; * Special ENVIRONMENT type operations
; -------------------------------------
(defun get-if-holds-class (self class)
  (obj-list-get-class (holdings self) class))

(defun get-if-visible-class (self class)
  (cond ((and (visible? self class)
              (class? (environment self) 'environment))
         (get-if-visible-class (environment self) class))
        (t (apply #'nconc
                  (obj-list-get-class (holdings self) class)
                  (mapcar #'(lambda (x) (si.gather-downward-visibles x class))
                          (obj-list-get-class (holdings self)
                                              'environment))))))

(defun si.gather-downward-visibles (self class)
  (cond ((visible? self class)
         (apply #'nconc
                (obj-list-get-class (holdings self) class)
                (mapcar #'(lambda (x) (si.gather-downward-visibles x class))
                        (obj-list-get-class (holdings self) 'environment))))
        (t nil)))

(defun held-by-environment? (self env-id)
  (cond ((eq (unique-id self) env-id) t)
        ((class? (environment self) 'environment)
         (held-by-environment? (environment self) env-id))
        (t nil)))

; --------------------------------------------------------------------
; * Generic object list accessing macros
; --------------------------------------
;
; These special forms are defined to do things to a list of objects.
;
; (OBJ-LIST-PUT list object)             => list + object
; (OBJ-LIST-GET-ID list id)              => object or ()
; (OBJ-LIST-GET-CLASS list class)        => object or ()
; (OBJ-LIST-HOLDS-ID? list id)           => T or ()
; (OBJ-LIST-HOLDS-CLASS? list class)     => T or ()
; (OBJ-LIST-HOLDS-CLASSES? list classes) => T or ()
; (OBJ-LIST-DEL-ID! list id)             => list - object
; WARNING: Alter these at risk!!  Destructive code relies on them.
(defun obj-list-put (object-list obj)
  (cons obj object-list))

(defun obj-list-get-id (object-list object-id)
  (cond ((null object-list) nil)
        ((eq object-id (unique-id (car object-list))) (car object-list))
        (t (obj-list-get-id (cdr object-list) object-id))))

(defun obj-list-get-class (object-list object-class)
  (cond ((null object-list) nil)
        ((class? (car object-list) object-class)
         (cons (car object-list)
               (obj-list-get-class (cdr object-list) object-class)))
        (t (obj-list-get-class (cdr object-list) object-class))))

(defun obj-list-holds-id? (object-list object-id)
  (not (null (obj-list-get-id object-list object-id))))

(defun obj-list-holds-class? (object-list object-class)
  (not (null (obj-list-get-class object-list object-class))))

(defun obj-list-holds-classes? (object-list class-list)
  (cond ((null class-list) t)
        ((obj-list-holds-class? object-list (car class-list))
         (obj-list-holds-classes? object-list (cdr class-list)))
        (t nil)))

(defun obj-list-del-id! (object-list object-id)
  (delete object-id
          object-list
          :test
          #'(lambda (x y) (eq x (unique-id y)))))

; --------------------------------------------------------------------
; * Things needed to deal with directions *
; -----------------------------------------
(defun map-direction? (direction)
  (member direction '(n s w e ne nw se sw) :test #'eq))

(defun map-direction-opposite (direction)
  (cond ((eq direction 'e) 'w)
        ((eq direction 'ne) 'sw)
        ((eq direction 'n) 's)
        ((eq direction 'nw) 'se)
        ((eq direction 'w) 'e)
        ((eq direction 'sw) 'ne)
        ((eq direction 's) 'n)
        ((eq direction 'se) 'nw)
        (t nil)))

(defun map-direction-left (direction)
  (cond ((eq direction 'e) 'ne)
        ((eq direction 'ne) 'n)
        ((eq direction 'n) 'nw)
        ((eq direction 'nw) 'w)
        ((eq direction 'w) 'sw)
        ((eq direction 'sw) 's)
        ((eq direction 's) 'se)
        ((eq direction 'se) 'e)
        (t nil)))

(defun map-direction-right (direction)
  (cond ((eq direction 'e) 'se)
        ((eq direction 'ne) 'e)
        ((eq direction 'n) 'ne)
        ((eq direction 'nw) 'n)
        ((eq direction 'w) 'nw)
        ((eq direction 'sw) 'w)
        ((eq direction 's) 'sw)
        ((eq direction 'se) 's)
        (t nil)))

(defun map-direction-index (direction)
  (cond ((eq direction 'e) 0)
        ((eq direction 'ne) 1)
        ((eq direction 'n) 2)
        ((eq direction 'nw) 3)
        ((eq direction 'w) 4)
        ((eq direction 'sw) 5)
        ((eq direction 's) 6)
        ((eq direction 'se) 7)
        (t nil)))

; --------------------------------------------------------------------
; * Error Reporting *
; -------------------

(defun check-class (obj class-name)
  (cond 
   ((or (class? obj class-name) 
		(eq obj *undefined-object*))
	 T)
   (t (cerror "To continue" "Object ~a not of class ~a as expected."
			  obj class-name))))

(defun speed? (value)
  (member value '(stop slow medium fast) :test #'eq))

(defun bumpiness? (value)
  (member value '(low high) :test #'eq))

(defun status? (value)
  (member value
          '(happy stuck broken rolled captured dead)
          :test
          #'eq))

(defun road-type? (value)
  (member value
          '(standard-road express-road
                          soft-road
                          winding-road
                          muddy-road
                          bumpy-road
                          bridge)
          :test
          #'eq))

(defun road-attribute? (value)
  (member value '(bumpy windy muddy) :test #'eq))

(defun error-message? (value)
  (and? (symbolp value)
        (member value
                '(arm-not-at arm-not-holding
                             arm-too-full
                             arm-interference
                             arm-cant-grip
                             arm-cant-move
                             arm-cant-find
                             arm-cant-toggle
                             container-full)
                :test
                #'eq)))

; --------------------------------------------------------------------
; * Random numbers, percentages and chances *
; -------------------------------------------

;;(let ((seed 143761))

;;;  (defvar get-random-number
;;          (comment t2cl-unhandled
;;;                   "Dunno how to convert this"
;;;                   (make-random seed)))
  
;;  (defun set-random-seed (x)
;;    (setf seed x))
;;  )

(defun random-number (n m)
  (let ((range (+ (- m n) 1)))
    (+ n (random range))))

(defun percent-adjust (percent value)
  (/ (* value percent) 100))

(defun random-chance? (percent)
  (>= percent (random-number 1 100)))

(defun random-chance-per? (chance delta interval)
  (cond ((null chance) nil)         ; ????
        ((<= chance 0) nil)
        ((>= chance 100) t)
        (t 
         (let ((percent (- 100
                           (truncate (* 100.0
                                        (exp (* (/ (coerce delta 'float)
                                                   (coerce interval 'float))
                                                (log (/ (coerce (- 100 chance) 'float)
                                                        100.0)))))))))
           (random-chance? percent)))))

(defun random-adjust (percent value)
  (/ (* value (random-number (- 100 percent) (+ 100 percent))) 100))

(defun random-distance (chance-per-hundred)
  (cond ((>= chance-per-hundred 100) 0)
        ((<= chance-per-hundred 0) most-positive-fixnum)
        (t (truncate 
            (* 100.0 (/ (log (/ (coerce (random-number 1 100) 'float)
                                100.0))
                      (log (- 1.0
                                (/ (coerce chance-per-hundred 'float)
                                     100.0)))))))))

; --------------------------------------------------------------------
; * Functions for building unique identifiers *
; ---------------------------------------------

(defvar *id-counter* 0)
  
(defun make-unique-id (kind)
  (incf *id-counter* 1)
  (intern (concatenate 'string (string kind) "-" (princ-to-string *id-counter*))))

(let ((name-table (make-hash-table))
      (spare-names (mapcar #'make-unique-id
                           '(obj obj obj obj obj obj obj obj obj)))
      (recycled-names nil))
  
  (defun fetch-id (thing)
    (let ((id (gethash thing name-table)))
      (cond ((or (null id) (eq id 'none))
             (cond ((null spare-names)
                    (setf (gethash thing name-table)
                          (make-unique-id 'obj)))
                   (t (let ((entry (random-number 0 (- (length spare-names) 1))))
                        (let ((entry (car (nthcdr entry spare-names))))
                          (setf (gethash thing name-table) entry)
                          (setf spare-names (delete entry spare-names))))))
             (gethash thing name-table))
            (t id))))

  (defun scramble-ids ()
    (maphash #'(lambda (a b)
                 (setf (gethash a name-table) 'none)
                 (if (not (eq b 'none))
                     (setf spare-names (cons b spare-names))))
             name-table)
    (if (null spare-names)
        (setf spare-names recycled-names)
        (setf (cdr (last spare-names)) recycled-names))
    (setf recycled-names nil))
  (defun recycle-id (thing)
    (let ((id (gethash thing name-table)))
      (cond ((or (null id) (eq id 'none)) (values))
            (t (setf (gethash thing name-table) nil)
               (setf recycled-names (cons id recycled-names))
               (values))))))

; --------------------------------------------------------------------
; * Functions for dealing with times *
; ------------------------------------
; A time can be either a number, or a list (days hours minutes)
(defun time->number (time)
  (+ (* (car time) 1440) (+ (* (cadr time) 60) (caddr time))))

(defun number->time (amount)
  (let* ((days (/ amount 1440))
         (hours (/ (- amount (* days 1440)) 60))
         (minutes (- amount (+ (* days 1440) (* hours 60)))))
    (list days hours minutes)))

