;**************************************************************************
;  MAP-DEFS.LISP

;  To the world, the map looks like a structure with its nodes 
;  and roads, and maybe some display info.

(defstruct map node-list road-list display-info)

;**************************************************************************

(defclass special-handler
  (thingoid)
  ())

;**************************************************************************

(defclass map-node  (container)
  ((handlers :accessor handlers :initform '())
   (my-map-sector :accessor my-map-sector :initarg :my-map-sector)
   (display-info :accessor display-info :initarg :display-info :initform nil)
   (map-roads :accessor map-roads :initform (fill (make-array 8) '()))))

;**************************************************************************

(defclass map-link (environment)
  ((environment :type environment :initform *undefined-object*)
   (road-length :accessor road-length :initarg :road-length)
   (road-direction :accessor road-direction :initarg :road-direction)
   (fuel-drag :accessor fuel-drag :initarg :fuel-drag)
   (speed-drag :accessor speed-drag :initarg :speed-drag)
   (my-bumpiness :accessor my-bumpiness :initarg :bumpiness)
   (fragility :accessor fragility  :initarg :fragility)
   (my-map-sector :accessor my-map-sector :initarg :my-map-sector)
   (start-node :accessor start-node :initform '())
   (start-node-id :accessor start-node-id :initform '())
   (start-node-direction :accessor start-node-direction :initform '())
   (finish-node :accessor finish-node :initform '())
   (finish-node-id :accessor finish-node-id :initform '())
   (finish-node-direction :accessor finish-node-direction :initform '())
   (sub-nodes :accessor sub-nodes)
   (road-backward :accessor road-backward)
   (display-info :accessor display-info :initarg :display-info :initform nil)))


;******************************************************************************
;  Methods

(defmethod update ((self map-node) number time)
  (mapc #'(lambda (x) (update x number time))
        (handlers self))
  (mapc #'(lambda (x) (update x number time))
        (holdings self))
  t)
                                         
(defmethod will-hold? ((self map-node) thing)
  (declare (ignore thing))
  t)

(defmethod put-in ((self map-node) (thing special-handler))
  (setf (handlers self) 
		(obj-list-put (handlers self) thing)))

(defmethod put-in ((self map-node) thing)
  (setf (holdings self) (obj-list-put (holdings self) thing))
  (if (displayer self) 
	  (dm-add (displayer self) 'current-location thing))
  (setf (environment thing) self)
  thing)
                                         
(defmethod take-out ((self map-node) thing)
  (cond 
	((eq (id self) (unique-id (environment thing)))
	  (setf (holdings self)
			(obj-list-del-id! (holdings self) (unique-id thing)))
	  (setf (environment thing) *undefined-object*)
	  (if (displayer self)
		  (dm-del (displayer self) 'current-location thing))
	  t)
	(t '())))

  
;*******************************************************************

(defmethod road-traverse ((self map-node) (truck truck) direction thingoid-classes)
  (let ((chosen-link (svref (map-roads self)
			    (map-direction-index direction))))
    (cond 
      (chosen-link
       (road-traverse (car chosen-link) truck
		      (cdr chosen-link) thingoid-classes))
      (t (values 0 0)))))

;*******************************************************************

(defun map-table-connection (the-node direction new-link rel-link-dir)  
  (let ((index (map-direction-index direction)))
    (when index
	  (vset (map-roads the-node)  index (cons new-link rel-link-dir)))))

(defmethod map-sector ((self map-node))
  (my-map-sector self))

;;; ** Operations needed for neatness

(defmethod (setf displayer) :after (new-displayer (self map-node)) 
  (declare (ignore new-displayer))
  (cond 
	((displayer self)
	 (dm-empty (displayer self) 'current-location)
	 (mapc #'(lambda (x) (dm-add (displayer self) 'current-location x))
		   (holdings self)))))
                                         
(defmethod show ((self map-node))
  (format t "Id:      ~a - Sector: ~a~&Classes: ~a~&"
          (id self) (my-map-sector self) (my-classes self))
  (format t "Links:   ")
  (map-link-list-printer (map-roads self))
  (format t "~&Holding: ~a~&Held-By: ~a~&"
		  (holdings self) (environment self))
  (format t "~&Handlers: ~a~&" (handlers self))
  t)

(defmethod show-contents ((self map-node))
  (format t "~a" (holdings self)))

    
; --------------------------------------------------------------------
; * Generic MAP-LINK-SUBNODE constructor *
; ----------------------------------------
;
; Map link subnodes are things that act like map nodes but represent
;  places along a link (ie. places of interest along a road).  A new
;  subnode is created when a truck tries to traverse a link, doesn't
;  make it and ends up where no subnode already exists.  When the last
;  truck in a subnode moves out and no things are left behind, then
;  the subnode is recycled.  If a subnode is left with things inside
;  it, then the subnode stays around for trucks to find later.
;
; Map link subnodes exist in a pool as they will often be created and
;  then tossed aside.

(defclass map-link-subnode (map-node)
  ())

(defmethod (setf environment) :after ((new-holder map-link) (self map-link-subnode))
   (setf (map-roads self) (fill (map-roads self) nil))
   (setf (svref (map-roads self) 
		(map-direction-index (road-direction new-holder)))
	 (cons new-holder (road-direction new-holder)))
   (setf (svref (map-roads self) 
		(map-direction-index 
		 (map-direction-opposite  (road-direction new-holder))))
	 (cons new-holder (map-direction-opposite (road-direction new-holder))))
   (setf (kind self) (cons (kind-id new-holder) (cdr (kind self)))))


(defmethod (setf environment) :before (new-holder (self map-link-subnode))
  (check-class new-holder 'map-link))

;;; ** Special MAP-LINK-SUBNODE type operations

(defmethod road-traverse ((self map-link-subnode) truck direction thingoid-classes)
  (road-traverse (environment self)
                 truck
                 direction
                 thingoid-classes))

(defmethod recycle ((self map-link-subnode))
  (setf (environment self) *undefined-object*)
  (setf (holdings self) '())
  (setf (kind self) (cdr (kind self)))  ;***
  (return-to-pool *si.map-link-subnode-pool* self))

(defmethod map-sector ((self map-link-subnode))
  (map-sector (environment self)))


(defmethod show ((self map-link-subnode))
  (format t
          "Id:    ~a~&Classes: ~a~&"
          (id self)
          (my-classes self))
  (format t
          "Holding: ~a~&Held-By: ~a~&"
          (holdings self)
          (environment self))
  (format t "~&Handlers: ~a~&" (handlers self))
  t)
 

(defmethod bumpiness ((self map-link))
  (my-bumpiness self))

(defmethod road-traverse ((self map-link) (truck truck) direction pattern)
  (multiple-value-bind (distance travel-time)
                       (cond ((or (eq direction (road-direction self) )
                                  (eq direction (road-backward self)))
                              (move-truck self truck direction pattern))
                             (t (values 0 0)))
    (when (> distance 0)
	 (cond 
       ((eq (unique-id (environment truck))
                       (start-node-id self))
         (setf (truck-heading truck)
               (map-direction-opposite (start-node-direction self))))
       ((eq (unique-id (environment truck))
            (finish-node-id self))
         (setf (truck-heading truck)
               (map-direction-opposite (finish-node-direction self))))
       (t (setf (truck-heading truck)
                direction))))
    (values distance travel-time)))

;****************************

(defun map-connect-start (the-link the-node connection-dir)
  (when (map-direction? connection-dir)
	(setf (start-node the-link) the-node)
	(setf (start-node-id the-link)  (unique-id the-node))
	(setf (start-node-direction the-link) connection-dir)
	(map-table-connection the-node 
						  connection-dir 
						  the-link 
						  (road-direction the-link))))

(defun map-connect-finish (the-link the-node connection-dir)
  (when (map-direction? connection-dir)
	(setf (finish-node the-link) the-node)
	(setf (finish-node-id the-link) (unique-id the-node))
	(setf (finish-node-direction the-link) connection-dir)
	(map-table-connection the-node
						  connection-dir
						  the-link
						  (road-backward the-link))))

;***********************************************************************

(defmethod map-sector ((self map-link))
  (my-map-sector self))
                    
(defmethod show ((self map-link))
  (format t
          "Id:      ~a  Sector: ~a~&Classes: ~a~&"
          (id self)
          (my-map-sector self)
          (my-classes self)
          (format t
                  "Start:   ~a  Finish: ~a  Direction: ~a  Length: ~a~&"
                  (start-node self)
                  (finish-node self)
                  (road-direction self) 
                  (road-length self))
          (format t "Held-By: ~a~&" (holder self))
          t))


