;;; Note that when the documentation refers to operations,
;;; everything is now implemented in CLOS classes, instances
;;; and generic functions

;;; Everything that was a PACKAGE is now a PARCEL because of 
;;; conflicts in common lisp

;;; check printing etc

; --------------------------------------------------------------------
; ** Simulator support for THINGOIDS, CONTAINERS and VESSELS
; ----------------------------------------------------------
;
; Things defined in this file:
;
; Functions:
;  (SI:DECLARE-LIQUID kind classes) => void
;
; Object builders:
;  (SI:MAKE-THING-GENERIC id kind bigness tool-needs)
;  (SI:MAKE-CONTAINER-GENERIC id kind capacity bigness tool-needs visibles)
;  (SI:MAKE-VESSEL-GENERIC id kind capacity bigness tool-needs visibles)
;
; This file makes reference to things in OPERATIONS LISP
;                                        LOWLEVEL.T
; --------------------------------------------------------------------
; ** Simulator support for THINGOID CONTAINERS
; --------------------------------------------
; THINGOIDS can be put in CONTAINERS.  A CONTAINER is a thingoid that
;  can hold other thingoids and a generic container is a thingoid 
;  itself.  When a thingoid is inside a container it is invisible to
;  all other objects in the simulation unless it is of a class for
;  which the container is invisible.  Examples of such classes are
;  paper containers and radioactive thingoids.
; CONTAINERS can be placed inside other containers but not VESSELS.
; Both CONTAINERS and VESSELS are PARCELS

; --------------------------------------------------------------------
; * Generic VESSEL constructor
; ----------------------------
; This function creates a single object of type VESSEL.  The object
;  created is also of type THINGOID and parcel (see the file
;  "OPERATIONS.T").  The properties of this object for the simulation
;  are given by the arguments.  Note that fuel drums and TRUCK fuel
;  bays are generic vessels.
;
; (SI:MAKE-VESSEL-GENERIC id kind capacity bigness tool-needs visibles)
;  -    id: A globally unique symbol to differentiate this object from
;            all others.
;  -  kind: A symbol or list of symbols giving additional classes that
;            this object will belong to.
;  -  capacity: An integer giving the maximum amount of liquid that this
;            object can hold.
;  - bigness: An integer giving the pseudo-size of this object
;            when empty.
;  - tool-needs: A list of object classes that a robot arm must already
;            hold before this object can be picked up.
;  - visibles: A list of liquid classes that this object is
;            transparent to.

(defclass vessel
  (parcel environment)      
  ((amount-held :accessor amount-held
                :initform 0)
   (liquid-held :accessor liquid-held
                :initform 'undefined-liquid)
   (extra-classes :accessor extra-classes
                  :initform '())))
                   
(defmethod examine ((self vessel))
  (post-sensor-datum 'vessel
                     (unique-id self)
                     (liquid-held self)
                     (space-full self)
                     '()))


;;; ** Special parcel type operations


(Defmethod space-full ((self vessel))
  (amount-held self))

(defmethod empty? ((self vessel)) 
  (<= (amount-held self) 0))

(defmethod gross-bigness ((self vessel))
  (+ (amount-held self) (bigness self)))


;;; ** Special VESSEL type operations
                    
(defmethod will-hold? ((self vessel) amount) 
  (<= amount (space-empty self)))

(defmethod pour-in ((self vessel) amount)
  (cond ((<= amount (space-empty self))
         (setf (amount-held self) (+ (amount-held self) amount))
         (setf (space-empty self) (- (capacity self) (amount-held self)))
         (cond ((displayer self)
                (dm-set (displayer self)
                        (id self)
                        (amount-held self))))
         t)
        (else '())))

(defmethod pour-out ((self vessel) amount)
  (cond ((< amount (amount-held self))
         (setf (amount-held self) (- (amount-held self) amount))
         (setf (space-empty self) (- (capacity self) (amount-held self))))
        (else (setf (amount-held self) 0)
              (setf (space-empty self) (capacity self))
              (setf (liquid-held self)
                   'undefined-liquid)))
  (cond ((displayer self)        
         (dm-set (displayer self) (id self) (amount-held self))))
  t)


;; display stuff

(defmethod show ((self vessel))
  (format t "Name:    ~a~&Classes: ~a~&" (id self) (extra-classes self))
  (format t "Bigness: ~a - Tools: ~a~&" (bigness self) (tool-needs self))
  (format t
          "Space:   ~a - Holding: ~a - Amount: ~a~&"
          (capacity self)
          (liquid-held self)
          (amount-held self))
  (format t "Let-See: ~a~&Held-By: ~a~&" (visibles self) (environment self))
  t)
 
(defmethod show-contents ((self vessel)) 
  (format t "~a" (holdings self)))
                    
(defmethod (setf liquid-held) :around (new-liquid-held (self vessel)) 
  (cond ((and (not (eq new-liquid-held (liquid-held self)))
              (<= (amount-held self) 0)
              (declared-liquid? new-liquid-held))
         (call-next-method)
         (setf (extra-classes self) '())
         (mapc #'(lambda (x)
                   (if (member x (visibles self) :test #'eq)
                     (push (extra-classes self) x)))
               (get-liquid-classes (liquid-held self))))
        ((not (declared-liquid? new-liquid-held))
         (cerror "To continue" "~a is not a liquid" new-liquid-held))
        (t '())
        ))
 
(defmethod (setf displayer) :after (new-displayer (self vessel))
  (cond ((displayer self)
         (dm-set (displayer self) (id self) (amount-held self)))))
 
