/* PD624 Code File
Mixed frame and rule-based medical diagnosis system from
Section 5.2 of PD624 'Knowledge Engineering'
*/
/* patients: similar to section 5.1, i.e. code file ch5-1.pl */

patient subclass_of person with
  symptoms: [].   /* deliberately empty as 'reminder' about this slot */

amy instance_of patient with
  symptoms: [cough, hoarse_voice].  /* amy has laryngitis */

bill instance_of patient with
  symptoms: [ runny_nose, red_itchy_watery_eyes]. /* bill has hay fever */

constance instance_of patient with
  symptoms: [ systemic_upset, purulent_sputum].
           /* constance has bronchiectasis */

deirdre instance_of patient with
  symptoms: [systemic_upset, purulent_sputum]. /* deirdre has lung abscess */
/* disease classes and instances */

respiratory_tract_disease subclass_of disease with  /* need ANY, not all!! */
  indicators: [cough, purulent_sputum, hoarse_voice, systemic_upset,
               red_itchy_watery_eyes, runny_nose],
  discriminators: [].

upper_rtd subclass_of respiratory_tract_disease with
  indicators: [cough, runny_nose, hoarse_voice, red_itchy_watery_eyes],
  discriminators: [].

lower_rtd subclass_of respiratory_tract_disease with
  indicators: [purulent_sputum, systemic_upset],
  discriminators: [].

pneumonia subclass_of lower_rtd with
  indicators: [ systemic_upset, purulent_sputum, mucoid_sputum ].

bacterial_pneumonia subclass_of pneumonia with
  indicators: [purulent_sputum].

non_bacterial_pneumonia subclass_of pneumonia with
  indicators: [mucoid_sputum].

fungal_pneumonia subclass_of pneumonia with
  indicators: [acquired_immune_deficiency_syndrome ].

pneumococcal_pneumonia instance_of bacterial_pneumonia with
  indicators: [blood_containing_sputum_in_productive_cough,
         sudden_fever, shivering, rigor,
             quick_short_respiration ],
  discriminators: ['paired diplococci seen in culture stain'].

staphylococcal_pneumonia instance_of bacterial_pneumonia with
  indicators: [fever, purulent_sputum, malaise ],
 discriminators: ['gram positive cocci seen in culture stain'].

bronchiectasis instance_of lower_rtd with
  indicators: [systemic_upset, purulent_sputum],
 discriminators: ['tubular shadows visible in x-ray'].

lung_abscess instance_of lower_rtd with
  indicators: [systemic_upset, purulent_sputum],
 discriminators: ['pulmonary cavitation with fluid levels visible in x-ray'].

hay_fever instance_of upper_rtd with
  indicators: [red_itchy_watery_eyes, runny_nose],
  discriminators: ['positive reaction to allergens'].

laryngitis instance_of upper_rtd with
  indicators: [cough, hoarse_voice],
  discriminators: [inflamed_larynx].

common_cold instance_of upper_rtd with
  indicators: [runny_nose, sneezing, fever],
  discriminators: [headache].

influenza instance_of upper_rtd with
  indicators: [fever, dry_cough, malaise],
  discriminators: [sore_throat_and_dry_persistent_cough].


/* control rules */
rule init forward
  if
    start
  then
    remove start &
    query the name of patient receives_answer X &
                            /* automatically notes the name! */
    add goal(hypothesize).  /* see course text for explanation of 'goals' */

rule switch_strategies_1 forward
  if
    goal(hypothesize)
  then
    remove goal(hypothesize) &
    add goal(refine).

rule switch_strategies_2 forward  /* this is the answer to exercise 5.5 */
  if
    goal(refine)
  then
    remove goal(refine) &
    add goal(discriminate).

rule switch_strategies_3 forward
  if
    goal(discriminate)
  then
    remove goal(discriminate) &
    add goal(choose_winner).

rule switch_strategies_4 forward
  if
    goal(choose_winner)
  then
    remove goal(choose_winner) &
    halt.                 /* This is where the whole thing ends  */

/* diagnosis rules */
rule ordinary_diagnosis forward
  if
    goal(hypothesize) &  /* in 'hypothesis' mode? */
    the name of patient is N &  /* retrieve name */
    the symptoms of N is Symp &  /* now find ANY symptom Symp */
    D subclass_of disease &  /* and any category of disease...  */
    -- already_did(D) &  /* that's not been suggested already.*/
    the indicators of D is Symp   /* which might be indicated by Symp */
  then
    announce ['now hypothesizing ',D] &
    add possible(D) &  /* place it in working memory */
    add already_did(D).

rule refinement_to_subclass forward
  if
    goal(refine) &
    possible(DiseaseClass) &  /* given this candidate */
    Subclass subclass_of DiseaseClass &   /* find a subclass of it...  */
    -- already_did(Subclass) &  /* which we haven't dealt with yet */
    deduce allowable(Subclass)  /* see if it passes further tests */
  then
    announce ['refining down to possible subclass ', Subclass] &
    add possible(Subclass) &  /* if so, add to set of 'possibles' */
    add already_did(Subclass).

rule refinement_to_instance forward  /* as above, but only for instances */
  if
    goal(refine) &
    possible(DiseaseClass) &  /* given this candidate */
    Disease instance_of DiseaseClass &   /* find an instance of it...  */
    -- already_did(Disease) &  /* which we haven't dealt with yet */
    deduce allowable(Disease)  /* see if it passes further tests */
  then
    announce ['refining down to disease instance ',Disease] &
    add possible_instance(Disease).  /* new addition to working memory */

rule eliminate forward  /* similar to the one in section 5.1 */
  if
    goal(discriminate) &
    possible_instance(Disease) &
    deduce passes_discriminating_test(Disease)
  then
    add likely(Disease).

rule simple_selection forward
  if
    goal(choose_winner) &
    likely(X)
  then
    announce ['A likely diagnosis is that the patient has ', X].


/* backward chaining 'filters' */
rule see_if_allowable backward
  if
    the indicators of Disease is Inds &  /* find any old indicator */
    the name of patient is N &
    the symptoms of N is Inds   /* which the patient also has */
  then
    allowable(Disease).

rule discriminatory_diagnosis backward
  if
      the discriminators of X is D &
      query ['Is there solid evidence of ', D, '?']
           receives_answer yes
  then
      passes_discriminating_test(X).

